///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/plugins/Plugin.h>
#include <core/plugins/PluginManager.h>

namespace Core {

/******************************************************************************
* Constructor for the Plugin class.
******************************************************************************/
Plugin::Plugin(const QString& manifestFile) :
	_manifestFilename(manifestFile), isManifestParsed(false),
	_isLoaded(false)
{
	// Load plugin manifest file into DOM.
	QFile file(manifestFile);
	if(!file.open(QIODevice::ReadOnly))
		throw Exception(tr("Failed to open plugin manifest file %1").arg(manifestFile));
	QString errorMsg;
	int errorLine, errorColumn;
	if(!_manifest.setContent(&file, true, &errorMsg, &errorLine, &errorColumn))
		throw Exception(tr("Failed to load plugin manifest file.\nXML File: %1\nError Message: %2\nLine %3, Column %4").arg(manifestFile, errorMsg).arg(errorLine).arg(errorColumn));

	// Extract the plugin identifier from the manifest.
	_pluginId = _manifest.documentElement().attribute("Plugin-Id");
	_pluginVendor = _manifest.documentElement().attribute("Plugin-Vendor");
	_pluginVersion = _manifest.documentElement().attribute("Plugin-Version");
}

/******************************************************************************
* Loads the plugin into memory.
******************************************************************************/
void Plugin::loadPlugin()
{
    if(isLoaded()) return;	// Plugin is already loaded.

	// Load other plugins this plugin depends on explicitly.
	for(QVector<QString>::const_iterator depName = _dependencies.begin(); depName != _dependencies.end(); ++depName) {
		Plugin* depPlugin = PLUGIN_MANAGER.plugin(*depName);
		if(depPlugin == NULL)
			throw Exception(QString("Cannot load plugin %1 because it depends on the plugin %2 which is not installed.").arg(pluginId(), *depName));
		_isLoaded = true;
		try {
			depPlugin->loadPlugin();
			_isLoaded = false;
		}
		catch(...) { _isLoaded = false; throw; }
	}

	// Load other plugins this plugin depends on implicitly.
	for(QSet<Plugin*>::const_iterator dep = _implicitDependencies.begin(); dep != _implicitDependencies.end(); ++dep) {
		_isLoaded = true;
		try {
			(*dep)->loadPlugin();
			_isLoaded = false;
		}
		catch(Exception& ex) {
			_isLoaded = false;
			throw ex.prependGeneralMessage(tr("Failed to load plugin %1 required by plugin %2.").arg((*dep)->pluginId(), pluginId()));
		}
		catch(...) { _isLoaded = false; throw; }
	}

	VerboseLogger() << logdate << "Loading plugin" << pluginId() << endl;

	// Do the plugin type specific work.
	loadPluginImpl();

	// Loading was successful.
	this->_isLoaded = true;
}

/******************************************************************************
* Parses the plugin's XML manifest.
******************************************************************************/
void Plugin::parseManifest()
{
	OVITO_ASSERT(!_manifest.isNull());
	OVITO_ASSERT(pluginId().isEmpty() == false);

	if(isManifestParsed) return;	// Is already parsed.
	isManifestParsed = true;		// Prevent re-entrance.

	for(QDomElement rootLevelNode = _manifest.documentElement().firstChildElement(); !rootLevelNode.isNull(); rootLevelNode = rootLevelNode.nextSiblingElement()) {
		if(rootLevelNode.localName() == "Classes") {
			parseClassDefinitions(rootLevelNode);
		}
		else if(rootLevelNode.localName() == "Plugin-Dependencies") {
			parsePluginDependencies(rootLevelNode);
		}
		else if(rootLevelNode.localName() == "Resource-File") {
			parseResourceFileReference(rootLevelNode);
		}
		else parseToplevelManifestElement(rootLevelNode);
	}
}

void Plugin::parsePluginDependencies(const QDomElement& parentNode)
{
	for(QDomElement depNode = parentNode.firstChildElement(); !depNode.isNull(); depNode = depNode.nextSiblingElement()) {
		if(depNode.localName() == "Plugin-Dependency") {
			// Parse plugin name.
			QString depPluginName = depNode.attribute("Plugin-Id");
			if(depPluginName.isEmpty())
				throw Exception(tr("Invalid plugin dependency attribute in manifest."));

			// Skip disabled elements.
			if(depNode.attribute("Enabled").compare("false", Qt::CaseInsensitive) == 0 ||
				depNode.attribute("Enabled").compare("off", Qt::CaseInsensitive) == 0 ||
				depNode.attribute("Enabled").compare("no", Qt::CaseInsensitive) == 0)
				continue;

			_dependencies.push_back(depPluginName);
		}
		else throw Exception(QString("Unknown element tag in XML file: <%1>").arg(depNode.localName()));
	}
}

/******************************************************************************
* Parses a resource file reference in the manifest file.
******************************************************************************/
void Plugin::parseResourceFileReference(const QDomElement& element)
{
	QString path = element.attribute("Path");
	if(path.isEmpty())
		throw Exception(QString("Element <Resource-File> has no Path attribute in manifest file %1.").arg(manifestFile()));

	// Resolve path.
	QDir baseDir = QFileInfo(manifestFile()).dir();
	QString fullPath = baseDir.absoluteFilePath(path);

	// Load resource file into memory.
	if(!QResource::registerResource(fullPath))
		throw Exception(QString("Could not load plugin resource file %1").arg(fullPath));
}


void Plugin::parseClassDefinitions(const QDomElement& parentNode)
{
    // Register the root class if this is the core plugin.
	if(isCore()) {
		OVITO_ASSERT(_classes.empty());
		// Register "PluginClass" class.
		PluginClassDescriptor::rootClass._plugin = this;
		_classes.push_back(&PluginClassDescriptor::rootClass);
	}

	for(QDomElement classNode = parentNode.firstChildElement(); !classNode.isNull(); classNode = classNode.nextSiblingElement()) {
		if(classNode.localName() == "Class") {
			// Parse class name.
			QString className = classNode.attribute("Name");
			if(className.isEmpty())
				throw Exception("Missing class name attribute in <Class> element.");

			// Skip disabled class elements.
			if(classNode.attribute("Enabled").compare("false", Qt::CaseInsensitive) == 0 ||
			   classNode.attribute("Enabled").compare("off", Qt::CaseInsensitive) == 0 ||
			   classNode.attribute("Enabled").compare("no", Qt::CaseInsensitive) == 0)
				continue;

			// Make sure the name is unique.
			if(classDeclarations.find(className) != classDeclarations.end())
				throw Exception(QString("Class name is not unique: %1").arg(className));

			// Register class declaration.
			classDeclarations[className] = classNode;
		}
		else throw Exception(QString("Unknown element tag in XML file: <%1>").arg(classNode.localName()));
	}

	// Parse the class definition elements.
	for(QMap<QString, QDomElement>::const_iterator iter = classDeclarations.begin(); iter != classDeclarations.end(); ++iter) {
		parseClassDefinition(iter.key(), iter.value());
	}
}

PluginClassDescriptor* Plugin::parseClassDefinition(const QString& className, const QDomElement& classNode)
{
	OVITO_ASSERT(className.isEmpty() == false);

	PluginClassDescriptor* descriptor = findClass(className);
	if(descriptor) return descriptor;	// Is already declared.

	try {
		bool isAbstract = false;
		if(classNode.attribute("Abstract") == "true") isAbstract = true;
		bool serializable = true;
		if(classNode.attribute("Serializable") == "false") serializable = false;

		// Get the base class.
		QDomElement baseClassElement = classNode.firstChildElement("Base-Class");
		if(baseClassElement.isNull())
			throw Exception("No base class specified in plugin class definition.");

		// Find base class.
		PluginClassDescriptor* baseClass = getRequiredClass(baseClassElement);

		// Create descriptor.
		descriptor = createClassDescriptor(className, baseClass, this, classNode, isAbstract, serializable);
		_classes.push_back(descriptor);
		return descriptor;
	}
	catch(Exception& ex) {
		throw ex.prependGeneralMessage(tr("Error in definition of plugin class '%1'.").arg(className));
	}
}

PluginClassDescriptor* Plugin::getRequiredClass(const QDomElement& classReferenceElement)
{
	QString className = classReferenceElement.attribute("Name");
	QString classPlugin = classReferenceElement.attribute("Plugin-Id");
	if(classPlugin.isEmpty()) classPlugin = pluginId();

	// Get required plugin.
	Plugin* otherPlugin = PLUGIN_MANAGER.plugin(classPlugin);
	if(otherPlugin == NULL)
		throw Exception(QString("Required plugin is not installed: %1").arg(classPlugin));
	CHECK_POINTER(otherPlugin);
	otherPlugin->parseManifest();

	// Find class.
	PluginClassDescriptor* descriptor = otherPlugin->lookupClass(className);
	if(descriptor == NULL)
		throw Exception(QString("Required class %1 not found in plugin %2.").arg(className, classPlugin));

	// Add the plugin that contrains the required class to the list of required plugins of this plugin.
	_implicitDependencies.insert(otherPlugin);

	return descriptor;
}

PluginClassDescriptor* Plugin::findClass(const QString& name) const
{
	Q_FOREACH(PluginClassDescriptor* descriptor, classes()) {
		if(descriptor->name() == name)
			return descriptor;
	}
	return NULL;
}

PluginClassDescriptor* Plugin::lookupClass(const QString& name)
{
	// Search class definitions.
    PluginClassDescriptor* descriptor = findClass(name);
    if(descriptor) return descriptor;

	// Search class declarations.
	for(QMap<QString, QDomElement>::const_iterator iter = classDeclarations.begin(); iter != classDeclarations.end(); ++iter) {
		if(iter.key() == name) {
			// Parse class definition.
			return parseClassDefinition(iter.key(), iter.value());
		}
	}
	return NULL;
}

/// Returns all plugins this plugin directly depends on.
QSet<Plugin*> Plugin::dependencies() const
{
	QSet<Plugin*> dep = _implicitDependencies;
	for(QVector<QString>::const_iterator depName = _dependencies.begin(); depName != _dependencies.end(); ++depName) {
		Plugin* depPlugin = PLUGIN_MANAGER.plugin(*depName);
		if(depPlugin != NULL)
			dep.insert(depPlugin);
	}
	return dep;
}

};

