///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/data/DataSetManager.h>
#include <core/scene/ObjectNode.h>
#include <core/scene/animation/AnimManager.h>

#include "AtomPicker.h"

namespace AtomViz {

/// Cached shape of the atom selection marker.
QVector<Point3> AtomPicker::atomMarker;

/******************************************************************************
* Picks the atom under the mouse cursor from an AtomsObject.
******************************************************************************/
bool AtomPicker::pickAtom(Viewport& vp, const QPoint& clickPoint, AtomsObject* atoms, TimeTicks time, const AffineTransformation& tm, PickAtomResult& result)
{
	PickAtomResult closestHit;

	Ray3 ray = vp.screenRay(clickPoint);

	// Get the atomic positions and radii.
	DataChannel* posChannel = atoms->getStandardDataChannel(DataChannel::PositionChannel);
	if(!posChannel) return false;
	TimeInterval iv;
	QVector<FloatType> radii = atoms->getAtomRadii(time, iv);
	OVITO_ASSERT(posChannel->size() == radii.size());

	// Hit test every atom.
	const Point3* p = posChannel->constDataPoint3();
	const FloatType* r = radii.constBegin();
	for(size_t index = 0; index < posChannel->size(); index++, ++p, ++r) {
		Point3 tp = tm * (*p);

		// Perform ray-sphere intersection test.
		Vector3 sphere_dir = tp - ray.base;
		FloatType b = DotProduct(ray.dir, sphere_dir);
		FloatType temp = DotProduct(sphere_dir, sphere_dir);
		FloatType disc = b*b + square(*r) - temp;

		// Only calculate the nearest intersection
		if(disc <= 0.0)
			continue; // Ray missed sphere entirely.

		// Calculate closest intersection.
		FloatType tnear = b - sqrt(disc);
		if(vp.isPerspectiveProjection() && tnear < 0.0)
			continue;

		if(tnear < closestHit.hitDistance) {
			closestHit.hitDistance = tnear;
			closestHit.index = (int)index;
			closestHit.localPos = *p;
			closestHit.worldPos = tp;
			closestHit.radius = *r;
		}
	}

	if(closestHit.index >= 0) {
		result = closestHit;
		result.atomsObject = atoms;
		return true;
	}
	else return false;
}

/******************************************************************************
* Picks an atom under the mouse cursor from any AtomsObject in the scene.
******************************************************************************/
bool AtomPicker::pickAtom(Viewport& vp, const QPoint& clickPoint, TimeTicks time, PickAtomResult& result)
{
	PickAtomResult closestHit;

	// Iterate over all object nodes in the scene that evaluate to an AtomsObject.
	SceneRoot* rootNode = DATASET_MANAGER.currentSet()->sceneRoot();
	if(rootNode == NULL) return false;

	for(SceneNodesIterator iter(rootNode); !iter.finished(); iter.next()) {
		ObjectNode* objNode = dynamic_object_cast<ObjectNode>(iter.current());
		if(!objNode) continue;

		const PipelineFlowState& flowState = objNode->evalPipeline(time);
		if(flowState.result() == NULL) continue;

		AtomsObject* atoms = dynamic_object_cast<AtomsObject>(flowState.result());
		if(!atoms) continue;

		// Setup transformation.
		TimeInterval iv;
		const AffineTransformation& nodeTM = objNode->getWorldTransform(time, iv);

		PickAtomResult rec;
		if(pickAtom(vp, clickPoint, atoms, time, nodeTM, rec)) {
			if(rec.hitDistance < closestHit.hitDistance) {
				closestHit = rec;
			}
		}
	}

	if(closestHit.index >= 0) {
		result = closestHit;
		return true;
	}
	else return false;
}

/******************************************************************************
* Renders the atom selection overlay in a viewport.
******************************************************************************/
void AtomPicker::renderSelectionMarker(Viewport* vp, const PickAtomResult& pickRecord)
{
	if(pickRecord.index < 0) return;

	// Prepare marker.
	if(atomMarker.empty()) {
		atomMarker.resize(32);
		for(int i=0; i<32; i++) {
			FloatType angle = (FloatType)i * 2.0 * FLOATTYPE_PI / 32;
			atomMarker[i] = Point3(cos(angle), sin(angle), 0);
		}
	}

	vp->setDepthTest(false);
	vp->setWorldMatrix(IDENTITY);
	AffineTransformation viewTM = vp->viewMatrix();
	glPushAttrib(GL_ENABLE_BIT);
	glDisable(GL_LIGHTING);
	vp->setRenderingColor(Viewport::getVPColor(Viewport::COLOR_SELECTION));
	Point3 tp = viewTM * pickRecord.worldPos;
	vp->setViewMatrix(AffineTransformation::translation(tp - ORIGIN) * AffineTransformation::scaling(pickRecord.radius));
	vp->renderPolyLine(atomMarker.size(), true, Box3(Point3(-1,-1,0), Point3(1,1,0)), atomMarker.constBegin());
	vp->setViewMatrix(AffineTransformation::translation(tp - ORIGIN) * AffineTransformation::scaling(pickRecord.radius*0.8));
	vp->renderPolyLine(atomMarker.size(), true, Box3(Point3(-1,-1,0), Point3(1,1,0)), atomMarker.constBegin());
	glPopAttrib();
	vp->setViewMatrix(viewTM);
	vp->renderMarker(Window3D::MARKER_BOX, pickRecord.worldPos);
}

};	// End of namespace AtomViz
