#ifndef _PROTO_LIST
#define _PROTO_LIST

/**
* @class ProtoList
*
* @brief The ProtoList class provides a simple double linked-list
* class with a "ProtoList::Item" base class to use for
* deriving your own classes you wish to store in a 
* ProtoList.  
* 
* I have some ideas for some more sophisticated
* list mechanisms (like that I put into the ProtoGraph
* class) to provide for a list item base class that can
* be a member of multiple lists at once, automatically get
* removed from all of its lists when destroyed, etc.  But,
* meanwhile this simple list puts no sophisticated demands
* on memory management other than that specified by the
* programmer.
*/

#include "protoDefs.h"

class ProtoList
{
    public:
        ProtoList();
        ~ProtoList();
        
        class Item;
        void Prepend(Item& item);
        void Append(Item& item);
        void Remove(Item& item);
        
        void Destroy();  // deletes contents
        
        bool IsEmpty() const
            {return (NULL == head);}
        Item* GetHead() const
            {return head;}
        Item* GetTail() const
            {return tail;}   
        
        Item* RemoveHead();
    
        class Iterator;
        friend class Iterator;
		/**
		* @class ProtoList::Item
		*
		* @brief Base class to use for deriving your own classes you wish to store in a 
        * ProtoList.  
		*/
        class Item
        {
            public:
                virtual ~Item();
                
            protected:
                Item();
                
            private:
                Item*   plist_prev;
                Item*   plist_next;
                
            friend class ProtoList;
            friend class Iterator;
        };  // end class ProtoList::Item()
        /**
		* @class ProtoList::Iterator
		*
		* @brief ProtoList::Item iterator.
		*/
        class Iterator
        {
            public:
                Iterator(const ProtoList& theList, bool reverse = false);
                ~Iterator();
                
                void Reset(bool reverse = false)
                {
                    reversed = reverse;
                    item = reverse ? list.tail : list.head;
                }
                Item* GetNextItem();
                Item* PeekNextItem() const
                    {return item;}
                Item* GetPrevItem();
                
            private:
                const ProtoList&    list;
                Item*               item;
                bool                reversed;
        };  // end class ProtoList::Iterator
        
    private:
        Item*   head;
        Item*   tail;
    
};  // end class ProtoList

/**
* @class ProtoListTemplate
*
* @brief The ProtoListTemplate definition lets you create new list variants that
* are type checked at compile time, etc.  
*
* Note the "ITEM_TYPE" _must_
*  be a class that is derived from "ProtoList::Item"
*  A simple example is provided in the comments.
*/
template <class ITEM_TYPE>
class ProtoListTemplate : public ProtoList
{
    public:
        ProtoListTemplate() {}
        virtual ~ProtoListTemplate() {}     
        
        ITEM_TYPE* GetHead() const
            {return static_cast<ITEM_TYPE*>(ProtoList::GetHead());}
        ITEM_TYPE* GetTail() const
            {return static_cast<ITEM_TYPE*>(ProtoList::GetTail());}
        
        ITEM_TYPE* RemoveHead()
            {return static_cast<ITEM_TYPE*>(ProtoList::RemoveHead());}
        
        class Iterator : public ProtoList::Iterator
        {
            public:
                Iterator(const ProtoListTemplate& theList, bool reverse = false)
                 : ProtoList::Iterator(theList, reverse) {}
                ~Iterator() {}
                
                void Reset(bool reverse = false)
                    {ProtoList::Iterator::Reset(reverse);}
                ITEM_TYPE* GetNextItem()
                    {return static_cast<ITEM_TYPE*>(ProtoList::Iterator::GetNextItem());}
                ITEM_TYPE* PeekNextItem() const
                    {return static_cast<ITEM_TYPE*>(ProtoList::Iterator::PeekNextItem());}
                ITEM_TYPE* GetPrevItem()
                    {return static_cast<ITEM_TYPE*>(ProtoList::Iterator::GetPrevItem());}

        };  // end class ProtoListTemplate::Iterator
        
};  // end ProtoListTemplate

/******
 * An example of how to use the ProtoListTemplate with 
 * your own type derived *from ProtoList::Item".
 *
 * Note that "ProtoList" itself may be used directly to do the same
 * thing, but using the template will save you some "static_casts", etc
 * for item retrieval, list iteration, etc and may have some other 
 * benefits.
 
 
class MyItem : public ProtoList::Item
{
    public:
        MyItem(int v) : value(v) {}
        int GetValue() const {return value;}
    private:
        int value;
};

class MyItemList : public ProtoListTemplate<MyItem> {};

MyItemList itemList;
MyItem item1(1), item2(2);

itemList.Append(item1);
itemList.Append(item2);

MyItem* headItem = itemList.GetHead();

...
        
************************************/
        
/**
* @class ProtoStack
* 
* @brief The ProtoStack class is like the ProtoList and similarly provides a
*  ProtoStackTemplate that can be used for deriving easier-to-
* use variants for custom items sub-classed from the ProtoStack::Item.
*
* The ProtoStack is a singly linked-list with Push() and Pop() methods
* but does keep track of both the list head and tail and so can be
*  optionally "appended" and used as a FIFO.  Additionally, an iterator
*  is provided to explore the list from head to tail.
*/
class ProtoStack
{
    public:
        ProtoStack();
        ~ProtoStack();
        
        class Item;
        
        // These methods manipulate the ProtoStack as a "stack"
        void Push(Item& item);  // prepend item to list head
        Item* Pop();            // removes/returns list head
        Item* Peek() const
            {return head;}
        
        // These methods manipulate the ProtoStack as a "FIFO"
        void Put(Item& item);  // append item to list tail
        Item* Get()            // remove/returns list head
            {return Pop();}
            
        bool IsEmpty() const
            {return (NULL == head);}
        Item* GetHead() const
            {return head;}
        Item* GetTail() const
            {return tail;}
        void Destroy();
        
        class Iterator;
        friend class Iterator;
		/** 
		* @class ProtoStack::Item
		*
		* @brief Base class to use for deriving your own classes you wish to store in a 
        * ProtoStack.
		*/
        class Item
        {
            public:
                virtual ~Item();
                
            protected:
                Item();
                
            private:
                Item*   pstack_next;
                
            friend class ProtoStack;
            friend class Iterator;
        };  // end class ProtoStack::Item()
        /**
		* @class ProtoStack::Iterator
		*
		* @brief ProtoStack::Item Iterator
		*/
        class Iterator
        {
            public:
                Iterator(const ProtoStack& theStack);
                ~Iterator();
                
                void Reset()
                    {next = stack.head;}
                
                Item* GetNextItem();
                Item* PeekNextItem() const
                    {return next;}
                
            private:
                const ProtoStack&   stack;
                Item*               next;
        };  // end class ProtoStack::Iterator
                         
    private:
        Item*   head;
        Item*   tail;
        
};  // end class ProtoStack

/**
* @class ProtoStackTemplate
*
* @brief The ProtoStackTemplate definition lets you create new list variants that
* are type checked at compile time, etc.  Note the "ITEM_TYPE" _must_
* be a class that is derived from "ProtoStack::Item"
*
* A simple example is provided in the comments.
*/
template <class ITEM_TYPE>
class ProtoStackTemplate : public ProtoStack
{
    public:
        ProtoStackTemplate() {}
        virtual ~ProtoStackTemplate() {}     
        
        ITEM_TYPE* Pop()
            {return static_cast<ITEM_TYPE*>(ProtoStack::Pop());}
        ITEM_TYPE* Get()
            {return static_cast<ITEM_TYPE*>(ProtoStack::Get());}
        ITEM_TYPE* GetHead() const
            {return static_cast<ITEM_TYPE*>(ProtoStack::GetHead());}
        ITEM_TYPE* GetTail() const
            {return static_cast<ITEM_TYPE*>(ProtoStack::GetTail());}
        
        class Iterator : public ProtoStack::Iterator
        {
            public:
                Iterator(const ProtoStackTemplate& theList, bool reverse = false)
                 : ProtoStack::Iterator(theList) {}
                ~Iterator() {}
                
                void Reset()
                    {ProtoStack::Iterator::Reset();}
                ITEM_TYPE* GetNextItem()
                    {return static_cast<ITEM_TYPE*>(ProtoStack::Iterator::GetNextItem());}
                ITEM_TYPE* PeekNextItem() const
                    {return static_cast<ITEM_TYPE*>(ProtoStack::Iterator::PeekNextItem());}

        };  // end class ProtoStackTemplate::Iterator
        
};  // end ProtoStackTemplate

#endif  // _PROTO_LIST
