/* Copyright (C) 2004 - 2007  db4objects Inc.  http://www.db4o.com */

package com.db4odoc.performance;

import java.io.File;
import java.util.ArrayList;

import com.db4o.Db4o;
import com.db4o.ObjectContainer;
import com.db4o.ObjectServer;
import com.db4o.ObjectSet;
import com.db4o.activation.ActivationPurpose;
import com.db4o.activation.Activator;
import com.db4o.config.Configuration;
import com.db4o.io.MemoryIoAdapter;
import com.db4o.query.Query;
import com.db4o.ta.Activatable;


public class UpdatePerformanceBenchmark {
    
    private static int _count = 100000;
    
    private static int _commitInterval = 1000;
    
    private static int _depth = 3;
    
    private static boolean _isClientServer = false;
    
    private static boolean TCP = true;
    
    private static String _filePath = "performance.db4o";
    
    private static String _host = "localhost";
	
    private static final int PORT = 4477;
    
    
    private ObjectContainer objectContainer;
    
    private ObjectServer objectServer;
    
    
    private long startTime;
    
    
    public static void main(String[] arguments) {
    	new UpdatePerformanceBenchmark().runConfigurationTest();
    	new UpdatePerformanceBenchmark().runDifferentObjectsTest();
    	new UpdatePerformanceBenchmark().runCommitTest();
    	new UpdatePerformanceBenchmark().runHardDriveTest();
    	new UpdatePerformanceBenchmark().runRamDiskTest();
    	new UpdatePerformanceBenchmark().runClientServerTest();
    	new UpdatePerformanceBenchmark().runInheritanceTest();
    	new UpdatePerformanceBenchmark().runIndexTest();
    }
    // end main
    
    private void runCommitTest(){
    	System.out.println("Update test with different commit frequency");
    	
    	initForCommitTest();
    	
    	clean();
    	System.out.println("Test update all:");
    	open(configureForCommitTest());
    	store();
    	updateItems(_count);
    	close();
    	
    	
    	
    	clean();
    	System.out.println("Test update all with commit after each " + _commitInterval + " objects:");
    	open(configureForCommitTest());
    	store();
    	updateWithCommit(_count);
    	close();
    	
    	
    }
    // end runCommitTest

	
	private void runRamDiskTest(){
		System.out.println("Update test: RAM disk");
		int objectsToUpdate = 90;
    	initForRamDriveTest();
    	clean();
    	open(configureDriveTest());
    	store();
    	System.out.println("Updating  " + objectsToUpdate + " objects on a RAM drive:");
    	updateItems(objectsToUpdate);
    	close();
	}
	// end runRamDiskTest
    	
	private void runHardDriveTest(){
		System.out.println("Update test: hard drive");
		int objectsToUpdate = 90;
		
    	initForHardDriveTest();
    	clean();
    	open(configureDriveTest());
    	store();
    	System.out.println("Updating  " + objectsToUpdate + " objects on a hard drive:");
    	updateItems(objectsToUpdate);
    	close();
    }
    // end runHardDriveTest

    private void runClientServerTest(){
    	System.out.println("Update test: Client/Server environment");
    	int objectsToUpdate = 30;
    	
    	
    	init();
    	clean();
    	open(configureClientServer());
    	store();
    	System.out.println("Update " + objectsToUpdate + " of " + _count + " objects locally:");
    	updateItems(objectsToUpdate);
    	close();
    	
    	initForClientServer();
    	clean();
    	open(configureClientServer());
    	store();
    	System.out.println("Update " + objectsToUpdate + " of " + _count + " objects remotely:");
    	updateItems(objectsToUpdate);
    	close();
    }
    // end runClientServerTest

    private void runInheritanceTest(){
    	System.out.println("Update test: objects with deep inheritance");
    	
    	int objectsToUpdate = 30;
    	init();
    	clean();
    	open(configure());
    	store();
    	System.out.println("Updating " + objectsToUpdate + " objects");
    	updateItems(objectsToUpdate);
    	close();
    	
    	clean();
    	open(configure());
    	storeInherited();
    	System.out.println("Updating " + objectsToUpdate + " inherited objects");
    	updateItems(objectsToUpdate);
    	close();
    	
    }
    // end runInheritanceTest

    private void runConfigurationTest(){
    	System.out.println("Update test with different configurations");
    	
    	//
    	clean();
    	init();
    	System.out.println("Update test: default configurations");
    	open(Db4o.newConfiguration());
    	store();
    	updateItems(90);
    	close();	
    	//
    	
    	clean();
    	System.out.println("Update test: memory IO adapter");
    	open(configure());
    	store();
    	updateItems(90);
    	close();
    	//
    	clean();
    	System.out.println("Update test: cascade on update");
    	open(configureCascade());
    	store();
    	updateTopLevelItems(90);
    	close();
    	
    	//
    	clean();
    	System.out.println("Update test: Transparent Persistence");
    	open(configureTP());
    	storeActivatableItems();
    	updateActivatableItems(90);
    	close();
    	
    }
    // end runConfigurationTest

	private void updateItems(int count) {
		startTimer();
		ObjectSet result = objectContainer.queryByExample(null);
		
		for (int i = 0; i < count; i ++){
			if (result.hasNext()){
				Item item = (Item)result.next();
		    	item._name = "Updated";
		    	update(item);	
			} else {
				count = i;
				break;
			}
		}
    	stopTimer("Updated " + count + " items");
	}
	// end updateItems

	private void updateWithCommit(int count) {
		startTimer();
		ObjectSet result = objectContainer.queryByExample(null);
		int j = 0;
		for (int i = 0; i < count; i ++){
			if (result.hasNext()){
				Item item = (Item)result.next();
		    	item._name = "Updated";
		    	update(item);
		    	if (j >= _commitInterval){
		    		j = 0;
		    		objectContainer.commit();
		    	} else {
		    		j ++;
		    	}
			} else {
				count = i;
				break;
			}
		}
    	stopTimer("Updated " + count + " items ");
	}
	// end updateWithCommit

	private void updateTopLevelItems(int count) {
		startTimer();
		Query query = objectContainer.query();
		query.constrain(Item.class);
		query.descend("_name").constrain("level0").startsWith(true);
		ObjectSet result = query.execute();
		
		for (int i = 0; i < count; i ++){
			if (result.hasNext()){
				Item item = (Item)result.next();
		    	item._name = "Updated";
		    	update(item);	
			} else {
				count = i;
				break;
			}
		}
    	stopTimer("Updated " + count + " items");
	}
	// end updateTopLevelItems

	private void updateActivatableItems(int count) {
		startTimer();
		Query query = objectContainer.query();
		query.constrain(ActivatableItem.class);
		query.descend("_name").constrain("level0").startsWith(true);
		ObjectSet result = query.execute();
		
		for (int i = 0; i < count; i ++){
			if (result.hasNext()){
				ActivatableItem item = (ActivatableItem)result.next();
		    	item.setName("Updated");
		    	update(item);	
			} else {
				count = i;
				break;
			}
		}
    	stopTimer("Updated " + count + " items");
	}
	// end updateActivatableItems

    private void update(Object item) {
    	objectContainer.store(item);
	}
    // end update
	
    
    private void runDifferentObjectsTest(){
    	System.out.println("Update test with different objects");
    	int objectsToUpdate = 90;
    	int updated = objectsToUpdate;
    			
    	initDifferentObjectsTest();
    	
    	clean();
    	System.out.println(" - primitive object with int field");
    	open(configure());
    	storeSimplest();
    	
		ObjectSet result = objectContainer.queryByExample(null);
		startTimer();
		for (int i = 0; i < objectsToUpdate; i ++){
			if (result.hasNext()){
				SimplestItem item = (SimplestItem)result.next();
				item._id = 1;
		    	update(item);	
			} else {
				updated = i;
				break;
			}
		}
    	stopTimer("Updated " + updated + " items");
    	close();
    
    	clean();
    	open(configure());
    	System.out.println(" - object with String field");
    	store();
    	updated = objectsToUpdate;
    	result = objectContainer.queryByExample(null);
    	startTimer();
    	for (int i = 0; i < objectsToUpdate; i ++){
			if (result.hasNext()){
				Item item = (Item)result.next();
				item._name = "Updated";
		    	update(item);	
			} else {
				updated = i;
				break;
			}
		}
    	stopTimer("Updated " + updated + " items");
    	close();
    	
    	clean();
    	open(configure());
    	System.out.println(" - object with StringBuffer field");
    	storeWithStringBuffer();
    	
    	updated = objectsToUpdate;
    	result = objectContainer.queryByExample(null);
    	startTimer();
    	for (int i = 0; i < objectsToUpdate; i ++){
			if (result.hasNext()){
				ItemWithStringBuffer item = (ItemWithStringBuffer)result.next();
				item._name = new StringBuffer("Updated");
		    	update(item);	
			} else {
				updated = i;
				break;
			}
		}
    	stopTimer("Updated " + updated + " items");
    	close();
    	
    	clean();
    	open(configure());
    	System.out.println(" - object with int array field");
    	storeWithArray();
    	updated = objectsToUpdate;
    	result = objectContainer.queryByExample(null);
    	startTimer();
    	for (int i = 0; i < objectsToUpdate; i ++){
			if (result.hasNext()){
				ItemWithArray item = (ItemWithArray)result.next();
				item._id = new int[]{1,2,3};
		    	update(item);	
			} else {
				updated = i;
				break;
			}
		}
    	stopTimer("Updated " + updated + " items");
    	close();
    	
    	clean();
    	open(configure());
    	System.out.println(" - object with ArrayList field");
    	storeWithArrayList();
    	updated = objectsToUpdate;
    	result = objectContainer.queryByExample(null);
    	startTimer();
    	for (int i = 0; i < objectsToUpdate; i ++){
			if (result.hasNext()){
				ItemWithArrayList item = (ItemWithArrayList)result.next();
				item._ids = new ArrayList();
		    	update(item);	
			} else {
				updated = i;
				break;
			}
		}
    	stopTimer("Updated " + updated + " items");
    	close();
    }
    // end runDifferentObjectsTest
    
    private void runIndexTest(){
    	System.out.println("Update test for objects with and without indexed fields");
    			
    	int objectsToUpdate = 100;
    	init();
    	System.out.println("Updating " + objectsToUpdate + " of " + _count + " objects");
    	clean();
    	open(configure());
    	store();
    	updateItems(objectsToUpdate);
    	close();
    	
    	clean();
    	init();
    	System.out.println("Updating " + objectsToUpdate + " of " + _count + " objects with indexed field");
    	open(configureIndexTest());
    	store();
    	updateItems(objectsToUpdate);
    	close();
    }
    // end runIndexTest

    
    private void init(){
    	_count = 1000;
        _depth = 90;
        _isClientServer = false;
        	
    }
    // end init
    
    private void initDifferentObjectsTest(){
    	_count = 1000;
        _depth = 1;
        _isClientServer = false;
        	
    }
    // end initDifferentObjectsTest
    

    
    private void initForClientServer(){
    	_count = 1000;
        _depth = 90;
        _isClientServer = true;
        _host = "localhost";	
    }
    // end initForClientServer
    
    private void initForRamDriveTest(){
    	_count = 30000;
        _depth = 1;
        _filePath = "r:\\performance.db4o";
        _isClientServer = false;
        	
    }
    // end initForRamDriveTest
    
    private void initForHardDriveTest(){
    	_count = 10000;
        _depth = 3;
        _filePath = "performance.db4o";
        _isClientServer = false;
    }
    // end initForHardDriveTest
    
    private void initForCommitTest(){
    	_count = 10000;
    	_commitInterval = 1000;
        _depth = 3;
        _isClientServer = false;
    }
    // end initForCommitTest
    
    private void clean(){
    	new File(_filePath).delete();
    }
    // end clean
    
    private Configuration configure(){
    	Configuration config = Db4o.newConfiguration();
    	// using MemoryIoAdapter improves the performance 
    	// by replacing the costly disk IO operations with 
    	// memory access
        config.io(new MemoryIoAdapter());
        return config;
    }
    // end configure

    private Configuration configureTP(){
    	Configuration config = Db4o.newConfiguration();
    	// With Transparent Persistence enabled only modified
    	// objects are written to disk. This allows to achieve 
    	// better performance
        config.objectClass(Item.class).cascadeOnUpdate(true);
    	return config;
    }
    // end configureTP

    private Configuration configureCascade(){
    	Configuration config = Db4o.newConfiguration(); 
    	// CascadeOnUpdate can be a performance-killer for 
    	// deep object hierarchies
        config.objectClass(Item.class).cascadeOnUpdate(true);
        return config;
    }
    // end configureCascade

    private Configuration configureIndexTest(){
    	Configuration config = Db4o.newConfiguration(); 
        config.io(new MemoryIoAdapter());
        config.objectClass(Item.class).objectField("_name").indexed(true);
        return config;
    }
    // end configureIndexTest
    
    private Configuration configureForCommitTest(){
    	Configuration config = Db4o.newConfiguration(); 
        config.lockDatabaseFile(false);
        // the commit information is physically written 
        // and in the correct order
        config.flushFileBuffers(true);
        return config;
    }
    // end configureForCommitTest


    private Configuration configureClientServer(){
    	Configuration config = Db4o.newConfiguration(); 
        config.clientServer().singleThreadedClient(true);
        return config;
    }
    // end configureClientServer

    private Configuration configureDriveTest(){
    	Configuration config = Db4o.newConfiguration(); 
        config.flushFileBuffers(true);
        return config;
    }
    // end configureDriveTest
    
    private void store(){
        startTimer();
        for (int i = 0; i < _count ;i++) {
            Item item = new Item("level" + i, null);
            for (int j = 1; j < _depth; j++) {
                item = new Item("level" + i + "/" + j, item);
            }
            objectContainer.store(item);
        }
        objectContainer.commit();
        stopTimer("Store "+ totalObjects() + " objects");
    }
    // end store

    private void storeActivatableItems(){
        startTimer();
        for (int i = 0; i < _count ;i++) {
            ActivatableItem item = new ActivatableItem("level" + i, null);
            for (int j = 1; j < _depth; j++) {
                item = new ActivatableItem("level" + i + "/" + j, item);
            }
            objectContainer.store(item);
        }
        objectContainer.commit();
        stopTimer("Store "+ totalObjects() + " objects");
    }
    // end storeActivatableItems

    private void storeInherited(){
        startTimer();
        for (int i = 0; i < _count ;i++) {
            ItemDerived item = new ItemDerived("level" + i, null);
            for (int j = 1; j < _depth; j++) {
                item = new ItemDerived("level" + i + "/" + j, item);
            }
            objectContainer.store(item);
        }
        objectContainer.commit();
        stopTimer("Store "+ totalObjects() + " objects");
    }
    // end storeInherited

    
    private void storeWithStringBuffer(){
        startTimer();
        for (int i = 0; i < _count ;i++) {
            ItemWithStringBuffer item = new ItemWithStringBuffer(new StringBuffer("level" + i), null);
            for (int j = 1; j < _depth; j++) {
                item = new ItemWithStringBuffer(new StringBuffer("level" + i + "/" + j), item);
            }
            objectContainer.store(item);
        }
        objectContainer.commit();
        stopTimer("Store "+ totalObjects() + " objects");
    }
    // end storeWithStringBuffer
    
    private void storeSimplest(){
        startTimer();
        for (int i = 0; i < _count ;i++) {
        	SimplestItem item = new SimplestItem(i, null);
            for (int j = 1; j < _depth; j++) {
                item = new SimplestItem(i, item);
            }
            objectContainer.store(item);
        }
        objectContainer.commit();
        stopTimer("Store "+ totalObjects() + " objects");
    }
    // end storeSimplest
    
    private void storeWithArray(){
        startTimer();
        int[] array = new int[]{1,2,3,4};
        for (int i = 0; i < _count ;i++) {
        	int[] id = new int[]{1,2,3,4};
        	ItemWithArray item = new ItemWithArray(id, null);
            for (int j = 1; j < _depth; j++) {
            	int[] id1 = new int[]{1,2,3,4};
                item = new ItemWithArray(id1, item);
            }
            objectContainer.store(item);
        }
        objectContainer.commit();
        stopTimer("Store "+ totalObjects() + " objects");
    }
    // end storeWithArray
    
    private void storeWithArrayList(){
    	startTimer();
    	ArrayList idList = new ArrayList();
    	idList.add(1);
    	idList.add(2);
    	idList.add(3);
    	idList.add(4);
        for (int i = 0; i < _count ;i++) {
        	ArrayList ids = new ArrayList();
        	ids.addAll(idList);
        	ItemWithArrayList item = new ItemWithArrayList(ids, null);
            for (int j = 1; j < _depth; j++) {
            	ArrayList ids1 = new ArrayList();
            	ids1.addAll(idList);
            	item = new ItemWithArrayList(ids1, item);
            }
            objectContainer.store(item);
        }
        objectContainer.commit();
        stopTimer("Store "+ totalObjects() + " objects");
    }
    // end storeWithArrayList
    
    private int totalObjects(){
    	return _count * _depth;
    }
    // end totalObjects
    
    private void open(Configuration config){
        if(_isClientServer){
        	int port = TCP ? PORT : 0;
        	String user = "db4o";
        	String password = user;
            objectServer = Db4o.openServer(_filePath, port);
            objectServer.grantAccess(user, password);
            objectContainer = TCP ? Db4o.openClient(_host, port, user,
					password) : objectServer.openClient();
        } else{
            objectContainer = Db4o.openFile(config, _filePath);
        }
    }
    // end open
    
    private void close(){
        objectContainer.close();
        if(_isClientServer){
            objectServer.close();
        }
    }
    //end close
    
    private void startTimer(){
    	startTime = System.currentTimeMillis();
    }
    // end startTimer

    
    private void stopTimer(String message){
        long stop = System.currentTimeMillis();
        long duration = stop - startTime;
        System.out.println(message + ": " + duration + "ms");
    }
    // end stopTimer
    
    public static class Item {

    	public String _name;
        public Item _child;

        public Item(){
            
        }
        
        public Item(String name, Item child){
            _name = name;
            _child = child;
        }
    }
    // end Item

    public static class ActivatableItem implements Activatable {

    	private String _name;
        public ActivatableItem _child;

    	transient Activator _activator;
    	
    	public void bind(Activator activator) {
        	if (_activator == activator) {
        		return;
        	}
        	if (activator != null && _activator != null) {
                throw new IllegalStateException();
            }
    		_activator = activator;
    	}
    	
    	public void activate(ActivationPurpose purpose) {
    		if (_activator == null) return;
    		_activator.activate(purpose);
    	}
    	
        
        public ActivatableItem(){
            
        }
        
        public ActivatableItem(String name, ActivatableItem child){
            setName(name);
            _child = child;
        }

		public void setName(String _name) {
			this._name = _name;
		}

		public String getName() {
			return _name;
		}
        
    }
    // end ActivatableItem

    public static class ItemCopy {
        
        public String _name;
        public ItemCopy _child;

        public ItemCopy(){
            
        }
        
        public ItemCopy(String name, ItemCopy child){
            _name = name;
            _child = child;
        }
    }
    // end ItemCopy

    public static class ItemDerived extends Item {
    	
    	public ItemDerived(String name, ItemDerived child){
            super(name, child);
        }
    }
    // end ItemDerived
    
    public static class ItemWithStringBuffer {
        
        public StringBuffer _name;
        public ItemWithStringBuffer _child;
        
        public ItemWithStringBuffer(){
        }
        
        public ItemWithStringBuffer(StringBuffer name, ItemWithStringBuffer child){
            _name = name;
            _child = child;
        }
    }
    // end ItemWithStringBuffer
    
    public static class SimplestItem {
        
        public int _id;
        public SimplestItem _child;

        public SimplestItem(){
        }
        
        public SimplestItem(int id, SimplestItem child){
            _id = id;
            _child = child;
        }
    }
    // end SimplestItem

    public static class ItemWithArray {
        
        public int[] _id;
        public ItemWithArray _child;
        
        public ItemWithArray(){
        }
        
        public ItemWithArray(int[] id, ItemWithArray child){
            _id = id;
            _child = child;
        }
    }
    // end ItemWithArray
    
    public static class ItemWithArrayList {
        
        public ArrayList _ids;
        public ItemWithArrayList _child;
        
        public ItemWithArrayList(){
        }
        
        public ItemWithArrayList(ArrayList ids, ItemWithArrayList child){
            _ids = ids;
            _child = child;
        }
    }
    // end ItemWithArrayList
}
