// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CC_SURFACES_COMPOSITOR_FRAME_SINK_SUPPORT_H_
#define CC_SURFACES_COMPOSITOR_FRAME_SINK_SUPPORT_H_

#include "base/compiler_specific.h"
#include "base/memory/weak_ptr.h"
#include "cc/output/compositor_frame.h"
#include "cc/scheduler/begin_frame_source.h"
#include "cc/surfaces/display.h"
#include "cc/surfaces/display_client.h"
#include "cc/surfaces/surface_factory.h"
#include "cc/surfaces/surface_factory_client.h"
#include "cc/surfaces/surfaces_export.h"

namespace cc {

class CompositorFrameSinkSupportClient;
class Display;
class SurfaceManager;

class CC_SURFACES_EXPORT CompositorFrameSinkSupport
    : public NON_EXPORTED_BASE(DisplayClient),
      public SurfaceFactoryClient,
      public BeginFrameObserver {
 public:
  CompositorFrameSinkSupport(
      CompositorFrameSinkSupportClient* client,
      SurfaceManager* surface_manager,
      const FrameSinkId& frame_sink_id,
      std::unique_ptr<Display> display,
      std::unique_ptr<BeginFrameSource> display_begin_frame_source);

  ~CompositorFrameSinkSupport() override;

  const FrameSinkId& frame_sink_id() const { return frame_sink_id_; }

  void EvictFrame();
  void SetNeedsBeginFrame(bool needs_begin_frame);
  void SubmitCompositorFrame(const LocalFrameId& local_frame_id,
                             CompositorFrame frame);
  void Require(const LocalFrameId& local_frame_id,
               const SurfaceSequence& sequence);
  void Satisfy(const SurfaceSequence& sequence);
  void AddChildFrameSink(const FrameSinkId& child_frame_sink_id);
  void RemoveChildFrameSink(const FrameSinkId& child_frame_sink_id);

  Display* display() { return display_.get(); }

 private:
  void DidReceiveCompositorFrameAck();

  // DisplayClient implementation.
  void DisplayOutputSurfaceLost() override;
  void DisplayWillDrawAndSwap(bool will_draw_and_swap,
                              const RenderPassList& render_passes) override;
  void DisplayDidDrawAndSwap() override;

  // SurfaceFactoryClient implementation.
  void ReturnResources(const ReturnedResourceArray& resources) override;
  void SetBeginFrameSource(BeginFrameSource* begin_frame_source) override;
  void WillDrawSurface(const LocalFrameId& local_frame_id,
                       const gfx::Rect& damage_rect) override;

  // BeginFrameObserver implementation.
  void OnBeginFrame(const BeginFrameArgs& args) override;
  const BeginFrameArgs& LastUsedBeginFrameArgs() const override;
  void OnBeginFrameSourcePausedChanged(bool paused) override;

  void UpdateNeedsBeginFramesInternal();

  CompositorFrameSinkSupportClient* const client_;

  SurfaceManager* const surface_manager_;

  const FrameSinkId frame_sink_id_;

  // GpuCompositorFrameSink holds a Display and its BeginFrameSource if it
  // created with non-null gpu::SurfaceHandle. In the window server, the display
  // root window's CompositorFrameSink will have a valid gpu::SurfaceHandle.
  std::unique_ptr<BeginFrameSource> display_begin_frame_source_;
  std::unique_ptr<Display> display_;

  LocalFrameId local_frame_id_;
  SurfaceFactory surface_factory_;
  // Counts the number of CompositorFrames that have been submitted and have not
  // yet received an ACK.
  int ack_pending_count_ = 0;
  ReturnedResourceArray surface_returned_resources_;

  // The begin frame source being observered. Null if none.
  BeginFrameSource* begin_frame_source_ = nullptr;

  // The last begin frame args generated by the begin frame source.
  BeginFrameArgs last_begin_frame_args_;

  // Whether a request for begin frames has been issued.
  bool needs_begin_frame_ = false;

  // Whether or not a frame observer has been added.
  bool added_frame_observer_ = false;

  // The set of BeginFrame children of this CompositorFrameSink.
  std::unordered_set<FrameSinkId, FrameSinkIdHash> child_frame_sinks_;

  base::WeakPtrFactory<CompositorFrameSinkSupport> weak_factory_;

  DISALLOW_COPY_AND_ASSIGN(CompositorFrameSinkSupport);
};

}  // namespace cc

#endif  // CC_SURFACES_COMPOSITOR_FRAME_SINK_SUPPORT_H_
