/*
 * scsidev.c 
 */

/* 
 * Program to assign static device names to SCSI devices as opposed to the
 * dynamic numbering of the Linux-1.0 -- 2.4 kernel.
 * Program written by Eric Youngdale <aric@aib.com>
 * Reworked 1/2000 Kurt Garloff <garloff@suse.de>
 * 
 * Copyright: GNU GPL.
 * Please read the file COPYING.
 *
 * Changes:
 * * 2000/01/16: Kurt Garloff <garloff@suse.de>: 
 *   - SCSI_DISK_MAJOR
 *   - Fixed lots of warnings
 *   - use tmp file in /dev/scsi
 *   Fix /etc/scsi-alias functionality
 *   - Parsing: Respect EOL \0
 *   - inquire: Request takes a few bytes, response buffer is somewhat smaller
 *		use getstr() to get and truncate strings
 *   - comparison: Empty Strings of failed inquiries do not fit
 * -> Released Version: 1.6
 * 
 * * 2000/01/17: Kurt Garloff <garloff@suse.de>:
 *   Major changes:
 *   - Now scan completely relying on sg. Does inquiry for every device
 *     immediately (it's not that expensive), and does interpret the 
 *     periph dev qualifier field to build the other high-level SCSI devs.
 *     -> removable devices will work, this way.
 *     For harddisks, the partitions will be scanned, for tapes the n version
 *     be created.
 *   Minor changes:
 *   - new inquiry () function replacing old inquire and get_version_no
 *   - added support for more than 16 SCSI disks
 *   - quiet flag -q
 *   - Link alias names flag -L
 *   - -c maxmiss option: Tolerate maxmiss missing sg devs.
 *   - rev and hostname fields
 *   - avoid unnecessary recreation of alias names
 *   - new options -n osanitize and -d elete undetected
 * -> Released Version: 2.0
 * 
 * * 2000/01/18: Kurt Garloff <garloff@suse.de>
 *   In principal, there is a chance to fool scsidev: Remove a device with
 *   a low SCSI ID and reinsert another device. Then the assumptions of
 *   scsidev about the dev no. ordering of the kernel are wrong.
 *   - Add a check for this
 *   - infrastructure changes: scsiname () and oldscsiname ()
 *   - struct regnames (sname) now has generic pointer to allow
 *     relationships between devs. name is the full name, now.
 * 
 *  * 2000/07/15: Kurt Garloff <garloff@suse.de>
 *    - Open device nodes (but sg) with O_NONBLOCK; unfortunately this is not
 *      honoured by most of Linux' SCSI high-level drivers :-(
 *    - Add support for OnStream tapes (osst)
 *  -> Version 2.20
 * 
 *  * 2000/09/04: Kurt Garloff <garloff@suse.de>
 *    - Bugfix for long hostnames from Doug Gilbert (=> 2.21)
 *    - Fix parsing of scsi.alias file: Broke on missing LF at the end
 *  -> 2.22
 * 
 *  * 2002/07/26: Kurt Garloff <garloff@suse.de>
 *    - optional (-e) cbtu naming scheme
 *    - bugfix WRT alias handling of subdevices (parititons, non-rew. tap)
 *    - Support for large SCSI IDs and LUNs. (from MGE)
 *    - Support WWID (INQ EVPD 0x83) report + aliases
 *    - Fix hex number parser
 *  -> 2.23
 * 
 *  * 2002/07/28: Kurt Garloff <garloff@suse.de>
 *    - More sane way of storing permissions
 *    - Agnostic of major numbers
 *    - Support many SDs (only up to 255 that is)
 *    - Support for SCSI changers
 *  -> 2.24
 *
 *  * 2002/07/29: Kurt Garloff <garloff@suse.de>
 *    - Support for /proc/scsi/scsi extensions
 *    - Support for really large no of disks
 *  -> 2.25
 * 
 */

#include <stdio.h>
//#include <linux/fs.h>
#include <sys/sysmacros.h>
#include <fcntl.h>
#include <linux/major.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <scsi/scsi_ioctl.h>


static char rcsid[] ="$Id: scsidev.c,v 1.28.2.9 2002/07/29 08:28:36 garloff Exp $";
static char *versid = "scsidev " VERSION " 2000/01/17";
static char *copyright = "Copyright: GNU GPL  (see file COPYING)\n" \
" (w)  1994--1997 Eric Youngdale <eric@andante.org>\n"\
"      2000--2002 Kurt Garloff   <garloff@suse.de>";


#include "config.h"

#ifdef HAVE_SCSI_SCSI_H					/* { */
#include <scsi/scsi.h>
#else							/* } { */
#ifdef HAVE_LINUX_SCSI_H				/* { */
#include <linux/scsi.h>
#else							/* } { */
#ifdef HAVE__USR_SRC_LINUX_DRIVERS_SCSI_SCSI_H 		/* { */
#include "/usr/src/linux/drivers/scsi/scsi.h"
#else							/* } { */
#error "Unable to include scsi.h"
#endif							/* } */
#endif							/* } */
#endif							/* } */

#include <getopt.h>

int use_symlink = 0;
int symlink_alias = 0;
int filemode = 0600;
int verbose = 0;
int quiet = 0;
int maxmiss = 8;
int force = 0;
int san_del = 0;
int no_san = 0;
int nm_cbtu = 0;
int supp_rmvbl = 0;
char * no_serial = "No serial number";
unsigned long long no_wwid = 0;

#define DEVSCSI "/dev/scsi"
#define TESTDEV DEVSCSI "/testdev"
#define PROCSCSI "/proc/scsi/scsi"
#define SHADOW ".shadow"

enum devtype_t { NONE=0, SG, SD, SR, ST, OSST, SCH, };

/*
 * This program builds entries in /dev/scsi that have names that are tied
 * to the actual device number and lun.   This is pretty solid as long as
 * you have only one scsi controller, but with multiple scsi controllers
 * this all falls down.  The problem is that you can have multiple controllers
 * from one manufacturer, so it is not sufficient to assign an id based upon
 * an index from the driver.  Currently the controller numbers are just the
 * internal index that is assigned by the kernel.
 *
 * A better solution would be for the kernel to assign id numbers for each
 * controller type, and the we could assign numbers based upon the number
 * of the controller located.  This would guarantee that we could generate
 * useful number.
 */

typedef struct regnames
{
    struct regnames * next;
    char * name;
    char * manufacturer;
    char * model;
    char * rev;
    char * serial;
    unsigned long long wwid;
    enum devtype_t devtp;
    char inq_devtp;
    char rmvbl;
    char unsafe;
    int  hostid;
    int  major;
    int  minor;
    char * hostname;
    int  hostnum;
    int  chan;
    int  id;
    int  lun;
    char partition;
    struct regnames * alias; // TBR
    struct regnames * related;
} sname;

/* The related pointer is new: It point for disks, tapes and CDRoms to
 * the corresponding generic sname; for generic entries, it points to
 * ZERO or to a disk/CDrom/tape */

sname * reglist = NULL;

void build_special();
int inquiry (int, sname *);

#ifndef OSST_MAJOR
# define OSST_MAJOR 206
#endif

#define OSST_SUPPORTS(spnt) (! ( memcmp (spnt->manufacturer, "OnStream", 8) || \
			       ( memcmp (spnt->model, "SC-", 3) && \
				 memcmp (spnt->model, "DI-", 3) && \
				 memcmp (spnt->model, "DP-", 3) && \
				 memcmp (spnt->model, "FW-", 3) && \
				 memcmp (spnt->model, "USB", 3) ) ) )


enum devtype_t inq_devtp_to_devtp (const char inq_devtp, const sname *spnt)
{
	switch (inq_devtp) {
	    case TYPE_DISK:
	    case TYPE_MOD:
		return SD;
		
	    case TYPE_TAPE:
		if (spnt && OSST_SUPPORTS(spnt)) 
			return OSST;
		else
			return ST;
		
	    case TYPE_ROM:
	    case TYPE_WORM:
		return SR;
		
	    case TYPE_MEDIUM_CHANGER:
		return SCH;
		
	    default:
		return SG;
	}
}

inline char isblk (enum devtype_t devtp)
{
	switch (devtp) {
	    case SD:
	    case SR:
		return 1;
	    default:
		return 0;
	}
}


/*
 * Used to maintain a list of the nodes that we have seen
 * which we know to be OK and active.
 */

void dumplist ()
{
    sname * pnt;
    for (pnt = reglist; pnt; pnt = pnt->next) {
	printf ("%s: %s %s %s (%s) %Lx\n", pnt->name,
		pnt->manufacturer, pnt->model, pnt->rev, pnt->serial,
		pnt->wwid);
	printf ("  on %s (%d-%x): c%di%dl%d", pnt->hostname,
		pnt->hostnum, pnt->hostid, pnt->chan, pnt->id, pnt->lun);
	if (pnt->partition != -1) 
	    printf ("p%d", pnt->partition);
	printf (" %c %x-%x\n", (isblk(pnt->devtp)? 'b': 'c'),
		pnt->major, pnt->minor);
    }
}

// Creates a dup scsi device registration, and a link from new to old
sname * sname_dup (sname * spnt)
{
    sname * spnt1 = malloc (sizeof (sname));
    memcpy (spnt1, spnt, sizeof (sname));
    spnt1->related = spnt; //spnt->related = spnt1;
    //spnt1->next = reglist; reglist = spnt1;
    return spnt1;
}

// compare two sname entries
char sname_cmp (sname *sp1, sname *sp2)
{
    // Host
    if (sp1->hostid != sp2->hostid || 
	sp1->hostnum != sp2->hostnum) return 1;
    if (strcmp (sp1->hostname, sp2->hostname)) return 1;
    // Channel, ID, LUN
    if (sp1->chan != sp2->chan) return 2;
    if (sp1->id != sp2->id) return 3;
    if (sp1->lun != sp2->lun) return 4;
    // Compare INQUIRY DATA
    if (sp1->inq_devtp != sp2->inq_devtp ||
	sp1->rmvbl != sp2->rmvbl) return 5;
    // FIXME: Don't compare devtp??
    if (strcmp (sp1->manufacturer, sp2->manufacturer)) return 6;
    if (strcmp (sp1->model, sp2->model)) return 7;
    if (strcmp (sp1->rev, sp2->rev)) return 8;
    if (strcmp (sp1->serial, sp2->serial)) return 9;
    if (sp1->wwid != sp2->wwid) return 10;
    return 0;
}

sname * register_dev (char * name, int major, int minor, 
		      int hnum, int hid, int chan, int id,
		      int lun, int part, char * hostname, 
		      sname * alias)
{
    sname * spnt;
    //char * pnt;

    spnt = (sname *) malloc (sizeof (sname));
    //pnt = strrchr (name, '/');
    //spnt->name = strdup (pnt + 1);
    spnt->name = strdup (name);
    spnt->major = major;
    spnt->minor = minor;
    spnt->hostnum = hnum;
    spnt->hostid = hid;
    spnt->chan = chan;
    spnt->id = id;
    spnt->lun = lun;
    spnt->partition = part;
    if (hostname) spnt->hostname = strdup (hostname); 
    else spnt->hostname = 0;
    spnt->alias = alias;
    spnt->related = 0;
    /*
     * Initialize this - they may be needed later.
     */
    spnt->model = spnt->manufacturer = spnt->serial = spnt->rev = NULL;
    spnt->wwid = no_wwid;
    spnt->next = reglist; reglist = spnt;
    return spnt;
}

/*
 * We need to "fix" any device nodes that are currently not used because
 * it is a sercurity risk to leave these lying about.  These are fixed
 * by storing a shadow file.  If these become active again,
 * we will be able to use them again because the minor number will be
 * set back again, and we are preserving the ownership and permissions.
 */
void sanitize_sdev ()
{
    struct dirent * de;
    char filename[64];
    DIR * sdir;
    sname * spnt;
    struct stat statbuf;
    int status;
    int i;

    /*
     * Next, delete all of the existing entries in the /dev/scsi directory.
     * The idea is that if we have added/removed devices, the numbers might have
     * changed.
     */
    sdir = opendir (DEVSCSI);
    while (1) {
	de = readdir (sdir);
	if (de == NULL) 
	    break;
	/* If it's a .shadow name, leave it alone */
	i = strlen (de->d_name) - strlen (SHADOW);
	if (i > 0 && !strcmp (de->d_name+i, SHADOW))
	    break;
	/*
	 * OK, we have the name.  See whether this is something
	 * we know about already.
	 */
	for( spnt = reglist; spnt; spnt = spnt->next ) {
	    if( strcmp(de->d_name, strrchr (spnt->name, '/') + 1) == 0 )
		break;
	}
	/* Didn't we find it? */
	if (spnt == NULL) {
	    strcpy (filename, DEVSCSI); strcat (filename, "/");
	    strcat (filename, de->d_name);
	    status = stat (filename, &statbuf);
	    if ( status == 0
		 && (S_ISCHR (statbuf.st_mode) || S_ISBLK (statbuf.st_mode)) ) {
		/*
		 * OK, this one is something new that we have to do something
		 * with.  No big deal, stat it so we get the particulars, then
		 * create a new one with a safe minor number.
		 */
		unlink (filename);
		if (!san_del) {
		    strcat (filename, SHADOW);
		    status = open (filename, O_RDWR | O_CREAT | O_EXCL);
		    status = chmod (filename, statbuf.st_mode);
		    chown (filename, statbuf.st_uid, statbuf.st_gid);
		}
	    }
	}
    }
    closedir (sdir);

}


/*
 * Next, delete all of the existing entries in the /dev/scsi directory.
 * The idea is that if we have added/removed devices, the numbers might have
 * changed.
 */
void flush_sdev ()
{
    struct dirent * de;
    char filename[60];
    DIR * sdir;
    sdir = opendir (DEVSCSI);
    while (1) {
	de = readdir (sdir);
	if (de == NULL) break;
	//if (de->d_name[0] != 's' && de->d_name[0] != 'n') continue;
	strcpy (filename, DEVSCSI); strcat (filename, "/");
	strcat (filename, de->d_name);
	unlink (filename);
    }
    closedir (sdir);
    if (!quiet) 
	printf ("Flushed old " DEVSCSI " entries...\n");

}

// Creates a /dev/scsi name from the info in sname
char * scsiname (sname *spnt)
{
    char nm[64]; char *genpart;
    char app[8];
    char *dnm = 0;
    enum devtype_t tp = spnt->devtp;

    *app = 0;
    strcpy (nm, DEVSCSI); strcat (nm, "/");
    /* FIXME */
    switch (tp) {
	case SG:
	    dnm = "sg"; break;
	case SR:
	    dnm = "sr"; break;
	case ST:
	    if (spnt->minor & 0x80) 
		dnm = "nst";
	    else
		dnm = "st";
	    break;
	case OSST:
	    if (spnt->minor & 0x80) 
		dnm = "nosst";
	    else 
		dnm = "osst";
	    break;
	case SD:
	    dnm = "sd";
	    if (spnt->minor & 0x0f) 
		sprintf (app, "p%d", spnt->minor % 0x10);
	    break;
	case SCH:
	    dnm = "sch";
	    break;
	default:
	    fprintf (stderr, "scsidev: PANIC: Illegal major 0x%02x!\n",
		     spnt->major);
	    abort ();
    }
    strcat (nm, dnm);
    genpart = nm + strlen (nm);
    if (nm_cbtu) 
	sprintf (genpart, "c%db%dt%du%d",
		 spnt->hostnum, 
		 spnt->chan, spnt->id, spnt->lun);
    else
	sprintf (genpart, "h%d-%xc%di%dl%d",
		 spnt->hostnum, spnt->hostid,
		 spnt->chan, spnt->id, spnt->lun);
    if (*app) 
	strcat (genpart, app);
    spnt->name = strdup (nm);
    return spnt->name;
}

/* Fortunately those are only needed for symlink mode
 * and if there's no exteneded /proc/scsi/scsi  KG. 
 */
int sd_major_to_disknum (const int major, const int minor)
{
	if (major == SCSI_DISK0_MAJOR)
		return minor >> 4;
	else if (major >= SCSI_DISK1_MAJOR && major <= SCSI_DISK7_MAJOR)
		return (minor >> 4) + ((major -  64) << 4);
	else if (major >= 128 && major <= 135) /* SCSI_DISK10_MAJOR -- SCSI_DISK17_MAJOR */
		return (minor >> 4) + ((major - 120) << 4);
	else if (major >= 144) /* This is only a guess :-( */
		return (minor >> 4) + ((major - 128) << 4);
	else if (major >= 72 && major < 128) /* This is only a guess :-( */
		return (minor >> 4) + ((major +  55) << 4);
	else if (major >= 136 && major < 144) /* This is only a guess :-( */
		return (minor >> 4) + ((major +  47) << 4);
	else if (major >= 12 && major <= 64) /* This is only a guess :-( */
		return (minor >> 4) + ((major + 179) << 4);
	else return -1;
}

int disknum_to_sd_major (const int diskno)
{
	int mj = diskno >> 4;
	if (mj == 0)
		return 8; /* SCSI_DISK0_MAJOR */
	else if (mj >= 1 && mj < 8)
		return 64 + mj;
	else if (mj >= 8 && mj < 16)
		return 120 + mj;
	else if (mj >= 16 && mj < 127)
		return 128 + mj;
	else if (mj >= 127 && mj < 183)
		return mj - 55;
	else if (mj >= 183 && mj < 191)
		return mj - 47;
	else if (mj >= 191 && mj < 244)
		return mj - 179;
	else return -1;
}


static void sd_devname(const unsigned int disknum, char *buffer)
{
	if (disknum < 26)
		sprintf(buffer, "sd%c", 'a' + disknum);
	else if (disknum < (26*27)) {
		unsigned int min1 = disknum / 26 - 1;
		unsigned int min2 = disknum % 26;
		sprintf(buffer, "sd%c%c", 'a' + min1, 'a' + min2);
	} else {
		unsigned int min1 = (disknum / 26 - 1) / 26 - 1;
		unsigned int min2 = (disknum / 26 - 1) % 26;
		unsigned int min3 = disknum % 26;
		sprintf(buffer, "sd%c%c%c", 'a' + min1, 'a' + min2, 'a' + min3);
	}
}


// Creates an old /dev/s? name from the info in sname
char * oldscsiname (sname *spnt)
{
    char nm[64]; char *genpart;
    int diskno;
    enum devtype_t tp = spnt->devtp;

    strcpy (nm, "/dev/"); genpart = nm + strlen (nm);
    /* FIXME */
    switch (tp) {
	case SG:
	    sprintf (genpart, "sg%d", spnt->minor); break;
	case SR:
	    sprintf (genpart, "sr%d", spnt->minor); break;
	case ST:
	    if (spnt->minor & 0x80) 
		sprintf (genpart, "nst%d", spnt->minor & 0x7f);
	    else
		sprintf (genpart, "st%d", spnt->minor & 0x7f);
	    break;
	case OSST:
	    if (spnt->minor & 0x80) 
		sprintf (genpart, "nosst%d", spnt->minor & 0x7f);
	    else
		sprintf (genpart, "osst%d", spnt->minor & 0x7f);
	    break;
	case SCH:
	    sprintf (genpart, "sch%d", spnt->minor);
	    break;
	case SD:
	    diskno = sd_major_to_disknum (spnt->major, spnt->minor);
	    sd_devname (diskno, genpart);
	    genpart += strlen (genpart);
	    if (spnt->minor & 0xf) 
		sprintf (genpart, "%d", (spnt->minor & 0xf));
	    break;
	default:
	    fprintf (stderr, "scsidev: PANIC: Illegal device type major 0x%02x!\n",
		     spnt->major);
	    abort ();
    }
    //spnt->name = strdup (nm);
    return strdup (nm);
}


/*
 * Check to see if a given entry exists.  If not, create it,
 * if it does make sure the major and minor numbers are correct
 * and save permissions and ownerships if this is the case.
 */
void update_device (char * path, int fmode, int major, int minor)
{
    struct stat statbuf;
    int recreate;
    int newmode;
    int uid, gid;
    int status;
    char shadow[64];
    strcpy (shadow, path);

    /* FIXME */
    newmode = fmode | 
	(major != SCSI_CDROM_MAJOR ? filemode : (filemode & ~0222));

    recreate = 1;
    uid = gid = -1;
    status = lstat (path, &statbuf);
    /* If dev node does not exist, try to look up .shadow file */
    if (status) {
	strcat (shadow, SHADOW);
	status = stat (shadow, &statbuf);
    }
    if (status == 0) {
	recreate = 0;
	uid = statbuf.st_uid;
	gid = statbuf.st_gid;
	/*
	 * We are NOT in symlink mode, when getting here.
	 */
	if ( S_ISLNK (statbuf.st_mode) ) {
	    recreate = 1;
	    if (verbose >= 2) 
		printf("is symbolic link ...\n");
	    /* Try to get permissions from somewhere */
	    strcat (shadow, SHADOW);
	    status = stat (shadow, &statbuf);
	    if (status) 
		status = stat (path, &statbuf);
	    uid = statbuf.st_uid;
	    gid = statbuf.st_gid;
	    newmode = fmode | (statbuf.st_mode & ~S_IFLNK);
	    unlink (path);
	}
	/*
	 * Make sure we have the correct device type too.
	 */
	if ( (statbuf.st_mode & S_IFMT) != fmode ) {
	    recreate = 1;
	    newmode = fmode | (statbuf.st_mode & ~S_IFLNK);
	    if (verbose >= 2) 
		printf ("mode: %08x vs. %08x\n", 
			statbuf.st_mode & S_IFMT, fmode);
	}
	/*
	 * Compare the device number.  If something changed, then 
	 * unlink it so that we can recreate it.  Save the mode of
	 * the thing so that we can keep the same permissions.
	 */
	if ( statbuf.st_rdev != makedev (major, minor) ) {
	    recreate = 1;
	    newmode = fmode | (statbuf.st_mode & ~S_IFLNK);
	    if (verbose >= 2) 
		printf ("r_dev: %08x vs. %08x\n", 
			(unsigned int)statbuf.st_rdev, makedev (major, minor));
	}

	if (verbose == 2 && recreate) 
	    printf ("mismatch: Recreate %s (%i-%i)\n", path, major, minor);
    }

    /*
     * If we need to recreate the device, then do it.
     */
    if( recreate ) {
	if (status == 0) 
	    unlink (shadow);
	status = mknod (path, newmode, makedev (major, minor));
	//printf("Recreate maj %i min %i\n", major, minor);
	if( status == -1 ) {
	    fprintf (stderr, "mknod (%s) failed\n", path);
	    exit (1);
	}
      
	if( uid != -1 )
	    status = chown (path, uid, gid);
	/*
	 * The mknod system call will not always use
	 * the right permissions because of umask.
	 * Fix it so that it is really correct.
	 */
	status = chmod (path, newmode);
	if( status == -1 ) {
	    fprintf (stderr, "chmod (%s) failed\n", path);
	    exit (1);
	}
      
    }  
}

void create_dev (sname *spnt)
{
    char *linkto;
    int status;
    int newmode;
    struct stat statbuf;
    int devtype = isblk (spnt->devtp)? S_IFBLK: S_IFCHR;;
	
    if (use_symlink) {
	char shadow[64];
	strcpy (shadow, spnt->name);
	strcat (shadow, SHADOW);

	linkto = oldscsiname (spnt);
	unlink (spnt->name);
	symlink (linkto, spnt->name);
		
	/*
	 * Now make sure that the device the symlink points to 
	 * actually exists.  If not, then create that device.
	 */
	status = stat (spnt->name, &statbuf);
	if (status == 0) {
	    if (statbuf.st_rdev != makedev(spnt->major, spnt->minor))
		fprintf (stderr,
			 "Warning - device %s does not point to expected device\n", linkto);
	} else {
	    newmode = devtype | filemode;
	    /*
	     * Unable to stat the file.  Assume this is 
	     * because it did not exist, so we create it.
	     */
	    status = mknod (linkto, newmode,
			    makedev (spnt->major, spnt->minor));
	    fprintf (stderr, "Creating %s\n", linkto);
	}
	/* If a shadow file exists, get perms from there */
	if (!stat (shadow, &statbuf)) {
	    chown (linkto, statbuf.st_uid, statbuf.st_gid);
	    chmod (linkto, devtype | statbuf.st_mode);
	}
	free (linkto);
    }
    else /* ! use_symlink */
	update_device (spnt->name, devtype, spnt->major, spnt->minor);
}

int getidlun (int fd, sname *spnt, int setidlun)
{
	int status;
	int id[2];

	status = ioctl (fd, SCSI_IOCTL_GET_IDLUN, &id);
	
	if (status == -1)	{
		if (verbose == 2)
			fprintf (stderr, "idlun(%x/%x) returned %d (%d)\n",
				 spnt->major, spnt->minor, status, errno);
		close (fd);
		return -2;
	}
	
	if (setidlun) {
		/* This unfortunately limits all the numbers to be <= 255 */
		spnt->hostnum = id[0] >> 24 & 0xff;
		spnt->chan    = id[0] >> 16 & 0xff;
		spnt->lun     = id[0] >>  8 & 0xff;
		spnt->id      = id[0]       & 0xff;
	}
	spnt->hostid  = id[1];
	
	if (verbose == 2)
		fprintf (stderr, "Found %x:%x with idlun %08x\n", 
			 spnt->major, spnt->minor, id[0]);

	return status;
}


int getscsihostname (int fd, sname *spnt)
{
	int status;
	char hostname[64];
	*(int*)hostname = 63;
	status = ioctl (fd, SCSI_IOCTL_PROBE_HOST, hostname);
	hostname[63] = '\0';
	
	if (status == -1)	{
		if (verbose == 2)
			fprintf (stderr, "probe host (%x/%x) returned %d (%d)\n",
				 spnt->major, spnt->minor, status, errno);
		spnt->hostname = 0;
		return -1;
	};
	spnt->hostname = strdup (hostname);
	return status;
}

int getscsiinfo (int fd, sname *spnt, int setidlun)
{
    int status;

    if ((status = getidlun (fd, spnt, setidlun)))
	return status;
    if ((status = getscsihostname (fd, spnt)) < 0)
	return status;	

    status = inquiry (fd, spnt);
    scsiname (spnt);
    return status;

#if 0
    spnt = register_dev (scsidev, major, minor, 
			 h_id, id[1], chan, scsi_id, lun, -1, hostname, NULL);
    create_dev (spnt);
    if (!quiet) printf ("Found %s (Type %02x) %c on %s \n", scsidev,
			spnt->inq_devtp, (spnt->rmvbl? 'R' : ' '),
			hostname);
#endif
}

int build_disk (sname * spnt, int no)
{
    int minor; int fd; int status;
    sname * spnt1 = sname_dup (spnt);
    spnt1->major = disknum_to_sd_major (no);
    spnt1->minor = (no << 4) & 0xf0; spnt->partition = -1;
    spnt1->devtp = SD;
    scsiname (spnt1);
    spnt1->next = reglist; reglist = spnt1;
    create_dev (spnt1);
    spnt->related = spnt1;
    /* Check if device is there (i.e. medium inside) */
    fd =  open (spnt1->name, O_RDONLY | O_NONBLOCK);
    /* No access to medium / part. table */
    if (fd < 0) {
	spnt1->unsafe = 1;
	/* If it is a removable device, we can't do much more than 
	 * trusting it or not support it at all */
	if (spnt1->rmvbl && supp_rmvbl) 
	    return 0;
	fprintf (stderr, "Can't access %sremovable %s, which should "
		 "be equal to %s!\n", (spnt1->rmvbl? "": "NON-"), 
		 strrchr (spnt1->name, '/') + 1,
		 strrchr (spnt->name, '/') + 1);
	/* We don't unlink spnt1->name! Let sanitize take care of it ... */
	reglist = spnt1->next; free (spnt1->name); 
	spnt->related = 0; free (spnt1);
	return 1;
    }
	
    /* Sanity checks */
    status = getscsiinfo (fd, spnt1, 1);
    close (fd);
    if (status) 
	fprintf (stderr, "scsidev: Strange: Could not get info from %s\n",
		 strrchr (spnt1->name, '/') + 1);
    if (sname_cmp (spnt, spnt1)) {
	fprintf (stderr, "scsidev: What's going on? Dev %s is different from %s\n", 
		 strrchr (spnt1->name, '/') + 1, strrchr (spnt->name, '/') + 1);
	spnt -> related = 0; spnt1 -> related = 0;
	/* And now ? */
	reglist = spnt1->next; free (spnt1->name); 
	spnt->related = 0; free (spnt1);
	return 1;
    }
    /* Now do a partition scan ... */
    spnt = spnt1;
    for (minor = spnt1->minor+1; minor % 16; minor++) {
	struct stat statbuf; int status;
	status = stat (TESTDEV, &statbuf);
	if (status == 0)
	    unlink (TESTDEV);
	    
	status = mknod ( TESTDEV, 0600 | S_IFBLK,
			 makedev (spnt1->major, minor) );
	fd = open (TESTDEV, O_RDONLY | O_NONBLOCK);
	unlink (TESTDEV);
	if (fd < 0) 
	    continue;
	// TO DO: Add sanity checks here ??
	close (fd);
	spnt1 = sname_dup (spnt);
	spnt1->partition = minor % 16;
	spnt1->minor = minor;
	scsiname (spnt1);
	spnt1->next = reglist; reglist = spnt1;
	create_dev (spnt1);
    }
    return 0;
}

int build_tape (sname * spnt, int no)
{
    int fd; int status;
    sname * spnt1 = sname_dup (spnt);
    spnt1->major = SCSI_TAPE_MAJOR;
    spnt1->minor = no;
    spnt1->devtp = ST;
    scsiname (spnt1);
    spnt1->next = reglist; reglist = spnt1;
    create_dev (spnt1);
    /* Check if device is there (i.e. medium inside) */
    fd =  open (spnt1->name, O_RDONLY | O_NONBLOCK);
    if (fd < 0) {
	/* Tapes are always accessible, as they are char devices */
	fprintf (stderr, "Can't access tape %s, which should "
		 "be equal to %s!\n", strrchr (spnt1->name, '/') + 1,
		 strrchr (spnt->name, '/') + 1);
	/* We don't unlink spnt1->name! Let sanitize take care of it ... */
	reglist = spnt1->next; free (spnt1->name); 
	spnt->related = 0; free (spnt1);
	return 1;
    }
    /* Do a sanity check here */
    status = getscsiinfo (fd, spnt1, 1);
    close (fd);
    if (status) 
	fprintf (stderr, "scsidev: Strange: Could not get info from %s\n",
		 strrchr (spnt1->name, '/') + 1);
    if (sname_cmp (spnt, spnt1)) {
	fprintf (stderr, "scsidev: What's going on? Dev %s is different from %s\n", 
		 strrchr (spnt1->name, '/') + 1, strrchr (spnt->name, '/') + 1);
	spnt -> related = 0; spnt1 -> related = 0;
	/* And now ? */
	reglist = spnt1->next; free (spnt1->name); 
	spnt->related = 0; free (spnt1);
	return 1;
    }
    /* And create the no-rewind alias */
    spnt1 = sname_dup (spnt1);
    spnt1->minor |= 0x80;
    scsiname (spnt1);
    spnt1->next = reglist; reglist = spnt1;
    create_dev (spnt1);
	
    return 0;
}

int build_os_tape (sname * spnt, int no)
{
    int fd; int status;
    sname * spnt1 = sname_dup (spnt);
    spnt1->major = OSST_MAJOR;
    spnt1->minor = no;
    spnt1->devtp = OSST;
    scsiname (spnt1);
    spnt1->next = reglist; reglist = spnt1;
    create_dev (spnt1);
    /* Check if device is there (i.e. medium inside) */
    fd =  open (spnt1->name, O_RDONLY | O_NONBLOCK);
    if (fd < 0) {
	/* OnStream tapes are NOT always accessible, as they have a heavy open() function */
	spnt1->unsafe = 1;
	if (!quiet || !supp_rmvbl) 
	    fprintf (stderr, "Can't access tape %s, which should "
		     "be equal to %s!\n", strrchr (spnt1->name, '/') + 1,
		     strrchr (spnt->name, '/') + 1);
	if (supp_rmvbl) 
	    goto osst_force_success;
	/* We don't unlink spnt1->name! Let sanitize take care of it ... */
	reglist = spnt1->next; free (spnt1->name); 
	spnt->related = 0; free (spnt1);
	return 1;
    }
    /* Do a sanity check here */
    status = getscsiinfo (fd, spnt1, 1);
    close (fd);
    if (status) 
	fprintf (stderr, "scsidev: Strange: Could not get info from %s\n",
		 strrchr (spnt1->name, '/') + 1);
    if (sname_cmp (spnt, spnt1)) {
	fprintf (stderr, "scsidev: What's going on? Dev %s is different from %s\n", 
		 strrchr (spnt1->name, '/') + 1, strrchr (spnt->name, '/') + 1);
	spnt -> related = 0; spnt1 -> related = 0;
	/* And now ? */
	reglist = spnt1->next; free (spnt1->name); 
	spnt->related = 0; free (spnt1);
	return 1;
    }
 osst_force_success:
    /* And create the no-rewind alias */
    spnt1 = sname_dup (spnt1);
    spnt1->minor |= 0x80;
    scsiname (spnt1);
    spnt1->next = reglist; reglist = spnt1;
    create_dev (spnt1);
	
    return 0;
}

int build_cdrom (sname * spnt, int no)
{
    int fd; int status;
    sname * spnt1 = sname_dup (spnt);
    spnt1->major = SCSI_CDROM_MAJOR;
    spnt1->minor = no;
    spnt1->devtp = SR;	
    scsiname (spnt1);
    spnt1->next = reglist; reglist = spnt1;
    create_dev (spnt1);
    fd =  open (spnt1->name, O_RDONLY | O_NONBLOCK);
    /* No access to medium / part. table */
    if (fd < 0) {
	spnt1->unsafe = 1;
	/* Removable block devs are hairy! */
	if (spnt1->rmvbl && supp_rmvbl) 
	    return 0;
	fprintf (stderr, "Can't access %sremovable %s, which should "
		 "be equal to %s!\n", (spnt1->rmvbl? "": "NON-"), 
		 strrchr (spnt1->name, '/') + 1,
		 strrchr (spnt->name, '/') + 1);
	/* We don't unlink spnt1->name! Let sanitize take care of it ... */
	reglist = spnt1->next; free (spnt1->name); 
	spnt->related = 0; free (spnt1);
	return 1;
    }
	
    /* Do a sanity check */
    status = getscsiinfo (fd, spnt1, 1);
    close (fd);
    if (status) 
	fprintf (stderr, "scsidev: Strange: Could not get info from %s\n",
		 strrchr (spnt1->name, '/') + 1);
    if (sname_cmp (spnt, spnt1)) {
	fprintf (stderr, "scsidev: What's going on? Dev %s is different from %s\n", 
		 strrchr (spnt1->name, '/') + 1, strrchr (spnt->name, '/') + 1);
	/* And now ? */
	reglist = spnt1->next; free (spnt1->name); 
	spnt->related = 0; free (spnt1);
	return 1;
    }
    return 0;
}

int build_changer (sname * spnt, int no)
{
    int fd; int status;
    sname * spnt1 = sname_dup (spnt);
    spnt1->major = 86;	/* SCSI_CHANGER_MAJOR; */
    spnt1->minor = no;
    spnt1->devtp = SCH;
    scsiname (spnt1);
    spnt1->next = reglist; reglist = spnt1;
    create_dev (spnt1);
    fd =  open (spnt1->name, O_RDONLY | O_NONBLOCK);
    /* No access to medium / part. table */
    if (fd < 0) {
	spnt1->unsafe = 1;
	/* Removable block devs are hairy! */
	if (spnt1->rmvbl && supp_rmvbl) 
	    return 0;
	fprintf (stderr, "Can't access %sremovable %s, which should "
		 "be equal to %s!\n", (spnt1->rmvbl? "": "NON-"), 
		 strrchr (spnt1->name, '/') + 1,
		 strrchr (spnt->name, '/') + 1);
	/* We don't unlink spnt1->name! Let sanitize take care of it ... */
	reglist = spnt1->next; free (spnt1->name); 
	spnt->related = 0; free (spnt1);
	return 1;
    }

    /* Do a sanity check */
    status = getscsiinfo (fd, spnt1, 1);
    close (fd);
    if (status) 
	fprintf (stderr, "scsidev: Strange: Could not get info from %s\n",
		 strrchr (spnt1->name, '/') + 1);
    if (sname_cmp (spnt, spnt1)) {
	fprintf (stderr, "scsidev: What's going on? Dev %s is different from %s\n",
		 strrchr (spnt1->name, '/') + 1, strrchr (spnt->name, '/') + 1);
	/* And now ? */
	reglist = spnt1->next; free (spnt1->name); 
	spnt->related = 0; free (spnt1);
	return 1;
    }
    return 0;
}



void build_sgdevlist ()
{
    int fd; 
    struct stat statbuf;
    int status;
    sname * spnt;
    int disks = 0, tapes = 0, cdroms = 0, changers = 0;
    int miss = 0;	
    int minor = 0;
    int major = SCSI_GENERIC_MAJOR; 
    int mode = O_RDWR;
    //    int devtype = (SCSI_BLK_MAJOR(major)? S_IFBLK: S_IFCHR);
    enum devtype_t devtp;
    
    status = stat (DEVSCSI, &statbuf);
    if (status == -1)
	return;

    status = stat (TESTDEV, &statbuf);
    if (status == 0)
	unlink (TESTDEV);

    if (verbose >= 1)
	fprintf (stderr, "Building list for sg (%s dev major %i)\n",
		 "char", major);

    while (minor <= 255) {
	errno = 0;
	status = mknod ( TESTDEV, 0600 | S_IFCHR, 
			 makedev (major, minor) );
	if (status) { 
	    perror ("scsidev: mknod"); 
	    exit (3); 
	}
	fd = open (TESTDEV, mode);
	unlink (TESTDEV);
	if (fd == -1) {
	    if (verbose == 2)
		fprintf (stderr, "open(%x/%x) returned %d (%d)\n",
			 major, minor, fd, errno);
	    miss++;
	    if (miss > maxmiss) 
		break;
	    else { 
		minor++; continue; 
	    }
	}
	spnt = (sname*) malloc (sizeof (sname));
	spnt->major = major;   spnt->minor = minor;
	spnt->devtp = SG;
	spnt->name  = TESTDEV; spnt->partition = -1;
	status = getscsiinfo (fd, spnt, 1);
	close (fd);

	if (status) { 
		free (spnt); miss++;
		if (miss > maxmiss) 
		    break;
		else { 
		    minor++; continue; 
		}
	}
	//scsiname (spnt);

	spnt->next = reglist; reglist = spnt;
	create_dev (spnt);

	devtp = inq_devtp_to_devtp (spnt->inq_devtp, spnt);

	if (!quiet) 
	    printf ("Found %s (Type %02x) %c on %s \n", spnt->name,
		    spnt->inq_devtp, (spnt->rmvbl? 'R' : ' '),
		    spnt->hostname);

	/* Now register cdroms, tapes, and disks as well */
	switch (devtp) {
	    case SD:
		if (!build_disk (spnt, disks)) 
			disks++;
		else if (!build_disk (spnt, disks+1)) 
			disks += 2;
		break;
	    case ST:
		if (!build_tape (spnt, tapes)) 
			tapes++;
		else if (!build_tape (spnt, tapes+1)) 
			tapes += 2;
		break;
	    case OSST:
		if (!build_os_tape (spnt, tapes)) 
			tapes++;
		else if (!build_os_tape (spnt, tapes+1)) 
			tapes += 2;
		break;
	    case SR:
		if (!build_cdrom (spnt, cdroms)) 
			cdroms++;
		else if (!build_cdrom (spnt, cdroms+1)) 
			cdroms += 2;
		break;
	    case SCH:
		if (!build_changer (spnt, changers)) 
			changers++;
		else if (!build_cdrom (spnt, changers+1)) 
			changers += 2;
		break;
		
	    default:
		;/* nothing to be done */
	}
	minor += 1;
    }
    //unlink (TESTDEV);
}

#if 0

/* Test for availability of /proc/scsi/scsi extensions */
int scsi_ext_status ()
{
}

/* Try to switch on extensions */
void scsi_ext_on ()
{
}

/* Try to switch on extensions */
void scsi_ext_off ()
{
}

void build_sgdevlist_procscsi ()
{
    int fd; 
    struct stat statbuf;
    int status;
    sname * spnt;
    int miss = 0;	
    int minor = 0; int major = SCSI_GENERIC_MAJOR; 
    char devchar = 'g'; int mode = O_RDWR;
    int devtype = (SCSI_BLK_MAJOR(major)? S_IFBLK: S_IFCHR);
    char* buf; int buflen;
    FILE* mapfile;

    status = stat (DEVSCSI, &statbuf);
    if (status == -1)
	return;

    status = stat (TESTDEV, &statbuf);
    if (status == 0)
	unlink (TESTDEV);

    if (verbose >= 1)
	fprintf (stderr, "Building list for s%c (%s dev major %i)\n",
		 devchar, (SCSI_BLK_MAJOR(major)? "block": "char"), major);
    
    mapfile = fopen (PROCSCSI, "r");
    if (!mapfile) {
	fprintf (stderr, "scsidev: could not open " PROCSCSI ": %s\n",
		 strerror (errno));
	return;
    }
    
    buf = malloc(128); buflen = 128;

	
    while (getline (&buf, &buflen, mapfile) != -1)
    {
	int h,c,i,l;
	int inqtp, onl;
	char *nm, *dev, *rest;
	if (buf[0] == '#')
		continue;
	
	errno = 0;
	status = mknod ( TESTDEV, 0600 | devtype ,
			 makedev (major, minor) );
	if (status) { perror ("scsidev: mknod"); exit (3); }
	fd = open (TESTDEV, mode);
	unlink (TESTDEV);
	if (fd == -1) {
	    if (verbose == 2)
	    {
		fprintf (stderr, "open(%x/%x) returned %d (%d)\n",
			 major, minor, fd, errno);
	    }
	    miss++;
	    if (miss > maxmiss) break;
	    else { minor++; continue; }
	}
	spnt = (sname*) malloc (sizeof (sname));
	spnt->major = major;   spnt->minor = minor;
	spnt->name  = TESTDEV; spnt->partition = -1;
	status = getscsiinfo (fd, spnt);
	close (fd);

	if (status) { free (spnt); return; };
	scsiname (spnt);

	spnt->next = reglist; reglist = spnt;
	create_dev (spnt);

	if (!quiet) printf ("Found %s (Type %02x) %c on %s \n", spnt->name,
			    spnt->inq_devtp, (spnt->rmvbl? 'R' : ' '),
			    spnt->hostname);

	/* Now register cdroms, tapes, and disks as well */
	switch (spnt->inq_devtp) {
	  case TYPE_DISK:
	  case TYPE_MOD:
	    if (!build_disk (spnt, disks)) disks++;
	    else if (!build_disk (spnt, disks+1)) disks += 2;
	    break;
	  case TYPE_TAPE:
	    if (OSST_SUPPORTS(spnt)) 
	    {
		if (!build_os_tape (spnt, tapes)) tapes++;
		else if (!build_os_tape (spnt, tapes+1)) tapes += 2;
	    }		    
	    else
	    {
		if (!build_tape (spnt, tapes)) tapes++;
		else if (!build_tape (spnt, tapes+1)) tapes += 2;
	    }
	    break;
	  case TYPE_ROM:
	  case TYPE_WORM:
	    if (!build_cdrom (spnt, cdroms)) cdroms++;
	    else if (!build_cdrom (spnt, cdroms+1)) cdroms += 2;
	    break;
	  default:
	    /* nothing to be done */
	}
	minor += 1;
    }
    //unlink (TESTDEV);
    free(buf);
}

int try_procscsi ()
{
	int se_status = scsi_ext_status();
	if (!se_status)
		scsi_ext_on();
	if (!scsi_ext_status())
		return -1;
	
	build_sgdevlist_procscsi();
	
	if (!se_status)
		scsi_ext_off();
}
#endif


void usage()
{
    fprintf (stderr, "%s\n", versid);
    fprintf (stderr, "Usage: scsidev [options]\n");
    fprintf (stderr, " -f     : Force deletion of all " DEVSCSI" entries\n");
    fprintf (stderr, " -n     : Nosanitize: leaved undetected entries untouched\n");
    fprintf (stderr, " -d     : sanitize by Deleting undetected entries (def: minor->255)\n");
    fprintf (stderr, " -l/-L  : create symLinks for device names / alias names\n");
    fprintf (stderr, " -m mode: permissions to create dev nodes with\n");
    fprintf (stderr, " -s     : list Serial numbers /WWIDs of devices (if available)\n");
    fprintf (stderr, " -c mxms: Continue scanning until mxms missing devs found\n");
    fprintf (stderr, " -r     : trust Removeable media (only safe after boot)\n");
    fprintf (stderr, " -e     : use dEvfs like naming  (cbtu chars)\n");
    fprintf (stderr, " -v/-q  : Verbose/Quiet operation\n");
    fprintf (stderr, " -h     : print Help and exit.\n");
}

int main (int argc, char * argv[])
{
    char c;
    int show_serial = 0;
    struct stat statbuf;
    sname * spnt;
    int status;

    status = stat(DEVSCSI, &statbuf);
    if ( status == -1 )
	mkdir (DEVSCSI, 0755);
    status = stat(DEVSCSI, &statbuf);
    if ( status == -1 || !S_ISDIR(statbuf.st_mode)) {
	fprintf(stderr, DEVSCSI " either does not exist, or is not a directory\n");
	exit(0);
    }
    while ((c = getopt(argc, argv, "flLvqshnderm:c:")) != EOF) {
	switch (c) {
	  case 'f':
	    force = 1; break;
	  case 'm':
	    filemode = strtoul (optarg, 0, 0); break;
	  case 'c':
	    maxmiss = strtoul (optarg, 0, 0); break;
	  case 'l':
	    use_symlink = 1; break;
	  case 'L':
	    symlink_alias = 1; break;
	  case 's':
	    show_serial = 1; break;
	  case 'n':
	    no_san = 1; break;
	  case 'd':
	    san_del = 1; break;
	  case 'r':
	    supp_rmvbl = 1; break;
	  case 'e':
	    nm_cbtu = 1; break;
	  case 'v':
	    verbose++; break;
	  case 'h':
	    fprintf (stderr, "%s\n%s\n", rcsid, copyright); 
	    usage (); exit (0); break;
	  case 'q':
	    quiet = 1; break;
	  default:
	    usage(); exit(1); break;
	}
    }

    if( verbose >= 1 ) 
	fprintf( stderr, "%s\n", versid );
    if( force ) 
	flush_sdev ();

#ifdef DEBUG
    register_dev("/dev/scsi/sdh4-334c0i0l0",  8,  0, 6, 0x334, 0, 0, 0, -1, "debug", NULL);
    register_dev("/dev/scsi/sdh4-334c0i0l0p1",8,  1, 6, 0x334, 0, 0, 0,  1, "debug", NULL);
    register_dev("/dev/scsi/sdh4-334c0i0l0p2",8,  2, 6, 0x334, 0, 0, 0,  2, "debug", NULL);
    register_dev("/dev/scsi/sdh4-334c0i0l0p3",8,  3, 6, 0x334, 0, 0, 0,  3, "debug", NULL);
    register_dev("/dev/scsi/sgh4-334c0i0l0", 21,  0, 6, 0x334, 0, 0, 0, -1, "debug", NULL);
    register_dev("/dev/scsi/sgh4-334c0i2l0", 21,  1, 6, 0x334, 0, 2, 0, -1, "debug", NULL);
    register_dev("/dev/scsi/sgh4-334c0i5l0", 21,  2, 6, 0x334, 0, 5, 0, -1, "debug", NULL);
    register_dev("/dev/scsi/srh4-334c0i2l0", 11,  0, 6, 0x334, 0, 2, 0, -1, "debug", NULL);
    register_dev("/dev/scsi/sth4-334c0i5l0",  9,  0, 6, 0x334, 0, 5, 0, -1, "debug", NULL);
    register_dev("/dev/scsi/rsth4-334c0i5l0", 9,128, 6, 0x334, 0, 5, 0, -1, "debug", NULL);
#else
    build_sgdevlist ();
#endif

    if( show_serial ) {
	if (verbose)
	    dumplist();
	for (spnt = reglist; spnt; spnt = spnt->next) {
	    if( spnt->partition != -1 ) 
		continue;
	    if( (spnt->devtp == ST || spnt->devtp == OSST)
		&& (spnt->minor & 0x80) != 0 )
		continue;

	    //if( spnt->serial == NULL ) inquiry (spnt);
	    if( spnt->serial == no_serial )
		printf("Device  %s has no serial number\n", spnt->name);
	    else
		printf("Serial number of %s: \"%s\"\n", spnt->name, spnt->serial);
	    if ( spnt->wwid != no_wwid )
		printf (" WWID: %Lx\n", spnt->wwid);
	}
    }

    //dumplist ();
    /*
     * Now, read the configuration file and see whether there
     * are any special device names we want to try and match.
     */
    build_special ();

    /* flush_sdev () has been changed to delete all, so the if is correct */
    if (!force)
	sanitize_sdev ();

    return 0;
}

char * get_string (char * pnt, char **result)
{
    char quote = 0;

    while (*pnt == ' ' || *pnt == '\t') pnt++;

    if( *pnt == '"' || *pnt == '\'') 
	quote = *pnt++;

    *result = pnt;

    if( quote != 0 ) {
	while ( *pnt != quote ) pnt++;
	*pnt++ = 0;
    } else {
	while ( *pnt != ',' && *pnt != ' ' && *pnt != '\t' && *pnt != '\0' ) 
	    pnt++;
	*pnt++ = 0;
    }

    while (*pnt == ' ' || *pnt == '\t') 
	pnt++;
    if (*pnt == ',') 
	pnt++;
    while (*pnt == ' ' || *pnt == '\t') 
	pnt++;
    return pnt;
}

char * get_llnumber (char * pnt, unsigned long long * result)
{
    int base = 10;
    unsigned long long num;

    while (*pnt == ' ' || *pnt == '\t') 
	pnt++;
    if (pnt[0] == '0' && pnt[1] == 'x') { 
	base = 16; pnt += 2;
    }

    num = 0;
    while (1) {
	if (base == 10 && *pnt >= '0' && *pnt <= '9' ) {
	    num = num * 10ULL + *pnt - '0';
	    pnt++;
	    continue;
	}
	else if ( base == 16 ) {
	    if (*pnt >= '0' && *pnt <= '9') {
		num = (num << 4) + *pnt - '0';
		pnt++;
		continue;
	    }
	    if (*pnt >= 'a' && *pnt <= 'f') {
		num = (num << 4) + *pnt - 'a' + 10;
		pnt++;
		continue;
	    }
	    if (*pnt >= 'A' && *pnt <= 'F') {
		num = (num << 4) + *pnt - 'A' + 10;
		pnt++;
		continue;
	    }
	    break;
	}
	/*
	 * Isn't a digit.  Must be the end of the number.
	 */
	break;
    }
    while (*pnt == ' ' || *pnt == '\t') 
	pnt++;
    if (*pnt == ',') 
	pnt++;
    while (*pnt == ' ' || *pnt == '\t') 
	pnt++;
    *result = num;
    return pnt;

}

char * get_number (char * pnt, int * result)
{
    int base = 10;
    int num;

    while (*pnt == ' ' || *pnt == '\t') 
	pnt++;
    if (pnt[0] == '0' && pnt[1] == 'x') { 
	base = 16; pnt += 2; 
    }

    num = 0;
    while (1) {
	if (base == 10 && *pnt >= '0' && *pnt <= '9' ) {
	    num = num * 10 + *pnt - '0';
	    pnt++;
	    continue;
	}
	else if ( base == 16 ) {
	    if (*pnt >= '0' && *pnt <= '9') {
		num = (num << 4) + *pnt - '0';
		pnt++;
		continue;
	    }
	    if (*pnt >= 'a' && *pnt <= 'f') {
		num = (num << 4) + *pnt - 'a' + 10;
		pnt++;
		continue;
	    }
	    if (*pnt >= 'A' && *pnt <= 'F') {
		num = (num << 4) + *pnt - 'A' + 10;
		pnt++;
		continue;
	    }
	    break;
	}
	/*
	 * Isn't a digit.  Must be the end of the number.
	 */
	break;
    }
    while (*pnt == ' ' || *pnt == '\t') 
	pnt++;
    if (*pnt == ',') 
	pnt++;
    while (*pnt == ' ' || *pnt == '\t') 
	pnt++;
    *result = num;
    return pnt;

}

/*
 * The configuration file is designed to be something that can match
 * any number of fields.  Thus we need to be able to specify a large
 * number of different things in the hope that we can uniquely match
 * to one specific device.
 *
 * Each match contains a series of tokens:
 *
 * ID=number
 * LUN=number
 * CHANNEL=number
 * HOSTID=number
 * HOST="string" (hostname)
 * MANUFACTURER="string"
 * MODEL="string"
 * SERIAL_NUMBER="string"	(for those devices that support this).
 * WWID=number			( "    "      "     "      "      " )
 * REV="string"
 * NAME="string" (alias)
 * DEVTYPE="disk", "tape", "osst", "generic", or "cdrom".
 */
	
void build_special ()
{
    FILE *	configfile;
    char buffer[256];
    char * pnt;
    char * pnt1;
    sname * spnt, *match;
    char scsidev[64];
    int type;

    int lun, chan, id, part, hostid, hostnum;
    int line;
    unsigned long long wwid;	/* host byte order ... */
    enum devtype_t devtype_i;
    char *manufacturer, *model, *serial_number, *name, *devtype, *rev, *host;
    char *scsialias;

#ifdef DEBUG
    scsialias = "scsi.alias";
#else
    scsialias = "/etc/scsi.alias";
#endif

    configfile = fopen (scsialias, "r");
    if (!configfile) {
	if (verbose) perror (scsialias);
	return;
    }

    line = 0;
    while (1) {
	*buffer = 0;
	fgets (buffer, sizeof(buffer), configfile);
	line++;
	if (feof (configfile) && !*buffer) 
	    break;
	
	/*
	 * Remove trailing \n, if present.
	 */
	pnt = buffer + strlen(buffer) - 1;
	if( *pnt == '\n' ) 
	    *pnt = '\0';

	/*
	 * First, tokenize the input line, and pick out the parameters.
	 */
	lun = -1; id = -1;
	chan = -1;
	hostid = -1; hostnum = -1;
	part = -1; wwid = no_wwid;
	host = NULL;
	manufacturer = NULL; model = NULL;
	serial_number = NULL; rev = NULL;
	name = NULL; devtype_i = NONE;
	devtype = NULL;
	pnt = buffer;
	while (*pnt == ' ' || *pnt == '\t') pnt++;

	/* allow blank lines and comments... */
	if( *pnt == '\0' ) 
	    continue;
	if( *pnt == '#'  ) 
	    continue;

	while (1) {
	    pnt1 = pnt;
	    while (*pnt1 != '=' && *pnt1 != '\0') 
		pnt1++;
	    if( *pnt1 == '\0' ) 
		break;

	    *pnt1 = '\0';
	    if( strncmp(pnt, "manu", 4) == 0 )
		pnt = get_string(pnt1 + 1, &manufacturer);
	    else if ( strncmp(pnt, "mode", 4) == 0 )
		pnt = get_string(pnt1 + 1, &model);
	    else if ( strncmp(pnt, "seri", 4) == 0 )
		pnt = get_string(pnt1 + 1, &serial_number);
	    else if ( strcmp(pnt, "wwid") == 0 )
		pnt = get_llnumber(pnt1 + 1, &wwid);
	    else if ( strncmp(pnt, "rev", 3) == 0 )
		pnt = get_string(pnt1 + 1, &rev);
	    else if ( strncmp(pnt, "hostname", 6) == 0 )
		pnt = get_string(pnt1 + 1, &host);
	    else if ( strcmp(pnt, "id") == 0 )
		pnt = get_number(pnt1 + 1, &id);
	    else if ( strcmp(pnt, "lun") == 0 )
		pnt = get_number(pnt1 + 1, &lun);
	    else if ( strncmp(pnt, "chan", 4) == 0 )
		pnt = get_number(pnt1 + 1, &chan);
	    else if ( strncmp(pnt, "part", 4) == 0 )
		pnt = get_number(pnt1 + 1, &part);
	    else if ( strcmp(pnt, "hostid") == 0 )
		pnt = get_number(pnt1 + 1, &hostid);
	    else if ( strcmp(pnt, "hostnum") == 0 )
		pnt = get_number(pnt1 + 1, &hostnum);
	    else if ( strncmp(pnt, "alia", 4) == 0 )
		pnt = get_string(pnt1 + 1, &name);
	    else if ( strncmp(pnt, "devt", 4) == 0 )
		pnt = get_string(pnt1 + 1, &devtype);
	    else {
		fprintf(stderr,"Unrecognized specifier \"%s\" on line %i\n", pnt,
			line);
		break;
	    }
	}

	/*
	 * OK, got one complete entry.  Make sure it has the required
	 * fields, and then attempt to match it to a real device.
	 */
	if( name == NULL ) {
	    fprintf(stderr,"Line %d is missing \"alias\" specifier\n", line);
	    continue;
	}
	if( devtype == NULL ) {
	    fprintf(stderr,"Line %d is missing \"devtype\" specifier\n", line);
	    continue;
	}
	if( strcmp(devtype, "disk") == 0 )
	    devtype_i = SD;
	else if( strcmp(devtype, "cdrom") == 0)
	    devtype_i = SR;
	else if( strcmp(devtype, "tape") == 0)
	    devtype_i = ST;
	else if( strcmp(devtype, "osst") == 0)
	    devtype_i = OSST;
	else if(strcmp(devtype, "generic") == 0 )
	    devtype_i = SG;
	else if(strcmp(devtype, "changer") == 0 )
	    devtype_i = SCH;
	else {
	    fprintf(stderr,"Line %d has invalid  \"devtype\" specifier(%s)\n", 
		    line, devtype);
	    continue;
	}

	/*
	 * OK, minimal requirements are met.  Try and match this to something
	 * we know about already.
	 */
	match = NULL;
	for (spnt = reglist; spnt; spnt = spnt->next) {
	    if( spnt->alias != NULL )
		continue;
	    /*
	     * Check the integers first.  Some of the strings we have to
	     * request, and we want to avoid this if possible.
	     */
	    if( id != -1 && id != spnt->id ) 
		continue;
	    if( chan != -1 && id != spnt->chan ) 
		continue;
	    if( lun != -1 && lun != spnt->lun ) 
		continue;
	    if( hostid != -1 && hostid != spnt->hostid ) 
		continue;
	    if( hostnum != -1 && hostnum != spnt->hostnum ) 
		continue;
	    if( spnt->devtp != devtype_i )
		continue;
	    if( part != spnt->partition )
		continue;
	    if( (spnt->devtp == ST || spnt->devtp == OSST)
		&& (spnt->minor & 0x80) != 0) 
		continue;
	    if( wwid != no_wwid && wwid != spnt->wwid ) 
		continue;

	    /*
	     * OK, that matches, now obtain some of the strings
	     * that might be needed.
	     */
	    if( manufacturer != NULL && (spnt->manufacturer == NULL ||
					 strcmp(spnt->manufacturer, manufacturer) != 0 ))
		continue;

	    if( model != NULL && (spnt->model == NULL ||
				  strcmp(spnt->model, model) != 0 ))
		continue;

	    if( serial_number != NULL && (spnt->serial == NULL ||
					  strcmp(spnt->serial, serial_number) != 0 ))
		continue;

	    if( rev != NULL && (spnt->rev == NULL ||
				strcmp(spnt->rev, rev) != 0 ))
		continue;

	    if( host != NULL && (spnt->hostname == NULL ||
				 strncmp(spnt->hostname, host, strlen(host)) != 0 ))
		continue;

	    /*
	     * We have a match.  Record it and keep looking just in
	     * case we find a duplicate.
	     */
	    if( match != NULL ) {
		fprintf (stderr, "Line %d not matched uniquely\n", line);
		fprintf (stderr, " Prev. match: %s\n", match->name);
		fprintf (stderr, " Curr. match: %s\n", spnt->name);
		break;
	    }
	    else
		match = spnt;
	}

	/*
	 * See if there was a non-unique mapping.  If so, then
	 * don't do anything for this one.
	 */
	    
	// detect break
	if( spnt != NULL )
	    continue;

	if( match != NULL ) {
	    /*
	     * OK, we have a unique match.  Create the device entries,
	     * as requested.
	     */
	    if (!quiet) {
		fprintf (stderr, "Alias device %s: %s=%s ", name,
			 strrchr (match->name, '/') + 1,
			 strrchr (oldscsiname (match), '/') + 1);
		if (match->related)
		    fprintf (stderr, "(%s=%s)\n",
			     strrchr (match->related->name, '/') + 1,
			     strrchr (oldscsiname (match->related), '/') + 1);
		else 
		    fprintf (stderr, "\n");
	    }

	    type = isblk (match->devtp)? S_IFBLK: S_IFCHR;

	    /*
	     * If this is just an ordinary single device type,
	     * Just create it.
	     */
	    sprintf (scsidev, DEVSCSI "/%s", name);
	    register_dev (scsidev, match->major, match->minor, 
			  0, 0, 0, 0, 0, 0, 0, match);
	    if (symlink_alias) {
		unlink (scsidev);
		symlink (match->name, scsidev);
	    }
	    else
		update_device (scsidev, type, 
			       match->major, match->minor);

	    if( devtype_i == ST || devtype_i == OSST ) {
		sprintf (scsidev, DEVSCSI "/n%s", name);
		register_dev (scsidev, match->major, match->minor | 0x80, 
			      0, 0, 0, 0, 0, 0, 0, match);
		if (symlink_alias) {
		    char nm2[128]; strcpy (nm2, "n"); strcat (nm2, match->name);
		    unlink (scsidev);
		    symlink (nm2, scsidev);
		}
		else
		    update_device (scsidev, S_IFCHR,
				   match->major, (match->minor | 0x80) );
	    }

	    if ( devtype_i == SD 
		 && match->partition == -1 ) {
		/*
		 * This is the master entry for a disk.
		 * we need to go through and generate entries
		 * for each partition.  The trick is to find
		 * all of the related entries so we know which
		 * ones we actually need to create.
		 */
		for( spnt = reglist; spnt; spnt = spnt->next ) {
		    if( spnt->alias != NULL ) continue;
		    if( spnt->partition == -1 ) continue;
		    if( spnt->major != devtype_i ) continue;
		    if( spnt->id != match->id ) continue;
		    if( spnt->lun != match->lun ) continue;
		    if( spnt->chan != match->chan ) continue;
		    if( spnt->hostnum != match->hostnum ) continue;
		    if (spnt->hostid != match->hostid ) continue;

		    sprintf(scsidev, DEVSCSI "/%s-p%d", name, 
			    spnt->partition);
		    register_dev (scsidev, match->major, spnt->minor, 
				  0, 0, 0, 0, 0, 0, 0, spnt);
		    if (symlink_alias) {
			unlink (scsidev);
			symlink (match->name, scsidev);
		    } else
			update_device (scsidev, S_IFBLK, 
				       match->major, spnt->minor);
		}
	    }
	}
	else
	{
	    if (!quiet) fprintf (stderr, "Unable to match device for line %d (alias %s)\n", 
				 line, name);
	}
    }

    fclose (configfile);
}

void dumppage (unsigned char* page)
{
    int ln = 4 + page[4];
    int i;
    for (i = 0; i <= ln; i++)
    {
	printf (" %02x", page[i]);
	if (!((i+1)%16)) printf ("\n");
    }
    if (i%16) printf ("\n");
}


char* getstr (char* page, int start, int stop)
{
    int ln = stop-start+1; int i;
    char *str = (char*) malloc (ln+1);
    memcpy (str, page+start, ln);
    str[ln] = 0;
    for (i = ln-1; i >= 0 && str[i] == ' '; i--) str[i] = 0;
    return str;
}

unsigned long long extract_wwid (unsigned char* page)
{
	unsigned int hi, lo;
	if (page[1] != 0x83)
		return no_wwid;
	/* We only support binary data */
	if ((page[4] & 0x0f) != 1)
		return no_wwid;
	/* We support identifier types 2, 3 */
	if ((page[5] & 0x0f) != 3 && (page[5] & 0x0f) != 2)
		return no_wwid;
	/* Id length should be 8 (64 bits) */
	if (page[7] != 8)
		return no_wwid;
	/* Now translate network byte order to host byte order */
	hi = ntohl (*(unsigned int*)(page+ 8));
	lo = ntohl (*(unsigned int*)(page+12));
	return ((unsigned long long) hi << 32) + (unsigned long long) lo;
}

#define INQBUFSZ 512
int get_inq_page (int file, int lun, unsigned char* buf, unsigned char page, char evpd)
{
	unsigned char *cmd;
	
	memset (buf, 0, INQBUFSZ);
	
	*( (int *)  buf)	= 0;			/* length of input data */
	*( ((int *) buf) + 1 )	= INQBUFSZ-32;	/* length of output buffer */
	
	cmd = (char *) ( ((int *) buf) + 2 );
	
	cmd[0] = 0x12;			/* INQUIRY */
	cmd[1] = (lun << 5) | (evpd? 1: 0);	/* lun, evpd=... */
	cmd[2] = page;			/* page code 0: std inquiry */
	cmd[3] = 0x00;			/* (reserved) */
	cmd[4] = 0xfc;			/* allocation length */
	cmd[5] = 0x00;			/* control */
	
	return ioctl (file, SCSI_IOCTL_SEND_COMMAND, buf);
}
	
int inquiry (int infile, sname * spnt)
{
#ifdef DEBUG
    /*
     * Fill in some entries just so that I can test this.
     */
    if(spnt->id == 0 ) {
	spnt->manufacturer = strdup("CONNER");
	spnt->model=strdup("CFP4207S");
    } else if(spnt->id == 2 ) {
	spnt->manufacturer = strdup("HP");
	spnt->model=strdup("C4324/C4325");
    } else if(spnt->id == 5 ) {
	spnt->manufacturer = strdup("WANGTEK");
	spnt->model=strdup("5150ES");
    }
#else
    int status;
    unsigned char buffer[INQBUFSZ];
    unsigned char * const pagestart = buffer + 8;
    //int infile;
    char have_ser_page = 0;
    char have_wwid_page = 0;
    int ln; int off;
    int lun; int ansi;
	
    //infile = open(spnt->name, O_RDWR);
    if( infile == -1 ) return -1;

    // Std. inquiry
    status = get_inq_page (infile, 0, buffer, 0, 0);

    if (status) { 
	fprintf (stderr, "INQUIRY failed for %s (%i-%i/%02x:%02x)!\n",
		 spnt->name, spnt->id, spnt->lun, spnt->major, spnt->minor);
	return -1;
    }
    if (verbose == 2) 
		dumppage (pagestart);
    spnt->manufacturer = getstr (pagestart, 8, 15); 
    spnt->model = getstr (pagestart, 16, 31);
    spnt->rev = getstr (pagestart, 32, 35);
    spnt->inq_devtp = pagestart[0] & 0x1f;
    spnt->rmvbl = (pagestart[1] & 0x80) >> 7;
    spnt->wwid = no_wwid; spnt->serial = no_serial;
    ansi = pagestart[2] & 7;
	
    if (ansi >= 3)
	lun = 0;
    else
	lun = spnt->lun;

    /* TODO: Extract serial number from bytes 36--43 ? */
    
    // List of supported EVPD pages ...
    if (get_inq_page (infile, lun, buffer, 0, 1))
	return 0;
    ln = pagestart[3];
    for (off = 0; off < ln; ++off) {
	if (pagestart[4+off] == 0x80)
		have_ser_page = 1;    
	if (pagestart[4+off] == 0x83)
		have_wwid_page = 1;    
    }
    
    if (have_ser_page && !get_inq_page (infile, lun, buffer, 0x80, 1)) {
	spnt->serial = getstr (pagestart, 4, 3+pagestart[3]);
	if (verbose == 2) 
	    printf ("Serial for %s: %s\n", spnt->name, spnt->serial);
    }

    if (have_wwid_page && !get_inq_page (infile, lun, buffer, 0x83, 1)) {
	spnt->wwid = extract_wwid (pagestart);

	if (verbose == 2)
	    printf ("WWID for %s: %Lx\n", spnt->name, spnt->wwid);
    }

    //close(infile);
    return 0;
#endif
}
