%{
/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: parse_y.y,v 1.3 94/06/19 11:22:12 nau Exp $";

/* grammar to parse ASCII input of PCB description
 */

#include <stdio.h>

#include "global.h"
#include "create.h"
#include "data.h"
#include "error.h"
#include "memory.h"
#include "misc.h"
#include "parse_l.h"

#ifdef	DBMALLOC_INCLUDE
#include DBMALLOC_INCLUDE
#endif

static	LayerTypePtr	Layer;
static	SymbolTypePtr	Symbol;
static	Boolean			LayerFlag[MAX_LAYER];

extern	char			*yytext;		/* defined by LEX */
extern	PCBTypePtr		yyPCB;
extern	ElementTypePtr	yyElement;
extern	FontTypePtr		yyFont;
extern	int				yylineno;		/* linenumber */
extern	char			*yyfilename;	/* in this file */
extern	short			ParseType;

%}

%union									/* define YYSTACK type */
{
	unsigned	number;
	char		*string;
}

%token	<number>	NUMBER CHAR_CONST	/* line thickness, coordinates ... */
%token	<string>	STRING				/* element names ... */

%token	T_PCB T_LAYER T_VIA T_LINE T_RECT T_TEXT T_ELEMENTLINE
%token	T_ELEMENT T_PIN T_GRID T_FLAGS T_SYMBOL T_SYMBOLLINE T_CURSOR
%token	T_ELEMENTARC T_MARK T_GROUPS

%type	<number>	symbolid

%%

pcb
		:
			{
				if (ParseType != PARSE_PCB)
				{
					MyWarningDialog("no PCB files allowed at this point");
					YYABORT;
				}
			}
		  parsepcb
		|  
			{
				if (ParseType != PARSE_ELEMENT)
				{
					MyWarningDialog("no ELEMENT files allowed at this point");
					YYABORT;
				}
			}
		  parseelement
		|  
			{
				if (ParseType != PARSE_FONT)
				{
					MyWarningDialog("no FONT files allowed at this point");
					YYABORT;
				}
			}
		  parsefont
		;

parsepcb
		: pcbname 
			{
					/* reset flags for 'used layers' and init font pointer */
				int	i;

				for (i = 0; i < MAX_LAYER; i++)
					LayerFlag[i] = False;
				yyFont = &yyPCB->Font;
			}
		  pcbgrid
		  pcbcursor
		  pcbflags
		  pcbgroups
		  parsefont
		  pcbdata
		;

parseelement
		: T_ELEMENT '(' STRING STRING NUMBER NUMBER NUMBER ')' '('
			{
				CreateNewElementData(yyElement, $3, $4, $5, $6, $7);
				SaveFree($3);
				SaveFree($4);
			}
		  elementdefinitions ')'
		;

parsefont
		:
			{
					/* mark all symbols invalid */
				int	i;

				yyFont->Valid = False;
				for (i = 0; i <= MAX_FONTPOSITION; i++)
					yyFont->Symbol[i].Valid = False;
			}
		  symbols
			{
				yyFont->Valid = True;
		  		SetFontInfo(yyFont);
			}
		|
			{
				MyWarningDialog("file '%s' has no font information\n"
					"default font will be used", yyfilename);
			}
		;

pcbname
		: T_PCB '(' STRING ')'
			{
				yyPCB->Name = $3;
			}
		;	

pcbgrid
		: T_GRID '(' NUMBER NUMBER NUMBER ')'
			{
				yyPCB->Grid = $3;
				yyPCB->GridOffsetX = $4;
				yyPCB->GridOffsetY = $5;
			}
		|
		;

pcbcursor
		: T_CURSOR '(' NUMBER NUMBER NUMBER ')'
			{
				yyPCB->CursorX = $3;
				yyPCB->CursorY = $4;
				yyPCB->Zoom = $5;
			}
		|
		;

pcbflags
		: T_FLAGS '(' NUMBER ')'
			{
				yyPCB->Flags = $3;
			}
		|
		;

pcbgroups
		: T_GROUPS '(' STRING ')'
			{
				ParseGroupString($3, &yyPCB->LayerGroups);
			}
		|
		;

pcbdata
		: pcbdefinitions
		|
		;

pcbdefinitions
		: pcbdefinition
		| pcbdefinitions pcbdefinition
		;

pcbdefinition
		: via
		| layer
		| element
		| error { YYABORT; }
		;

via
			/* x, y, thickness, name, flags */
		: T_VIA '(' NUMBER NUMBER NUMBER STRING NUMBER ')'
			{
				CreateNewVia(yyPCB, $3, $4, $5, $6, ($7 & 0xfc) | VIAFLAG);
				SaveFree($6);
			}
		;

layer
			/* name */
		: T_LAYER '(' NUMBER STRING ')' '('
			{
				if ($3 <= 0 || $3 > MAX_LAYER)
				{
					yyerror("Layernumber out of range");
					return(1);
				}
				if (LayerFlag[$3-1])
				{
					yyerror("Layernumber used twice");
					return(1);
				}
				Layer = &yyPCB->Layer[$3-1];

					/* memory for name is alread allocated */
				Layer->Name = $4;
				LayerFlag[$3-1] = True;
			}
		  layerdata ')'
		;

layerdata
		: layerdefinitions
		|
		;

layerdefinitions
		: layerdefinition
		| layerdefinitions layerdefinition
		;

layerdefinition
			/* x1, y1, x2, y2, thickness, flags */
		: T_LINE '(' NUMBER NUMBER NUMBER NUMBER NUMBER NUMBER ')'
			{
				CreateNewLineOnLayer(yyPCB, Layer, $3, $4, $5, $6, $7, $8);
			}
			/* x1, y1, x2, y2, flags */
		| T_RECT '(' NUMBER NUMBER NUMBER NUMBER NUMBER ')'
			{
				CreateNewRect(yyPCB, Layer, $3, $4, $5, $6, $7);
			}
			/* x, y, direction, text, flags */
		| T_TEXT '(' NUMBER NUMBER NUMBER STRING NUMBER ')'
			{
				CreateNewText(yyPCB, Layer, $3, $4, $5, $6, $7);
				SaveFree($6);
			}
		;

element
			/* canonical-name, pcb-name, text_x, text_y, text_direction */
		: T_ELEMENT '(' STRING STRING NUMBER NUMBER NUMBER ')' '('
			{
				yyElement = CreateNewElement(yyPCB, $3, $4, $5, $6, $7);
				SaveFree($3);
				SaveFree($4);
			}
		  elementdefinitions ')'
		;

elementdefinitions
		: elementdefinition
		| elementdefinitions elementdefinition
		;

elementdefinition
			/* x1, y1, x2, y2, thickness */
		: T_ELEMENTLINE '(' NUMBER NUMBER NUMBER NUMBER NUMBER ')'
			{
				CreateNewLineInElement(yyElement, $3, $4, $5, $6, $7);
			}
			/* x, y, thickness, name, flags */
		| T_PIN '(' NUMBER NUMBER NUMBER STRING NUMBER ')'
			{
				CreateNewPin(yyElement, $3, $4, $5, $6, ($7 & 0xfc) | PINFLAG);
				SaveFree($6);
			}
			/* x, y, width, height, Startangle, anglediff, thickness */
		| T_ELEMENTARC '(' NUMBER NUMBER NUMBER NUMBER NUMBER NUMBER NUMBER ')'
			{
				CreateNewArcInElement(yyElement, $3, $4, $5, $6, $7, $8, $9);
			}
		| T_MARK '(' NUMBER NUMBER ')'
			{
				CreateNewMarkInElement(yyElement, $3, $4);
			}
		;

symbols
		: symbol
		| symbols symbol
		;

symbol
		: T_SYMBOL '(' symbolid NUMBER ')' '('
			{
				if ($3 <= 0 || $3 > MAX_FONTPOSITION)
				{
					yyerror("fontposition out of range");
					return(1);
				}
				Symbol = &yyFont->Symbol[$3];
				if (Symbol->Valid)
				{
					yyerror("symbol ID used twice");
					return(1);
				}
				Symbol->Valid = True;
				Symbol->Delta = $4;
			}
		  symboldata ')'
		;

symbolid
		: NUMBER
		| CHAR_CONST
		;

symboldata
		: symboldefinitions
		|
		;

symboldefinitions
		: symboldefinition
		| symboldefinitions symboldefinition
		;

symboldefinition
			/* x1, y1, x2, y2, thickness */
		: T_SYMBOLLINE '(' NUMBER NUMBER NUMBER NUMBER NUMBER ')'
			{
				CreateNewLineInSymbol(Symbol, $3, $4, $5, $6, $7);
			}
		;

%%

/* ---------------------------------------------------------------------------
 * error routine called by parser library
 */
int yyerror(s)
char *s;
{
	MyWarningDialog("ERROR parsing file '%s', line %i\nMessage: '%s'",
		yyfilename, yylineno, s);
	return(0);
}

