#include "wily.h"
#include "view.h"
#include <errno.h>

/* Toggle whether or not files starting with "." are displayed
 * in directory listings.
 */
static void
dotfiles(View *v, char *arg) {
	show_dot_files = !show_dot_files;
}

/* Write the address of the window we hit, or the current selection in the
 * tag of the window of the current selection of the court of King
 * Caractacus.
 */
static void
anchor(View *v, char *arg) {
	Tile	*win;

	if ( (win=view_win(v)) || (win=view_win(last_selection)) ) {
		win_anchor(win, arg);
	}
}

static void
undo_ops(View *v, Bool doall, Range (*allfn)(Text*), Range(*onefn)(Text*))
{
	Range	r;

	r = (doall?(*allfn):(*onefn))(v->t);
	if ROK(r) {
		view_show(v, r);
		view_select(v, r);
	}
}

static void
undo(View *v, char *arg)
{
	View	*body;

	if ((body = view_body(v)))
		undo_ops(body, (Bool)arg, undo_undoall, undo_undo);
}

static void
redo(View *v, char *arg)
{
	View	*body;

	if ((body = view_body(v)))
		undo_ops(body, (Bool)arg, undo_redoall, undo_redo);
}

static void
clear(View *v, char *arg) {
	View	*body;

	if ((body = view_body(v))) 
		text_replace(body->t, text_all(body->t), rstring(0,0));
}

static void
del(View *v, char *arg)
{
	win_del(view_win(v));
}

static void
delcol(View*v, char *arg)
{
	col_del(tile_col(v->tile));
}

/* Read or write the window associated with 'v', possibly using 'arg' */
static void
getorput(View*v, char*arg, int(*fn)(Data*,char*))
{
	Data *d = view_data(v);
	Path	dest;
	char	*s=0;

	if (!d)
		return;

	if (arg && strlen(arg)) {
		pathexpand(arg, v, dest);
		s = dest;
	}
	(*fn)(d, s);
}

static void	put(View *v, char *arg)	{getorput(v, arg, data_put);}
static void	get(View *v, char *arg)	{getorput(v, arg, data_get);}

/* Save all dirty files.
 */
void
putall(View *v, char *arg)
{
	data_putall();
}

/* If 'v' or 'last_selection' are in a window, split that window */
static void
split(View *v, char *arg)
{
	Tile	*win;

	if( (win = view_win(v)) || (win = view_win(last_selection)) ) {
		win_clone(win);
	}
}

/* Cut from last_selection */
static void
cut(View *v, char *arg)
{
	if((v=last_selection))
		view_cut(v, v->sel);
}

static void
builtin_snarf(View *v, char *arg)
{
	if((v=last_selection))
		snarf(v->t, v->sel);
}

static void
builtin_paste(View *v, char *arg)
{
	if((v=last_selection))
		view_select(v, paste(v->t, v->sel));
}

/* Look for 'arg' or the current selection in the body of 'v',
 * select and show the found thing if you find it.
 */
static void
look(View *v, char *arg) {
	if((v=view_body(v)))
		view_look(v, arg);
}

static void
quit(View *v, char *arg)
{
	cleanup_and_die(0);
}

/* Make a New window, possibly called 'arg'.  Try to use a window
 * for context.
 */
static void
new(View *v, char *arg)
{
	Path	path;

	if(!arg)
		arg = "New";
		
	/* If 'v' isn't part of a window, maybe last_selection is */
	if(!view_win(v))
		v = last_selection;
	pathexpand( arg, v, path);
	if(!data_find(path))
		data_open(path, true);
}

static void
builtin_font(View *v, char *arg)
{
	tile_setfont(view_tile(v), arg);
}

/* Toggle autoindent mode either for a window or the default */
static void
builtin_autoindent(View *v, char *arg)
{
	View *body;
	
	if ((body = view_body(v))) {
		body->autoindent = ! body->autoindent;
	} else {
		autoindent_enabled = !autoindent_enabled;
		/* also toggle autoindent for all existing views?  think not */
	}
}

typedef struct Cmd Cmd;
struct Cmd {
	char	*name;
	void	(*cmd)(View *, char *);
};

/* _Must_ be kept in sorted order, we use bsearch on it */
static Cmd builtins[] = {
	{"Anchor", anchor},
	{"Clear", clear},
	{"Cut", cut},
	{"Del", del},
	{"Delcol", delcol},
	{"Dotfiles", dotfiles},
	{"Font", builtin_font},
	{"Get", get},
	{"Indent", builtin_autoindent},
	{"Kill", ex_kill},
	{"Look", look},
	{"New", new},
	{"Newcol", col_new},
	{"Paste", builtin_paste},
	{"Put", put},
	{"Putall", putall},
	{"Quit", quit},
	{"Redo", redo},
	{"Snarf", builtin_snarf},
	{"Split", split},
	{"Undo", undo},
};

static int
cmd_compare(Cmd *a,Cmd *b)
{
	return strcmp(a->name, b->name);
}

/*
 * PRE: 'v' is the context, 'cmd' the command.
 *	The first character of cmd is not 0 or whitespace
 * POST: return true if we recognise the builtin, false otherwise.
 */
Bool
builtin(View *v, char *cmd, char *arg)
{
	Cmd	key, *c;

	assert(v);

	key.name = cmd;
	c = bsearch( &key, builtins, sizeof(builtins)/sizeof(Cmd),
				 sizeof(Cmd), 
			(int	(*)(const void *,const void*))cmd_compare);

	if (c) 
		(*c->cmd)(v, arg);
	return (Bool) c;
}
