/********************************************************************************
* Copyright (c) Des Herriott 1993, 1994
*               Erik Kunze   1995 - 1999
*
* Permission to use, distribute, and sell this software and its documentation
* for any purpose is hereby granted without fee, provided that the above
* copyright notice appear in all copies and that both that copyright notice and
* this permission notice appear in supporting documentation, and that the name
* of the copyright holder not be used in advertising or publicity pertaining to
* distribution of the software without specific, written prior permission.  The
* copyright holder makes no representations about the suitability of this
* software for any purpose.  It is provided "as is" without express or implied
* warranty. THE CODE MAY NOT BE MODIFIED OR REUSED WITHOUT PERMISSION!
*
* THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
* INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
* EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY SPECIAL, INDIRECT OR
* CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
* DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
* TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
* PERFORMANCE OF THIS SOFTWARE.
*
* Authors: Martin Smith (msmith@lssec.bt.co.uk)
*          Erik Kunze
*******************************************************************************/
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#ifdef XZX_IF1
#ifndef lint
static char rcsid[] = "$Id: if1.c,v 4.18 1999/04/27 13:56:19 erik Rel $";
#endif
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <assert.h>
#include "z80.h"
#include "debug.h"
#include "resource.h"
#include "machine.h"
#ifdef REGISTERED
#include "menu.h"
#endif
#include "util.h"
#include "dialog.h"
#include "main.h"
#include "if1.h"
#define CART_SIZE		137923
#define CART_SECSIZE	543
#define CART_SECNUM		254
#define GAP_CLOCK_RATE	10
#define WRITE_PROT_BIT	0x01
#define SYNC_BIT		0x02
#define GAP_BIT			0x04
#define DTR_BIT			0x08
#define BUSY_BIT		0x10
#define COMMS_DATA_BIT	0x01
#define COMMS_CLK_BIT	0x02
#define RW_BIT			0x04
#define ERASE_BIT		0x08
#define CTS_BIT			0x10
#define WAIT_BIT		0x20
typedef struct {
char *fn;
int readonly;
uns8 *mmp;
size_t mmsize;
} cartDef;
typedef struct {
int id;
cartDef cart;
unsigned int pos;
} driveDef;
#define DRIVE			(activeDrive)
#define CART			((cartDef *)(&(activeDrive->cart)))
#ifdef DEBUG
#define DEB(x)			{ if (GETCFG(debug) & D_IF1) { x } }
#else
#define DEB(x)
#endif
static void drvMotorOn(int);
static void drvMotorOff(int);
static int cartOpenImage(int, char *);
static void cartCloseImage(int);
static void if1PatchRom(void);
static driveDef drives[IF1_DRIVES] = {
{ 0, { NULL, 0, NULL, 0 }, 0 },
#if IF1_DRIVES > 1
{ 1, { NULL, 0, NULL, 0 }, 0 },
#endif
#if IF1_DRIVES > 2
{ 2, { NULL, 0, NULL, 0 }, 0 },
#endif
#if IF1_DRIVES > 3
{ 3, { NULL, 0, NULL, 0 }, 0 },
#endif
#if IF1_DRIVES > 4
{ 4, { NULL, 0, NULL, 0 }, 0 },
#endif
#if IF1_DRIVES > 5
{ 5, { NULL, 0, NULL, 0 }, 0 },
#endif
#if IF1_DRIVES > 6
{ 6, { NULL, 0, NULL, 0 }, 0 },
#endif
#if IF1_DRIVES > 7
{ 7, { NULL, 0, NULL, 0 }, 0 },
#endif
};
static driveDef *activeDrive = NULL;
static int gapClock = 0;
static int gapBit = 0;
static int preamble = 0;
static int driveCount = 0;
void
If1Init(void)
{
int i;
if (LoadRomImage(GETCFG(if1Rom), IF1ROM, 1))
{
Msg(M_INFO, "IF1 emulation disabled");
SETCFG(if1Active, -1);
}
else
{
if1PatchRom();
for (i = 0; i < IF1_DRIVES; i++)
{
if (cartOpenImage(i, GETCFG(if1Carts[i])) == -1)
{
free(GETCFG(if1Carts[i]));
}
SETCFG(if1Carts[i], NULL);
}
}
#ifdef REGISTERED
UpdateCartridgeMenu();
#endif
}
int
If1InsertCart(int drive)
{
int types[2] = { FT_MDR, FT_END };
char *filename;
int ret = -1;
if ((filename = FileSelector("Insert Cartridge", 1, types)))
{
if (!cartOpenImage(drive, filename))
{
ret = 0;
}
else
{
free(filename);
}
#ifdef REGISTERED
UpdateCartridgeMenu();
#endif
}
return ret;
}
void
If1EjectCart(int drive)
{
cartCloseImage(drive);
#ifdef REGISTERED
UpdateCartridgeMenu();
#endif
}
char *
If1CartName(int drive)
{
return (drives[drive].cart.fn);
}
void
If1ControlOutByte(uns8 byte)
{
static int lastByte = 0;
if (driveCount > 7 || driveCount < 0)
{
driveCount = 0;
}
if (byte == 0xe2 && !preamble)
{
preamble = 12;
}
if (byte == 0xee && lastByte == 0xee)
{
driveCount = 0;
byte = 0;
DEB(Msg(M_DEBUG, "IF1: drive count reset"););
}
else if (byte == 0xec && lastByte == 0xee)
{
++driveCount;
drvMotorOn(8 - driveCount);
}
else if (byte == 0xed && lastByte == 0xef)
{
++driveCount;
drvMotorOff(8 - driveCount);
}
lastByte = byte;
}
void
If1DataOutByte(uns8 byte)
{
if (DRIVE)
{
if (!preamble)
{
unsigned int pos = DRIVE->pos % CART_SECSIZE;
if ((!pos && (byte & 1) != 1) || (pos == 15 && (byte & 1) != 0))
{
Msg(M_ERR, "microdrive write sync lost %d", DRIVE->pos);
CART->readonly = 1;
}
if (!CART->readonly)
{
CART->mmp[DRIVE->pos] = byte;
}
if (pos == (CART_SECSIZE - 1) || pos == 14)
{
preamble = 3;
}
if (++(DRIVE->pos) == CART_SIZE - 1)
{
DRIVE->pos = 0;
DEB(Msg(M_DEBUG, "IF1: drive %d revolution (write)",
DRIVE->id + 1););
}
}
else
{
--preamble;
}
}
}
uns8
If1ControlInByte(void)
{
uns8 res = 0xe0;
if (DRIVE)
{
unsigned int pos = DRIVE->pos % CART_SECSIZE;
uns8 syncBit = SYNC_BIT;
if (pos == 0 || pos == 15)
{
syncBit = 0;
}
else
{
if (++(DRIVE->pos) == CART_SIZE - 1)
{
DRIVE->pos = 0;
DEB(Msg(M_DEBUG, "IF1: drive %d revolution (sync)",
DRIVE->id + 1););
}
}
if (!CART->readonly)
{
res |= WRITE_PROT_BIT;
}
if (!(++gapClock % GAP_CLOCK_RATE))
{
gapBit = !gapBit;
}
if (gapBit)
{
res |= GAP_BIT;
}
res |= syncBit;
}
else
{
res |= GAP_BIT | SYNC_BIT | WRITE_PROT_BIT;
}
return res;
}
uns8
If1DataInByte(void)
{
uns8 res = 0xff;
if (DRIVE)
{
res = CART->mmp[DRIVE->pos];
if (++(DRIVE->pos) == CART_SIZE - 1)
{
DRIVE->pos = 0;
DEB(Msg(M_DEBUG, "IF1: drive %d revolution (read)", DRIVE->id + 1););
}
}
return res;
}
static void
drvMotorOn(int drive)
{
DEB(Msg(M_DEBUG, "IF1: drive %d on", drive + 1););
if (drive < IF1_DRIVES)
{
gapClock = 0;
gapBit = 0;
if (DRIVE  && DRIVE->id != drive)
{
drvMotorOff(DRIVE->id);
}
if (drives[drive].cart.fn || !If1InsertCart(drive))
{
DRIVE = &drives[drive];
DRIVE->pos /= CART_SECSIZE;
DRIVE->pos *= CART_SECSIZE;
}
}
}
static void
drvMotorOff(int drive)
{
DEB(Msg(M_DEBUG, "IF1: drive %d stopped", drive + 1););
if (drive < IF1_DRIVES && DRIVE && drive == DRIVE->id)
{
DRIVE = NULL;
}
}
static int
cartOpenImage(int d, char *fname)
{
driveDef *drive = &drives[d];
cartDef *cart = (cartDef *)&(drive->cart);
int readonly = 0;
size_t size;
int i;
assert(fname != NULL);
cartCloseImage(d);
for (i = 0; i < IF1_DRIVES; i++)
{
if (drives[i].cart.fn && !strcmp(fname, drives[i].cart.fn))
{
Msg(M_WARN, "cartridge image <%s> allready assigned to drive %d",
fname,  1 + i);
return -1;
}
}
if ((cart->mmp = (uns8 *)Mmap(fname, 1, &readonly, &size)) == (uns8 *)-1)
{
return -1;
}
if (size != CART_SIZE)
{
Msg(M_WARN, "invalid cartridge image <%s>", fname);
Munmap((void *)cart->mmp, cart->mmsize);
return -1;
}
cart->fn = fname;
cart->readonly = cart->mmp[size - 1] ? 1 : readonly;
cart->mmsize = size;
drive->pos = 0;
return 0;
}
static void
cartCloseImage(int drive)
{
cartDef *cart = (cartDef *)&(drives[drive].cart);
if (cart->fn)
{
assert(cart->mmp != NULL);
Munmap((void *)cart->mmp, cart->mmsize);
free(cart->fn);
cart->fn = NULL;
}
}
static void
if1PatchRom(void)
{
uns16 rs323inAddr, rs232outAddr;
int checksum, patchFormat;
checksum = GetRomChecksum(IF1ROM, XZX_PAGESIZE);
switch (checksum)
{
case 54778:
DEB(Msg(M_DEBUG, "IF1: found standard ROM v1"););
rs323inAddr = 0x0b9a;
rs232outAddr = 0x0c5a;
patchFormat = 0;
break;
case 52730:
DEB(Msg(M_DEBUG, "IF1: found standard ROM v2"););
rs323inAddr = 0x0b98;
rs232outAddr = 0x0d07;
patchFormat = 1;
break;
case 10427:
DEB(Msg(M_DEBUG, "IF1: found Ian Collier's ROM"););
rs323inAddr = 0x0b98;
rs232outAddr = 0x0d07;
patchFormat = 1;
break;
default:
Msg(M_WARN, "unknown Interface I ROM image - checksum %u", checksum);
return;
}
#ifndef NO_ED_TRAPS
RealMemory[IF1ROM][rs323inAddr] = 0xed;
RealMemory[IF1ROM][rs323inAddr + 1] = RS232_IP_BPT;
RealMemory[IF1ROM][rs232outAddr] = 0xed;
RealMemory[IF1ROM][rs232outAddr + 1] = RS232_OP_BPT;
#endif
if (patchFormat)
{
RealMemory[IF1ROM][0x16a5] = 0;
RealMemory[IF1ROM][0x16a6] = 0;
RealMemory[IF1ROM][0x16cd] = 0;
RealMemory[IF1ROM][0x16ce] = 0;
RealMemory[IF1ROM][0x16cf] = 0;
RealMemory[IF1ROM][0x16d0] = 0;
}
}
#ifndef NO_ED_TRAPS
void
If1Rs232In(void)
{
char c;
int ret;
if ((ret = (int)read(fileno(stdin), (void *)&c, 1)) == 1)
{
DEB(Msg(M_DEBUG, "IF1: RS232 read <%02x>", (unsigned int)c););
A = (c == 0x0a && GETCFG(if1TranslateNl)) ? 0x0d : c;
}
else
{
switch (StdinFileType)
{
case FD_PIPE:
case FD_TTY:
if (errno == 0 || errno == EAGAIN)
{
CLR(C_FLAG);
SET(Z_FLAG);
goto end;
}
break;
case FD_FILE:
if (!ret)
{
CLR(C_FLAG);
CLR(Z_FLAG);
goto end;
}
break;
case FD_UNKNOWN:
default:
break;
}
Msg(M_PERR, "read error on stdin");
}
end:
RET();
}
#endif
#ifndef NO_ED_TRAPS
void
If1Rs232Out(void)
{
char c;
if (A == 13 && GETCFG(if1StripNl))
{
return;
}
c = (A == 0x0d && GETCFG(if1TranslateNl)) ? 0x0a : A;
if (write(fileno(stdout), (void *)&c, 1) == 1)
{
(void)fflush(stdout);
DEB(Msg(M_DEBUG, "IF1: RS232 write <%02x>", (unsigned int)A););
}
else
{
Msg(M_PERR, "IF1: write error on stdout");
}
RET();
}
#endif
#endif
