
/*
 * LIB/NEWSFEED.C
 *
 * (c)Copyright 1997-1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution 
 *    for specific rights granted.
 *
 * NOTE: hlabel, once found, may not be changed.
 */

#include "defs.h"

Prototype int FeedAdd(const char *msgid, time_t t, History *h, const char *nglist, const char *npath, const char *dist, int headerOnly);
Prototype int FeedWrite(int (*callback)(const char *hlabel, const char *msgid, const char *path, const char *offsize, int plfo, int headOnly), const char *msgid, const char *path, const char *offsize, const char *nglist, const char *npath, const char *dist, const char *headOnly, int spamArt);
Prototype void FeedFlush(void);
Prototype void LoadNewsFeed(time_t t, int force, const char *hlabel);
Prototype void TouchNewsFeed(void);
Prototype int FeedValid(const char *hlabel, int *pcount);
Prototype int IsFiltered(const char *hlabel, const char *nglist);
Prototype int FeedQuerySpam(const char *hlabel, const char *hostInfo);
Prototype int PathElmMatches(const char *hlabel, const char *p, int bytes, int *pidx);
Prototype int CommonElmMatches(const char *common, const char *p, int bytes);
Prototype void FeedGetThrottle(const char *hlabel, int *delay, int *lines);
Prototype int FeedCanReadOnly(const char *hlabel);

Prototype FILE *FeedFo;

/*
 * NewsFeed - a newsfeeds label or group.
 *
 *	For most integer options, 0 is the default (usually off), -1 is
 *	off, and +1 is on.  Other values are also possible depending on the
 *	option.
 */

typedef struct NewsFeed {
    struct NewsFeed	*nf_Next;
    char		*nf_Label;
    struct Node		*nf_PathAliasBase;
    struct Node		*nf_GroupAcceptBase;
    struct Node		*nf_FilterBase;
    struct Node		*nf_SpamBase;
    struct Node		*nf_RequireGroupsBase;
    int			nf_MaxCrossPost;	/* 0=unlimited 		*/
    int			nf_MinCrossPost;	/* 0=unlimited 		*/
    int			nf_MaxArtSize;		/* 0=unlimited 		*/
    int			nf_MinArtSize;		/* 0=unlimited 		*/
    int			nf_MaxPathLen;		/* maximum outgoing path len */
    int			nf_MinPathLen;		/* minimum outgoing path len */
    int			nf_MaxConnect;		/* max connections	*/
    char		*nf_Dist;		/* distribution		*/
    char		*nf_NoDist;		/* !distribution	*/
    char		nf_PerLineFlushOpt;
    char		nf_NoMisMatch;
    char		nf_SpamFeedOpt;
    char		nf_Resolved;
    int			nf_ThrottleDelay;
    int			nf_ThrottleLines;
    char		nf_AllowReadOnly;
} NewsFeed;

#define MAXRECURSION	16

#define RTF_ENABLED	0x01			/* enable realtime feed  */
#define RTF_NOBATCH	0x02			/* do not generate batch */

NewsFeed 	*NFBase;
NewsFeed	*GLBase;
NewsFeed 	*NFCache;
NewsFeed	*NFGlob;

char	NFBuf[16384];
int	NFIdx;
FILE	*FeedFo;
MemPool	*NFMemPool;
const char *SaveHLabel;

static int RecurCount = 0;
static int RecurWarn = 0;


int feedQueryPaths(NewsFeed *feed, const char *npath, int size);
int feedQueryGroups(NewsFeed *feed, const char *nglist);
int feedQueryDists(NewsFeed *feed, const char *dist);
int filterRequireGroups(Node *node, const char *nglist);
int filterQueryGroups(NewsFeed *feed, const char *nglist);
int feedPathQuery(NewsFeed *feed, const char *path);
int feedDistQuery(const char *item, int len, const char *list);
int feedSpamFeedOK(int feed, int article);

void resolveGroupList(const char *label, Node *scan);
void AltFeed(NewsFeed *nf, FILE *fi);
int TooNear(time_t t);
void resolveDefaults(NewsFeed *nf, Node *no);

int cbWildCmpStopWhenFound(Node *node, const void *data, int *pabort, int def);
int cbWildCmpNoStop(Node *node, const void *data, int *pabort, int def);
int cbFeedGroupQuery(Node *node, const void *data, int *pabort, int def);
int cbFilterRequireGroups(Node *node, const void *data, int *pabort, int def);
int recursiveScan(NewsFeed *feed, int off, const void *data, int (*callback)(Node *node, const void *data, int *pabort, int def), int def);

/*
 * FEEDADD()  - given message-id, history file data, newsgroup list, pathlist,
 *		and article size, commit the article to outgoing feeds.
 *
 *		NOTE: none of the string objects is allowed to contain a tab,
 *		this is checked prior to the call.
 */

int 
FeedAdd(const char *msgid, time_t t, History *h, const char *nglist, const char *npath, const char *dist, int headerOnly)
{
    int r = 0;

    LoadNewsFeed(t, 0, (char *)-1);

    if (FeedFo) {
	char path[256];

	ArticleFileName(path, sizeof(path), h, 0);

	fprintf(FeedFo, "SOUT\t%s\t%ld,%ld\t%s\t%s\t%s\t%s\t%d\n",
	    path, (long)h->boffset, (long)h->bsize, msgid, nglist, dist, npath,
	    headerOnly
	);
	fflush(FeedFo);	/* simplify the parent reader on the pipe */
	if (ferror(FeedFo)) {
	    syslog(LOG_CRIT, "lost backchannel to master server");
	    r = -1;
	}
    }
    return(r);
}

int
FeedWrite(
    int (*callback)(const char *hlabel, const char *msgid, const char *path, const char *offsize, int plfo, int headOnly), 
    const char *msgid, 
    const char *path, 
    const char *offsize,
    const char *nglist,
    const char *npath,
    const char *dist,
    const char *headOnly,
    int spamArt
) {
    NewsFeed *nf;
    int r = 0;
    int bytes = 0;

    {
	char *p;
	if ((p = strchr(offsize, ',')) != NULL)
	    bytes = strtol(p + 1, NULL, 0);
    }

    for (nf = NFBase; nf; nf = nf->nf_Next) {
	if (feedQueryPaths(nf, npath, bytes) == 0 && 
	    feedQueryGroups(nf, nglist) == 0 &&
	    feedSpamFeedOK(nf->nf_SpamFeedOpt, spamArt) == 0 &&
	    feedQueryDists(nf, dist) == 0
	) {
	    r += callback(nf->nf_Label, msgid, path, offsize, nf->nf_PerLineFlushOpt, strtol(headOnly, NULL, 10));
	}
    }
    return(r);
}

void
FeedFlush(void)
{
/*    NewsFeed *nf;*/

    /*
    if (FeedFo)
	fflush(FeedFo);
    */
}

int
FeedValid(const char *hlabel, int *pcount)
{
    NewsFeed *nf;

    if ((nf = NFCache) == NULL) {
	for (nf = NFBase; nf; nf = nf->nf_Next) {
	    if (strcmp(hlabel, nf->nf_Label) == 0)
		break;
	}
	NFCache = nf;
    }
    if (nf == NULL)
	return(FEED_MISSINGLABEL);
    if (nf->nf_MaxConnect && *pcount > nf->nf_MaxConnect) {
	*pcount = nf->nf_MaxConnect;
	return(FEED_MAXCONNECT);
    }
    return(0);
}

/*
 * IsFiltered() - return 0 if the article should be scrapped because
 *		  one of the gropus is filtered out, -1 otherwise.
 */

int 
IsFiltered(const char *hlabel, const char *nglist)
{
    NewsFeed *nf;
    int r;

    if (hlabel[0] == 0)
	hlabel = "DEFAULT";

    if ((nf = NFCache) == NULL) {
	for (nf = NFBase; nf; nf = nf->nf_Next) {
	    if (strcmp(hlabel, nf->nf_Label) == 0)
		break;
	}
	NFCache = nf;
    }

    if (nf == NULL) {
	r = 0;
    } else {
	r = filterQueryGroups(nf, nglist);
    }
    if (DebugOpt > 1)
	printf("IsFiltered: %d (%s,%s)\n", r, hlabel, nglist);
    return(r);
}

/*
 * Return 0 if the feed does NOT have an alias for an element in the path
 *	AND the article is not too large, otherwise return -1.
 */

int
feedQueryPaths(NewsFeed *feed, const char *npath, int size)
{
    const char *p;
    int cnt = 0;

    if (feed->nf_MaxArtSize && size > feed->nf_MaxArtSize)
	return(-1);
    if (feed->nf_MinArtSize && size <= feed->nf_MinArtSize)
	return(-1);

    while (*npath == ' ' || *npath == '\t')
	++npath;

    for (p = npath; p; p = strchr(p, '!')) {
	char pat[MAXGNAME];
	int l;

	++cnt;

	if (*p == '!')
	    ++p;
	for (l = 0; l < MAXGNAME - 1 && p[l] && p[l] != '!' && p[l] != ' ' && p[l] != '\t' && p[l] != '\n' && p[l] != '\r'; ++l)
	    ;
	strncpy(pat, p, l);
	pat[l] = 0;
	if (feedPathQuery(feed, pat) == 0)
	    break;
    }
    if (p == NULL) {
	/*
	 * no path aliases matched, we are ok... unless MaxPathLen 
	 * or MinPathLen is set.
	 */
	if (feed->nf_MaxPathLen && feed->nf_MaxPathLen < cnt)
	    return(-1);
	if (feed->nf_MinPathLen && feed->nf_MinPathLen > cnt)
	    return(-1);
	return(0);
    }
    /*
     * path alias found, skip this feed
     */

    return(-1);
}

/*
 * Return 0 if the feed has a group match against a group in the group list
 * Return -1 if there is no match
 */

int
feedQueryGroups(NewsFeed *feed, const char *nglist)
{
    const char *p;
    int r = -1;
    int count = 0;

    while (*nglist == ' ' || *nglist == '\t')
	++nglist;

    for (p = nglist; p && r != -2; p = strchr(p, ',')) {
	int l;
	int nr;
	char group[MAXGNAME];

	if (*p == ',')
	    ++p;

	for (l = 0; l < MAXGNAME - 1 && p[l] && p[l] != ',' && p[l] != ' ' && p[l] != '\t' && p[l] != '\n' && p[l] != '\r'; ++l)
	    ;

	/*
	 * r:
	 *	0	feed based on group
	 *	-1	do not feed based on group
	 *	-2	do not feed based on group if group appears at all
	 */

	strncpy(group, p, l);
	group[l] = 0;
	nr = recursiveScan(feed, offsetof(NewsFeed, nf_GroupAcceptBase), group, cbFeedGroupQuery, -1);
	++count;
	if (nr != -1) {
	    r = nr;
	}
    }
    if (r >= 0 && feed->nf_MaxCrossPost && count > feed->nf_MaxCrossPost)
	r = -1;
    if (r >= 0 && feed->nf_MinCrossPost && count < feed->nf_MinCrossPost)
	r = -1;
    if (r >= 0 && recursiveScan(feed, offsetof(NewsFeed, nf_RequireGroupsBase), nglist, cbFilterRequireGroups, 0) < 0)
	r = -2;
    return(r);
}

/*
 * Check whether a distribution is valid.  Return 0 if so, -1 if we should
 * drop the article.
 *
 * Original by larso@ifi.uio.no (=?iso-8859-1?Q?Lars_Henrik_B=F8ler_Olafsen?=),
 * this routine rewritten by Matt.
 *
 * The algorithm works like this:
 *
 *	If no feed distributions defined, Pass.
 *
 *	If positive feed distributions defined, require that we match at
 *	least one of them to return 0, else we return -1.  But if no
 *	positive feed distributions have been defined we will return 0 unless
 *	a negative feed distribution match occurs.
 *
 *	If negative feed distributions defined and we match *any* of them,
 *	we dump, even if we had other positive matches.  I.e. negative rules
 *	have precedence.
 *
 * Individual distributions cannot be more then 30 characters long.
 */

int
feedQueryDists(NewsFeed *feed, const char *dist)
{
    int r = 0;

    if (feed->nf_Dist != NULL || feed->nf_NoDist != NULL) {
	/*
	 * If match distributions exist, the default return value is -1
	 * and we MUST match something.
	 */
	int i = 0;

	if (feed->nf_Dist != NULL)
	    r = -1;
	
	while (dist[i]) {
	    int j;

	    i += strspn(dist + i, " \t\r\n,"); /* skip ws 		*/
	    j = strcspn(dist + i, " \t\r\n,"); /* parse distribution	*/
	    if (j) {
		/*
		 * &dist[i] for j characters
		 *
		 * If we find a match in nodist, that's it.
		 */
		if (feedDistQuery(dist + i, j, feed->nf_NoDist) == 0) {
		    r = -1;
		    break;
		}
		/*
		 * If we find a match in dist, set r = 0, but a nodist match
		 * later on can still dump us.
		 */
		if (feedDistQuery(dist + i, j, feed->nf_Dist) == 0) {
		    r = 0;
		}
	    }
	    i += j;

	    /*
	     * after skipping past the distribution string, the next character
	     * MUST be a comma if we are going to have more distributions, with
	     * no white space.  If someone puts whitespace here they're bozos
	     * anyway.
	     */
	    if (dist[i] != ',')
		break;
	}
    }
    return(r);
}

int
feedDistQuery(const char *item, int len, const char *list)
{
    int i = 0;
    int r = -1;

    if (list) {
	while (list[i]) {
	    int j = strcspn(list + i, ",");

	    if (j == len && strncasecmp(item, list + i, j) == 0) {
		r = 0;
		break;
	    }
	    i += j;
	    if (list[i] != ',')
		break;
	    ++i;
	}
    }
    return(r);
}

/*
 * cbFilterRequireGroups()
 *
 *	If no requiregroups nodes specified, allow any group.  Otherwise
 *	group must be in requiregroups list.  abort if def > 0.  For the
 *	recursion, if someone else sets def > 0, we are already done.
 */

int
cbFilterRequireGroups(Node *node, const void *data, int *pabort, int def)
{
    const char *nglist = data;

    if (def <= 0) {
	/*
	 * 0 -> -1, indicates that we have at leaset one requiregroup
	 * so the default has changed.. we MUST find the group.
	 */
	const char *p;

	def = -1;

	while (*nglist == ' ' || *nglist == '\t')
	    ++nglist;

	for (p = nglist; p; p = strchr(p, ',')) {
	    int l;
	    char group[MAXGNAME];

	    if (*p == ',')
		++p;

	    for (l = 0; l < MAXGNAME - 1 && p[l] && p[l] != ',' && p[l] != ' ' && p[l] != '\t' && p[l] != '\n' && p[l] != '\r'; ++l)
		;
	    strncpy(group, p, l);
	    group[l] = 0;
	    if (WildCmp(node->no_Name, group) == 0) {
		def = 1;
		break;
	    }
	}
    }
    if (def > 0)
	*pabort = 1;
    return(def);
}

/*
 * FeedQuerySpam() - scan addspam/delspam filters on NNTP-Posting-Host:
 *
 *	0 - could be either, use normal spam filter
 *	-1 - definitely spam
 *      +1 - definitely not spam
 */

int
FeedQuerySpam(const char *hlabel, const char *hostInfo)
{
    NewsFeed *feed;
    int r = 0;

    if ((feed = NFCache) == NULL) {
	for (feed = NFBase; feed; feed = feed->nf_Next) {
	    if (strcmp(hlabel, feed->nf_Label) == 0)
		break;
	}
	NFCache = feed;
    }

    if (feed) {
	r = recursiveScan(
	    feed, 
	    offsetof(NewsFeed, nf_SpamBase),
	    hostInfo, 
	    cbWildCmpNoStop,
	    r 
	);
    }

    /*
     * My directives are weird.  'delspam' means 'not spam' but has a node
     * value of -1.  'addspam' means 'spam' but has a node value of +1. 
     */

    if (r)
	r = -r;

    return(r);
}

/*
 * PathElmMatches() - match first path element against aliases.
 *		      Return -1 on failure, 0 on success.  Set
 *		      *pidx to the length of the first path element.
 */

int
PathElmMatches(const char *hlabel, const char *p, int bytes, int *pidx)
{
    NewsFeed *feed;
    int i;
    int r = 0;

    if ((feed = NFCache) == NULL) {
	for (feed = NFBase; feed; feed = feed->nf_Next) {
	    if (strcmp(hlabel, feed->nf_Label) == 0)
		break;
	}
	NFCache = feed;
    }
    for (i = 0; i < bytes && p[i] != '!'; ++i)
	;
    *pidx = i;

    if (feed && feed->nf_NoMisMatch <= 0) {
	char buf[256];

	if (i >= sizeof(buf))
	    i = sizeof(buf) - 1;
	bcopy(p, buf, i);
	buf[i] = 0;

	r = recursiveScan(
	    feed, 
	    offsetof(NewsFeed, nf_PathAliasBase),
	    buf, 
	    cbWildCmpStopWhenFound,
	    -1
	);
    }
    return(r);
}

/*
 * Return 0 if the common path element exists in the given
 * path string.
 */

int 
CommonElmMatches(const char *common, const char *p, int bytes)
{
    int l = strlen(common);

    while (bytes >= l) {
	if (strncmp(common, p, l) == 0 &&
	    (bytes == l || p[l] == '!' || p[l] == '\n')
	) {
	    return(0);
	}
	while (bytes && *p != '!') {
	    --bytes;
	    ++p;
	}
	if (bytes && *p == '!') {
	    --bytes;
	    ++p;
	}
    }
    return(-1);
}

/*
 * Write the throttling delay and line count into the appropriate
 * memory locations.
 */

void
FeedGetThrottle(const char *hlabel, int *delay, int *lines)
{
    NewsFeed *feed;

    if ((feed = NFCache) == NULL) {
	for (feed = NFBase; feed; feed = feed->nf_Next) {
	    if (strcmp(hlabel, feed->nf_Label) == 0)
		break;
	}
	NFCache = feed;
    }

    if (feed) {
	*delay = feed->nf_ThrottleDelay;
	*lines = feed->nf_ThrottleLines;
    } else {
	*delay = 0;
	*lines = 0;
    }
}

/*
 * Return 1 if read-only-dhistory ``reader'' connections are allowed,
 * 0 otherwise.
 */

int
FeedCanReadOnly(const char *hlabel)
{
    NewsFeed *feed;

    if ((feed = NFCache) == NULL) {
	for (feed = NFBase; feed; feed = feed->nf_Next) {
	    if (strcmp(hlabel, feed->nf_Label) == 0)
		break;
	}
	NFCache = feed;
    }

    if (feed)
	return feed->nf_AllowReadOnly;
    else
	return 0;
}

/*
 * filterQueryGroups()
 *
 *	0	if no filter commands matched
 *	-1	if last matching filter command was 'nofilter'
 *	+1	if last matching filter command was 'filter'
 *
 * If one group returns -1 and another returns +1, we always return +1.
 */

int
filterQueryGroups(NewsFeed *feed, const char *nglist)
{
    const char *p;
    int r = 0;

    while (*nglist == ' ' || *nglist == '\t')
	++nglist;

    for (p = nglist; p; p = strchr(p, ',')) {
	int l;
	char group[MAXGNAME];

	if (*p == ',')
	    ++p;
	for (l = 0; l < MAXGNAME - 1 && p[l] && p[l] != ',' && p[l] != ' ' && p[l] != '\t' && p[l] != '\n' && p[l] != '\r'; ++l)
	    ;
	strncpy(group, p, l);
	group[l] = 0;

	r = recursiveScan(
	    feed, 
	    offsetof(NewsFeed, nf_FilterBase),
	    group, 
	    cbWildCmpNoStop,
	    r
	);

	if (r > 0)
	    break;
    }
    return(r);
}

/*
 * cbFeedGroupQuery()
 *
 *	-2	do not feed ANY group if this group appears in the newsgroups
 *		line.
 *
 *	-1	do not feed this group
 *
 *	0	feed this group
 *
 */

int
cbFeedGroupQuery(Node *node, const void *data, int *pabort, int def)
{
    const char *group = data;

    if (WildCmp(node->no_Name, group) == 0) {
	if (node->no_Value < 0) {
	    def = node->no_Value;	/* -1 or -2 */
	} else if (node->no_Value > 0) {
	    def = 0;			/* +1	    */
	}
    }
    return(def);
}

/*
 * Return 0 if we get a match
 */

int
feedPathQuery(NewsFeed *feed, const char *path)
{
    int r = -1;

    if (feed) {
	r = recursiveScan(
	    feed, 
	    offsetof(NewsFeed, nf_PathAliasBase),
	    path, 
	    cbWildCmpStopWhenFound,
	    r 
	);
    }
    return(r);
}

time_t NFGmtMin = (time_t)-1;
time_t NFMTime = 0;

void
TouchNewsFeed(void)
{
    FILE *fi = fopen(PatLibExpand(DNewsfeedsPat), "r+");
    if (fi) {
	int c;
	if ((c = fgetc(fi)) != EOF) {
	    fseek(fi, 0L, 0);
	    fputc(c, fi);
	    fflush(fi);
	}
	fclose(fi);
    }
}

/*
 * LoadNewsFeed() - [re]load dnewsfeeds file
 */

void
LoadNewsFeed(time_t t, int force, const char *hlabel)
{
    int timeChanged = 0;

    /*
     * check for dnewsfeeds file modified once a minute
     */

    if (hlabel != (void *)-1)
	SaveHLabel = hlabel;

    if (force || t == 0 || t / 60 != NFGmtMin) {
	struct stat st = { 0 };
	FILE *fi;

	timeChanged = 1;

	if (t)
	    NFGmtMin = t / 60;

	errno = 0;

	fi = fopen(PatLibExpand(DNewsfeedsPat), "r");

	if (fi == NULL)
	    syslog(LOG_EMERG, "%s: %s", PatLibExpand(DNewsfeedsPat), strerror(errno));

	if (fi == NULL || 
	    (fstat(fileno(fi), &st) == 0 && st.st_mtime != NFMTime && !TooNear(st.st_mtime)) ||
	    force ||
	    NFMTime == 0
	) {
	    char buf[MAXGNAME+256];

	    NFMTime = st.st_mtime;	/* may be 0 if file failed to open */

	    /*
	     * flush existing feed information
	     */

	    if (DebugOpt) {
		printf("Reloading dnewsfeeds file hlabel=%s\n",
		    ((hlabel) ? hlabel : "?")
		);
	    }

	    FeedFlush();

	    /*
	     * free up feed structures
	     */

	    freePool(&NFMemPool);
	    NFCache = NULL;
	    NFGlob = NULL;
	    NFBase = NULL;
	    GLBase = NULL;

	    /*
	     * Reset MaxPerRemote if it wasn't specified on the command line.
	     *	0 = disabled
	     * -1 = disabled but there may be per-feed limits
	     * +N = some global limit
	     */

	    if (SaveHLabel == NULL && MaxPerRemote == -1)
		MaxPerRemote = 0;

	    /*
	     * load up new feed structures
	     */

	    {
		NewsFeed *nf = NULL;
		Node **paNode = NULL;
		Node **pgNode = NULL;
		Node **psNode = NULL;
		Node **pfNode = NULL;
		Node **prNode = NULL;
		int lineNo = 0;
		int inGroupDef = 0;
		FILE *altFi = NULL;

		while (fi && fgets(buf, sizeof(buf), fi) != NULL) {
		    char *s1 = strtok(buf, " \t\n");
		    char *s2 = (s1) ? strtok(NULL, " \t\n") : NULL;
		    int err = 1;

		    ++lineNo;

		    if (s1 == NULL || *s1 == '#')
			continue;

		    if (strcmp(s1, "label") == 0) {
			if (nf) {
			    syslog(LOG_CRIT, 
				"Config line %d, no end before new label!\n",
				lineNo
			    );

			    if (pfNode) {
				*pfNode = NULL;
				pfNode = NULL;
			    }
			    if (paNode) {
				*paNode = NULL;
				paNode = NULL;
			    }
			    if (pgNode) {
				*pgNode = NULL;
				pgNode = NULL;
			    }
			    if (psNode) {
				*psNode = NULL;
				psNode = NULL;
			    }
			    if (prNode) {
				*prNode = NULL;
				prNode = NULL;
			    }

			    if (nf) {
				if (inGroupDef) {
				    nf->nf_Next = GLBase;
				    GLBase = nf;
				} else {
				    nf->nf_Next = NFBase;
				    NFBase = nf;
				}
				if (altFi) {
				    AltFeed(nf, altFi);
				    fclose(altFi);
				    altFi = NULL;
				}
			    }
			    nf = NULL;
			}

			inGroupDef = 0;

			/*
			 * If we are loading a particular label, it must exist.
			 */

			if (s2 && (SaveHLabel == NULL || strcmp(s2, SaveHLabel) == 0 ||
				strcmp(s2, "GLOBAL") == 0 || strcmp(s2, "DEFAULT") == 0
			)) {
			    char path[256];

			    nf = zalloc(&NFMemPool, sizeof(NewsFeed) + strlen(s2) + 1);
			    nf->nf_Label = (char *)(nf + 1);
			    strcpy(nf->nf_Label, s2);
			    pfNode = &nf->nf_FilterBase;
			    paNode = &nf->nf_PathAliasBase;
			    pgNode = &nf->nf_GroupAcceptBase;
			    psNode = &nf->nf_SpamBase;
			    prNode = &nf->nf_RequireGroupsBase;

			    snprintf(path, sizeof(path), "%s/%s", PatExpand(FeedsHomePat), s2);
			    altFi = fcdopen(path, "r");
			    err = 0;
			    if (strcmp(nf->nf_Label, "GLOBAL") == 0)
				NFGlob = nf;
			}
		    } else if (strcmp(s1, "alias") == 0) {
			if (nf && s2) {
			    (void)MakeNodeAppList(&paNode, &NFMemPool, s2, 0);
			    err = 0;
			}
		    } else if (strcmp(s1, "adddist") == 0) {
			if (nf && s2) {
			    zappendStr(&NFMemPool, &nf->nf_Dist, ",", s2);
			    err = 0;
			}
		    } else if (strcmp(s1, "deldist") == 0) {
			if (nf && s2) {
			    zappendStr(&NFMemPool, &nf->nf_NoDist, ",", s2);
			    err = 0;
			}
		    } else if (strcmp(s1, "rtflush") == 0) {
			if (nf) {
			    err = 0;
			    nf->nf_PerLineFlushOpt = 1;
			}
		    } else if (strcmp(s1, "nortflush") == 0) {
			if (nf) {
			    err = 0;
			    nf->nf_PerLineFlushOpt = -1;
			}
		    } else if (strcmp(s1, "nospam") == 0) {
			if (nf) {
			    err = 0;
			    nf->nf_SpamFeedOpt = 1;
			}
		    } else if (strcmp(s1, "onlyspam") == 0) {
			if (nf) {
			    err = 0;
			    nf->nf_SpamFeedOpt = 2;
			}
		    } else if (strcmp(s1, "nomismatch") == 0) {
			if (nf) {
			    nf->nf_NoMisMatch = 1;
			    err = 0;
			}
		    } else if (strcmp(s1, "domismatch") == 0) {
			if (nf) {
			    nf->nf_NoMisMatch = -1;
			    err = 0;
			}
		    } else if (strcmp(s1, "filter") == 0) {
			if (nf && s2) {
			    (void)MakeNodeAppList(&pfNode, &NFMemPool, s2, 1);
			    err = 0;
			}
		    } else if (strcmp(s1, "nofilter") == 0) {
			if (nf && s2) {
			    (void)MakeNodeAppList(&pfNode, &NFMemPool, s2, -1);
			    err = 0;
			}
		    } else if (strcmp(s1, "maxconnect") == 0) {
			if (nf && s2) {
			    nf->nf_MaxConnect = strtol(s2, NULL, 0);
			    err = 0;
			    if (nf->nf_MaxConnect && MaxPerRemote == 0)
				MaxPerRemote = -1;
			}
		    } else if (strcmp(s1, "mincross") == 0) {
			if (nf && s2) {
			    nf->nf_MinCrossPost = strtol(s2, NULL, 0);
			    err = 0;
			}
		    } else if (strcmp(s1, "maxcross") == 0) {
			if (nf && s2) {
			    nf->nf_MaxCrossPost = strtol(s2, NULL, 0);
			    err = 0;
			}
		    } else if (strcmp(s1, "minpath") == 0) {
			if (nf && s2) {
			    nf->nf_MinPathLen = strtol(s2, NULL, 0);
			    err = 0;
			}
		    } else if (strcmp(s1, "maxpath") == 0) {
			if (nf && s2) {
			    nf->nf_MaxPathLen = strtol(s2, NULL, 0);
			    err = 0;
			}
		    } else if (strcmp(s1, "minsize") == 0) {
			if (nf && s2) {
			    char *p;

			    nf->nf_MinArtSize = strtol(s2, &p, 0);

			    switch(*p) {
			    case 'g':
			    case 'G':
				nf->nf_MinArtSize *= 1024;
				/* fall through */
			    case 'm':
			    case 'M':
				nf->nf_MinArtSize *= 1024;
				/* fall through */
			    case 'k':
			    case 'K':
				nf->nf_MinArtSize *= 1024;
				break;
			    }
			    err = 0;
			}
		    } else if (strcmp(s1, "maxsize") == 0) {
			if (nf && s2) {
			    char *p;

			    nf->nf_MaxArtSize = strtol(s2, &p, 0);

			    switch(*p) {
			    case 'g':
			    case 'G':
				nf->nf_MaxArtSize *= 1024;
				/* fall through */
			    case 'm':
			    case 'M':
				nf->nf_MaxArtSize *= 1024;
				/* fall through */
			    case 'k':
			    case 'K':
				nf->nf_MaxArtSize *= 1024;
				break;
			    }
			    err = 0;
			}
		    } else if (strcmp(s1, "groupdef") == 0) {
			if (nf) {
			    syslog(LOG_CRIT,
			       "Config line %d, no end before new groupdef!\n", 
				lineNo
			    );
			    if (pfNode) {
				*pfNode = NULL;
				pfNode = NULL;
			    }
			    if (paNode) {
				*paNode = NULL;
				paNode = NULL;
			    }
			    if (pgNode) {
				*pgNode = NULL;
				pgNode = NULL;
			    }
			    if (psNode) {
				*psNode = NULL;
				psNode = NULL;
			    }
			    if (prNode) {
				*prNode = NULL;
				prNode = NULL;
			    }

			    if (inGroupDef) {
				nf->nf_Next = GLBase;
				GLBase = nf;
			    } else {
				nf->nf_Next = NFBase;
				NFBase = nf;
			    }
			    if (altFi) {
				AltFeed(nf, altFi);
				fclose(altFi);
				altFi = NULL;
			    }
			    nf = NULL;
			}

			inGroupDef = 1;

			if (s2) {
			    nf = zalloc(&NFMemPool, sizeof(NewsFeed) + strlen(s2) + 1);
			    nf->nf_Label = (char *)(nf + 1);
			    strcpy(nf->nf_Label, s2);
			    pfNode = &nf->nf_FilterBase;
			    paNode = &nf->nf_PathAliasBase;
			    pgNode = &nf->nf_GroupAcceptBase;
			    psNode = &nf->nf_SpamBase;
			    prNode = &nf->nf_RequireGroupsBase;
			    err = 0;
			}
		    } else if (strcmp(s1, "groupref") == 0) {
			if (nf && s2) {
			    (void)MakeNodeAppList(&pgNode, &NFMemPool, s2, 2);
			    (void)MakeNodeAppList(&paNode, &NFMemPool, s2, 2);
			    (void)MakeNodeAppList(&psNode, &NFMemPool, s2, 2);
			    (void)MakeNodeAppList(&prNode, &NFMemPool, s2, 2);
			    (void)MakeNodeAppList(&pfNode, &NFMemPool, s2, 2);
			}
			if (nf && s2)
			    err = 0;
		    } else if (strcmp(s1, "delgroupany") == 0) {
			if (pgNode && s2) {
			    (void)MakeNodeAppList(&pgNode, &NFMemPool, s2, -2);
			}
			if (pgNode && s2)
			    err = 0;
		    } else if (strcmp(s1, "delgroup") == 0) {
			if (pgNode && s2) {
			    (void)MakeNodeAppList(&pgNode, &NFMemPool, s2, -1);
			}
			if (pgNode && s2)
			    err = 0;
		    } else if (strcmp(s1, "requiregroup") == 0) {
			if (prNode && s2) {
			    (void)MakeNodeAppList(&prNode, &NFMemPool, s2, 1);
			}
			if (pgNode && s2)
			    err = 0;
		    } else if (strcmp(s1, "addgroup") == 0) {
			if (pgNode && s2) {
			    (void)MakeNodeAppList(&pgNode, &NFMemPool, s2, 1);
			}
			if (pgNode && s2)
			    err = 0;
		    } else if (strcmp(s1, "delspam") == 0) {
			if (psNode && s2) {
			    (void)MakeNodeAppList(&psNode, &NFMemPool, s2, -1);
			}
			if (psNode && s2)
			    err = 0;
		    } else if (strcmp(s1, "addspam") == 0) {
			if (psNode && s2) {
			    (void)MakeNodeAppList(&psNode, &NFMemPool, s2, 1);
			}
			if (psNode && s2)
			    err = 0;
		    } else if (strcmp(s1, "throttle_delay") == 0) {
			if (nf && s2) {
			    nf->nf_ThrottleDelay = strtol(s2, NULL, 0);
			    err = 0;
			}
		    } else if (strcmp(s1, "throttle_lines") == 0) {
			if (nf && s2) {
			    nf->nf_ThrottleLines = strtol(s2, NULL, 0);
			    err = 0;
			}
		    } else if (strcmp(s1, "allow_readonly") == 0) {
			if (nf) {
			    nf->nf_AllowReadOnly = 1;
			    err = 0;
			}
		    } else if (strcmp(s1, "end") == 0) {
			if (nf) {
			    *paNode = NULL;
			    *pgNode = NULL;
			    *psNode = NULL;
			    *prNode = NULL;
			    *pfNode = NULL;

			    paNode = NULL;
			    pgNode = NULL;
			    psNode = NULL;
			    prNode = NULL;
			    pfNode = NULL;

			    if (inGroupDef) {
				nf->nf_Next = GLBase;
				GLBase = nf;
			    } else {
				nf->nf_Next = NFBase;
				NFBase = nf;
			    }
			    if (altFi) {
				AltFeed(nf, altFi);
				fclose(altFi);
				altFi = NULL;
			    }
			    nf = NULL;
			    err = 0;
			}
		    } else {
			syslog(LOG_CRIT, 
			    "Config line %d, unknown command\n", 
			    lineNo
			);
			err = 0;
		    }

		    /*
		     * deal with errors inside active labels (nf != NULL) or
		     * general errors when parsing the entire file (hlabel == NULL)
		     */

		    if (err && (nf != NULL || SaveHLabel == NULL)) {
			syslog(LOG_CRIT, "Config line %d, command in unexpected position or command requires argument\n", lineNo);
		    }
		}
		if (nf) {
		    *paNode = NULL;
		    *pgNode = NULL;
		    *psNode = NULL;
		    *prNode = NULL;
		    *pfNode = NULL;

		    paNode = NULL;
		    pgNode = NULL;
		    psNode = NULL;
		    prNode = NULL;
		    pfNode = NULL;

		    if (inGroupDef) {
			nf->nf_Next = GLBase;
			GLBase = nf;
		    } else {
			nf->nf_Next = NFBase;
			NFBase = nf;
		    }
		    nf = NULL;
		}
		if (altFi) {
		    fclose(altFi);
		    altFi = NULL;
		}
	    }
	}
	if (fi)
	    fclose(fi);

	/*
	 * Resolve NewsFeed references after the fact.  This allows
	 * grouplist definitions to put after newsfeed definitions.
	 */
	{
	    NewsFeed *nf;

	    for (nf = NFBase; nf; nf = nf->nf_Next) {
		resolveGroupList(nf->nf_Label, nf->nf_PathAliasBase);
		resolveGroupList(nf->nf_Label, nf->nf_GroupAcceptBase);
		resolveGroupList(nf->nf_Label, nf->nf_FilterBase);
		resolveGroupList(nf->nf_Label, nf->nf_SpamBase);
		resolveGroupList(nf->nf_Label, nf->nf_RequireGroupsBase);
	    }
	    for (nf = GLBase; nf; nf = nf->nf_Next) {
		resolveGroupList(nf->nf_Label, nf->nf_PathAliasBase);
		resolveGroupList(nf->nf_Label, nf->nf_GroupAcceptBase);
		resolveGroupList(nf->nf_Label, nf->nf_FilterBase);
		resolveGroupList(nf->nf_Label, nf->nf_SpamBase);
		resolveGroupList(nf->nf_Label, nf->nf_RequireGroupsBase);
	    }

	    /*
	     * This recursively resolves defaults for all groupdef's, we
	     * can scan any of our lists to find the groups so we use the
	     * one that is most likely to be short.
	     *
	     * We must also resolve global defaults in
	     */

	    for (nf = NFBase; nf; nf = nf->nf_Next) {
		resolveDefaults(nf, nf->nf_FilterBase);
		if (NFGlob)
		    resolveDefaults(nf, NFGlob->nf_FilterBase);
	    }

	    /*
	     * This isn't really necessary, we should have resolved them
	     * all above, but if we use groups for something else in the
	     * future we need to make sure the recursive grouprefs are
	     * resolved.
	     */
	    for (nf = GLBase; nf; nf = nf->nf_Next) {
		resolveDefaults(nf, nf->nf_FilterBase);
	    }
	}
    }
}

void
resolveGroupList(const char *label, Node *scan)
{
    for (; scan; scan = scan->no_Next) {
	if (scan->no_Value == 2) {
	    NewsFeed *gl;

	    for (gl = GLBase; gl; gl = gl->nf_Next) {
		if (strcmp(scan->no_Name, gl->nf_Label) == 0) {
		    scan->no_Data = gl;
		    break;
		}
	    }
	    if (gl == NULL) {
		syslog(LOG_CRIT, "Error: grouplist %s does not exist (from %s)\n", scan->no_Name, label);
	    }
	}
    }
}

void
AltFeed(NewsFeed *nf, FILE *fi)
{
    char buf[MAXGNAME+256];
    Node **pgNode = &nf->nf_GroupAcceptBase;
    Node **psNode = &nf->nf_SpamBase;
    Node **prNode = &nf->nf_RequireGroupsBase;

    while (*pgNode)
	pgNode = &(*pgNode)->no_Next;
    while (*psNode)
	psNode = &(*psNode)->no_Next;

    while (fgets(buf, sizeof(buf), fi) != NULL) {
	char *s1 = strtok(buf, " \t\n");
	char *s2 = strtok(NULL, " \t\n");

	if (s1 == NULL)
	    continue;
	if (s2 == NULL)
	    continue;
	if (strcmp(s1, "addgroup") == 0) {
	    (void)MakeNodeAppList(&pgNode, &NFMemPool, s2, 1);
	} else if (strcmp(s1, "delgroup") == 0) {
	    (void)MakeNodeAppList(&pgNode, &NFMemPool, s2, -1);
	} else if (strcmp(s1, "requiregroup") == 0) {
	    (void)MakeNodeAppList(&prNode, &NFMemPool, s2, 1);
	} else if (strcmp(s1, "addspam") == 0) {
	    (void)MakeNodeAppList(&psNode, &NFMemPool, s2, 1);
	} else if (strcmp(s1, "delspam") == 0) {
	    (void)MakeNodeAppList(&psNode, &NFMemPool, s2, -1);
	} else if (strcmp(s1, "delgroupany") == 0) {
	    (void)MakeNodeAppList(&pgNode, &NFMemPool, s2, -2);
	}
    }
    *pgNode = NULL;
    *psNode = NULL;
    *prNode = NULL;
}

int
TooNear(time_t t)
{
    time_t now = time(NULL);
    int32 dt = (int32)(now - t);

    if (dt > -10 && dt < 10)
	return(1);
    return(0);
}

int 
feedSpamFeedOK(int feed, int article)
{
      if (! feed) {
              return(0);
      }
      if (feed == 1) {
              if (article == 1) {
                      return(1);
              }
              return(0);
      }
      if (feed == 2) {
              if (article == 0) {
                      return(1);
              }
              return(0);
      }
      return(0);
}

/*
 * resolveDefaults() - resolve groupref recursion for simple defaults.
 *
 *	We have to recurse our resolution.  This code resolves simple
 *	defaults:  Distribution patterns, integer values, and so forth.
 */

void
resolveDefaults(NewsFeed *nf, Node *no)
{
    nf->nf_Resolved = 1;

    while (no) {
	if (no->no_Value == 2 && no->no_Data != NULL) {
	    NewsFeed *gl = no->no_Data;

	    if (gl->nf_Resolved == 0)
		resolveDefaults(gl, gl->nf_FilterBase);

	    if (!nf->nf_MaxCrossPost && gl->nf_MaxCrossPost)
		nf->nf_MaxCrossPost = gl->nf_MaxCrossPost;
	    if (!nf->nf_MinCrossPost && gl->nf_MinCrossPost)
		nf->nf_MinCrossPost = gl->nf_MinCrossPost;
	    if (!nf->nf_MaxArtSize && gl->nf_MaxArtSize)
		nf->nf_MaxArtSize = gl->nf_MaxArtSize;
	    if (!nf->nf_MinArtSize && gl->nf_MinArtSize)
		nf->nf_MinArtSize = gl->nf_MinArtSize;
	    if (!nf->nf_MaxPathLen && gl->nf_MaxPathLen)
		nf->nf_MaxPathLen = gl->nf_MaxPathLen;
	    if (!nf->nf_MinPathLen && gl->nf_MinPathLen)
		nf->nf_MinPathLen = gl->nf_MinPathLen;
	    if (!nf->nf_MaxConnect && gl->nf_MaxConnect)
		nf->nf_MaxConnect = gl->nf_MaxConnect;

	    if (nf->nf_Dist == NULL && gl->nf_Dist)
		zappendStr(&NFMemPool, &nf->nf_Dist, NULL, gl->nf_Dist);
	    if (nf->nf_NoDist == NULL && gl->nf_NoDist)
		zappendStr(&NFMemPool, &nf->nf_NoDist, NULL, gl->nf_NoDist);

	    if (!nf->nf_PerLineFlushOpt && gl->nf_PerLineFlushOpt)
		nf->nf_PerLineFlushOpt = gl->nf_PerLineFlushOpt;
	    if (!nf->nf_NoMisMatch && gl->nf_NoMisMatch)
		nf->nf_NoMisMatch = gl->nf_NoMisMatch;
	    if (!nf->nf_SpamFeedOpt && gl->nf_SpamFeedOpt)
		nf->nf_SpamFeedOpt = gl->nf_SpamFeedOpt;

	    if (!nf->nf_ThrottleDelay && gl->nf_ThrottleDelay)
		nf->nf_ThrottleDelay = gl->nf_ThrottleDelay;
	    if (!nf->nf_ThrottleLines && gl->nf_ThrottleLines)
		nf->nf_ThrottleLines = gl->nf_ThrottleLines;
	    if (!nf->nf_AllowReadOnly && gl->nf_AllowReadOnly)
		nf->nf_AllowReadOnly = gl->nf_AllowReadOnly;
	}
	no = no->no_Next;
    }
}

int
recursiveScan(NewsFeed *feed, int off, const void *data, int (*callback)(Node *node, const void *data, int *pabort, int def), int def)
{
    int r = def;
    int abortMe = 0;
    Node *node;

    if (NFGlob && NFGlob != feed)
	r = recursiveScan(NFGlob, off, data, callback, r);

    for (node = *(Node **)((char *)feed + off); node; node = node->no_Next) {
	if (node->no_Value == 2) {
	    if (++RecurCount == MAXRECURSION) {
		if (RecurWarn == 0) {
		    RecurWarn = 1;
		    syslog(LOG_EMERG, "Infinite recursion in dnewsfeeds file!");
		}
	    } else {
		if (node->no_Data != NULL)
		    r = recursiveScan(node->no_Data, off, data, callback, r);
	    } 
	    --RecurCount;
	} else {
	    r = callback(node, data, &abortMe, r);
	}
	if (abortMe)
	    break;
    }
    return(r);
}

int
cbWildCmpStopWhenFound(Node *node, const void *data, int *pabort, int def)
{
    if (WildCmp(node->no_Name, (char *)data) == 0) {
	def = node->no_Value;
	*pabort = 1;
    }
    return(def);
}

int
cbWildCmpNoStop(Node *node, const void *data, int *pabort, int def)
{
    if (WildCmp(node->no_Name, (char *)data) == 0) {
	def = node->no_Value;
    }
    return(def);
}

