;==============================================================================

; file: "_num1.scm"

; Copyright (C) 1994-1998 by Marc Feeley, All Rights Reserved.

(##include "header.scm")

;------------------------------------------------------------------------------

; Numerical type predicates

(define (##complex? x)
  (number-dispatch x #f #t #t #t #t #t))

(define (number? x)  (force-vars (x) (##complex? x)))
(define (complex? x) (force-vars (x) (##complex? x)))

(define (##real? x)
  (number-dispatch x #f #t #t #t #t (cpxnum-real? x)))

(define (real? x) (force-vars (x) (##real? x)))

(define (##rational? x)
  (number-dispatch x #f #t #t #t (flonum-rational? x) (cpxnum-rational? x)))

(define (rational? x) (force-vars (x) (##rational? x)))

(define (##integer? x)
  (number-dispatch x #f #t #t #f (flonum-int? x) (cpxnum-int? x)))

(define (integer? x) (force-vars (x) (##integer? x)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Exactness predicates

(define (##exact? x)

  (define (type-error) #f)

  (number-dispatch x (type-error) #t #t #t #f
    (and (##not (##flonum? (cpxnum-real x)))
         (##not (##flonum? (cpxnum-imag x))))))

(define (exact? x)
  (force-vars (x)
    (let ()

      (define (type-error)
        (##trap-check-number 'exact? x))

      (number-dispatch x (type-error) #t #t #t #f
        (and (##not (##flonum? (cpxnum-real x)))
             (##not (##flonum? (cpxnum-imag x))))))))

(define (inexact? x)
  (force-vars (x)
    (let ()

      (define (type-error)
        (##trap-check-number 'inexact? x))

      (number-dispatch x (type-error) #f #f #f #t
        (or (##flonum? (cpxnum-real x))
            (##flonum? (cpxnum-imag x)))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Numerical comparison predicates

(define (##eqv? x y)
  (number-dispatch x (##eq? x y)
    (and (##fixnum? y) (##fixnum.= x y))
    (and (##bignum? y) (##bignum.= x y))
    (and (##ratnum? y) (##ratnum.= x y))
    (and (##flonum? y) (##bvector=? x y))
    (and (##cpxnum? y)
         (##eqv? (cpxnum-real x) (cpxnum-real y))
         (##eqv? (cpxnum-imag x) (cpxnum-imag y)))))

(define (##= x y)

  (##define-macro (type-error) ''())

  (number-dispatch x (type-error)

    (number-dispatch y (type-error) ; x = fixnum
      (##fixnum.= x y)
      #f
      #f
      (##flonum.= (##flonum.<-fixnum x) y) ; assuming exact conversion
      ;alternative: (and (##flonum.finite? y)
      ;                  (##ratnum.= (##ratnum.<-exact-int x) (##flonum.->ratnum y)))
      (##cpxnum.= (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = bignum
      #f
      (##bignum.= x y)
      #f
      (and (##flonum.finite? y)
           (##ratnum.= (##ratnum.<-exact-int x) (##flonum.->ratnum y)))
      (##cpxnum.= (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = ratnum
      #f
      #f
      (##ratnum.= x y)
      (and (##flonum.finite? y)
           (##ratnum.= x (##flonum.->ratnum y)))
      (##cpxnum.= (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = flonum
      (##flonum.= x (##flonum.<-fixnum y)) ; assuming exact conversion
      ;alternative: (and (##flonum.finite? x)
      ;                  (##ratnum.= (##flonum.->ratnum x) (##ratnum.<-exact-int y)))
      (and (##flonum.finite? x)
           (##ratnum.= (##flonum.->ratnum x) (##ratnum.<-exact-int y)))
      (and (##flonum.finite? x)
           (##ratnum.= (##flonum.->ratnum x) y))
      (##flonum.= x y)
      (##cpxnum.= (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = cpxnum
      (##cpxnum.= x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.= x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.= x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.= x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.= x y))))

(define-nary-bool (= x y)
  #t
  (if (##complex? x) #t '())
  (##= x y)
  force-vars
  no-check
  (##null? ##trap-check-number*))

(define (##< x y #!optional (nan-result #f))

  (##define-macro (type-error) ''())

  (number-dispatch x (type-error)

    (number-dispatch y (type-error) ; x = fixnum
      (##fixnum.< x y)
      (bignum-positive? y)
      (##ratnum.< (##ratnum.<-exact-int x) y)
      (cond ((##flonum.finite? y)
             (##flonum.< (##flonum.<-fixnum x) y)) ; assuming exact conversion
             ;alternative: (##ratnum.< (##ratnum.<-exact-int x) (##flonum.->ratnum y))
            ((##flonum.nan? y)
             nan-result)
            (else
             (##flonum.positive? y)))
      (if (cpxnum-real? y)
        (##< x (cpxnum-real y) nan-result)
        (type-error)))

    (number-dispatch y (type-error) ; x = bignum
      (bignum-negative? x)
      (##bignum.< x y)
      (##ratnum.< (##ratnum.<-exact-int x) y)
      (cond ((##flonum.finite? y)
             (##ratnum.< (##ratnum.<-exact-int x) (##flonum.->ratnum y)))
            ((##flonum.nan? y)
             nan-result)
            (else
             (##flonum.positive? y)))
      (if (cpxnum-real? y)
        (##< x (cpxnum-real y) nan-result)
        (type-error)))

    (number-dispatch y (type-error) ; x = ratnum
      (##ratnum.< x (##ratnum.<-exact-int y))
      (##ratnum.< x (##ratnum.<-exact-int y))
      (##ratnum.< x y)
      (cond ((##flonum.finite? y)
             (##ratnum.< x (##flonum.->ratnum y)))
            ((##flonum.nan? y)
             nan-result)
            (else
             (##flonum.positive? y)))
      (if (cpxnum-real? y)
        (##< x (cpxnum-real y) nan-result)
        (type-error)))

    (number-dispatch y (type-error) ; x = flonum
      (cond ((##flonum.finite? x)
             (##flonum.< x (##flonum.<-fixnum y))) ; assuming exact conversion
             ;alternative: (##ratnum.< (##flonum.->ratnum x) (##ratnum.<-exact-int y))
            ((##flonum.nan? x)
             nan-result)
            (else
             (##flonum.negative? x)))
      (cond ((##flonum.finite? x)
             (##ratnum.< (##flonum.->ratnum x) (##ratnum.<-exact-int y)))
            ((##flonum.nan? x)
             nan-result)
            (else
             (##flonum.negative? x)))
      (cond ((##flonum.finite? x)
             (##ratnum.< (##flonum.->ratnum x) y))
            ((##flonum.nan? x)
             nan-result)
            (else
             (##flonum.negative? x)))
      (if (or (##flonum.nan? x) (##flonum.nan? y))
        nan-result
        (##flonum.< x y))
      (if (cpxnum-real? y)
        (##< x (cpxnum-real y) nan-result)
        (type-error)))

    (if (cpxnum-real? x) ; x = cpxnum
      (number-dispatch y (type-error)
        (##< (cpxnum-real x) y nan-result)
        (##< (cpxnum-real x) y nan-result)
        (##< (cpxnum-real x) y nan-result)
        (##< (cpxnum-real x) y nan-result)
        (if (cpxnum-real? y)
          (##< (cpxnum-real x) (cpxnum-real y) nan-result)
          (type-error)))
      (type-error))))

(define-nary-bool (< x y)
  #t
  (if (##real? x) #t '())
  (##< x y #f)
  force-vars
  no-check
  (##null? ##trap-check-real*))

(define-nary-bool (> x y)
  #t
  (if (##real? x) #t '())
  (##< y x #f)
  force-vars
  no-check
  (##null? ##trap-check-real*))

(define-nary-bool (<= x y)
  #t
  (if (##real? x) #t '())
  (##not (##< y x #t))
  force-vars
  no-check
  (##null? ##trap-check-real*))

(define-nary-bool (>= x y)
  #t
  (if (##real? x) #t '())
  (##not (##< x y #t))
  force-vars
  no-check
  (##null? ##trap-check-real*))

(define (##zero? x)

  (define (type-error)
    (##trap-check-number 'zero? x))

  (number-dispatch x (type-error)
    (##fixnum.zero? x)
    #f
    #f
    (##flonum.zero? x)
    (and (let ((imag (cpxnum-imag x)))
           (and (##flonum? imag) (##flonum.zero? imag)))
         (let ((real (cpxnum-real x)))
           (if (##fixnum? real)
             (##fixnum.zero? real)
             (and (##flonum? real) (##flonum.zero? real)))))))

(define (zero? x) (force-vars (x) (##zero? x)))

(define (##positive? x)

  (define (type-error)
    (##trap-check-real 'positive? x))

  (number-dispatch x (type-error)
    (##fixnum.positive? x)
    (bignum-positive? x)
    (##positive? (ratnum-numerator x))
    (##flonum.positive? x)
    (if (cpxnum-real? x)
      (##positive? (cpxnum-real x))
      (type-error))))

(define (positive? x) (force-vars (x) (##positive? x)))

(define (##negative? x)

  (define (type-error)
    (##trap-check-real 'negative? x))

  (number-dispatch x (type-error)
    (##fixnum.negative? x)
    (bignum-negative? x)
    (##negative? (ratnum-numerator x))
    (##flonum.negative? x)
    (if (cpxnum-real? x)
      (##negative? (cpxnum-real x))
      (type-error))))

(define (negative? x) (force-vars (x) (##negative? x)))

(define (odd? x)
  (force-vars (x)
    (let ()

      (define (type-error)
        (##trap-check-integer 'odd? x))

      (define (exact-int-odd? x)
        (if (##fixnum? x)
          (##fixnum.odd? x)
          (bignum-odd? x)))

      (number-dispatch x (type-error)
        (##fixnum.odd? x)
        (bignum-odd? x)
        (type-error)
        (if (flonum-int? x)
          (exact-int-odd? (##flonum.inexact->exact x))
          (type-error))
        (if (cpxnum-int? x)
          (exact-int-odd? (##inexact->exact (cpxnum-real x)))
          (type-error))))))

(define (even? x)
  (force-vars (x)
    (let ()

      (define (type-error)
        (##trap-check-integer 'even? x))

      (define (exact-int-even? x)
        (if (##fixnum? x)
          (##not (##fixnum.odd? x))
          (##not (bignum-odd? x))))

      (number-dispatch x (type-error)
        (##not (##fixnum.odd? x))
        (##not (bignum-odd? x))
        (type-error)
        (if (flonum-int? x)
          (exact-int-even? (##flonum.inexact->exact x))
          (type-error))
        (if (cpxnum-int? x)
          (exact-int-even? (##inexact->exact (cpxnum-real x)))
          (type-error))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Max and min

(define (##max x y)

  (##define-macro (type-error) ''())

  (number-dispatch x (type-error)

    (number-dispatch y (type-error) ; x = fixnum
      (##fixnum.max x y)
      (if (##< x y) y x)
      (if (##< x y) y x)
      (##flonum.max (##flonum.<-fixnum x) y) ; assuming exact conversion
      (if (cpxnum-real? y)
        (##max x (cpxnum-real y))
        (type-error)))

    (number-dispatch y (type-error) ; x = bignum
      (if (##< x y) y x)
      (if (##< x y) y x)
      (if (##< x y) y x)
      (##flonum.max (##flonum.<-bignum x) y)
      (if (cpxnum-real? y)
        (##max x (cpxnum-real y))
        (type-error)))

    (number-dispatch y (type-error) ; x = ratnum
      (if (##< x y) y x)
      (if (##< x y) y x)
      (if (##< x y) y x)
      (##flonum.max (##flonum.<-ratnum x) y)
      (if (cpxnum-real? y)
        (##max x (cpxnum-real y))
        (type-error)))

    (number-dispatch y (type-error)
      (##flonum.max x (##flonum.<-fixnum y)) ; assuming exact conversion
      (##flonum.max x (##flonum.<-bignum y))
      (##flonum.max x (##flonum.<-ratnum y))
      (##flonum.max x y)
      (if (cpxnum-real? y)
        (##max x (cpxnum-real y))
        (type-error)))

    (if (cpxnum-real? x) ; x = cpxnum
      (number-dispatch y (type-error)
        (##max (cpxnum-real x) y)
        (##max (cpxnum-real x) y)
        (##max (cpxnum-real x) y)
        (##max (cpxnum-real x) y)
        (if (cpxnum-real? y)
          (##max (cpxnum-real x) (cpxnum-real y))
          (type-error)))
      (type-error))))

(define-nary (max x y)
  ()
  (if (##real? x) x '())
  (##max x y)
  force-vars
  no-check
  (##null? ##trap-check-real*))

(define (##min x y)

  (##define-macro (type-error) ''())

  (number-dispatch x (type-error)

    (number-dispatch y (type-error) ; x = fixnum
      (##fixnum.min x y)
      (if (##< x y) x y)
      (if (##< x y) x y)
      (##flonum.min (##flonum.<-fixnum x) y) ; assuming exact conversion
      (if (cpxnum-real? y)
        (##min x (cpxnum-real y))
        (type-error)))

    (number-dispatch y (type-error) ; x = bignum
      (if (##< x y) x y)
      (if (##< x y) x y)
      (if (##< x y) x y)
      (##flonum.min (##flonum.<-bignum x) y)
      (if (cpxnum-real? y)
        (##min x (cpxnum-real y))
        (type-error)))

    (number-dispatch y (type-error) ; x = ratnum
      (if (##< x y) x y)
      (if (##< x y) x y)
      (if (##< x y) x y)
      (##flonum.min (##flonum.<-ratnum x) y)
      (if (cpxnum-real? y)
        (##min x (cpxnum-real y))
        (type-error)))

    (number-dispatch y (type-error)
      (##flonum.min x (##flonum.<-fixnum y)) ; assuming exact conversion
      (##flonum.min x (##flonum.<-bignum y))
      (##flonum.min x (##flonum.<-ratnum y))
      (##flonum.min x y)
      (if (cpxnum-real? y)
        (##min x (cpxnum-real y))
        (type-error)))

    (if (cpxnum-real? x) ; x = cpxnum
      (number-dispatch y (type-error)
        (##min (cpxnum-real x) y)
        (##min (cpxnum-real x) y)
        (##min (cpxnum-real x) y)
        (##min (cpxnum-real x) y)
        (if (cpxnum-real? y)
          (##min (cpxnum-real x) (cpxnum-real y))
          (type-error)))
      (type-error))))

(define-nary (min x y)
  ()
  (if (##real? x) x '())
  (##min x y)
  force-vars
  no-check
  (##null? ##trap-check-real*))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; +, *, -, /

(define (##+ x y)

  (##define-macro (type-error) ''())

  (number-dispatch x (type-error)

    (number-dispatch y (type-error) ; x = fixnum
      (if (##fixnum.negative? x)
        (if (##fixnum.negative? y)
          (let ((r (##fixnum.+ x y)))
            (if (##fixnum.negative? r)
              r
              (##bignum.+/bignum-fixnum ##bignum.2*min-fixnum32 r)))
          (##fixnum.+ x y))
        (if (##fixnum.negative? y)
          (##fixnum.+ x y)
          (let ((r (##fixnum.+ x y)))
            (if (##fixnum.negative? r)
              (##bignum.-/fixnum-bignum r ##bignum.2*min-fixnum32)
              r))))
      (##bignum.+/bignum-fixnum y x)
      (##ratnum.+ (##ratnum.<-exact-int x) y)
      (if #f;;;(##fixnum.zero? x)
        y
        (##flonum.+ (##flonum.<-fixnum x) y)) ; assuming exact conversion
      (##cpxnum.+ (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = bignum
      (##bignum.+/bignum-fixnum x y)
      (##bignum.+ x y)
      (##ratnum.+ (##ratnum.<-exact-int x) y)
      (##flonum.+ (##flonum.<-bignum x) y)
      (##cpxnum.+ (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = ratnum
      (##ratnum.+ x (##ratnum.<-exact-int y))
      (##ratnum.+ x (##ratnum.<-exact-int y))
      (##ratnum.+ x y)
      (##flonum.+ (##flonum.<-ratnum x) y)
      (##cpxnum.+ (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = flonum
      (if #f;;;(##fixnum.zero? y)
        x
        (##flonum.+ x (##flonum.<-fixnum y))) ; assuming exact conversion
      (##flonum.+ x (##flonum.<-bignum y))
      (##flonum.+ x (##flonum.<-ratnum y))
      (##flonum.+ x y)
      (##cpxnum.+ (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = cpxnum
      (##cpxnum.+ x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.+ x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.+ x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.+ x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.+ x y))))

(define-nary (+ x y)
  0
  (if (##complex? x) x '())
  (##+ x y)
  force-vars
  no-check
  (##null? ##trap-check-number*))

(define (##* x y)

  (##define-macro (type-error) ''())

  (number-dispatch x (type-error)

    (number-dispatch y (type-error)
      (cond ((and (##not (##fixnum.< x (min-fixnum32-div-radix)))
                  (##not (##fixnum.< (max-fixnum32-div-radix) x))
                  (##fixnum.< (minus-radix) y)
                  (##not (##fixnum.< (radix) y)))
             (##fixnum.* x y))
            ((or (##fixnum.zero? x) (##fixnum.zero? y))
             0)
            ((##fixnum.= x 1)
             y)
            ((##fixnum.= y 1)
             x)
            (else
             (##bignum.* (##bignum.<-fixnum x) (##bignum.<-fixnum y))))
      (##bignum.*/bignum-fixnum y x)
      (##ratnum.* (##ratnum.<-exact-int x) y)
      (if #f;;;(##fixnum.zero? x)
        x
        (##flonum.* (##flonum.<-fixnum x) y)) ; assuming exact conversion
      (##cpxnum.* (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = bignum
      (##bignum.*/bignum-fixnum x y)
      (##bignum.* x y)
      (##ratnum.* (##ratnum.<-exact-int x) y)
      (##flonum.* (##flonum.<-bignum x) y)
      (##cpxnum.* (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = ratnum
      (##ratnum.* x (##ratnum.<-exact-int y))
      (##ratnum.* x (##ratnum.<-exact-int y))
      (##ratnum.* x y)
      (##flonum.* (##flonum.<-ratnum x) y)
      (##cpxnum.* (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = flonum
      (if #f;;;(##fixnum.zero? y)
        y
        (##flonum.* x (##flonum.<-fixnum y))) ; assuming exact conversion
      (##flonum.* x (##flonum.<-bignum y))
      (##flonum.* x (##flonum.<-ratnum y))
      (##flonum.* x y)
      (##cpxnum.* (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = cpxnum
      (##cpxnum.* x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.* x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.* x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.* x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.* x y))))

(define-nary (* x y)
  1
  (if (##complex? x) x '())
  (##* x y)
  force-vars
  no-check
  (##null? ##trap-check-number*))

(define (##negate x)

  (##define-macro (type-error) ''())

  (number-dispatch x (type-error)
    (if (##fixnum.negative? x)
      (let ((r (##fixnum.- x)))
        (if (##fixnum.negative? r)
          (##bignum.-/fixnum-bignum r ##bignum.2*min-fixnum32)
          r))
      (##fixnum.- x))
    (##bignum.-/fixnum-bignum 0 x)
    (ratnum-make (##negate (ratnum-numerator x)) (ratnum-denominator x))
    (##flonum.- x)
    (##make-rectangular (##negate (cpxnum-real x))
                        (##negate (cpxnum-imag x)))))

(define (##- x y)

  (##define-macro (type-error) ''())

  (number-dispatch x (type-error)

    (number-dispatch y (type-error) ; x = fixnum
      (if (##fixnum.negative? x)
        (if (##fixnum.negative? y)
          (##fixnum.- x y)
          (let ((r (##fixnum.- x y)))
            (if (##fixnum.negative? r)
              r
              (##bignum.+/bignum-fixnum ##bignum.2*min-fixnum32 r))))
        (if (##fixnum.negative? y)
          (let ((r (##fixnum.- x y)))
            (if (##fixnum.negative? r)
              (##bignum.-/fixnum-bignum r ##bignum.2*min-fixnum32)
              r))
          (##fixnum.- x y)))
      (##bignum.-/fixnum-bignum x y)
      (##ratnum.- (##ratnum.<-exact-int x) y)
      (if #f;;;(##fixnum.zero? x)
        (##flonum.- y)
        (##flonum.- (##flonum.<-fixnum x) y)) ; assuming exact conversion
      (##cpxnum.- (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = bignum
      (##bignum.-/bignum-fixnum x y)
      (##bignum.- x y)
      (##ratnum.- (##ratnum.<-exact-int x) y)
      (##flonum.- (##flonum.<-bignum x) y)
      (##cpxnum.- (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = ratnum
      (##ratnum.- x (##ratnum.<-exact-int y))
      (##ratnum.- x (##ratnum.<-exact-int y))
      (##ratnum.- x y)
      (##flonum.- (##flonum.<-ratnum x) y)
      (##cpxnum.- (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = flonum
      (if #f;;;(##fixnum.zero? y)
        x
        (##flonum.- x (##flonum.<-fixnum y))) ; assuming exact conversion
      (##flonum.- x (##flonum.<-bignum y))
      (##flonum.- x (##flonum.<-ratnum y))
      (##flonum.- x y)
      (##cpxnum.- (##cpxnum.<-non-cpxnum x) y))

    (number-dispatch y (type-error) ; x = cpxnum
      (##cpxnum.- x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.- x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.- x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.- x (##cpxnum.<-non-cpxnum y))
      (##cpxnum.- x y))))

(define-nary (- x y)
  ()
  (##negate x)
  (##- x y)
  force-vars
  no-check
  (##null? ##trap-check-number*))

(define (##/ x y)

  (##define-macro (type-error) ''())

  (##define-macro (divide-by-zero-error) '#f)

  (number-dispatch y (type-error)

    (number-dispatch x (type-error) ; y = fixnum
      (if (##fixnum.zero? y)
        (divide-by-zero-error)
        (##ratnum./ (##ratnum.<-exact-int x) (##ratnum.<-exact-int y)))
      (if (##fixnum.zero? y)
        (divide-by-zero-error)
        (##ratnum./ (##ratnum.<-exact-int x) (##ratnum.<-exact-int y)))
      (if (##fixnum.zero? y)
        (divide-by-zero-error)
        (##ratnum./ x (##ratnum.<-exact-int y)))
      (if (##fixnum.zero? y)
        (divide-by-zero-error)
        (##flonum./ x (##flonum.<-fixnum y))) ; assuming exact conversion
      (if (##fixnum.zero? y)
        (divide-by-zero-error)
        (##cpxnum./ x (##cpxnum.<-non-cpxnum y))))

    (number-dispatch x (type-error) ; y = bignum
      (##ratnum./ (##ratnum.<-exact-int x) (##ratnum.<-exact-int y))
      (##ratnum./ (##ratnum.<-exact-int x) (##ratnum.<-exact-int y))
      (##ratnum./ x (##ratnum.<-exact-int y))
      (##flonum./ x (##flonum.<-bignum y))
      (##cpxnum./ x (##cpxnum.<-non-cpxnum y)))

    (number-dispatch x (type-error) ; y = ratnum
      (##ratnum./ (##ratnum.<-exact-int x) y)
      (##ratnum./ (##ratnum.<-exact-int x) y)
      (##ratnum./ x y)
      (##flonum./ x (##flonum.<-ratnum y))
      (##cpxnum./ x (##cpxnum.<-non-cpxnum y)))

    (number-dispatch x (type-error) ; y = flonum, no error possible
      (if #f;;;(##fixnum.zero? x)
        x
        (##flonum./ (##flonum.<-fixnum x) y)) ; assuming exact conversion
      (##flonum./ (##flonum.<-bignum x) y)
      (##flonum./ (##flonum.<-ratnum x) y)
      (##flonum./ x y)
      (##cpxnum./ x (##cpxnum.<-non-cpxnum y)))

    (number-dispatch x (type-error) ; y = cpxnum
      (##cpxnum./ (##cpxnum.<-non-cpxnum x) y)
      (##cpxnum./ (##cpxnum.<-non-cpxnum x) y)
      (##cpxnum./ (##cpxnum.<-non-cpxnum x) y)
      (##cpxnum./ (##cpxnum.<-non-cpxnum x) y)
      (##cpxnum./ x y))))

(define-nary (/ x y)
  ()
  (##/ 1 x)
  (##/ x y)
  force-vars
  no-check
  (##null? ##trap-check-number*)
  (##not ##trap-divide-by-zero*))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; abs

(define (##abs x)

  (define (type-error)
    (##trap-check-real 'abs x))

  (number-dispatch x (type-error)
    (if (##fixnum.negative? x)
      (##bignum.-/fixnum-fixnum 0 x)
      x)
    (if (bignum-negative? x)
      (let ((r (##bignum.copy x)))
        (bignum-set-positive! r)
        r)
      x)
    (if (##negative? (ratnum-numerator x))
      (ratnum-make (##negate (ratnum-numerator x)) (ratnum-denominator x))
      x)
    (##flonum.abs x)
    (if (cpxnum-real? x)
      (##make-rectangular (##abs (cpxnum-real x))
                          (##abs (cpxnum-imag x)))
      (type-error))))

(define (abs x) (force-vars (x) (##abs x)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; quotient, remainder, modulo

(define (##quotient x y)

  (define (type-error)
    (##trap-check-integer 'quotient x y))

  (define (divide-by-zero-error)
    (##trap-divide-by-zero 'quotient x y))

  (define (inexact-quotient)
    (##exact->inexact (##quotient (##inexact->exact x) (##inexact->exact y))))

  (number-dispatch y (type-error)

    (number-dispatch x (type-error) ; y = fixnum
      (if (##fixnum.zero? y)
        (divide-by-zero-error)
        (if (##fixnum.= y -1)
          (##bignum.-/fixnum-fixnum 0 x)
          (##fixnum.quotient x y)))
      (if (##fixnum.zero? y)
        (divide-by-zero-error)
        (##bignum.quotient/bignum-fixnum x y))
      (type-error)
      (if (flonum-int? x)
        (if (##fixnum.zero? y)
          (divide-by-zero-error)
          (inexact-quotient))
        (type-error))
      (if (cpxnum-int? x)
        (if (##fixnum.zero? y)
          (divide-by-zero-error)
          (inexact-quotient))
        (type-error)))

    (number-dispatch x (type-error) ; y = bignum
      (##bignum.quotient/fixnum-bignum x y)
      (##bignum.quotient x y)
      (type-error)
      (if (flonum-int? x)
        (inexact-quotient)
        (type-error))
      (if (cpxnum-int? x)
        (inexact-quotient)
        (type-error)))

    (type-error) ; y = ratnum

    (number-dispatch x (type-error) ; y = flonum
      (if (flonum-int? y)
        (inexact-quotient)
        (type-error))
      (if (flonum-int? y)
        (inexact-quotient)
        (type-error))
      (type-error)
      (if (flonum-int? x)
        (if (flonum-int? y)
          (inexact-quotient)
          (type-error))
        (type-error))
      (if (cpxnum-int? x)
        (if (flonum-int? y)
          (inexact-quotient)
          (type-error))
        (type-error)))

    (number-dispatch x (type-error) ; y = cpxnum
      (if (cpxnum-int? y)
        (inexact-quotient)
        (type-error))
      (if (cpxnum-int? y)
        (inexact-quotient)
        (type-error))
      (type-error)
      (if (flonum-int? x)
        (if (cpxnum-int? y)
          (inexact-quotient)
          (type-error))
        (type-error))
      (if (cpxnum-int? x)
        (if (cpxnum-int? y)
          (inexact-quotient)
          (type-error))
        (type-error)))))

(define (quotient x y) (force-vars (x y) (##quotient x y)))

(define (##remainder x y)

  (define (type-error)
    (##trap-check-integer 'remainder x y))

  (define (divide-by-zero-error)
    (##trap-divide-by-zero 'remainder x y))

  (define (inexact-remainder)
    (##exact->inexact (##remainder (##inexact->exact x) (##inexact->exact y))))

  (number-dispatch y (type-error)

    (number-dispatch x (type-error) ; y = fixnum
      (if (##fixnum.zero? y)
        (divide-by-zero-error)
        (##fixnum.remainder x y))
      (if (##fixnum.zero? y)
        (divide-by-zero-error)
        (##bignum.remainder/bignum-fixnum x y))
      (type-error)
      (if (flonum-int? x)
        (if (##fixnum.zero? y)
          (divide-by-zero-error)
          (inexact-remainder))
        (type-error))
      (if (cpxnum-int? x)
        (if (##fixnum.zero? y)
          (divide-by-zero-error)
          (inexact-remainder))
        (type-error)))

    (number-dispatch x (type-error) ; y = bignum
      (##bignum.remainder/fixnum-bignum x y)
      (##bignum.remainder x y)
      (type-error)
      (if (flonum-int? x)
        (inexact-remainder)
        (type-error))
      (if (cpxnum-int? x)
        (inexact-remainder)
        (type-error)))

    (type-error) ; y = ratnum

    (number-dispatch x (type-error) ; y = flonum
      (if (flonum-int? y)
        (inexact-remainder)
        (type-error))
      (if (flonum-int? y)
        (inexact-remainder)
        (type-error))
      (type-error)
      (if (flonum-int? x)
        (if (flonum-int? y)
          (inexact-remainder)
          (type-error))
        (type-error))
      (if (cpxnum-int? x)
        (if (flonum-int? y)
          (inexact-remainder)
          (type-error))
        (type-error)))

    (number-dispatch x (type-error) ; y = cpxnum
      (if (cpxnum-int? y)
        (inexact-remainder)
        (type-error))
      (if (cpxnum-int? y)
        (inexact-remainder)
        (type-error))
      (type-error)
      (if (flonum-int? x)
        (if (cpxnum-int? y)
          (inexact-remainder)
          (type-error))
        (type-error))
      (if (cpxnum-int? x)
        (if (cpxnum-int? y)
          (inexact-remainder)
          (type-error))
        (type-error)))))

(define (remainder x y) (force-vars (x y) (##remainder x y)))

(define (##modulo x y)

  (define (type-error)
    (##trap-check-integer 'modulo x y))

  (define (divide-by-zero-error)
    (##trap-divide-by-zero 'modulo x y))

  (define (inexact-modulo)
    (##exact->inexact (##modulo (##inexact->exact x) (##inexact->exact y))))

  (number-dispatch y (type-error)

    (number-dispatch x (type-error) ; y = fixnum
      (if (##fixnum.zero? y)
        (divide-by-zero-error)
        (##fixnum.modulo x y))
      (if (##fixnum.zero? y)
        (divide-by-zero-error)
        (##bignum.modulo/bignum-fixnum x y))
      (type-error)
      (if (flonum-int? x)
        (if (##fixnum.zero? y)
          (divide-by-zero-error)
          (inexact-modulo))
        (type-error))
      (if (cpxnum-int? x)
        (if (##fixnum.zero? y)
          (divide-by-zero-error)
          (inexact-modulo))
        (type-error)))

    (number-dispatch x (type-error) ; y = bignum
      (##bignum.modulo/fixnum-bignum x y)
      (##bignum.modulo x y)
      (type-error)
      (if (flonum-int? x)
        (inexact-modulo)
        (type-error))
      (if (cpxnum-int? x)
        (inexact-modulo)
        (type-error)))

    (type-error) ; y = ratnum

    (number-dispatch x (type-error) ; y = flonum
      (if (flonum-int? y)
        (inexact-modulo)
        (type-error))
      (if (flonum-int? y)
        (inexact-modulo)
        (type-error))
      (type-error)
      (if (flonum-int? x)
        (if (flonum-int? y)
          (inexact-modulo)
          (type-error))
        (type-error))
      (if (cpxnum-int? x)
        (if (flonum-int? y)
          (inexact-modulo)
          (type-error))
        (type-error)))

    (number-dispatch x (type-error) ; y = cpxnum
      (if (cpxnum-int? y)
        (inexact-modulo)
        (type-error))
      (if (cpxnum-int? y)
        (inexact-modulo)
        (type-error))
      (type-error)
      (if (flonum-int? x)
        (if (cpxnum-int? y)
          (inexact-modulo)
          (type-error))
        (type-error))
      (if (cpxnum-int? x)
        (if (cpxnum-int? y)
          (inexact-modulo)
          (type-error))
        (type-error)))))

(define (modulo x y) (force-vars (x y) (##modulo x y)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; gcd, lcm

(define (##gcd x y)

  (##define-macro (type-error) ''())

  (define (exact-gcd x y)
    (let loop ((x (##abs x)) (y (##abs y)))
      (if (##eq? y 0) x (loop y (##remainder x y)))))

  (if (and (##integer? x) (##integer? y))
    (if (and (##exact? x) (##exact? y))
      (exact-gcd x y)
      (##exact->inexact (exact-gcd (##inexact->exact x) (##inexact->exact y))))
    (type-error)))

(define-nary (gcd x y)
  0
  (if (##integer? x) x '())
  (##gcd x y)
  force-vars
  no-check
  (##null? ##trap-check-integer*))

(define (##lcm x y)

  (##define-macro (type-error) ''())

  (define (exact-gcd x y)
    (let loop ((x (##abs x)) (y (##abs y)))
      (if (##eq? y 0) x (loop y (##remainder x y)))))

  (define (exact-lcm x y)
    (if (or (##eq? x 0) (##eq? y 0))
      0
      (##quotient (##abs (##* x y)) (exact-gcd x y))))

  (if (and (##integer? x) (##integer? y))
    (if (and (##exact? x) (##exact? y))
      (exact-lcm x y)
      (##exact->inexact (exact-lcm (##inexact->exact x) (##inexact->exact y))))
    (type-error)))

(define-nary (lcm x y)
  1
  (if (##integer? x) x '())
  (##lcm x y)
  force-vars
  no-check
  (##null? ##trap-check-integer*))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; numerator, denominator

(define (##numerator x)

  (define (type-error)
    (##trap-check-rational 'numerator x))

  (number-dispatch x (type-error)
    x
    x
    (ratnum-numerator x)
    (if (flonum-rational? x)
      (##exact->inexact (##numerator (##flonum.inexact->exact x)))
      (type-error))
    (if (cpxnum-rational? x)
      (##numerator (cpxnum-real x))
      (type-error))))

(define (numerator x) (force-vars (x) (##numerator x)))

(define (##denominator x)

  (define (type-error)
    (##trap-check-rational 'denominator x))

  (number-dispatch x (type-error)
    1
    1
    (ratnum-denominator x)
    (if (flonum-rational? x)
      (##exact->inexact (##denominator (##flonum.inexact->exact x)))
      (type-error))
    (if (cpxnum-rational? x)
      (##denominator (cpxnum-real x))
      (type-error))))

(define (denominator x) (force-vars (x) (##denominator x)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; floor, ceiling, truncate, round

(define (##floor x)

  (define (type-error)
    (##trap-check-real 'floor x))

  (number-dispatch x (type-error)
    x
    x
    (##ratnum.floor x)
    (##flonum.floor x)
    (if (cpxnum-real? x)
      (##floor (cpxnum-real x))
      (type-error))))

(define (floor x) (force-vars (x) (##floor x)))

(define (##ceiling x)

  (define (type-error)
    (##trap-check-real 'ceiling x))

  (number-dispatch x (type-error)
    x
    x
    (##ratnum.ceiling x)
    (##flonum.ceiling x)
    (if (cpxnum-real? x)
      (##ceiling (cpxnum-real x))
      (type-error))))

(define (ceiling x) (force-vars (x) (##ceiling x)))

(define (##truncate x)

  (define (type-error)
    (##trap-check-real 'truncate x))

  (number-dispatch x (type-error)
    x
    x
    (##ratnum.truncate x)
    (##flonum.truncate x)
    (if (cpxnum-real? x)
      (##truncate (cpxnum-real x))
      (type-error))))

(define (truncate x) (force-vars (x) (##truncate x)))

(define (##round x)

  (define (type-error)
    (##trap-check-real 'round x))

  (number-dispatch x (type-error)
    x
    x
    (##ratnum.round x)
    (##flonum.round x)
    (if (cpxnum-real? x)
      (##round (cpxnum-real x))
      (type-error))))

(define (round x) (force-vars (x) (##round x)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; rationalize

(define (##rationalize x y)

  (define (type-error)
    (##trap-check-real 'rationalize x y))

  (define (simplest-rational1 x y)
    (if (##< y x)
      (simplest-rational2 y x)
      (simplest-rational2 x y)))

  (define (simplest-rational2 x y)
    (cond ((##not (##< x y))
           x)
          ((##positive? x)
           (simplest-rational3 x y))
          ((##negative? y)
           (##negate (simplest-rational3 (##negate y) (##negate x))))
          (else
           0)))

  (define (simplest-rational3 x y)
    (let ((fx (##floor x))
          (fy (##floor y)))
      (cond ((##not (##< fx x))
             fx)
            ((##= fx fy)
             (##+ fx
                  (##/ 1
                       (simplest-rational3
                         (##/ 1 (##- y fy))
                         (##/ 1 (##- x fx))))))
            (else
             (##+ fx 1)))))

  (if (and (##rational? x) (##rational? y))
    (simplest-rational1 (##- x y) (##+ x y))
    (type-error)))

(define (rationalize x y) (force-vars (x y) (##rationalize x y)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; trigonometry and complex numbers

(define (##exp x)

  (define (type-error)
    (##trap-check-number 'exp x))

  (number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      1
      (##flonum.exp (##flonum.<-fixnum x))) ; assuming exact conversion
    (##flonum.exp (##flonum.<-bignum x))
    (##flonum.exp (##flonum.<-ratnum x))
    (##flonum.exp x)
    (##make-polar (##exp (cpxnum-real x)) (cpxnum-imag x))))

(define (exp x) (force-vars (x) (##exp x)))

(define (##log x)

  (define (type-error)
    (##trap-check-number 'log x))

  (define (range-error)
    (##trap-check-range 'log x))

  (define (negative-log x)
    (##make-rectangular (##log (##negate x)) (inexact-+pi)))

  (number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      (range-error)
      (if (##fixnum.negative? x)
        (negative-log x)
        (if (##eq? x 1)
          0
          (##flonum.log (##flonum.<-fixnum x))))) ; assuming exact conversion
    (if (bignum-negative? x)
      (negative-log x)
      (##flonum.log (##flonum.<-bignum x)))
    (if (##negative? (ratnum-numerator x))
      (negative-log x)
      (##flonum.log (##flonum.<-ratnum x)))
    (if (##flonum.negative? (##flonum.copysign (inexact-+1) x))
      (negative-log x)
      (##flonum.log x))
    (##make-rectangular (##log (##magnitude x)) (##angle x))))

(define (log x) (force-vars (x) (##log x)))

(define (##sin x)

  (define (type-error)
    (##trap-check-number 'sin x))

  (number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      0
      (##flonum.sin (##flonum.<-fixnum x))) ; assuming exact conversion
    (##flonum.sin (##flonum.<-bignum x))
    (##flonum.sin (##flonum.<-ratnum x))
    (##flonum.sin x)
    (##/ (##- (##exp (##make-rectangular
                      (##negate (cpxnum-imag x))
                      (cpxnum-real x)))
              (##exp (##make-rectangular
                      (cpxnum-imag x)
                      (##negate (cpxnum-real x)))))
         (cpxnum-+2i))))

(define (sin x) (force-vars (x) (##sin x)))

(define (##cos x)

  (define (type-error)
    (##trap-check-number 'cos x))

  (number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      1
      (##flonum.cos (##flonum.<-fixnum x))) ; assuming exact conversion
    (##flonum.cos (##flonum.<-bignum x))
    (##flonum.cos (##flonum.<-ratnum x))
    (##flonum.cos x)
    (##/ (##+ (##exp (##make-rectangular
                      (##negate (cpxnum-imag x))
                      (cpxnum-real x)))
              (##exp (##make-rectangular
                      (cpxnum-imag x)
                      (##negate (cpxnum-real x)))))
         2)))

(define (cos x) (force-vars (x) (##cos x)))

(define (##tan x)

  (define (type-error)
    (##trap-check-number 'tan x))

  (number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      0
      (##flonum.tan (##flonum.<-fixnum x))) ; assuming exact conversion
    (##flonum.tan (##flonum.<-bignum x))
    (##flonum.tan (##flonum.<-ratnum x))
    (##flonum.tan x)
    (let ((a (##exp (##make-rectangular
                     (##negate (cpxnum-imag x))
                     (cpxnum-real x))))
          (b (##exp (##make-rectangular
                     (cpxnum-imag x)
                     (##negate (cpxnum-real x))))))
      (let ((c (##/ (##- a b) (##+ a b))))
        (##make-rectangular (##imag-part c) (##negate (##real-part c)))))))

(define (tan x) (force-vars (x) (##tan x)))

(define (##asin x)

  (define (type-error)
    (##trap-check-number 'asin x))

  (define (safe-case x)
    (##* (cpxnum--i)
         (##log (##+ (##* (cpxnum-+i) x)
                     (##sqrt (##- 1 (##* x x)))))))

  (define (unsafe-case x)
    (##negate (safe-case (##negate x))))

  (define (real-case x)
    (cond ((##< x -1)
           (unsafe-case x))
          ((##< 1 x)
           (safe-case x))
          (else
           (##flonum.asin (##exact->inexact x)))))

  (number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      0
      (real-case x))
    (real-case x)
    (real-case x)
    (real-case x)
    (let ((imag (cpxnum-imag x)))
      (if (or (##positive? imag)
              (and (##flonum? imag)
                   (##flonum.zero? imag)
                   (##negative? (cpxnum-real x))))
        (unsafe-case x)
        (safe-case x)))))

(define (asin x) (force-vars (x) (##asin x)))

(define (##acos x)

  (define (type-error)
    (##trap-check-number 'acos x))

  (define (complex-case x)
    (##* (cpxnum--i)
         (##log (##+ x
                     (##* (cpxnum-+i) (##sqrt (##- 1 (##* x x))))))))

  (define (real-case x)
    (if (or (##< x -1) (##< 1 x))
      (complex-case x)
      (##flonum.acos (##exact->inexact x))))

  (number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      (inexact-+pi/2)
      (real-case x))
    (real-case x)
    (real-case x)
    (real-case x)
    (complex-case x)))

(define (acos x) (force-vars (x) (##acos x)))

(define (##atan x)

  (define (type-error)
    (##trap-check-number 'atan x))

  (define (range-error)
    (##trap-check-range 'atan x))

  (number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      0
      (##flonum.atan (##flonum.<-fixnum x))) ; assuming exact conversion
    (##flonum.atan (##flonum.<-bignum x))
    (##flonum.atan (##flonum.<-ratnum x))
    (##flonum.atan x)
    (let ((r (cpxnum-imag x))
          (i (cpxnum-real x)))
      (if (and (##eq? r 1) (##zero? i))
        (range-error)
        (let ((a (##make-rectangular (##negate r) i)))
          (##/ (##- (##log (##+ a 1)) (##log (##- 1 a)))
               (cpxnum-+2i)))))))

(define (##atan2 y x)

  (define (type-error)
    (##trap-check-real 'atan y x))

  (if (and (##real? x) (##real? y))
    (let ((x (##exact->inexact (##real-part x)))
          (y (##exact->inexact (##real-part y))))
      (##flonum.atan y x))
    (type-error)))

(define (atan x #!optional (y (absent-obj)))
  (force-vars (x)
    (if (##eq? y (absent-obj))
      (##atan x)
      (force-vars (y)
        (##atan2 x y)))))

(define (##sqrt x)

  (define (type-error)
    (##trap-check-number 'sqrt x))

  (define (exact-int-sqrt x)
    (cond ((##eq? x 0)
           0)
          ((##negative? x)
           (##make-rectangular 0 (exact-int-sqrt (##negate x))))
          (else
           (let ((y (##exact-int.root x 2)))
             (if (##= x (##* y y))
               y
               (##flonum.sqrt (##exact->inexact x)))))))

  (number-dispatch x (type-error)
    (exact-int-sqrt x)
    (exact-int-sqrt x)
    (##/ (exact-int-sqrt (ratnum-numerator x))
         (exact-int-sqrt (ratnum-denominator x)))
    (if (##flonum.negative? x)
      (##make-rectangular 0 (##flonum.sqrt (##flonum.- x)))
      (##flonum.sqrt x))
    (##make-polar (##sqrt (##magnitude x)) (##/ (##angle x) 2))))

(define (sqrt x) (force-vars (x) (##sqrt x)))

(define (##expt x y)

  (define (type-error)
    (##trap-check-number 'expt x y))

  (define (exact-int-expt x y) ; y >= 0
    (cond ((##eq? y 0)
           1)
          ((or (##zero? x) (##= x 1))
           x)
          (else
           (let loop ((x x) (y y) (result 1))
             (if (##eq? y 1)
               (##* x result)
               (loop (##* x x)
                     (##quotient y 2)
                     (if (##zero? (##modulo y 2))
                       result
                       (##* x result))))))))

  (define (real-expt x y) ; y > 0
    (if (##zero? y)
      (if (##eq? y 0) 1 (inexact-+1))
      (if (##zero? x)
        (if (##eq? x 0) 0 (inexact-0))
        (complex-expt x y))))

  (define (complex-expt x y)
    (##exp (##* (##log x) y)))

  (define (invert z)
    (let ((result (##/ 1 z)))
      (if (##not result)
        (##trap-check-range 'expt x y)
        result)))

  (if (##complex? x)
    (number-dispatch y (type-error)
      (if (and (##flonum? x) (##flonum.nan? x))
        x
        (if (##fixnum.negative? y)
          (invert (exact-int-expt x (##negate y)))
          (exact-int-expt x y)))
      (if (and (##flonum? x) (##flonum.nan? x))
        x
        (if (bignum-negative? y)
          (invert (exact-int-expt x (##negate y)))
          (exact-int-expt x y)))
      (if (and (##flonum? x) (##flonum.nan? x))
        x
        (if (##negative? (ratnum-numerator y))
          (invert (real-expt x (##negate y)))
          (real-expt x y)))
      (if (and (##flonum? x) (##flonum.nan? x))
        x
        (if (##flonum.nan? y)
          y
          (if (##flonum.negative? y)
            (invert (real-expt x (##flonum.- y)))
            (real-expt x y))))
      (if (and (##flonum? x) (##flonum.nan? x))
        x
        (complex-expt x y)))
    (type-error)))

(define (expt x y) (force-vars (x y) (##expt x y)))

(define (##make-rectangular x y)

  (define (type-error)
    (##trap-check-real 'make-rectangular x y))

  (if (and (##real? x) (##real? y))
    (let ((r (##real-part x)) (i (##real-part y)))
      (cond ((##eq? i 0)
             r)
;            ((and (##flonum? r) (##flonum? i) (##flonum.zero? i))
;             r)
            (else
             (cpxnum-make r i))))
    (type-error)))

(define (make-rectangular x y)
  (force-vars (x y) (##make-rectangular x y)))

(define (##make-polar x y)

  (define (type-error)
    (##trap-check-real 'make-polar x y))

  (if (and (##real? x) (##real? y))
    (let ((x* (##real-part x)) (y* (##real-part y)))
      (##make-rectangular (##* x* (##cos y*))
                          (##* x* (##sin y*))))
    (type-error)))

(define (make-polar x y) (force-vars (x y) (##make-polar x y)))

(define (##real-part x)

  (define (type-error)
    (##trap-check-number 'real-part x))

  (number-dispatch x (type-error)
    x x x x (cpxnum-real x)))

(define (real-part x) (force-vars (x) (##real-part x)))

(define (##imag-part x)

  (define (type-error)
    (##trap-check-number 'imag-part x))

  (number-dispatch x (type-error)
    0 0 0 0 (cpxnum-imag x)))

(define (imag-part x) (force-vars (x) (##imag-part x)))

(define (##magnitude x)

  (define (type-error)
    (##trap-check-number 'magnitude x))

  (number-dispatch x (type-error)
    (if (##fixnum.negative? x)
      (##bignum.-/fixnum-fixnum 0 x)
      x)
    (let ((r (##bignum.copy x)))
      (bignum-set-positive! r)
      r)
    (if (##negative? (ratnum-numerator x))
      (ratnum-make (##negate (ratnum-numerator x)) (ratnum-denominator x))
      x)
    (##flonum.abs x)
    (let ((r (##real-part x)) (i (##imag-part x)))

      (define (complex-magn a b)
        (cond ((##eq? a 0)
               b)
              ((and (##flonum? a) (##flonum.zero? a))
               (##exact->inexact b))
              (else
               (let ((c (##/ a b)))
                 (##* b (##sqrt (##+ (##* c c) 1)))))))

      (cond ((and (##flonum? r) (##not (##flonum.finite? r)))
             (if (##flonum.nan? r)
               r
               (if (and (##flonum? i) (##flonum.nan? i))
                 i
                 (inexact-+inf))))
            ((and (##flonum? i) (##not (##flonum.finite? i)))
             (if (##flonum.nan? i)
               i
               (inexact-+inf)))
            (else
             (let ((abs-r (##abs r))
                   (abs-i (##abs i)))
               (if (##< abs-r abs-i)
                 (complex-magn abs-r abs-i)
                 (complex-magn abs-i abs-r))))))))

(define (magnitude x) (force-vars (x) (##magnitude x)))

(define (##angle x)

  (define (type-error)
    (##trap-check-number 'angle x))

  (number-dispatch x (type-error)
    (if (##fixnum.negative? x)
      (inexact-+pi)
      0)
    (if (bignum-negative? x)
      (inexact-+pi)
      0)
    (if (##negative? (ratnum-numerator x))
      (inexact-+pi)
      0)
    (if (##flonum.negative? (##flonum.copysign (inexact-+1) x))
      (inexact-+pi)
      (inexact-0))
    (if (##zero? x)
      (inexact-0)
      (##atan2 (cpxnum-imag x) (cpxnum-real x)))))

(define (angle x) (force-vars (x) (##angle x)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; exact->inexact, inexact->exact

(define (##exact->inexact x)

  (define (type-error)
    (##trap-check-number 'exact->inexact x))

  (number-dispatch x (type-error)
    (##flonum.<-fixnum x) ; assuming exact conversion
    (##flonum.<-bignum x)
    (##flonum.<-ratnum x)
    x
    (##make-rectangular (##exact->inexact (cpxnum-real x))
                        (##exact->inexact (cpxnum-imag x)))))

(define (exact->inexact x)
  (force-vars (x) (##exact->inexact x)))

(define (##inexact->exact x)

  (define (type-error)
    (##trap-check-number 'inexact->exact x))

  (define (range-error)
    (##trap-check-range 'inexact->exact x))

  (number-dispatch x (type-error)
    x
    x
    x
    (if (flonum-rational? x)
      (##flonum.inexact->exact x)
      (range-error))
    (let ((real (cpxnum-real x))
          (imag (cpxnum-imag x)))
      (if (and (non-cpxnum-rational? real)
               (non-cpxnum-rational? imag))
        (##make-rectangular (##inexact->exact real)
                            (##inexact->exact imag))
        (range-error)))))

(define (inexact->exact x)
  (force-vars (x)
    (##inexact->exact x)))

;------------------------------------------------------------------------------
