#include <GL/gl.h>

#include "global.h"
#include "texture.h"


static TextureCacheEntry *texture_cache_first;
static MM_Image *default_image;
#ifdef WITHTHREAD
static pthread_mutex_t texture_mutex;
#endif /* WITHTHREAD */
static texture_quality;

Public
void initTextureCache(int quality)
{
  unsigned char *pix;
  int i;

  texture_cache_first = NULL;
  default_image = MM_newImage(256, 256, MM_IMAGE_RGB);
  pix = default_image->pixmap;
  for (i=0; i < 256*256; i++) {
    pix[3*i+0] = pix[3*i+1] = pix[3*i+2] = 0x80;
  }
  texture_quality = quality;
#ifdef WITHTHREAD
  pthread_mutex_init(&texture_mutex, NULL);
#endif /* WITHTHREAD */
}

Public
void closeTextureCache(void)
{
  /* TODO: librer les textures */

  MM_freeImage(default_image);
}

Public
void updateTextures(void)
{
  TextureCacheEntry *tc;
  MM_Image *image, *image1;

#ifdef WITHTHREAD
  pthread_mutex_lock(&texture_mutex);
#endif /* WITHTHREAD */
  for (tc = texture_cache_first; tc != NULL; tc = tc->next) {
    if ((image = tc->image) != NULL) {
      glBindTexture(GL_TEXTURE_2D, tc->texture);
      /* resizing */
      if (image->xsize != 256 || image->ysize != 256) {
        image1 = MM_resizeImage(256, 256, image);
        glTexImage2D(GL_TEXTURE_2D, 0, 3, image1->xsize, image1->ysize, 0,
                     GL_RGB,GL_UNSIGNED_BYTE,image1->pixmap);
        MM_freeImage(image1);
      } else {
        glTexImage2D(GL_TEXTURE_2D, 0, 3, image->xsize, image->ysize, 0,
                     GL_RGB,GL_UNSIGNED_BYTE,image->pixmap);
      }
      if (tc->load_finished) {
        MM_freeImage(image);
      }
      tc->image = NULL;
    }
  }
#ifdef WITHTHREAD
  pthread_mutex_unlock(&texture_mutex);
#endif /* WITHTHREAD */
}

Private
int readImage(void *tc1, char *buf, int length)
{
  TextureCacheEntry *tc = tc1;
  int ret_length;

  ret_length = httpRead(tc->handle, buf, length);
  return ret_length;
}

Private
void displayImage(void *tc1, MM_Image *image)
{
  TextureCacheEntry *tc = tc1;

#ifdef WITHTHREAD
  pthread_mutex_lock(&texture_mutex);
#endif /* WITHTHREAD */
  tc->image = image;
#ifdef WITHTHREAD
  pthread_mutex_unlock(&texture_mutex);
#endif /* WITHTHREAD */
}

Private
int netOpenCallBack(void *tc1, void *handle)
{
  TextureCacheEntry *tc = tc1;
  int mm_type;
  MM_Image *image;

  trace(DBG_ZV, "texture: handle=%X", (int) handle);

  /* error while opening */
  if (handle == NULL) return 0;

  tc->handle = handle;
  mm_type = MM_getType(tc->url);
  image = NULL;
  switch (mm_type) {
  case MM_TYPE_PPM:
    image = MM_loadPPM(tc, readImage, displayImage);
    break;
  case MM_TYPE_GIF:
    image = MM_loadGIF(tc, readImage, displayImage);
    break;
  case MM_TYPE_PCX:
    image = MM_loadPCX(tc, readImage, displayImage);
    break;
  default:
    trace(DBG_ZV, "texture: unknown type %s", tc->url);
    break;
  }
#ifdef WITHTHREAD
  pthread_mutex_lock(&texture_mutex);
#endif /* WITHTHREAD */
  tc->image = image;
  tc->load_finished = 1;
#ifdef WITHTHREAD
  pthread_mutex_unlock(&texture_mutex);
#endif /* WITHTHREAD */
  return 0;
}

Public
int getTextureFromCache(char *url)
{
  TextureCacheEntry *tc;

#ifdef WITHTHREAD
  pthread_mutex_lock(&texture_mutex);
#endif /* WITHTHREAD */
  for (tc = texture_cache_first; tc != NULL; tc = tc->next) {
    if (strcmp(tc->url, url) == 0) {
#ifdef WITHTHREAD
      pthread_mutex_unlock(&texture_mutex);
#endif /* WITHTHREAD */
      return tc->texture;
    }
  }
  trace(DBG_ZV, "texture: loading %s", url);

  /* new entry in cache */
  tc = malloc(sizeof(TextureCacheEntry));
  tc->next = texture_cache_first;
  texture_cache_first = tc;
  tc->image = NULL;
  tc->load_finished = 0;
  strcpy(tc->url, url);
  glGenTextures(1, &tc->texture);
  glBindTexture(GL_TEXTURE_2D, tc->texture);
  glTexImage2D(GL_TEXTURE_2D, 0, 3,
	       default_image->xsize, default_image->ysize, 0,
	       GL_RGB, GL_UNSIGNED_BYTE, default_image->pixmap);
  glTexEnvi( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL );
  
  if (texture_quality) {
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  } else {
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  }

  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
#ifdef WITHTHREAD
  pthread_mutex_unlock(&texture_mutex);
#endif /* WITHTHREAD */
  httpOpen(tc->url, netOpenCallBack, tc, NO_NONBLOCK); 
  return tc->texture;
}

