/***************************************************************************
                          store.h  -  general purpose file storage class
                             -------------------                                         
    begin                : Thu Jun 3 1999                                           
    copyright            : (C) 1999 by David Johnson                         
    email                : arandir@meer.net                                     

    This software licensed under the Berkeley Software Distribution License
 ***************************************************************************/

// This is version 0.3 of the Store class

#ifndef STORE_H
#define STORE_H

#include <qfile.h>
#include <qstring.h>

/////////////////////////////////////////////////////////////////////////////
// Miscellaneous definitions
const unsigned MAX_LENGTH	= 512;
const char STORE_DELIM		= '=';
const char STORE_COMMENT	= '#';

/**
  * A C++ class to store and retrieve information to and from a human-readable text file.
  * It will be usable for program configuration options or for simple object serialization.
  * It is based on Name/Value pairs.
  *
  * @author David Johnson
  */

class Store {
public:
	/** A minimal constructor.*/
	Store();
	/**
	  * The typical constructor for the Store class.
	  * @param mode		QFile mode (IO_ReadOnly or IO_WriteOnly)
	  * @param filename	The name of the file to open or create
	  * @param format	Optional format string to identify text files
	  * @param version	Optional version numbers to identify version of file
	  */
	Store(int mode, const QString &filename, const QString &format, QString version="");
	/** The destructor for the Store class.
	  * Will close any opened files from the constructor
	  */
	~Store();
	
	/** Returns true if no file errors exist.*/
	bool good();
	/** Returns the file major version as an string */
	const QString getVersion();
	
	/** processes the next line of text. Returns false if at end of file */
	bool getLine();
	/** returns the last read element group */
	const QString getGroup();
	/** returns the name for the current line */
	const QString getName();
	/** returns the value for the current line */
	const QString getValue();
	/** Write a new element group to file. Returns false if error. */
	bool putGroup(const QString &group);
	/** Write name and value to file. Returns false if file error. */
	bool putLine(const QString &name, const QString &value);
	/** Write name and value to file. Returns false if file error. */
	bool putLine(const QString &name, const int &value);
	/** Write a comment line to file. Returns false if file error. */
	bool putComment(const QString &comment);
	/** Write an additional comma and element to the line */
	bool putElement(const QString &element);
	/** Write an additional comma and element to the line */
	bool putElement(const int &element);

	/* I eventually want to add methods to parse individual comma separated
	 * elements within the value field
	 */
	
private:
	bool getHeader();	// check for valid header
	bool putHeader();	// write a header
	QString nextValidLine();	// get next valid line
	
private:
	QString sfilename;	// file name
	QString sformat;	// file format
	QString sversion;	// file format major version
	int smode;			// current file mode
	bool sgood;			// current error status
	QFile sfile;		// store file
	QString sgroup;		// current group
	QString sname;		// current name element
	QString svalue;		// current value element
};

#endif // STORE_H













