/* pagefuncs.c: HTML `functions' can be executed with page_process_page (). */

/* Author: Brian J. Fox (bfox@ai.mit.edu) Tue Jul 18 17:50:42 1995.

   This file is part of <Meta-HTML>(tm), a system for the rapid deployment
   of Internet and Intranet applications via the use of the Meta-HTML
   language.

   Copyright (c) 1995, 1996, Brian J. Fox (bfox@ai.mit.edu).
   Copyright (c) 1996, Universal Access Inc. (http://www.ua.com).

   Meta-HTML is free software; you can redistribute it and/or modify
   it under the terms of the UAI Free Software License as published
   by Universal Access Inc.; either version 1, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   UAI Free Software License for more details.

   You should have received a copy of the UAI Free Software License
   along with this program; if you have not, you may obtain one by
   writing to:

   Universal Access Inc.
   129 El Paseo Court
   Santa Barbara, CA
   93101  */

/*
################### CAUTION ##################################
# This file has been modified for inclusion into WML
# D. Barbier   16/01/1999
##############################################################
*/

#include "language.h"

#if defined (__cplusplus)
extern "C"
{
#endif
/************************************************************/
/*							    */
/*		Language Manipulation Functions		    */
/*							    */
/************************************************************/

/* <comment> ... </comment> */
static void pf_comment (PFunArgs);

/* <verbatim> .... </verbatim> Insert the contents verbatim. */
static void pf_verbatim (PFunArgs);

/* <include filename alt="page value if FILENAME isn't found"> */
static void pf_include (PFunArgs);

/* <subst-in-page this that> Replace occurrences of THIS with THAT. */
static void pf_subst_in_page (PFunArgs);

/* <page-insert location string> */
static void pf_page_insert (PFunArgs);

/************************************************************/
/*							    */
/*			HTML "Helper" Functions		    */
/*							    */
/************************************************************/

static PFunDesc func_table[] =
{
  /* File manipulation functions. */
  { "INCLUDE",    0, 0, pf_include },

  /* Block manipulation. */
  { "COMMENT",		1, 0, pf_comment },
  { "VERBATIM",		1, 0, pf_verbatim },

  /* Page manipulation.  The following functions operate on the page as it
     exists in its current state at the time the function was called. */
  { "SUBST-IN-PAGE",   0, 0, pf_subst_in_page },
  { "PAGE-INSERT",  0, 0, pf_page_insert },

  { (char *)NULL,	0, 0, (PFunHandler *)NULL }
};

PACKAGE_INITIALIZER (initialize_pagefunc_functions)

DEFINE_SECTION (LANGUAGE-OPERATORS, self-modif; language;
self-referential, "<Meta-HTML> provides various functions and
variables which are used to operate on or within the language
environment itself, rather than within the application environment.

Such constructs make it possible to create self-modifying code, to
change the way in which a builtin function operates, and other such
language dependent features.", "")

DEFVAR (MHTML::INHIBIT-COMMENT-PARSING, "When this variable is set to
a non-empty value, the triple semi-colon comment feature is disabled.

You may need this for complex applications where you are deliberately
shipping the source of a <meta-html> document to the client.")

DEFVAR (MHTML::VERSION, "Contains the version identification of the
currently running interpreter.  This variable is available whereever
the interpreter is running, including the <meta-html> server, Engine,
and in <code>mdb</code>.

<complete-example>
<get-var mhtml::version>
</complete-example>")

DEFMACRO (pf_comment, ,
"Simply discard the <var body>.  No processing is done, and no output
is produced.

A shorthand for commenting source on a single line exists; when the
sequence of 3 semicolon character is seen
(<code>;</code><code>;;</code>), then the text from this sequence to 
the end of the line inclusive is discarded.

Example:<set-var mhtml::inhibit-comment-parsing=1>
<example>
<comment>
  This text is simply ignored.

  All of it.
</comment>
;;; So is this text, up to the end of the line.
;; But NOT this line -- it only has two semi-colons on it.
</example><unset-var mhtml::inhibit-comment-parsing>")
{
  /* Contents already deleted by caller. */
}

DEFINE_SECTION (PAGE-OPERATORS, page commands; page search; search;
insert; modify page,
"<meta-html> contains a few commands which
operate on the entire <i>page</i> as it currently exists.

You may find, modify, or delete text which has already been looked at by the interpreter, as well as text which hasn't yet been looked at.

Most of the time, you won't need to use such commands.  They can make
it hard for other people to understand the sequence of code in your
page, and they can sometimes produce unexpected results which are
difficult to debug because much of the information has been modified.
", "")

static int include_recursive_calls = 0;

DEFUN (pf_include, webpath &key alt=altcode verbatim=true,
"Insert the contents of the file named by <var webpath> into the
document at the point where the <code>include</code> form was read.
If the keyword argument <var verbatim=true> is given, then the contents
of the file are not executed, only inserted.  Otherwise, execution
resumes at the point where the file was inserted.

<var webpath> can be given as an absolute pathname, or a relative
pathname.  If the path is relative, it is considered to be relative to
the location of the document which contains the <code>include</code>
form.  If the path given is not relative, it is appended to the
directory found in <b>MHTML::INCLUDE-PREFIX</b>.

If the named file could not be found on the server, and an <var alt=altcode>
value is given, then that value is placed into the page instead.")
{
  int verbatim_p = var_present_p (vars, "VERBATIM");
  char *pathname = (char *)NULL;
  int found = 0;

  include_recursive_calls++;
  if (include_recursive_calls < MHTML_INCLUDE_RECURSION_LIMIT)
    {
      char *arg = mhtml_evaluate_string (get_positional_arg (vars, 0));
      char *incpref = pagefunc_get_variable ("%%::incpref");
      char *relpref = pagefunc_get_variable ("%%::relpref");

      if (debug_level > 10)
	page_debug ("<include %s> [relpref: %s]",
		    arg ? arg : "\"\"",
		    relpref ? relpref : "\"\"");
		    
      if (!incpref) incpref = pagefunc_get_variable ("mhtml::include-prefix");
      if (!relpref) relpref = pagefunc_get_variable ("mhtml::relative-prefix");

      if (arg != (char *)NULL)
	{
	  PAGE *file_contents = (PAGE *)NULL;
	  char *wr = (char *)NULL;

	  pathname = mhtml_canonicalize_file_name (arg, incpref, relpref, &wr);

	  if (!empty_string_p (pathname))
	    file_contents = page_read_template (pathname);

	  /* Did the user specify some alternate HTML if the file
	     couldn't be found? */
	  if (!file_contents)
	    {
	      char *alt = mhtml_evaluate_string
		(get_one_of (vars, "ALT", "ALTERNATE", (char *)0));

	      found = 0;
	      if (alt != (char *)NULL)
		{
		  verbatim_p = 0;
		  file_contents = page_create_page ();
		  bprintf (file_contents, "%s", alt);
		  free (alt);
		}
	      else if (debug_level)
		page_debug ("include %s: Unable to read %s", arg, pathname);
	    }
	  else
	    {
	      char digits[40];
	      found = 1;
	      sprintf (digits, "%ld", page_most_recent_modification_time);
	      pagefunc_set_variable ("mhtml::last-modification-time", digits);
	    }

	  if (file_contents)
	    {
	      bprintf_insert_binary
		(page, start, file_contents->buffer, file_contents->bindex);

	      if (verbatim_p)
		*newstart += file_contents->bindex;

	      page_free_page (file_contents);
	    }
	  free (arg);
	}
    }
  xfree (pathname);
  include_recursive_calls--;
}

DEFINE_SECTION (HTML-HELPERS, HTML; helper; convenience, 
"The following functions all produce HTML as output and are defined in
order to help with the creation of forms and tables.", "")

char *
html_quote_string (char *string)
{
  BPRINTF_BUFFER *newstring = bprintf_create_buffer ();
  char *result;

  if (string != (char *)NULL)
    {
      register int i;

      for (i = 0; string[i] != '\0'; i++)
	{
	  if (string[i] == '<')
	    bprintf (newstring, "&lt;");
	  else if (string[i] == '>')
	    bprintf (newstring, "&gt;");
	  else if (string[i] == '&')
	    bprintf (newstring, "&amp;");
	  else
	    bprintf (newstring, "%c", string[i]);
	}
    }

  result = newstring->buffer;
  free (newstring);

  return (result);
}

DOC_SECTION (PAGE-OPERATORS)

DEFUN (pf_subst_in_page, this-string with-that,
"Replaces all occurrences of <var this-string> with <var with-that> in the
current page.  Both <var this-string> and <var with-that> are evaluated
before the replacement is done.  <var this-string> can be any regular
expression allowed by the POSIX extended regular expression matching.

This command can be useful when included as part of a footer
<code>include</code> file.  For example, you could replace all
occurrences of <code>%name%</code> with a the value of the variable
<code>FirstName</code> by writing:
<example>
  <subst-in-page %name% <get-var forms::FirstName>>
</example>")
{
  int arg = 0, done = 0;
  PagePDL *top_page = page_pdl_page (0);

  while (!done)
    {
      char *this_string, *with_that;

      this_string = get_positional_arg (vars, arg++);
      with_that = get_positional_arg (vars, arg++);

      if (this_string == (char *)NULL)
	done = 1;
      else
	{
	  this_string = mhtml_evaluate_string (this_string);
	  with_that = mhtml_evaluate_string (with_that);

	  if (this_string)
	    page_subst_in_page_pivot
	      (top_page->page, this_string, with_that, &(top_page->start));

	  xfree (this_string);
	  xfree (with_that);
	}
    }

  *top_page->search_start_modified = top_page->start;
  if (top_page->page == page) *newstart = top_page->start;
}

static void
pf_page_insert (PFunArgs)
{
  int arg = 0;
  char *insert_loc = mhtml_evaluate_string (get_positional_arg (vars, arg++));
  char *insertion = mhtml_evaluate_string (get_positional_arg (vars, arg++));
  PagePDL *top_page = page_pdl_page (0);

  if ((!empty_string_p (insert_loc)) && (!empty_string_p (insertion)))
    {
      int loc = atoi (insert_loc);

      if (loc < 0)
          loc += top_page->page->bindex + 1;
      if ((loc > -1) && (loc < top_page->page->bindex))
        {
          bprintf_insert_text (top_page->page, loc, insertion);
          if (loc < start)
              *newstart += strlen (insertion);
        }
    }
}

DOC_SECTION (HTML-HELPERS)
DEFUN (pf_verbatim, &key quote,
"Insert <var body> verbatim, avoiding doing any processing on
the contents.  If the keyword argument <var quote> is given,
occurrences of characters with special meaning to <b>HTML</b> are
replaced with the <b>HTML</b> code to produce that character in the
output page.

Contrast this with <funref page-operators comment>.")
{
  int quote_p = var_present_p (vars, "quote");

  /* Insert the contents, and then skip past them. */
  if (body && body->buffer)
    {
      char *string = body->buffer;

      if (quote_p)
	string = html_quote_string (string);

      if (string != (char *)NULL)
	{
	  bprintf_insert (page, start, "%s", string);
	  *newstart = start + strlen (string);

	  if (quote_p)
	    free (string);
	}
    }
}

#if defined (__cplusplus)
}
#endif
