/*  Screem:  preview.c,
 *  code for creating/updating the preview window
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */


#include <config.h>
#include <gnome.h>
#include <gtk-xmhtml/gtk-xmhtml.h>

#include "preferences.h"
#include "preview.h"

#ifdef HAVE_GTKHTML
#include <gtkhtml/gtkhtml.h>

static void url_requested( GtkHTML *html, const gchar *url,
			   GtkHTMLStreamHandle handle );
#endif

extern Preferences *cfg;

static GtkWidget *display;

static GtkWidget *screem_default_preview( void );
static GtkWidget *screem_gtkhtml_preview( void );
static gchar* screem_load_splash( void );

/*
 * screem_preview_new:
 *
 * create the editor area
 *
 * return values: a GtkWidget
 */
GtkWidget *screem_preview_new()
{
	GtkWidget *preview;

	/* insert switch statement here to select from possible
	   preview widgets */
#ifndef HAVE_GTKHTML
	preview = screem_default_preview();
#else
	preview = screem_gtkhtml_preview();
#endif
	gtk_widget_set_usize(preview, cfg->preview_width, cfg->preview_height);

	return preview;
}

static GtkWidget *screem_default_preview()
{
	GtkWidget *box;
	GtkWidget *xmhtml;
	
	box = gtk_vbox_new( FALSE, 4 );

	xmhtml = gtk_xmhtml_new();
	gtk_widget_show( xmhtml );
	gtk_box_pack_start( GTK_BOX( box ), xmhtml, TRUE, TRUE, 0 );

	gtk_object_set_data( GTK_OBJECT( box ), "preview", xmhtml );

	/* remember the widget for use in this file */
	display = xmhtml;

	return box;
}

#ifdef HAVE_GTKHTML
static GtkWidget *screem_gtkhtml_preview()
{
	GtkWidget *sw;
	GtkWidget *html;

	html = gtk_html_new();
	gtk_widget_show( html );

	sw = gtk_scrolled_window_new( GTK_LAYOUT( html )->hadjustment,
				      GTK_LAYOUT( html )->vadjustment );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ), 
                                        GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC );

	gtk_container_add( GTK_CONTAINER( sw ), html );

	display = html;

	gtk_signal_connect( GTK_OBJECT( html ), "url_requested",
			    GTK_SIGNAL_FUNC( 0 ), NULL );

	return sw;
}

static void url_requested( GtkHTML *html, const gchar *url,
			   GtkHTMLStreamHandle handle )
{
	FILE *file;
}

#endif

/* returns text to show in the preview window when screem first loads up */
static gchar* screem_load_splash()
{
	gchar* splash = g_strdup_printf( _("\
<html>\
<head>\
<title>SCREEM</title>\
</head>\
<body bgcolor=\"black\" text=\"white\">\
<h1 align=\"center\">SCREEM</h1>\
<h3 align=\"center\">Site CReating and Editing EnvironMent</h3>\
<hr>\
<div align=\"center\">\
Welcome to %s version %s\
</div>\
<br>\
<h3>Release Notes:</h3>\
<ul>\
<li>Enhanced HTML attribute insertion: eg to insert &amp;amp; you just need to\
 type an ampersand twice.  Inserting a &amp;quot; attribute is automatically\
 done when typing the character.  Try out others to see what is supported.\
<li>Auto close tag insertion:  When a close tag is typed, eg &lt;/&gt; then\
 it will automatically become a closing tag for the current open tag.\
<li>Multiple pages open at once, switchable via tabs\
<li>Multiple sites open at once\
<li>A new browse file tree tab\
<li>Numerous bug fixes\
<li>Support for the GtkHTML widget for the preview window (don't go installing\
 the widget unless you are a developer as it isn't ready for general use yet)\
<li>New Tools menu\
<li>CSS Wizard - able to:\
<ul>\
<li>save stylesheets\
<li>dump stylesheets into the editor at the current position\
<li>dump individual styles into the editor at the current position\
</ul>\
<li><em>NOTE:</em>Netscape 4.x doesn't support CSS very well, I reccomend \
grabbing the latest Mozilla milestone build if you want to check your pages \
that use CSS\
</ul>"),PACKAGE,VERSION);
	return splash;
}

/*
 * screem_preview_display:
 *
 * displays the given text
 *
 * return values: none
 */
void screem_preview_display( gchar *text )
{
	gboolean free_me = FALSE;
#ifndef HAVE_GTKHTML
	if( text == NULL )
		text = screem_load_splash();

	gtk_xmhtml_freeze( GTK_XMHTML( display ) );
	gtk_xmhtml_source( GTK_XMHTML( display ), text );
	gtk_xmhtml_thaw( GTK_XMHTML( display ) );
#else
	GtkHTMLStreamHandle handle;

	if( text == NULL )
		free_me = (gboolean)text = screem_load_splash();

	handle = gtk_html_begin( GTK_HTML( display ), "" );
	gtk_html_write( GTK_HTML( display ), handle, text, strlen( text ) );
	gtk_html_end( GTK_HTML( display ), handle, GTK_HTML_STREAM_OK );

	gtk_html_parse( GTK_HTML( display ) );
#endif
	if( free_me )
		g_free( text );
}


/*
 * screem_preview_update :
 *
 * updates the preview, keeping the window in the same position
 *
 * return values: none
 */
void screem_preview_update( gchar *text )
{
	GtkAdjustment *vadj;
	GtkAdjustment *hadj;
	gfloat vval;
	gfloat hval;

     
#ifndef HAVE_GTKHTML
	vadj = GTK_ADJUSTMENT( GTK_XMHTML( display )->vsba );
	hadj = GTK_ADJUSTMENT( GTK_XMHTML( display )->hsba );
#else
	vadj = GTK_ADJUSTMENT( GTK_LAYOUT( display )->vadjustment );
	hadj = GTK_ADJUSTMENT( GTK_LAYOUT( display )->hadjustment );
#endif
	vval = vadj->value;
	hval = hadj->value;

	screem_preview_display( text );

	gtk_adjustment_set_value( vadj, vval );
	gtk_adjustment_set_value( hadj, hval );
}
