/*  Screem:  uploadWizard.c,
 *  This file handles the uploading of a site
 *
 *  This file Copyright (C) 1999 Lee Mallabone.
 *  Original 'sitecopy' by Joe Orton.
 *  Sitecopy functionality integrated into transfer.c by Lee Mallabone.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <errno.h>
#include <gmodule.h>
#include <gnome.h>
#include <unistd.h>
#include <stdio.h>
#include <pthread.h>

#include <sys/stat.h>

/* add sitecopy functionality */
#include "common.h"
#include "sites.h"
#include "rcfile.h"
#include "frontend.h"

#include "uploadWizard.h"
#include "site.h"
#include "page.h"

extern GtkWidget *app;
extern Site *current_site;

/* Pointers to all the stuff we modify/destroy in the callbacks */
GtkWidget *connection_label;
GtkWidget *op_label;
GtkWidget *file_label;
GtkWidget *dir_label;
GtkWidget *main_progressbar;
GtkWidget *job_progressbar;
GtkWidget *update_dialog;
GtkWidget *upload_button;
GtkWidget *error_button;
GtkWidget *close_button;
GtkWidget *keep_going_button;

GtkWidget *error_log_window;
GtkWidget *error_log_list;

/* Closing the dialog while a thread is still going would be a
 * very Bad Thing. */
gboolean upload_in_progress = FALSE;
pthread_t update_thread_id = 0;

/* The site we're working with during all these functions. */
struct site_t *tmp_site;
float uploaded_bytes = 0.0;

/* Function that updates the site view with any necessary changes after an
 * upload. Remove all references to it if it's not actually required. */
/*extern void screem_refresh_site_view(void);*/

#ifdef USE_DAV
extern const struct proto_driver dav_driver;
#endif /* USE_DAV */
extern const struct proto_driver ftp_driver;

void dump_site_info(void) {
   printf("siteName is: \t%s\n", screem_site_get_name( current_site ) );
   printf("pathname is: \t%s\n", screem_site_get_pathname( current_site ) );
   printf("remote server is: \t%s\n",screem_site_get_remote_url(current_site));
   printf("remotePath is: \t%s\n", screem_site_get_remote_path(current_site));
   printf("username is: \t%s\n", screem_site_get_remote_user( current_site ) );
   /* this would be a big security flaw if we displayed the password on 
      the screen */
   /*   printf("password is: \t%s\n", site->password);*/
   printf("cvsRoot is: \t%s\n", screem_site_get_cvs_root( current_site ) );
   printf("Method is \t%s\n", screem_site_get_remote_method( current_site ) );
}

/** Plugin stuff **/

G_MODULE_EXPORT const gchar*
g_module_check_init( GModule *module )
{
           g_print("uploadWizard: check-init\n");
           return NULL;
}

G_MODULE_EXPORT void
g_module_unload( GModule *module )
{
           g_print( "uploadWizard: unloaded\n" );
}

G_MODULE_EXPORT void
init()
{
   GtkWidget *uploadButton;
   GtkWidget *toolbar;

   GnomeUIInfo menuinfo[] = {
	{
	   GNOME_APP_UI_ITEM, N_( "Upload" ),
	   N_( "Upload the site" ),
	   uploadSite, NULL, NULL,
	   GNOME_APP_PIXMAP_STOCK,
	   GNOME_STOCK_MENU_BLANK,
	   0,
	   GDK_CONTROL_MASK, NULL
	},
      GNOMEUIINFO_END
   };

   
   /* place menu item after image under insert */
   gnome_app_insert_menus( GNOME_APP( app ),
			   _("_Tools/"), menuinfo);

      toolbar = gtk_object_get_data( GTK_OBJECT( app ), "wizardbar" );

   /* place a button on the wizards toolbar,
    *            I need to expose another toolbar as it shouldn't really be
    *            put on the wizard one */
   uploadButton = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_REDO );
   gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
			   _("Upload Wizard"), "", uploadButton ,
			   uploadSite, 0 );
   
   g_print( "uploadWizard: initialised\n" );
}

/******************************************************************/

void upload_button_clicked                  (GtkButton       *button,
					     gpointer         user_data)
{

  /* We don't want to begin again. Also, there doesn't seem to be any way
   * to explictly terminate a pthread, so we don't want the dialog to be closed
   * while the upload would carry on in the background.
   * We also don't want the user changing files while we're doing an upload, so
   * we make the upload dialog modal.
   */
	/*gtk_widget_set_sensitive (close_button, FALSE);*/

  if ((upload_in_progress == TRUE) && (update_thread_id != 0)) {
	 pthread_kill (update_thread_id, 15);
	 site_writefiles(tmp_site);
	 upload_in_progress = FALSE;
	 return;
  } else {
	 gtk_widget_set_sensitive (upload_button, FALSE);
    gtk_window_set_modal (GTK_WINDOW(update_dialog), TRUE);
	 
	 if ( pthread_create (&update_thread_id, NULL, update_thread,
								 NULL) != 0) {
		gnome_error_dialog ("There was a problem creating the thread to do an update.\n");
      return;
	 }
	 if (update_thread_id != 0)
		pthread_join (update_thread_id, NULL);
			
	 gtk_window_set_modal (GTK_WINDOW(update_dialog), FALSE);
	 gtk_widget_set_sensitive (close_button, TRUE);
	 gtk_widget_set_sensitive (upload_button, TRUE);
	 upload_in_progress = FALSE;
  }
}

void error_button_clicked                  (GtkButton       *button,
					    gpointer         user_data)
{
  gtk_widget_show (error_log_window);
}

void close_button_clicked                   (GtkButton       *button,
					     gpointer         user_data)
{
	if (!upload_in_progress)
		gtk_widget_destroy (update_dialog);
}

/*********************************************************************/
/*** GUI creation function(s). I used glade (http://glade.pn.org/) ***/
/*** to create the upload dialog, simply because it rocks so much. ***/
/*********************************************************************/

void create_update_dialog (void)
{
  GtkWidget *dialog_vbox1;
  GtkWidget *vbox1;
  GtkWidget *hseparator3;
  GtkWidget *table1;
  GtkWidget *label3;
  GtkWidget *label6;
  GtkWidget *hseparator1;
  GtkWidget *label2;
  GtkWidget *label1;
  GtkWidget *dialog_action_area1;

  update_dialog = gnome_dialog_new ("Update Progress", NULL);
  gtk_object_set_data (GTK_OBJECT (update_dialog), "update_dialog", update_dialog);
  gtk_widget_set_usize (update_dialog, 480, -2);

  dialog_vbox1 = GNOME_DIALOG (update_dialog)->vbox;
  gtk_object_set_data (GTK_OBJECT (update_dialog), "dialog_vbox1", dialog_vbox1);
  gtk_widget_show (dialog_vbox1);

  vbox1 = gtk_vbox_new (FALSE, 1);
  gtk_object_set_data (GTK_OBJECT (update_dialog), "vbox1", vbox1);
  gtk_widget_show (vbox1);
  gtk_box_pack_start (GTK_BOX (dialog_vbox1), vbox1, TRUE, TRUE, 0);

  hseparator3 = gtk_hseparator_new ();
  gtk_object_set_data (GTK_OBJECT (update_dialog), "hseparator3", hseparator3);
  gtk_widget_show (hseparator3);
  gtk_box_pack_start (GTK_BOX (vbox1), hseparator3, TRUE, TRUE, 3);

  table1 = gtk_table_new (4, 2, FALSE);
  gtk_object_set_data (GTK_OBJECT (update_dialog), "table1", table1);
  gtk_widget_show (table1);
  gtk_box_pack_start (GTK_BOX (vbox1), table1, TRUE, TRUE, 0);
  gtk_table_set_row_spacings (GTK_TABLE (table1), 1);
  gtk_table_set_col_spacings (GTK_TABLE (table1), 2);

  connection_label = gtk_label_new ("Click Upload to begin.");
  gtk_object_set_data (GTK_OBJECT (update_dialog), "connection_label", connection_label);
  gtk_widget_show (connection_label);
  gtk_table_attach (GTK_TABLE (table1), connection_label, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (connection_label), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (connection_label), 7.45058e-09, 0.5);

  op_label = gtk_label_new (" ");
  gtk_object_set_data (GTK_OBJECT (update_dialog), "op_label", op_label);
  gtk_widget_show (op_label);
  gtk_table_attach (GTK_TABLE (table1), op_label, 0, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (op_label), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (op_label), 7.45058e-09, 0.5);

  file_label = gtk_label_new ("");
  gtk_object_set_data (GTK_OBJECT (update_dialog), "file_label", file_label);
  gtk_widget_show (file_label);
  gtk_table_attach (GTK_TABLE (table1), file_label, 0, 2, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (file_label), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (file_label), 7.45058e-09, 0.5);

  label3 = gtk_label_new ("Status: ");
  gtk_object_set_data (GTK_OBJECT (update_dialog), "label3", label3);
  gtk_widget_show (label3);
  gtk_table_attach (GTK_TABLE (table1), label3, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (label3), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label3), 7.45058e-09, 0.5);

  label6 = gtk_label_new ("To: ");
  gtk_object_set_data (GTK_OBJECT (update_dialog), "label6", label6);
  gtk_widget_show (label6);
  gtk_table_attach (GTK_TABLE (table1), label6, 0, 1, 3, 4,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (label6), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label6), 7.45058e-09, 0.5);

  dir_label = gtk_label_new (" ");
  gtk_object_set_data (GTK_OBJECT (update_dialog), "dir_label", dir_label);
  gtk_widget_show (dir_label);
  gtk_table_attach (GTK_TABLE (table1), dir_label, 1, 2, 3, 4,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (dir_label), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (dir_label), 7.45058e-09, 0.5);

  hseparator1 = gtk_hseparator_new ();
  gtk_object_set_data (GTK_OBJECT (update_dialog), "hseparator1", hseparator1);
  gtk_widget_show (hseparator1);
  gtk_box_pack_start (GTK_BOX (vbox1), hseparator1, TRUE, TRUE, 5);

  label2 = gtk_label_new ("Current Progress");
  gtk_object_set_data (GTK_OBJECT (update_dialog), "label2", label2);
  gtk_widget_show (label2);
  gtk_box_pack_start (GTK_BOX (vbox1), label2, TRUE, FALSE, 2);
  gtk_misc_set_alignment (GTK_MISC (label2), 0.5, 1);

  main_progressbar = gtk_progress_bar_new ();
  gtk_object_set_data (GTK_OBJECT (update_dialog), "main_progressbar", main_progressbar);
  gtk_widget_show (main_progressbar);
  gtk_box_pack_start (GTK_BOX (vbox1), main_progressbar, TRUE, FALSE, 0);
  gtk_progress_set_show_text (GTK_PROGRESS (main_progressbar), TRUE);

  label1 = gtk_label_new ("Total Progress");
  gtk_object_set_data (GTK_OBJECT (update_dialog), "label1", label1);
  gtk_widget_show (label1);
  gtk_box_pack_start (GTK_BOX (vbox1), label1, TRUE, FALSE, 2);
  gtk_misc_set_alignment (GTK_MISC (label1), 0.5, 1);

  job_progressbar = gtk_progress_bar_new ();
  gtk_object_set_data (GTK_OBJECT (update_dialog), "job_progressbar", job_progressbar);
  gtk_widget_show (job_progressbar);
  gtk_box_pack_start (GTK_BOX (vbox1), job_progressbar, TRUE, FALSE, 0);
  gtk_progress_set_show_text (GTK_PROGRESS (job_progressbar), TRUE);

  keep_going_button = gtk_check_button_new_with_label ("Keep going if there is a problem creating directories.");
  gtk_object_set_data (GTK_OBJECT (update_dialog), "keep_going_button", keep_going_button);
  gtk_widget_show (keep_going_button);
  gtk_box_pack_start (GTK_BOX (vbox1), keep_going_button, TRUE, TRUE, 0);

  dialog_action_area1 = GNOME_DIALOG (update_dialog)->action_area;
  gtk_object_set_data (GTK_OBJECT (update_dialog), "dialog_action_area1", dialog_action_area1);
  gtk_widget_show (dialog_action_area1);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area1), GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (dialog_action_area1), 8);

  gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (update_dialog),
                                          "Upload", GNOME_STOCK_PIXMAP_CONVERT);
  upload_button = g_list_last (GNOME_DIALOG (update_dialog)->buttons)->data;
  gtk_object_set_data (GTK_OBJECT (update_dialog), "upload_button", upload_button);
  gtk_widget_show (upload_button);
  GTK_WIDGET_SET_FLAGS (upload_button, GTK_CAN_DEFAULT);

  gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (update_dialog),
                                          "View Errors", GNOME_STOCK_PIXMAP_SEARCH);
  error_button = g_list_last (GNOME_DIALOG (update_dialog)->buttons)->data;
  gtk_object_set_data (GTK_OBJECT (update_dialog), "error_button", error_button);
  gtk_widget_show (error_button);
  gtk_widget_set_sensitive (error_button, FALSE);
  GTK_WIDGET_SET_FLAGS (error_button, GTK_CAN_DEFAULT);

  gnome_dialog_append_button (GNOME_DIALOG (update_dialog), GNOME_STOCK_BUTTON_CLOSE);
  close_button = g_list_last (GNOME_DIALOG (update_dialog)->buttons)->data;
  gtk_object_set_data (GTK_OBJECT (update_dialog), "close_button", close_button);
  gtk_widget_show (close_button);
  GTK_WIDGET_SET_FLAGS (close_button, GTK_CAN_DEFAULT);

  gtk_signal_connect (GTK_OBJECT (upload_button), "clicked",
                      GTK_SIGNAL_FUNC (upload_button_clicked),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (error_button), "clicked",
                      GTK_SIGNAL_FUNC (error_button_clicked),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (close_button), "clicked",
                      GTK_SIGNAL_FUNC (close_button_clicked),
                      NULL);

  gtk_widget_show (update_dialog);
}

void
create_error_log_window (void)
{
  GtkWidget *dialog_vbox2;
  GtkWidget *label9;
  GtkWidget *sc_win;
  GtkWidget *label7;
  GtkWidget *label8;
  GtkWidget *dialog_action_area2;
  GtkWidget *cancel;

  error_log_window = gnome_dialog_new ("Errors during the recent update", NULL);
  gtk_object_set_data (GTK_OBJECT (error_log_window), "error_log_window", error_log_window);
  gtk_widget_set_usize (error_log_window, 421, 248);
  gtk_window_set_policy (GTK_WINDOW (error_log_window), TRUE, TRUE, FALSE);

  dialog_vbox2 = GNOME_DIALOG (error_log_window)->vbox;
  gtk_object_set_data (GTK_OBJECT (error_log_window), "dialog_vbox2", dialog_vbox2);
  gtk_widget_show (dialog_vbox2);

  label9 = gtk_label_new ("There were errors with the following files and/or directories:");
  gtk_object_set_data (GTK_OBJECT (error_log_window), "label9", label9);
  gtk_widget_show (label9);
  gtk_box_pack_start (GTK_BOX (dialog_vbox2), label9, FALSE, FALSE, 0);
  gtk_label_set_justify (GTK_LABEL (label9), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label9), 7.45058e-09, 0.5);

  sc_win = gtk_scrolled_window_new (NULL, NULL);
  gtk_object_set_data (GTK_OBJECT (error_log_window), "sc_win", sc_win);
  gtk_widget_show (sc_win);
  gtk_box_pack_start (GTK_BOX (dialog_vbox2), sc_win, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sc_win), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

  error_log_list = gtk_clist_new (2);
  gtk_object_set_data (GTK_OBJECT (error_log_window), "error_log_list", error_log_list);
  gtk_widget_show (error_log_list);
  gtk_container_add (GTK_CONTAINER (sc_win), error_log_list);
  gtk_clist_set_column_width (GTK_CLIST (error_log_list), 0, 164);
  gtk_clist_set_column_width (GTK_CLIST (error_log_list), 1, 80);
  gtk_clist_column_titles_show (GTK_CLIST (error_log_list));

  label7 = gtk_label_new ("File/Directory Name");
  gtk_object_set_data (GTK_OBJECT (error_log_window), "label7", label7);
  gtk_widget_show (label7);
  gtk_clist_set_column_widget (GTK_CLIST (error_log_list), 0, label7);

  label8 = gtk_label_new ("Error code/message");
  gtk_object_set_data (GTK_OBJECT (error_log_window), "label8", label8);
  gtk_widget_show (label8);
  gtk_clist_set_column_widget (GTK_CLIST (error_log_list), 1, label8);

  dialog_action_area2 = GNOME_DIALOG (error_log_window)->action_area;
  gtk_object_set_data (GTK_OBJECT (error_log_window), "dialog_action_area2", dialog_action_area2);
  gtk_widget_show (dialog_action_area2);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area2), GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (dialog_action_area2), 8);

  gnome_dialog_append_button (GNOME_DIALOG (error_log_window), GNOME_STOCK_BUTTON_CLOSE);
  cancel = g_list_last (GNOME_DIALOG (error_log_window)->buttons)->data;
  gtk_object_set_data (GTK_OBJECT (error_log_window), "cancel", cancel);
  gtk_widget_show (cancel);
  GTK_WIDGET_SET_FLAGS (cancel, GTK_CAN_DEFAULT);

  gtk_signal_connect_object (GTK_OBJECT (error_log_window), "delete_event",
                             GTK_SIGNAL_FUNC (gtk_widget_hide),
                             GTK_OBJECT (error_log_window));
  gtk_signal_connect_object (GTK_OBJECT (cancel), "clicked",
                             GTK_SIGNAL_FUNC (gtk_widget_hide),
                             GTK_OBJECT (error_log_window));
}

static void confirmInit( GtkWidget *widget, gint button, gpointer data) {
  gpointer *prompt = gtk_object_get_data(GTK_OBJECT( widget ), "dialog");
  gint *b = (gint*)data;
  /*  gtk_widget_destroy(*prompt); */
  *prompt = NULL;
  *b = button;
}

gboolean should_init ()
{
  GtkWidget *state_question;
  GtkWidget *dialog_vbox2;
  GtkWidget *button3;
  GtkWidget *button4;
  GtkWidget *dialog_action_area2;
  gint button;

  state_question = gnome_message_box_new ("This appears to be the first time you have attempted to upload\nthis website. What is the state of the remote site?",
                              GNOME_MESSAGE_BOX_GENERIC, NULL);
  gtk_object_set_data (GTK_OBJECT (state_question), "state_question", state_question);
  GTK_WINDOW (state_question)->type = GTK_WINDOW_DIALOG;
  gtk_window_set_modal (GTK_WINDOW (state_question), TRUE);
  gtk_window_set_policy (GTK_WINDOW (state_question), FALSE, FALSE, FALSE);
  gnome_dialog_set_close (GNOME_DIALOG (state_question), TRUE);

  dialog_vbox2 = GNOME_DIALOG (state_question)->vbox;
  gtk_object_set_data (GTK_OBJECT (state_question), "dialog_vbox2", dialog_vbox2);
  gtk_widget_show (dialog_vbox2);

  gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (state_question),
                                          "Empty", GNOME_STOCK_PIXMAP_NEW);
  button3 = g_list_last (GNOME_DIALOG (state_question)->buttons)->data;
  gtk_widget_ref (button3);
  gtk_object_set_data_full (GTK_OBJECT (state_question), "button3", button3,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (button3);
  GTK_WIDGET_SET_FLAGS (button3, GTK_CAN_DEFAULT);

  gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (state_question),
                                          "Already uploaded", GNOME_STOCK_PIXMAP_MULTIPLE);
  button4 = g_list_last (GNOME_DIALOG (state_question)->buttons)->data;
  gtk_widget_ref (button4);
  gtk_object_set_data_full (GTK_OBJECT (state_question), "button4", button4,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (button4);
  GTK_WIDGET_SET_FLAGS (button4, GTK_CAN_DEFAULT);

  dialog_action_area2 = GNOME_DIALOG (state_question)->action_area;
  gtk_widget_ref (dialog_action_area2);
  gtk_object_set_data_full (GTK_OBJECT (state_question), "dialog_action_area2", dialog_action_area2,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_object_set_data (GTK_OBJECT(state_question), "dialog", &state_question);
  gtk_signal_connect (GTK_OBJECT(state_question), "clicked", 
							 confirmInit, &button);
  gtk_widget_show(state_question);
  button = -1;
  while (state_question)
	 gtk_main_iteration();
  
  if (button == 1)
	 return FALSE;
  return TRUE;
}

void uploadSite(void)
{
   gchar *method, *home;
   struct stat s;
	struct exclude_t *tmp_exclude;
	struct exclude_t *tmp_exclude2;
   /* Used for return values */
   int ret;
   
   /* We fill this with the single site to upload. */
   extern struct site_t *all_sites;
   
   if( ! current_site )
     return;

   dump_site_info();
   /* identify transfer method */
   method = screem_site_get_remote_method( current_site );
   
   /* Allocate memory for a sitecopy data structure and begin mapping screem
    * site attributes onto it.
    */
   tmp_site = malloc (sizeof (struct site_t));
   memset (tmp_site, 0, sizeof (struct site_t));
   all_sites = tmp_site;
   
   tmp_site->prev = NULL;
   tmp_site->next = NULL;
   
   tmp_site->name = screem_site_get_name( current_site );
   tmp_site->server = screem_site_get_remote_url( current_site );
   tmp_site->username = screem_site_get_remote_user( current_site );
   tmp_site->password = screem_site_get_remote_pass( current_site );
   tmp_site->url = screem_site_get_http_url( current_site );
   /* Here I'm constraining a site's method to one of 3 strings:
    * either "FTP", "Local" or "Webdav". The exact strings are pretty arbitrary.
    */
   if (strcmp ("FTP", method) == 0) {
      tmp_site->protocol = siteproto_ftp;
      tmp_site->driver = &ftp_driver;
#ifdef USE_DAV
   } else if (strcmp ("Webdav", method) == 0) {
      tmp_site->protocol = siteproto_http;
      tmp_site->driver = &dav_driver;
#endif /* USE_DAV */
   } else if (strcmp ("Local", method) == 0) {
      /* Here we can either call a manual local_upload() function, or
       * build something new into sitecopy. I think adding something to
       * sitecopy would be better.
       * Joe did say the same functionality we need for local has been requested
       * so we can either wait for him to add it, or implement it first. I don't
       * know how urgent/essential it is though, so I'll defer on a decision. :)
       */
     gnome_error_dialog("\"Local\" upload method is not currently supported.");
     return;
   } else {
	  return;
	}

   tmp_site->remote_root_user = screem_site_get_remote_path( current_site );
   if (*(tmp_site->remote_root_user) == '~') {
      tmp_site->remote_isrel = TRUE;
   } else if (*(tmp_site->remote_root_user) == '/') {
      tmp_site->remote_isrel = FALSE;
   } else {
      gnome_error_dialog("Your site's remote directory must begin with either a '/' or '~/'.");
      free (tmp_site);
      all_sites = NULL;
      return;
   }
   tmp_site->local_root_user = screem_site_get_pathname( current_site );
   if (*(tmp_site->local_root_user) == '~') {
     tmp_site->local_isrel = TRUE;
   } else if (*(tmp_site->local_root_user) == '/') {
     tmp_site->local_isrel = FALSE;
   } else {
     /* The software itself should already have ensured that the local dir
	begins with ~ or / - this error should never be seen by the user. */
      gnome_error_dialog ("The local directory must begin with either a '/' or '~/'.");
      return;
   }
   /* info_filename is where sitecopy keeps a record of which files have
    * been modified since the last update.
    */
   home = g_get_home_dir();
   printf ("home dir is %s\n", home);

   /* does home/.sitecopy exist? */
   tmp_site->infofile = g_strdup_printf( "%s/.sitecopy", home );
   if( stat( tmp_site->infofile, &s ) != 0 ) {
	   if( errno != ENOENT ) {
		   gnome_error_dialog ("Couldn't access $HOME/.sitecopy directory" );
		   g_free( tmp_site->infofile );
		   return;
	   }
	   /* it doesn't exist so lets make it */
	   if( mkdir( tmp_site->infofile, S_IRWXU ) != 0 ) {
		   /* couldn't make it */
		   gnome_error_dialog ("Couldn't create $HOME/.sitecopy directory" );
		   g_free( tmp_site->infofile );
		   return;
	   }
   }
   g_free( tmp_site->infofile );

   tmp_site->infofile = g_strdup_printf("%s/.sitecopy/%s", home, 
					tmp_site->name);
   printf ("infofile is: \t%s\n", tmp_site->infofile);

   /* stat the infofile */
	memset(&s, 0, sizeof(struct stat));
   if (stat(tmp_site->infofile, &s) != 0) {
	  /* Displays a dialog asking if the user wants to init or catchup their website. */
	  if (should_init()) {
		 printf("Initializing site...");
		 site_initialize(tmp_site);
		 printf("done.\n");
	  } else {
		 printf("Catching up this site...");
		 site_catchup(tmp_site);
		 site_writefiles(tmp_site);
		 printf("done.\n");
		 /* If the site is up to date, we really don't need to upload it! */
		 return;
	  }
	}

   tmp_site->files = NULL;
   tmp_site->files_tail = NULL;

   tmp_site->ftp_pasv_mode = TRUE; /* make this optional probably */   
   tmp_site->perms = sitep_exec; /* can make this an option */
   tmp_site->symlinks = sitesym_ignore;  /* can make this an option */
   
   /* Sensible defaults. If required, these could be configured with an
    * 'advanced' config dialog, but stuff would need adding to screem's
    * _Site struct to do that nicely I think.
    */
   tmp_site->nodelete = FALSE;
   tmp_site->checkmoved = TRUE;
   tmp_site->nooverwrite = FALSE;
   /* we need to exclude the screem project file otherwise its a major
      security problem because some FTP servers won't set permissions */
	tmp_exclude2 = malloc (sizeof(struct exclude_t));
	tmp_exclude2->haspath = FALSE;
	tmp_exclude2->pattern = strdup("CVS");
	tmp_exclude2->prev = NULL;

	tmp_exclude = malloc (sizeof(struct exclude_t));
	tmp_exclude->haspath = FALSE;
	tmp_exclude->pattern = strdup("*.screem");
	tmp_exclude->prev = tmp_exclude2;
	tmp_exclude->next = NULL;

	tmp_exclude2->next = tmp_exclude;

   tmp_site->excludes = tmp_exclude2;
   tmp_site->use_this = TRUE;
   
   /* Sets up things like the port number, internally used directory names
    * and also ensures that passwords/usernames etc. are all valid.
    * Additionally stops sitecopy from attempting operations it doesn't support.
    */
   switch (ret = rc_verifysite (tmp_site)) {
   case SITE_NONAME:
     gnome_error_dialog ("No site name given.\n");
     break;
   case SITE_NOSERVER:
     gnome_error_dialog ("No server name given.\n");
     break;
   case SITE_NOUSERNAME:
     gnome_error_dialog ("No username given.\n");
     break;
   case SITE_NOPASSWORD:
     gnome_error_dialog ("No password given.\n");
     break;
   case SITE_NOREMOTEDIR:
     gnome_error_dialog ("No remote directory given.\n");
     break;
   case SITE_NOLOCALDIR:
     gnome_error_dialog ("No local directory given.\n");
     break;
   case SITE_ACCESSLOCALDIR:
     gnome_error_dialog ("The local dir couldn't be accessed.\n");
     break;
   case SITE_INVALIDPORT:
     gnome_error_dialog ("There was a problem with the port number for this site.");
     break;
   case SITE_NOMAINTAIN:
      gnome_error_dialog ("Sorry, the symbolic links option you chose is not supported\nby this transfer protocol. Please choose another option.");
      break;
    case SITE_NOREMOTEREL:
      if (tmp_site->remote_isrel) {
	gnome_error_dialog ("This protocol does not support relative remote directories.\n");
      }
      break;
    case SITE_NOLOCALREL:
      gnome_error_dialog ("The local dir is invalid.\n");
      break;
    case 0:
      /* Site integrity is okay. */
      break;
    default:
      gnome_error_dialog ("There was an undetermined problem verifying the correctness of your site definition. Please report this to the maintainer.");
      break;
   }
   
   /* Let's die if the site was invalid. */
   if (ret != 0) {
     return;
   }
      
   /* Do some scanning to find out about the local site's files */
   if ( (site_readfiles (tmp_site)) != 0) {
      gnome_error_dialog ("Could not retrieve information about your local files.");
      return;
   }
	if ( !tmp_site->is_different) {
	  gnome_ok_dialog("The remote site is already up to date!");
	  return;
	}
   /* Now we're pretty sure that we have all the info we need, let's show the
    * pretty upload dialog box.
    */
   create_update_dialog();
}

void *update_thread(void *bler) {
   int ret;
	gboolean tmp;
	
	upload_in_progress = TRUE;   
   
	/*pthread_detach(pthread_self());*/

	if (GTK_TOGGLE_BUTTON(keep_going_button)->active) {
	  ret = site_update(tmp_site, TRUE);
	} else {
	  ret = site_update(tmp_site, FALSE);
	}
   
   printf ("update_thread started.\n");

   switch (ret) {
    case SITE_OK:
      /* be sure to add some 'free' commands in here */
      site_writefiles(tmp_site);
      break;
    case SITE_AUTH:
      gnome_warning_dialog ("There was a problem authenticating with the remote server.");
      return NULL;
    case SITE_LOOKUP:
      gnome_warning_dialog ("Could not resolve the remote site's hostname.");
      return NULL;
    case SITE_CONNECT:
      gnome_warning_dialog ("Could not establish a connection to the remote site.");
      return NULL;
    case SITE_ERRORS:
      /* replace this with a custom dialog containing the errors, a la Xsitecopy */
      gnome_warning_dialog ("There were errors.");      
      return NULL;
   }

   /* Free the memory created in site_readfiles. */
   site_destroy (tmp_site);
   
   /* clean up */
   free (tmp_site);
   all_sites = NULL;
   upload_in_progress = FALSE;
   printf("upload thread just finishing...\n");
   return NULL;
}

/********************************************************************/
/**** Implementations of all the functions defined in frontend.h ****/
/**** These functions should be considered 'private' and never   ****/
/**** be called directly.                                        ****/
/********************************************************************/

/** Informs the user when we're trying to (re)establish a connection
 *  or lookup a hostname. 
 */
void fe_connection( fe_conn_status status ) {
   switch(status) {
    case (0):
      gtk_label_set (GTK_LABEL (connection_label), "Looking up hostname...");
      break;
    case (1):
      gtk_label_set (GTK_LABEL (connection_label), "Attempting to connect...");
      break;
    case (2):
      gtk_label_set (GTK_LABEL (connection_label), "Connected.");
      break;
   }
   while (gtk_events_pending() > 0)
     gtk_main_iteration();
}

/* FIXME: use David's blocking routine here */

bool fe_can_update (const struct site_file_t *file) {
   return TRUE;
}

void fe_updating (const struct site_file_t *file) {
   char *file_status = NULL;
  
   file_status = g_strdup_printf ("Commiting updates to %s...", tmp_site->server);
   gtk_label_set (GTK_LABEL (connection_label), file_status);
   g_free(file_status);
   if (file->dir) {
     if (file->diff == file_new) {
       gtk_label_set (GTK_LABEL (op_label), "Creating directory...");
       gtk_label_set (GTK_LABEL (file_label), file->full_local);
       gtk_label_set (GTK_LABEL (dir_label), "");
     } else {
       /* can we move dirs yet? */
       gtk_label_set (GTK_LABEL (op_label), "Deleting directory...");
       gtk_label_set (GTK_LABEL (dir_label), "");
     }
   } else {
     switch (file->diff) {
     case file_changed:
       gtk_label_set (GTK_LABEL (op_label), "Uploading...");
       gtk_label_set (GTK_LABEL (file_label), file->full_local);
       gtk_label_set (GTK_LABEL (dir_label), file->full_remote);
       break;
     case file_new:
       gtk_label_set (GTK_LABEL (op_label), "Uploading...");
       gtk_label_set (GTK_LABEL (file_label), file->full_local);
       gtk_label_set (GTK_LABEL (dir_label), file->full_remote);
       break;
     case file_deleted:
       gtk_label_set (GTK_LABEL (op_label), "Deleting...");
       gtk_label_set (GTK_LABEL (file_label), file->full_local);
       gtk_label_set (GTK_LABEL (dir_label), file->full_remote);
       break;
     case file_moved:
       gtk_label_set (GTK_LABEL (op_label), "Moving...");
       gtk_label_set (GTK_LABEL (file_label), file->old->full_local);
       gtk_label_set (GTK_LABEL (dir_label), file->directory);
       break;
     case file_unchanged:
       gtk_label_set (GTK_LABEL (op_label), "ARG! The file hasn't changed, we shouldn't be doing anything!");
     }
     
   }
   while (gtk_events_pending() > 0)
     gtk_main_iteration();
}

/* Once a file has been updated, any errors with it are recorded in the 
 * error list, and the progress bar is reset. 
 */
void fe_updated (const struct site_file_t *file, const bool success,
		 const char *error) {
   gchar *error_item[2];
   
   if (!success) {
      error_item[0] = g_strdup(file->full_local);
      error_item[1] = g_strdup(error);
      gtk_clist_append (GTK_CLIST (error_log_list), error_item);
      g_free(error_item[0]);
      g_free(error_item[1]);
   }
   gtk_progress_bar_update (GTK_PROGRESS_BAR (main_progressbar), 0.0);
   uploaded_bytes += (float) file->localsize;
   while (gtk_events_pending() > 0)
     gtk_main_iteration();
}

void fe_transfer_progress (size_t progress, size_t total) {
   
   float percentage, div1, div2;
   size_t upload_size;

   /* Calculate the amount to display the current file's progress */
   div1 = (float) progress;
   div2 = (float) total;

   percentage = div1 / div2;
   gtk_progress_bar_update (GTK_PROGRESS_BAR (main_progressbar), percentage);
   
   /* Calculate the amount to display for total job progress */
   upload_size = tmp_site->totalnew + tmp_site->totalchanged;
   gtk_progress_bar_update (GTK_PROGRESS_BAR (job_progressbar),
			    (uploaded_bytes + (float) progress) / (float) upload_size);

   /* Keep the interface sensitive */
   while (gtk_events_pending() > 0)
     gtk_main_iteration();
}

/* These can be implemented if we ever want to use the respective 
 * functionality.
 */
void fe_synching( const struct site_file_t *file ){
}
void fe_synched( const struct site_file_t *file, const bool success,
	       const char *error ){
}
void fe_fetch_found( const struct site_file_t *file ){
}
