/* 
   sitecopy, for managing remote web sites.
   Copyright (C) 1998-99, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   $Id: rcfile.c,v 1.1.1.1 1999/11/21 19:47:17 davek Exp $
*/

#include <config.h>

#include <sys/types.h>

#include <sys/stat.h>

#include <ctype.h>
#include <errno.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif 
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include "common.h"
#include "netrc.h"
#include "rcfile.h"
#include "socket.h"
#include "sites.h"

/** Global variables **/
char *copypath;
char *rcfile;
char *netrcfile;
char *home;
bool havenetrc;

/* These are used for reporting errors back to the calling procedures. */
int rcfile_linenum; 
char *rcfile_err;

/** Not quite so global variables **/

/* These are appeneded to $HOME */
#define RCNAME "/." PACKAGE "rc"
#define COPYNAME "/." PACKAGE "/"
#define NETRCNAME "/.netrc"

/* Stores the list of entries in the ~/.netrc */
netrc_entry *netrc_list;

const char *rc_get_netrc_password( const char *server, const char *username );

/* rcfile_read will read the rcfile and fill given sites list.
 * This returns 0 on success, RC_OPENFILE if the rcfile could not
 * be read, or RC_CORRUPT if the rcfile was corrupt.
 * If it is corrupt, rcfile_linenum and rcfile_line are set to the
 * the corrupt line.
 */
int rcfile_read( struct site_t **sites ) {
    FILE *fp;
    int state, last_state=8, ret=0;
    bool alpha, hash;
    char buf[BUFSIZ];
    char *ch;
    char *ptr, key[BUFSIZ], val[BUFSIZ], val2[BUFSIZ];
    /* Holders for the site info */
    struct site_t *this_site, *last_site;

    if( (fp = fopen( rcfile, "r" )) == NULL ) {
	rcfile_err = strerror( errno );
	return RC_OPENFILE;
    } 

    last_site = this_site = NULL;
    rcfile_linenum = 0;
    rcfile_err = NULL;

    while( fgets( buf, sizeof(buf), fp ) != NULL ) {
	rcfile_linenum++;
	/* Put the line without the LF into the error buffer */
	if( rcfile_err != NULL ) free( rcfile_err );
	rcfile_err = strdup( buf );
	ptr = strchr( rcfile_err, '\n' );
	if( ptr != NULL ) *ptr = '\0';
	state = 0;
	ptr = key;
	memset( key, 0, BUFSIZ );
	memset( val, 0, BUFSIZ );
	memset( val2, 0, BUFSIZ );
	for( ch=buf; *ch!='\0'; ch++ ) {
	    alpha = !isspace((unsigned)*ch); /* well, alphaish */
	    hash = (*ch == '#');
	    switch( state ) {
	    case 0: /* whitespace at beginning of line */
		if(hash) {
		    state = 8;
		} else if(alpha) {
		    *(ptr++) = *ch;
		    state = 1;
		}
		break;
	    case 1: /* key */
		if(hash) {
		    state = 8;
		} else if(!alpha) {
		    ptr = val;
		    state = 2;
		} else {
		    *(ptr++) = *ch;
		}
		break;
	    case 2: /* whitespace after key */
		if(hash) {
		    state = 8;
		} else if( *ch == '"' ) {
		    state = 4; /* begin quoted value */
		} else if ( alpha ) {
		    *(ptr++) = *ch;
		    state = 3;
		} 
		break;
	    case 3: /* unquoted value 1 */
		if(hash) {
		    state = 8;
		} else if( !alpha ) {
		    ptr = val2;
		    state = 5;
		} else {
		    *(ptr++) = *ch;
		}
		break;
	    case 4: /* quoted value 1 */
		if( *ch == '"' ) {
		    ptr = val2;
		    state = 5;
		} else if( *ch == '\\' ) {
		    last_state = 4;
		    state = 9;
		} else {
		    *(ptr++) = *ch;
		}
		break;
	    case 5: /* whitespace after value 1 */
		if(hash) {
		    state = 8;
		} else if( *ch == '"' ) {
		    state = 6; /* begin quoted value 2 */
		} else if ( alpha ) {
		    *(ptr++) = *ch;
		    state = 7; /* begin unquoted value 2 */
		} 
		break;
	    case 6: /* quoted value 2 */
		if( *ch == '"' ) {
		    state = 8;
		} else if( *ch == '\\' ) {
		    last_state = 4;
		    state = 9;
		} else {
		    *(ptr++) = *ch;
		}
		break;
	    case 7: /* unquoted value 2 */
		if(hash) {
		    state = 8;
		} else if( !alpha ) {
		    state = 8;
		} else {
		    *(ptr++) = *ch;
		}
		break;
	    case 8: /* ignore till end of line */
		break;
	    case 9: /* a literal (\-slashed) in a value */
		*(ptr++) = *ch;
		state = last_state;
		break;
	    }
	}
	
	DEBUG( DEBUG_RCFILE, "Key [%s] Value: [%s] Value2: [%s]\n", key, val, val2);
	
	if( strlen( key ) == 0 ) {
	    continue;
	}
	if( strlen( val ) == 0 ) {
	    /* A key with no value. */
	    if( this_site == NULL ) {
		/* Need to be in a site! */
		ret = RC_CORRUPT;
		break;
	    }
	    if( strcmp( key, "nodelete" ) == 0 ) {
		this_site->nodelete = true;
	    } else if( strcmp( key, "checkmoved" ) == 0 ) {
		this_site->checkmoved = true;
	    } else if( strcmp( key, "nooverwrite" ) == 0 ) {
		this_site->nooverwrite = true;
	    } else {
		ret = RC_CORRUPT;
		break;
	    }
	} else if( strlen( val2 ) == 0 ) {
	    /* A key with a single value. */
	    if( strcmp( key, "site" ) == 0 ) {
		/* Beginning of a new Site */
		last_site = this_site;
		/* Allocate new item */
		this_site = malloc( sizeof( struct site_t ) );
		/* Zero it out */
		memset( this_site, 0, sizeof( struct site_t ) );
		this_site->prev = last_site;
		if( last_site != NULL ) { /* next site */
		    last_site->next = this_site;
		} else { /* First site */
		    *sites = this_site;
		}
		this_site->name = strdup( val );
		this_site->files = NULL;
		/* defaults... */
		this_site->perms = sitep_ignore;
		this_site->symlinks = sitesym_follow;
		this_site->nodelete = false;
		this_site->checkmoved = false;
		this_site->driver = &ftp_driver;
		this_site->ftp_pasv_mode = true;
		/* Now work out the info filename */
		this_site->infofile = malloc( strlen(copypath) + strlen(val) + 1);
		strcpy( this_site->infofile, copypath );
		strcat( this_site->infofile, val );
	    } else if( this_site == NULL ) {
		ret = RC_CORRUPT;
		break; 
	    } else if( strcmp( key, "username" ) == 0 ) {
		/* username */
		this_site->username = strdup( val );
	    } else if( strcmp( key, "server" ) == 0 ) {
		this_site->server = strdup( val );
	    } else if( strcmp( key, "port" ) == 0 ) {
		this_site->port = atoi( val );
	    } else if( strcmp( key, "password" ) == 0 ) {
		this_site->password = strdup( val );
	    } else if( strcmp( key, "url" ) == 0 ) {
	        this_site->url = strdup( val );
	    } else if( strcmp( key, "remote" ) == 0 ) {
		/* Relative filenames must start with "~/" */
		if( val[0] == '~' ) {
		    if( val[1] == '/' ) {
			this_site->remote_isrel = true;
		    } else {
			ret = RC_CORRUPT;
			break; 
		    }
		} else {
		    /* Dirname doesn't begin with "~/" */
		    this_site->remote_isrel = false;
		}
		if( val[strlen(val)-1] != '/' )
		    strcat( val, "/" );
		this_site->remote_root_user = strdup( val );
	    } else if( strcmp( key, "local" ) == 0 ) {
		/* Relative filenames must start with "~/" */
		if( val[0] == '~' ) {
		    if( val[1] == '/' ) {
			this_site->local_isrel = true;
		    } else {
			ret = RC_CORRUPT;
			break; 
		    }
		} else { 
		    /* Dirname doesn't begin with a "~/" */
		    this_site->local_isrel = false;
		}
		if( val[strlen(val)-1] != '/' )
		    strcat( val, "/" );
		this_site->local_root_user = strdup( val );
	    } else if( strcmp( key, "permissions") == 0 ) {
		if( strcmp( val, "ignore" ) == 0 ) {
		    this_site->perms = sitep_ignore;
		} else if( strcmp( val, "exec" ) == 0 ) {
		    this_site->perms = sitep_exec;
		} else if( strcmp( val, "all" ) == 0 ) {
		    this_site->perms = sitep_all;
		} else {
		    ret = RC_CORRUPT;
		    break;
		}
	    } else if( strcmp( key, "symlinks" ) == 0 ) {
		if( strcmp( val, "follow" ) == 0 ) {
		    this_site->symlinks = sitesym_follow;
		} else if( strcmp( val, "maintain" ) == 0 ) {
		    this_site->symlinks = sitesym_maintain;
		} else if( strcmp( val, "ignore" ) == 0 ) {
		    this_site->symlinks = sitesym_ignore;
		} else {
		    ret = RC_CORRUPT;
		    break;
		}
	    } else if( strcmp( key, "exclude" ) == 0 ) {
		struct exclude_t *newex;
		newex = malloc( sizeof( struct exclude_t ) );
		newex->next = this_site->excludes;
		newex->prev = NULL;
		if( this_site->excludes != NULL )
		    this_site->excludes->prev = newex;
		this_site->excludes = newex;
		newex->pattern = strdup( val );
		if( strchr( val, '/' ) != NULL ) {
		    newex->haspath = true;
		} else {
		    newex->haspath = false;
		}
	    } else if( strcmp( key, "ascii") == 0 ) {
		/* Find the position in the array */
		this_site->ascii[this_site->numascii] = strdup(val);
		this_site->numascii++;
	    } else if( strcmp( key, "protocol" ) == 0 ) {
		if( strcmp( val, "ftp" ) == 0 ) {
		    this_site->protocol = siteproto_ftp;
		    this_site->driver = &ftp_driver;
		}
#ifdef USE_DAV
		else if( strcmp( val, "http" ) == 0 ) {
		    this_site->protocol = siteproto_http;
		    this_site->driver = &dav_driver;
		}
#endif /* USE_DAV */
	        else {
		    ret = RC_CORRUPT;
		    break;
		}
	    } else if( strcmp( key, "ftp" ) == 0 ) {
		if( strcmp( val, "nopasv" ) == 0 ) {
		    this_site->ftp_pasv_mode = false;
		} else {
		    ret = RC_CORRUPT;
		}
	    } else {
		/* Unknown key! */
		ret = RC_CORRUPT;
		break;
	    }
	} else {
	    {
		ret = RC_CORRUPT;
		break; 
	    }
	}
    }

    fclose( fp );
    return ret;
}

const char *rc_get_netrc_password( const char *server, const char *username ) {
    netrc_entry *found;
    found = search_netrc( netrc_list, server );
    if( found == NULL ) {
	return NULL;
    }
    if( strcmp( found->account, username ) == 0 ) {
	return found->password;
    } else {
	return NULL;
    }
}

/* Returns zero if site is properly defined, else non-zero */
int rc_verifysite( struct site_t *any_site ) {
    struct stat localst;
    char *temp;
    int ret;

    /* Check they specified everything in the rcfile */

    if( any_site->server == NULL ) {
	return SITE_NOSERVER;
    } else if( any_site->username == NULL ) {
	return SITE_NOUSERNAME;
    } else if( any_site->password == NULL ) {
	if( havenetrc ) {
	    const char *pass;
	    DEBUG( DEBUG_RCFILE, "Checking netrc for password for %s@%s...",
		   any_site->username, any_site->server );
	    pass = rc_get_netrc_password( any_site->server, any_site->username );
	    if( pass == NULL ) {
		DEBUG( DEBUG_RCFILE, "none found.\n" );
		return SITE_NOPASSWORD;
	    } else {
		DEBUG( DEBUG_RCFILE, "found!\n" );
		any_site->password = (char *) pass;
	    }
	} else {
	    return SITE_NOPASSWORD;
	}
    }

    if( any_site->remote_root_user == NULL ) {
	return SITE_NOREMOTEDIR;
    } else if( any_site->local_root_user == NULL ) {
	return SITE_NOLOCALDIR;
    }
    
    /* Need a home directory if we're using relative local root */
    if( home == NULL && any_site->local_root )
	return SITE_NOLOCALREL;

    if( any_site->remote_isrel ) {
	any_site->remote_root = strdup( any_site->remote_root_user + 2 );
    } else {
	any_site->remote_root = strdup( any_site->remote_root_user );
    }
    if( any_site->local_isrel ) {
	/* We skip the first char ('~') of l_r_u */
	temp = malloc( strlen( any_site->local_root_user ) + 
		       strlen( home ) );
	strcpy( temp, home );
	strcat( temp, any_site->local_root_user + 1 );
	any_site->local_root = temp;
    } else {
	any_site->local_root = any_site->local_root_user;
    }

    /* Now check the local directory actually exists.
     * To do this, stat `/the/local/root/.', which will fail if the
     * can't read the directory or if it's a file not a directory */
    temp = malloc( strlen( any_site->local_root ) + 2 );
    strcpy( temp, any_site->local_root ); /* must have trailing / */
    strcat( temp, "." ); 
    ret = stat( temp, &localst );
    free( temp );
    if( ret != 0 ) {
	return SITE_ACCESSLOCALDIR;
    }
    /* Protocol-specific checks */
    switch( any_site->protocol ) {
    case siteproto_ftp:
	/* FTP checks */
	if( any_site->symlinks == sitesym_maintain ) {
	    return SITE_NOMAINTAIN;
	}
	break;
    case siteproto_http:
	/* HTTP checks */
	if( any_site->remote_isrel ) { 
	    return SITE_NOREMOTEREL;
	}
	/* fall through */
    default:
	break;
    }
	  
    /* Assign default ports if they didn't bother to */
    if( any_site->port == 0 ) {
	DEBUG( DEBUG_RCFILE, "No port given... finding default.\n" );
	any_site->port = get_tcp_port( any_site->driver->service_name );
	if( any_site->port == 0 ) {
	    ret = SITE_INVALIDPORT;
	} else {
	    DEBUG( DEBUG_RCFILE, "Assigned default port: %d\n", any_site->port );
	}
    }
    return 0;
}

int init_netrc( ) {
    if( !havenetrc ) return 0;
    netrc_list = parse_netrc( netrcfile );
    if( netrc_list == NULL ) {
	/* Couldn't parse it */
	return 1;
    } else {
	/* Could parse it */
	return 0;
    }
}

/* Checks the perms of the rcfile and site storage directory. */
int init_paths( ) {
    struct stat st;
    if( stat( rcfile, &st ) < 0 ) {
	DEBUG( DEBUG_RCFILE, "stat failed on %s: %s\n", 
	       rcfile, strerror(errno) );
	return RC_OPENFILE;
    }
#if !defined (__EMX__) && !defined(__CYGWIN__)
    if( (st.st_mode & ~(S_IFREG | S_IREAD | S_IWRITE )) > 0 ) {
	return RC_PERMS;
    }
#endif
    if( (netrcfile == 0) || (stat( netrcfile, &st ) < 0) ) {
	havenetrc = false;
#if !defined (__EMX__) && !defined(__CYGWIN__)
    } else if( (st.st_mode & ~(S_IFREG | S_IREAD | S_IWRITE )) > 0 ) {
	return RC_NETRCPERMS;
#endif
    } else {
	havenetrc = true;
    }
    if( stat( copypath, &st ) < 0 ) {
	DEBUG( DEBUG_RCFILE, "stat failed on %s: %s\n", 
	       copypath, strerror(errno) );
	return RC_DIROPEN;
    }
#if !defined (__EMX__) && !defined(__CYGWIN__)
    if( (st.st_mode & ~(S_IFDIR | S_IREAD | S_IWRITE | S_IEXEC )) > 0 ) {
	return RC_DIRPERMS;
    }
#endif
    return 0;
}

int init_env( ) {
    /* Assign default filenames if they didn't give us any */
    home = getenv("HOME");
    if( home == NULL ) {
	if( ( rcfile == NULL ) || ( copypath == NULL ) ) {
	    /* We need a $HOME or both rcfile and info dir path */
	    return 1;
	} else {
	    /* No $HOME, but we've got the rcfile and info dir path */
	    return 0;
	}
    }
    if( rcfile == NULL ) {
	rcfile = malloc( strlen(RCNAME) + strlen(home) + 1);
	strcpy( rcfile, home );
	strcat( rcfile, RCNAME );
    }
    if( copypath == NULL ) {
	copypath = malloc( strlen(COPYNAME) + strlen(home) + 1);
	strcpy( copypath, home );
	strcat( copypath, COPYNAME );
    }
    netrcfile = malloc( strlen(NETRCNAME) + strlen(home) + 1 );
    strcpy( netrcfile, home );
    strcat( netrcfile, NETRCNAME );
    return 0;
}

/* rcfile_write() by Lee Mallabone, cleaned by JO.
 * Write the contents of list_of_sites to the specified 'filename'
 * in the standard sitecopy rc format.
 *
 * Any data already in 'filename' is over-written.
 */
int rcfile_write (char *filename, struct site_t *list_of_sites) {
   
   struct site_t *current;
   struct exclude_t *excl;

   FILE *fp;
   
   if ( (fp = fopen (filename, "w")) == NULL) {
      printf ("There was a problem writing to the sitecopy configuration file.\n\nCheck permissions on %s.", filename);
      return RC_OPENFILE;
   }
   /* Set rcfile permissions properly */
#if !defined (__EMX__) && !defined(__CYGWIN__)
   if (fchmod (fileno(fp), 00600) == -1) {
      return RC_PERMS;
   }
#endif
   
   for (current=list_of_sites; current!=NULL; current=current->next) {
      /* Okay so this maybe isn't the most intuitive thing to look at.
       * With any luck though, the rcfile's it produces will be. :) */
      if (fprintf (fp, "site %s\n", current->name) == -1) {
	 return RC_CORRUPT;
      }
      if (fprintf (fp, "  server %s\n  %s%susername %s\n  password %s\n  remote %s\n  local %s\n%s%s",
	       current->server, 
	       current->ftp_pasv_mode?"":"ftp nopasv\n  ",
	       current->nooverwrite?"nooverwrite\n  ":"",
	       current->username,
	       current->password, current->remote_root_user,
	       current->local_root_user, current->nodelete?"  nodelete\n":"",
	       current->checkmoved?"  checkmoved\n":"") == -1) {
	 return RC_CORRUPT;
      }
      if (fprintf (fp, "  protocol %s\n", current->protocol==siteproto_ftp?"ftp":"http") == -1) {
	 return RC_CORRUPT;
      }
      if (current->port > 0)  { /* Sanity check */
	 if (fprintf (fp, "  port %d\n", current->port) == -1) {
	    return RC_CORRUPT;
	 }
      }
      
      /* Add the site's URL if one has been supplied. */
      if (current->url) {
	 if (fprintf (fp, "  url %s\n", current->url) == -1) {
	    return RC_CORRUPT;
	 }
      }
      
      /* Permissions now */
      switch (current->perms) {
       case (sitep_ignore): 
	 if (fprintf (fp, "  permissions ignore\n") == -1) {
	    return RC_CORRUPT;
	 }
	 break;
       case (sitep_exec):
	 if (fprintf (fp, "  permissions exec\n") == -1) {
	    return RC_CORRUPT;
	 }
	 break;
       case (sitep_all):
	 if (fprintf (fp, "  permissions all\n") == -1) {
	    return RC_CORRUPT;
	 }
	 break;
      }

      /* Sym link mode */
      switch (current->symlinks) {
       case (sitesym_ignore): 
	 if (fprintf (fp, "  symlinks ignore\n") == -1) {
	    return RC_CORRUPT;
	 }
	 break;
       case (sitesym_follow):
	 if (fprintf (fp, "  symlinks follow\n") == -1) {
	    return RC_CORRUPT;
	 }
	 break;
       case (sitesym_maintain):
	 if (fprintf (fp, "  symlinks maintain\n") == -1) {
	    return RC_CORRUPT;
	 }
	 break;
      }

      /* Excludes */
      for (excl=current->excludes; excl!=NULL; excl=excl->next) {
	  if (fprintf (fp, "  exclude \"%s\"\n", excl->pattern) == -1) {
		return RC_CORRUPT;
	    }
      }
   }
   if (fclose (fp) != 0)
      return RC_CORRUPT;
   return 0;
}
