/* gxset - GTK interface to xset(1)

   Copyright (C) 1999 Ren Seindal (rene@seindal.dk)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/resource.h>
#include <sys/wait.h>

#include "interface.h"
#include "support.h"
#include "callbacks.h"

#include "main.h"
#include "guistuff.h"

XsetInfo *
xset_info_new(void)
{
    XsetInfo *info;

    info = (XsetInfo *)g_malloc(sizeof(struct XsetInfo));

    info->keyrepeat_onoff = 0;
    info->keyrepeat_delay = 0;
    info->keyrepeat_rate = 0;
    info->keyclick_volume = 0;

    info->bell_volume = 0;
    info->bell_pitch = 0;
    info->bell_duration = 0;

    info->mouse_acceleration = 1;
    info->mouse_divisor = 1;
    info->mouse_threshold = 1;

    info->scrsaver_onoff = 0;
    info->scrsaver_blank = 0;
    info->scrsaver_expose = 0;
    info->scrsaver_delay = 0;
    info->scrsaver_cycle = 0;

    info->dpms_onoff = 0;
    info->dpms_standby = 0;
    info->dpms_suspend = 0;
    info->dpms_off = 0;

    info->font_path = g_ptr_array_new();

    return info; 
}

static void
init_string_vector(GPtrArray *dest, gchar **strs)
{
    int i;

    if (dest != NULL)
	g_ptr_array_free(dest, TRUE);

    dest = g_ptr_array_new();
    for (i = 0; strs[i]; i++) {
	g_ptr_array_add(dest, g_strdup(strs[i]));
    }
}

static GPtrArray *
copy_string_vector(GPtrArray *vec)
{
    GPtrArray *dest;
    int i;

    dest = g_ptr_array_new();
    for (i = 0; i < vec->len; i++) {
	g_ptr_array_add(dest, g_strdup((gchar *)g_ptr_array_index(vec, i)));
    }

    return dest;
}

XsetInfo *
xset_info_dup(XsetInfo *orig)
{
    XsetInfo *info;

    info = (XsetInfo *)g_malloc(sizeof(struct XsetInfo));

    *info = *orig;		/* copies all simple fields */
    info->font_path = copy_string_vector(orig->font_path);

    return info; 
}

void
xset_info_assign(XsetInfo *lval, XsetInfo *rval)
{
    g_ptr_array_free(lval->font_path, TRUE);

    *lval = *rval;		/* copies all simple fields */
    lval->font_path = copy_string_vector(rval->font_path);
}

void
xset_info_delete(XsetInfo *info)
{
    g_ptr_array_free(info->font_path, TRUE);
    g_free(info);
}


enum XsetInputType {
    TYPE_FALSE,
    TYPE_TRUE,
    TYPE_ON_OFF,
    TYPE_INT,
    TYPE_FRACTION,
};
typedef enum XsetInputType XsetInputType;

static void
contains(const char *s, const char *kw, XsetInputType t, int *p1, int *p2)
{
    const char *found;
    const char stmp[128];
    int itmp, itmp2;

    found = strstr(s, kw);
    if (found) {
	switch (t) {
	case TYPE_FALSE:
	    *p1 = 0;
	    break;
	case TYPE_TRUE:
	    *p1 = 1;
	    break;
	case TYPE_ON_OFF:
	    if (sscanf(found+strlen(kw), "%s", stmp) == 1) {
		if ((g_strcasecmp(stmp, "on") == 0 
		     || g_strcasecmp(stmp, "yes") == 0))
		    *p1 = 1;
		if ((g_strcasecmp(stmp, "off") == 0
		     || g_strcasecmp(stmp, "no") == 0))
		    *p1 = 0;
	    }
	    break;
	case TYPE_INT:
	    if (sscanf(found+strlen(kw), "%d", &itmp) == 1) {
		*p1 = itmp;
	    }
	    break;
	case TYPE_FRACTION:
	    if (sscanf(found+strlen(kw), "%d/%d", &itmp, &itmp2) == 2) {
		*p1 = itmp;
		*p2 = itmp2;
	    }
	    break;
	default:
	    g_error("Internal botch in parse:contains()");
	    break;
	}
    }

    return;
}

XsetInfo *
xset_info_read(void)
{
    FILE *fp;
    char buf[1024];
    XsetInfo *info;
    gchar **tmp;

    fp = popen("xset q", "r");
    if (fp == NULL)
	g_error(_("Could not run xset: %s"), g_strerror(errno));

    info = xset_info_new();

    while (fgets(buf, sizeof buf, fp) != NULL) {
	if (strcmp(buf, "Font Path:\n") == 0) {
	    if (fgets(buf, sizeof buf, fp) == NULL)
		break;

	    g_strstrip(buf);
	    tmp = g_strsplit(buf, ",", 0);
	    init_string_vector(info->font_path, tmp);
	    g_strfreev(tmp);

	    continue;
	}

	contains(buf, "auto repeat: ", TYPE_ON_OFF,
		 &info->keyrepeat_onoff, 0);
	contains(buf, "repeat rate: ", TYPE_INT,
		 &info->keyrepeat_rate, 0);
	contains(buf, "repeat delay: ", TYPE_INT,
		 &info->keyrepeat_delay, 0);
	contains(buf, "key click percent: ", TYPE_INT,
		 &info->keyclick_volume, 0);

	contains(buf, "bell percent: ", TYPE_INT,
		 &info->bell_volume, 0);
	contains(buf, "bell pitch: ", TYPE_INT,
		 &info->bell_pitch, 0);
	contains(buf, "bell duration: ", TYPE_INT,
		 &info->bell_duration, 0);

	contains(buf, "acceleration: ", TYPE_FRACTION,
		 &info->mouse_acceleration, &info->mouse_divisor);
	contains(buf, "threshold: ", TYPE_INT,
		 &info->mouse_threshold, 0);

	contains(buf, "prefer blanking: ", TYPE_ON_OFF,
		 &info->scrsaver_blank, 0);
	contains(buf, "allow exposures: ", TYPE_ON_OFF,
		 &info->scrsaver_expose, 0);
	contains(buf, "timeout: ", TYPE_INT,
		 &info->scrsaver_delay, 0);
	contains(buf, "cycle: ", TYPE_INT,
		 &info->scrsaver_cycle, 0);

	contains(buf, "DPMS is Enabled", TYPE_TRUE,
		 &info->dpms_onoff, 0);
	contains(buf, "Standby: ", TYPE_INT,
		 &info->dpms_standby, 0);
	contains(buf, "Suspend: ", TYPE_INT,
		 &info->dpms_suspend, 0);
	contains(buf, "Off: ", TYPE_INT,
		 &info->dpms_off, 0);

    }
    info->scrsaver_onoff = (info->scrsaver_delay > 0);


    pclose(fp);
    return info;
}


gchar *
font_get_current(ApplicationState *appstate)
{
    gint i = appstate->font_path_selected;

    if (i < 0 || i >= appstate->cur->font_path->len)
	return NULL;

    return g_strdup(g_ptr_array_index(appstate->cur->font_path, i));
}

void
font_set_current(ApplicationState *appstate, gchar *replacement)
{
    gint i = appstate->font_path_selected;

    if (i < 0 || i >= appstate->cur->font_path->len)
	return;

    g_free(g_ptr_array_index(appstate->cur->font_path, i));
    g_ptr_array_index(appstate->cur->font_path, i) = g_strdup(replacement);
}



gboolean
font_move_up(ApplicationState *appstate)
{
    gpointer tmp;
    gint i;

    i = appstate->font_path_selected;
    if (i <= 0 || i >= appstate->cur->font_path->len)
	return FALSE;

    tmp = g_ptr_array_index(appstate->cur->font_path, i);
    g_ptr_array_index(appstate->cur->font_path, i) =
	g_ptr_array_index(appstate->cur->font_path, i-1);
    g_ptr_array_index(appstate->cur->font_path, i-1) = tmp;

    appstate->font_path_selected--;

    return TRUE;
}

gboolean
font_move_down(ApplicationState *appstate)
{
    gpointer tmp;
    gint i;

    i = appstate->font_path_selected;
    if (i < 0 || i >= appstate->cur->font_path->len-1)
	return FALSE;

    tmp = g_ptr_array_index(appstate->cur->font_path, i);
    g_ptr_array_index(appstate->cur->font_path, i) =
	g_ptr_array_index(appstate->cur->font_path, i+1);
    g_ptr_array_index(appstate->cur->font_path, i+1) = tmp;

    appstate->font_path_selected++;

    return TRUE;
}

void
font_add_new(ApplicationState *appstate, gchar *new_path)
{
    g_ptr_array_add(appstate->cur->font_path, g_strdup(new_path));
    appstate->font_path_selected = appstate->cur->font_path->len - 1;
}

gboolean
font_remove(ApplicationState *appstate)
{
    gint i;

    i = appstate->font_path_selected;
    if (i < 0 || i >= appstate->cur->font_path->len)
	return FALSE;

    g_free(g_ptr_array_index(appstate->cur->font_path, i));
    g_ptr_array_remove_index(appstate->cur->font_path, i);
    appstate->font_path_selected = -1;

    return TRUE;
}



static gchar *
build_font_path(XsetInfo *info)
{
    int i;
    int size = info->font_path->len;
    GString *s;
    gchar *path;

    s = g_string_new("");
    for (i = 0; i < size; i++) {
	g_string_append(s, g_ptr_array_index(info->font_path, i));
	g_string_append_c(s, ',');
    }
    s->str[s->len-1] = '\0';
    path = s->str;

    g_string_free(s, FALSE);
    return path;
}

static GString *
build_xset_command(ApplicationState *appstate, gboolean complete)
{
    GString *s;
    XsetInfo *orig = appstate->orig;
    XsetInfo *info = appstate->cur;

    s = g_string_new("");

    if ((complete 
	 || (info->keyrepeat_onoff != orig->keyrepeat_onoff)
	 || (info->keyrepeat_delay != orig->keyrepeat_delay)
	 || (info->keyrepeat_rate != orig->keyrepeat_rate))) {
	if (info->keyrepeat_onoff == 0)
	    info->keyrepeat_delay = 0; /* force off */

	g_string_sprintfa(s, " r rate %d %d",
			  info->keyrepeat_delay,
			  info->keyrepeat_rate);
    }

    if ((complete
	 || (info->keyclick_volume != orig->keyclick_volume))) {
	g_string_sprintfa(s, " c %d",
			  info->keyclick_volume);
    }

    if ((complete 
	 || (info->bell_volume != orig->bell_volume)
	 || (info->bell_pitch != orig->bell_pitch)
	 || (info->bell_duration != orig->bell_duration))) {
	g_string_sprintfa(s, " b %d %d %d",
			  info->bell_volume, 
			  info->bell_pitch,
			  info->bell_duration);
    }

    if ((complete 
	 || (info->mouse_acceleration != orig->mouse_acceleration)
	 || (info->mouse_divisor != orig->mouse_divisor)
	 || (info->mouse_threshold != orig->mouse_threshold))) {
	g_string_sprintfa(s, " m %d/%d %d",
			  info->mouse_acceleration,
			  info->mouse_divisor,
			  info->mouse_threshold);
    }

    if ((complete 
	 || (info->scrsaver_onoff != orig->scrsaver_onoff)
	 || (info->scrsaver_blank != orig->scrsaver_blank)
	 || (info->scrsaver_expose != orig->scrsaver_expose)
	 || (info->scrsaver_delay != orig->scrsaver_delay)
	 || (info->scrsaver_cycle != orig->scrsaver_cycle))) {
	if (info->scrsaver_onoff == 0)
	    info->scrsaver_delay = 0;

	g_string_sprintfa(s, " s %d %d s %s s %s",
			  info->scrsaver_delay,
			  info->scrsaver_cycle,
			  (info->scrsaver_blank ? "blank" : "noblank"),
			  (info->scrsaver_expose ? "expose" : "noexpose"));
    }

    if ((complete 
	 || (info->dpms_onoff != orig->dpms_onoff)
	 || (info->dpms_standby != orig->dpms_standby)
	 || (info->dpms_suspend != orig->dpms_suspend)
	 || (info->dpms_off != orig->dpms_off))) {
	g_string_sprintfa(s, " dpms %d %d %d %cdpms",
			  info->dpms_standby,
			  info->dpms_suspend,
			  info->dpms_off,
			  (info->dpms_onoff > 0 ? '+' : '-'));
    }

    if (complete) {
	gchar *fp_info = build_font_path(info);
	g_string_sprintfa(s, " fp= %s", fp_info);
	g_free(fp_info);
    } else {
	gchar *fp_orig = build_font_path(orig);
	gchar *fp_info = build_font_path(info);

	if (strcmp(fp_orig, fp_info) != 0)
	    g_string_sprintfa(s, " fp= %s", build_font_path(info));

	g_free(fp_orig);
	g_free(fp_info);
    }

    if (s->len > 0)
	s = g_string_prepend(s, "xset");

    return s;
}


GString *
apply_settings(ApplicationState *appstate)
{
    GString *cmd;
    FILE *fp;
    GString *output;
    GString *msg;
    gint ch;
    gint status;

    cmd = build_xset_command(appstate, FALSE);

    msg = g_string_new("");

    if (cmd->len) {
	output = g_string_new("");

	g_string_append(cmd, " 2>&1"); /* catch error messages */
	fp = popen(cmd->str, "r");
	if (fp == NULL) {
	    g_string_sprintf(msg, _("The 'xset' command failed.\nReason: %s"),
			     g_strerror(errno));
	} else {
	    while ((ch = fgetc(fp)) != EOF)
		g_string_append_c(output, ch);
	    status = pclose(fp);

	    if (WIFEXITED(status)) {
		if (WEXITSTATUS(status))
		    g_string_sprintf(msg, _(
			"The 'xset' command exited abnormally\nExit code %d"),
				     WEXITSTATUS(status));
		else {
		    g_string_assign(msg, _("Settings applied succesfully."));
		}

		if (output->len > 0) {
		    g_string_sprintfa(msg, _("\nError message:\n%s"),
				      output->str);
		}
	    } else if (WIFSIGNALED(status)) {
		g_string_sprintf(msg, _(
		    "The 'xset' command was killed by a signal: %s"),
				 g_strsignal(WTERMSIG(status)));
	    } else
		g_string_assign(msg, _("Something weird went wrong!"));
	}

	g_string_free(cmd, TRUE);
	g_string_free(output, TRUE);
    } else
	g_string_assign(msg, _("No changes to apply."));


    return msg;
}

GString *
save_settings(ApplicationState *appstate)
{
    FILE *fp;
    GString *msg;
    GString *cmd;
    gchar *filename = appstate->save_file_name;

    msg = g_string_new("");

    fp = fopen(filename, "w");

    if (fp == NULL) {
	g_string_sprintf(msg, _("Failed to open the file ~/.xset.sh.\n" 
				"Reason: %s"), g_strerror(errno));
    } else {
	cmd = build_xset_command(appstate, TRUE);
	fprintf(fp, _("# This file is written by gxset\n"));
	fprintf(fp, (_("# Any changes will be lost "
		     "the next time you do a save in gxset\n")));
	fprintf(fp, "%s\n", cmd->str);
	fclose(fp);

	g_string_free(cmd, TRUE);
	g_string_sprintf(msg, _("Settings saved to %s"), filename);
    }

    return msg;
}



int
main (int argc, char *argv[])
{
    GtkWidget *gxset;
    ApplicationState appstate;

    bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
    textdomain (PACKAGE);

    gtk_set_locale ();
    gtk_init (&argc, &argv);

    appstate.orig = xset_info_read();
    appstate.cur = xset_info_dup(appstate.orig);
    appstate.font_path_selected = -1;
    appstate.save_file_name = g_strdup("~/.xset.sh");

    gxset = create_gxset();
    set_gui_state(gxset, &appstate);
    gtk_object_set_user_data(GTK_OBJECT(gxset), &appstate);
    gtk_widget_show(gxset);
    gtk_main();
    return 0;
}

