#ifndef __autil_hash_hh__
#define __autil_hash_hh__

#include <iterator>
#include <utility>

namespace autil {

  //
  // This hash table is implemnted as a Open Address Hash Table
  // which uses a Vector like object to store its data.  So
  // it might even be considered an adapter
  //
  
  ////////////////////////////////////////////////////////
  //                                                    //
  //          Vector Hash Table Interator               //
  //                                                    //
  ////////////////////////////////////////////////////////
  //
  // This is an internal object and should not be created
  // directly.  Instead use VectorHashTable::begin() and end()

  template <class Parms>
  // Parms is expected to have:
  //  typename HashTable;
  //  typename TableIter;
  //  typename Value;
  class VHTIterator 
  {
    friend bool operator==<> (VHTIterator, VHTIterator);
#if !defined(__GNUC__) || (__GNUC__ >= 2 && __GNUC_MINOR__ >= 95)
    friend bool operator!=<> (VHTIterator, VHTIterator);
#endif
  private:
    typedef typename Parms::TableIter       TableIter;
    typedef typename Parms::HashTable       HashTable;
    TableIter   pos;
    HashTable * hash_table; 
  public:
    typedef bidirectional_iterator_tag                  iterator_category;
    typedef typename Parms::Value                       value_type;
    typedef iterator_traits<TableIter>::difference_type difference_type;
    typedef iterator_traits<TableIter>::pointer         pointer;
    typedef iterator_traits<TableIter>::reference       reference;

    //VHTIterator vector_iterator() const {return pos;}
  public:
    VHTIterator(TableIter p, HashTable *ht) : pos(p), hash_table(ht) {}
    VHTIterator(TableIter p, HashTable *ht, bool) 
      : pos(p), hash_table(ht) 
    {
      while (pos != hash_table->vector().end()
	     && hash_table->parms().is_nonexistent(*pos) )
	++pos;
    }
    
    value_type & operator * () const  {return *pos;}
    value_type * operator -> () const {return &*pos;}
    
    bool at_end() const {return pos == hash_table->vector().end();}
    
    VHTIterator& operator++ () {
      ++pos;
      for (;;) {
	if (pos == hash_table->vector().end()) break;
	if (!hash_table->parms().is_nonexistent(*pos)) break;
	++pos;
      }
      return *this;
    }
    VHTIterator operator++ (int) {
      VHTIterator temp = *this; 
      operator++();
      return temp;
    }

    VHTIterator& operator-- () {
      --pos;
      for (;;) {
	if (pos < hash_table->vector().begin()) break;
	if (!hash_table->parms().is_nonexistent_func(*pos)) break;
	--pos;
      }
      return *this;
    }
    VHTIterator operator-- (int) {
      VHTIterator temp = *this; 
      operator--();
      return temp;
    }
  };

  template <class Parms>
  inline 
  bool operator== (VHTIterator<Parms> rhs, VHTIterator<Parms> lhs)
  {
    return rhs.pos == lhs.pos;
  }

#if !defined(__GNUC__) || (__GNUC__ >= 2 && __GNUC_MINOR__ >= 95)
  
  template <class Parms>
  inline
  bool operator!= (VHTIterator<Parms> rhs, VHTIterator<Parms> lhs)
  {
    return rhs.pos != lhs.pos;
  }

#endif
  
  ////////////////////////////////////////////////////////
  //                                                    //
  //                Vector Hash Table                   //
  //                                                    //
  ////////////////////////////////////////////////////////

  // Parms is expected to have the following methods
  //   typename Vector
  //   typedef Vector::value_type Value
  //   typedef Vector::size_type  Size
  //   typename Key
  //   Size hash(Key)
  //   bool equal(Key, Key)
  //   Key key(Value)
  //   bool is_nonexistent(Value)
  //   void make_nonexistent(Value &)

  template <class Parms>
  class VectorHashTable {
    typedef typename Parms::Vector           Vector;
  public:
    typedef typename Parms::Vector           vector_type;
    typedef typename Vector::value_type      value_type;
    typedef typename Vector::size_type       size_type;
    typedef typename Vector::difference_type difference_type;

    typedef typename Vector::pointer         pointer;
    typedef typename Vector::reference       reference;
    typedef typename Vector::const_reference const_reference;

    typedef typename Parms::Key              key_type;
  private:
    Parms parms_;

  public:
    typedef Parms parms_type;
    const parms_type & parms() const {return parms_;}

  public:
    // These public functions are very dangerous and should be used with
    // great care as the modify the internal structure of the object
    Vector & vector()       {return vector_;}
    const Vector & vector() const {return vector_;}
    parms_type & parms() {return parms_;}
    void recalc_size();
    void set_size(size_type s) {size_  = s;} 

  private:
    Vector      vector_;
    size_type   size_;

    typedef typename Vector::iterator       vector_iterator;
    typedef typename Vector::const_iterator const_vector_iterator;
  
    int hash1(const key_type &d) const {
      return parms_.hash(d) % bucket_count();
    }
    int hash2(const key_type &d) const {
      return 1 + (parms_.hash(d) % (bucket_count() - 2));
    }

    struct NonConstParms {
      typedef VectorHashTable HashTable;
      typedef vector_iterator  TableIter;
      typedef value_type      Value;
    };
    struct ConstParms {
      typedef const VectorHashTable HashTable;
      typedef const_vector_iterator  TableIter;
      typedef const value_type      Value;
    };
  public:
    typedef VHTIterator<NonConstParms> iterator;
    typedef VHTIterator<ConstParms>    const_iterator;

  private:
    pair<int,bool> find_item(const key_type &d) const;
    void nonexistent_vector();
  
  public:
    VectorHashTable(size_type i = 7, const Parms & p = Parms())
      : parms_(p), vector_(i < 7 ? 7 : i), size_(0) {nonexistent_vector();}
    VectorHashTable(const Parms & p)
      : parms_(p), vector_(7), size_(0) {nonexistent_vector();}
 
    iterator begin() {return iterator(vector_.begin(), this, 1);}
    iterator end()   {return iterator(vector_.end(), this);}
    const_iterator begin() const {return const_iterator(vector_.begin(), this, 1);}
    const_iterator end()   const {return const_iterator(vector_.end(), this);}

    pair<iterator, bool> insert(const value_type &);
    bool count(const key_type &) const;

    iterator find(const key_type&);
    const_iterator find(const key_type&) const;
  
    size_type erase(const key_type &key);
    void erase(const iterator &p);
  
    size_type size() const {return size_;}
    bool      empty() const {return !size_;}

    void swap(VectorHashTable &);
    void resize(size_type);
    size_type bucket_count() const {return vector_.size();}
    double load_factor() const {return static_cast<double>(size())/bucket_count();}
  };

#if 0

  ////////////////////////////////////////////////////////
  //                                                    //
  //             Pair Adapter Functions                 //
  //                                                    //
  ////////////////////////////////////////////////////////

  template <class Pair, class Func>
  class unary_on_first {
    Func func;
  public:
    typedef Pair                         pair_type;
    typedef Func                         func_type;
    typedef typename Func::result_type   result_type;
    typedef typename Func::argument_type argument_type;

    unary_on_first() {}
    unary_on_first(const Func &f) : func(f) {}

    result_type operator() (argument_type p) const {return func(p);}
    result_type operator() (pair_type p)     const {return func(p.first);}
  };

  template <class Pair1, class Pair2, class Func>
  class binary_on_first {
    Func func;

  public:

    typedef Pair1                               first_pair_type;
    typedef Pair2                               second_pair_type;
    typedef Func                                func_type;
    typedef typename Func::result_type          result_type;
    typedef typename Func::first_argument_type  first_argument_type;
    typedef typename Func::second_argument_type second_argument_type;

    binary_on_first() {}
    binary_on_first(const Func &f) : func(f) {}

    result_type operator() (first_argument_type p1, second_argument_type p2) const
    {return func(p1,p2);}
    result_type operator() (first_pair_type p1, second_argument_type p2) const
    {return func(p1.first,p2);}
    result_type operator() (first_argument_type p1, second_pair_type p2) const
    {return func(p1,p2.first);}
    result_type operator() (first_pair_type p1, second_pair_type p2) const
    {return func(p1.first,p2.first);}
  };

  ////////////////////////////////////////////////////////
  //                                                    //
  //                  vector_hash_set                   //
  //                                                    //
  ////////////////////////////////////////////////////////


#define htable  VectorHashTable \
  < Vector, Value, const Value, \
    HashFunc, EqualKey, \
    IsNonExistent, MakeNonExistent \
  >

  template <class Value, class HashFunc, 
    class IsNonExistent, class MakeNonExistent,
    class EqualKey = equal_to<Value>, 
    class Vector = vector<Value> >
  class vector_hash_set : public htable
  {
  private:
    typedef htable derived;
  public:
    vector_hash_set() : derived() {}
    vector_hash_set(size_type i) : derived(i) {}
  };


  ////////////////////////////////////////////////////////
  //                                                    //
  //                  vector_hash_map                   //
  //                                                    //
  ////////////////////////////////////////////////////////


#define ppair pair<Key, Data>
#define cpair pair<const Key, Data>

#undef  htable
#define htable  VectorHashTable \
  <Vector, Key, ppair, \
   unary_on_first<const ppair&, HashFunc>,\
   pair_equal_to_with_key<ppair, EqualKey>,\
   unary_on_first<const ppair&, IsNonExistent>, \
   unary_on_first<ppair, MakeNonExistent> \
  >

  template <class Key, class Data, class HashFunc, 
    class IsNonExistent, class MakeNonExistent,
    class EqualKey = equal_to< Key >, 
    class Vector = vector< ppair >
         >
  class vector_hash_map : public htable
  {
  private:
    typedef htable derived;
  public:
    vector_hash_map() : derived() {}
    vector_hash_map(size_type i) : derived(i) {}
    typedef Data data_type;
    data_type& operator[] (const key_type& k) 
    {return (*((insert(typename vector_hash_map::value_type(k, data_type()))).first)).second;}
  };
#undef ppair
#undef cpair
#undef htable

#endif

}

#endif
