#!/bin/sh
#  -*-tcl-*-\
exec wish8.0 "$0" ${1:+"$@"}

# Copyright (C) 1998, DGA - part of the Transcriber program
# distributed under the GNU General Public License (see COPYING file)

# Sound file server through sockets
#
# Syntax :
#    SoundServer.tcl [server_port]
#
# Action :
#    Open sound file and dialog with client through sockets, returning
#    result of sound commands along with error flag.

# Default port
set port 8032

# List of authorized pathnames
set snd_paths {*/sounds/*}

# List of authorized sound extensions
set exts {".au" ".wav" ".snd" ".sph" ".sig" ".sd" ".smp" ".aif" ".aiff" ".mp3" ".raw"}

# Default path for storing shapes
set shp_path "/var/tmp"

set base [file dir [file dir [file join [pwd] [info script]]]]
lappend auto_path $base [file dir $base]
package require sound
catch {
   package require snackSphere
}
package require trans

# Socket server
proc SocketServer {port} {
   Msg "Sound server port $port"
   if [catch {
      set ::servsock [socket -server SocketAccept $port]
      #vwait forever
   } error] {
      puts $error
      exit
   }
}

# Connection to server
proc SocketAccept {sock addr port} {
   # We could reject here foreign hosts or have some restricted list
   Msg "Accept $sock from $addr port $port"
   fconfigure $sock -buffering full -translation binary
   fileevent $sock readable [list FirstCmd $sock]
}

# First command has to be "sound [options]"
proc FirstCmd {sock} {
   Busy $sock ""
   set line ""
   if {[catch {gets $sock line} error] || ([string length $line]==0)} {
      #Msg "in $sock line $line error $error"
      Close $sock
   } else {
      set line [string trimright $line \r]
      Msg $line
      switch [lindex $line 0] {
	 "sound" {
	    # Test that access is authorized for this file
	    array set opt {-file "" -load "" -channel ""}
	    array set opt [lrange $line 1 end]

	    # Test that pathname/extension for soundfile is authorized
	    set path_ok 0
	    foreach path $::snd_paths {
	       if {[string match $path [file dirname $opt(-file)]/]} {
		  set path_ok 1
	       }
	    }
	    if {$opt(-load) != "" || $opt(-channel) != ""
		|| ! $path_ok
		|| [lsearch -exact $::exts [file extension $opt(-file)]] < 0
		|| ![file exists $opt(-file)]} {
	       ExecCmd $sock [list error "Can't open remote file $opt(-file)"]
	       Close $sock
	    } else {
	       set snd [ExecCmd $sock $line]
	       fileevent $sock readable [list NextCmd $sock $snd]
	    }
	 } 
	 default {
	    ExecCmd $sock [list error "Wrong command: $line"]
	    Close $sock
	 }
      }
   }
   Free
}

proc NextCmd {sock snd} {

   Busy $sock $snd

   set error "eof $sock"
   if {[catch {fconfigure $sock} error]
       || [eof $sock] 
       || [catch {gets $sock line} error]} {
      Close $sock $snd
   } else {
      set line [string trimright $line \r]
      if {[string length $line]!=0} {
	 Msg "$line"
	 switch -glob -- [lindex $line 0] {
	    "dump" {
	       global v
	       set v(-start) 0
	       set v(-end) -1
	       array set v [lrange $line 1 end]
	       if {$v(-end)<0} {
		  set v(-end) [$snd length]
	       }
	       puts $sock "CODE ok LEN -1"
	       flush $sock
	       fconfigure $sock -blocking 0 
	       fileevent $sock writable [list PlayHandler $sock $snd]
	    }
	    "shname" {
	       ExecCmd $sock [list LookForShape [lindex $line 1]]
	    }
	    "shape" -
	    "centi" -
	    "get" -
	    "cget" -
	    "order" -
	    "info" -
	    "destroy" -
	    "length" {
	       ExecCmd $sock [concat $snd $line]
	    }
	    default {
	       ExecCmd $sock [list error "Non authorized sub-command [lindex $line 0]"]
	    }
	 }
      }
   }

   Free
}

# Execute command and write result to the socket channel in the format :
#   CODE $code LEN $len <RETURN> $result
proc ExecCmd {sock cmd} {
   if [catch {eval $cmd} res] {
      set code "error" 
   } else {
      set code "ok"
   }
   set len [string length $res]
   puts $sock "CODE $code LEN $len"
   puts -nonewline $sock $res
   flush $sock
   return $res
}

proc PlayHandler {sock snd} {
   global v

   Busy $sock $snd
   Msg "playing"

   if {$v(-start) >= $v(-end) || [catch {
      set end $v(-end)
      if {$end > [expr $v(-start) + 10000]} {
	 set end [expr $v(-start) + 10000]
      }
      puts -nonewline $sock [$snd get $v(-start) [expr $end-$v(-start)]]
      flush $sock
      set v(-start) $end
   }]} {
      Close $sock $snd
   }

   Free
}

# Taken from Signal.tcl
proc LookForShape {sigName} {
   global v

   set base [file root [file tail $sigName]]

   # Search for an existing matching shape
   # (in default dir, shp sub-dir or signal dir)
   foreach path [concat $::shp_path "shp ../shp ."] {
      # Relative paths are relative to signal path
      set path [file join [file dirname $sigName] $path]
      set shape [file join $path $base.shp]
      # Verify that the shape is newer than the signal
      if {[file isfile $shape] && [file readable $shape] 
	  && [file mtime $shape] >= [file mtime $sigName]} {
	 return $shape
      }
   }
   # Return new shape name in default shape path
   set shape [file join $::shp_path $base.shp]
   file delete $shape
   return $shape
}

proc Busy {sock {snd ""}} {
   set ::busy 1
   if {[catch {
      set ::peer [lindex [fconfigure $sock -peername] 1]
   }]} {
      set ::peer ""
   }
   if {$snd == "" || [catch {
      set ::name [$snd cget -file]
   }]} {
      set ::name ""
   }
   update idletasks
}

proc Close {sock {snd ""}} {
   Msg "closing connection"
   catch {$snd destroy}
   catch {close $sock}
}

proc Msg {txt} {
   set ::msg $txt
   update idletasks
}

proc Free {} {
   set ::busy 0
}

proc Quit {} {
   catch {close $::servsock}
   exit
}

proc Restart {} {
   catch {close $::servsock}
   SocketServer $::port
}

proc Interface {} {
   wm title . "Transcriber's Sound Server"
   wm protocol . WM_DELETE_WINDOW {Quit}

   set w [frame .top -relief raised -bd 1]
   pack $w -side top -fill both -expand true
   
   foreach i {1 2 3} n {
      "Port" "Shape storage" "Authorized paths"
   } var {port shp_path snd_paths} {
      pack [frame $w.$i] -side top -fill x
      label $w.$i.l -text "$n:" -width 20 -anchor e
      pack $w.$i.l -side left -padx 1m -pady 1m
      entry $w.$i.e -textvariable $var -width 30
      pack $w.$i.e -side left -padx 1m -pady 1m -fill x -expand true
   }
   
   set w [frame .mid -relief raised -bd 1]
   pack $w -side top -fill both -expand true
   
   pack [frame $w.1] -side top -fill x
   label $w.1.l1 -text "Client:" -width 10
   label $w.1.l2 -textvariable peer -width 20 -relief sunken -anchor w
   pack $w.1.l1 $w.1.l2 -side left -padx 1m -pady 1m

   pack [frame $w.2] -side top -fill x
   label $w.2.l3 -text "Sound:" -width 10
   label $w.2.l4 -textvariable name -width 40 -relief sunken -anchor w
   pack $w.2.l3 $w.2.l4 -side left -padx 1m -pady 1m
   pack $w.2.l4 -fill x -expand true

   pack [frame $w.3] -side top -fill x
   checkbutton $w.3.c -text busy -var busy -anchor w
   pack $w.3.c -side left -padx 1m -pady 1m
   label $w.3.m -textvariable msg -width 40 -relief sunken -anchor w
   pack $w.3.m -side left -padx 1m -pady 1m -fill x -expand true
   
   set w [frame .bot -relief raised -bd 1]
   pack $w -side top -fill x

   button $w.r -text Restart -command Restart
   button $w.q -text Quit -command Quit
   pack $w.r $w.q -side left -expand true -padx 3m -pady 2m
}

# Start file server; if port already in use return else loop
Interface
SocketServer $port
