
/*
 * The Real SoundTracker - main user interface handling
 *
 * Copyright (C) 1998-1999 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <stdio.h>
#include <string.h>
#include <math.h>

#include <unistd.h>

#include "poll.h"

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <glib.h>
#ifdef USE_GNOME
#include <gnome.h>
#endif

#include "i18n.h"
#include "gui.h"
#include "gui-subs.h"
#include "xm.h"
#include "st-subs.h"
#include "audio.h"
#include "xm-player.h"
#include "tracker.h"
#include "main.h"
#include "keys.h"
#include "instrument-editor.h"
#include "sample-editor.h"
#include "track-editor.h"
#include "scope-group.h"
#include "module-info.h"
#include "preferences.h"
#include "menubar.h"
#include "time-buffer.h"
#include "tips-dialog.h"
#include "gui-settings.h"
#include "file-operations.h"
#include "playlist.h"

int gui_playing_mode = 0;
GtkWidget *editing_toggle;
GtkWidget *gui_curins_name, *gui_cursmpl_name;
GtkWidget *mainwindow;
ScopeGroup *scopegroup;

static GSList *text_entries;

static gint pipetag = -1;
static GtkWidget *mainwindow_upper_hbox, *mainwindow_second_hbox;
static GtkWidget *notebook;
static GtkWidget *spin_editpat, *spin_patlen, *spin_numchans;
static GtkWidget *cursmpl_spin;
static GtkAdjustment *adj_amplification, *adj_pitchbend;
static GtkWidget *spin_jump, *curins_spin, *spin_octave;
static Playlist *playlist;

static void tempo_slider_changed (int value);
static void bpm_slider_changed (int value);

static gui_subs_slider tempo_slider = {
    N_("Tempo"), 1, 31, tempo_slider_changed, GUI_SUBS_SLIDER_SPIN_ONLY
};
static gui_subs_slider bpm_slider = {
    "BPM", 32, 255, bpm_slider_changed, GUI_SUBS_SLIDER_SPIN_ONLY
};

static GdkColor gui_clipping_led_on, gui_clipping_led_off;
static GtkWidget *gui_clipping_led;
static gboolean gui_clipping_led_status;

static int set_songpos_count = 0, startstop_count = 0;
static double set_songpos_wait_time = -1.0;

static int editing_pat = 0;
static int songpos = 0;
static int capture_keys = 1;
static int notebook_current_page;

/* gui event handlers */
static void file_selected(GtkWidget *w, GtkFileSelection *fs);
static void current_instrument_changed(GtkSpinButton *spin);
static void current_instrument_name_changed(void);
static void current_sample_changed(GtkSpinButton *spin);
static void current_sample_name_changed(void);
static int keyevent(GtkWidget *widget, GdkEventKey *event, gpointer data);
static void spin_editpat_changed(GtkSpinButton *spin);
static void spin_patlen_changed(GtkSpinButton *spin);
static void spin_numchans_changed(GtkSpinButton *spin);
static void notebook_page_switched(GtkNotebook *notebook, GtkNotebookPage *page, int page_num);
static void gui_adj_amplification_changed(GtkAdjustment *adj);
static void gui_adj_pitchbend_changed(GtkAdjustment *adj);

/* mixer / player communication */
static void read_mixer_pipe(gpointer data, gint source, GdkInputCondition condition);
static void wait_for_player(void);
static void play_song(void);
static void play_pattern(void);

/* gui initialization / helpers */
static void text_entry_selected(void);
static void text_entry_unselected(void);

static void gui_enable(int enable);
static void offset_current_pattern(int offset);
static void offset_current_instrument(int offset);
static void offset_current_sample(int offset);
static void programlist_initialize(void);

static void
gui_mixer_play_pattern (int p,
			int pp)
{
    audio_ctlpipe_id i = AUDIO_CTLPIPE_PLAY_PATTERN;
    write(audio_ctlpipe, &i, sizeof(i));
    write(audio_ctlpipe, &p, sizeof(p));
    write(audio_ctlpipe, &pp, sizeof(pp));
}

static void
gui_mixer_stop_playing (void)
{
    audio_ctlpipe_id i = AUDIO_CTLPIPE_STOP_PLAYING;
    write(audio_ctlpipe, &i, sizeof(i));
}

static void
gui_mixer_set_songpos (int songpos)
{
    audio_ctlpipe_id i = AUDIO_CTLPIPE_SET_SONGPOS;
    write(audio_ctlpipe, &i, sizeof(i));
    write(audio_ctlpipe, &songpos, sizeof(songpos));
}

static void
gui_mixer_set_pattern (int pattern)
{
    audio_ctlpipe_id i = AUDIO_CTLPIPE_SET_PATTERN;
    write(audio_ctlpipe, &i, sizeof(i));
    write(audio_ctlpipe, &pattern, sizeof(pattern));
}

static void
gui_load_callback (gint reply,
		   gpointer data)
{
    if(reply == 0) {
	gui_load_xm((gchar*)data);
    }
}

static void
gui_save_callback (gint reply,
		   gpointer data)
{
    if(reply == 0) {
	XM_Save(xm, (gchar*)data);
	xm->modified = 0;
    }
}

static void
gui_save_wav_callback (gint reply,
		       gpointer data)
{
    if(reply == 0) {
	int l = strlen(data);
	audio_ctlpipe_id i = AUDIO_CTLPIPE_RENDER_SONG_TO_FILE;

	gui_play_stop();

	write(audio_ctlpipe, &i, sizeof(i));
	write(audio_ctlpipe, &l, sizeof(l));
	write(audio_ctlpipe, data, l + 1);

	gui_set_current_pattern(xm->pattern_order_table[0]);
	wait_for_player();
    }
}

static void
file_selected (GtkWidget *w,
	       GtkFileSelection *fs)
{
    gchar *fn = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));

    gtk_widget_hide(GTK_WIDGET(fs));

    if(fs == GTK_FILE_SELECTION(fileops_dialogs[DIALOG_LOAD_MOD])) {
	file_selection_save_path(fn, gui_settings.loadmod_path);
	if(xm->modified) {
	    gnome_app_ok_cancel_modal(GNOME_APP(mainwindow),
				      _("Are you sure you want to free the current project?\nAll changes will be lost!"),
				      gui_load_callback,
				      fn);
	} else {
	    gui_load_callback(0, fn);
	}
    } else if(fs == GTK_FILE_SELECTION(fileops_dialogs[DIALOG_SAVE_MOD])) {
	FILE *f = fopen(fn, "r");

	file_selection_save_path(fn, gui_settings.savemod_path);

	if(f != NULL) {
	    fclose(f);
	    gnome_app_ok_cancel_modal(GNOME_APP(mainwindow),
				      _("Are you sure you want to overwrite the file?"),
				      gui_save_callback,
				      fn);
	} else {
	    gui_save_callback(0, fn);
	}
    } else {
	FILE *f = fopen(fn, "r");

	file_selection_save_path(fn, gui_settings.savemodaswav_path);

	if(f != NULL) {
	    fclose(f);
	    gnome_app_ok_cancel_modal(GNOME_APP(mainwindow),
				      _("Are you sure you want to overwrite the file?"),
				      gui_save_wav_callback,
				      fn);
	} else {
	    gui_save_wav_callback(0, fn);
	}
    }
}

static void
current_instrument_changed (GtkSpinButton *spin)
{
    int m = xm_get_modified();
    STInstrument *i = &xm->instruments[gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin))-1];
    STSample *s = &i->samples[gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin))];

    instrument_editor_set_instrument(i);
    sample_editor_set_sample(s);
    modinfo_set_current_instrument(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin)) - 1);
    xm_set_modified(m);
}

static void
current_instrument_name_changed (void)
{
    STInstrument *i = &xm->instruments[gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin))-1];

    strncpy(i->name, gtk_entry_get_text(GTK_ENTRY(gui_curins_name)), 22);
    i->name[22] = 0;
    modinfo_update_instrument(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin))-1);
    xm_set_modified(1);
}

static void
current_sample_changed (GtkSpinButton *spin)
{
    int m = xm_get_modified();
    STInstrument *i = &xm->instruments[gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin))-1];
    STSample *s = &i->samples[gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin))];

    gtk_entry_set_text(GTK_ENTRY(gui_cursmpl_name), s->name);
    sample_editor_set_sample(s);
    modinfo_set_current_sample(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin)));
    xm_set_modified(m);
}

static void
current_sample_name_changed (void)
{
    STInstrument *i = &xm->instruments[gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin))-1];
    STSample *s = &i->samples[gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin))];

    strncpy(s->name, gtk_entry_get_text(GTK_ENTRY(gui_cursmpl_name)), 22);
    s->name[22] = 0;
    modinfo_update_sample(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin)));
    xm_set_modified(1);
}

static gboolean
gui_handle_standard_keys (int shift,
			  int ctrl,
			  int alt,
			  guint32 keyval)
{
    gboolean handled = FALSE;

    switch (keyval) {
    case GDK_F1 ... GDK_F7:
	if(!shift && !ctrl && !alt) {
	    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_octave), keyval - GDK_F1);
	    handled = TRUE;
	}
	break;
    case '1' ... '8':
	if(ctrl) {
	    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_jump), keyval - '0');
	    handled = TRUE;
	}
	break;
    case GDK_Left:
	if(ctrl) {
	    /* previous instrument */
	    offset_current_instrument(shift ? -5 : -1);
	    handled = TRUE;
	} else if(alt) {
	    /* previous pattern */
	    offset_current_pattern(shift ? -10 : -1);
	    handled = TRUE;
	}
	break;
    case GDK_Right:
	if(ctrl) {
	    /* next instrument */
	    offset_current_instrument(shift ? 5 : 1);
	    handled = TRUE;
	} else if(alt) {
	    /* next pattern */
	    offset_current_pattern(shift ? 10 : 1);
	    handled = TRUE;
	}
	break;
    case GDK_Up:
	if(ctrl) {
	    /* next sample */
	    offset_current_sample(shift ? 4 : 1);
	    handled = TRUE;
	}
	break;
    case GDK_Down:
	if(ctrl) {
	    /* previous sample */
	    offset_current_sample(shift ? -4 : -1);
	    handled = TRUE;
	}
	break;
    case GDK_Alt_R:
    case GDK_Meta_R:
    case GDK_Super_R:
    case GDK_Hyper_R:
    case GDK_Mode_switch: /* well... this is X :D */
	play_pattern();
	handled = TRUE;
	break;
    case GDK_Control_R:
    case GDK_Multi_key:
	play_song();
	handled = TRUE;
	break;
    case ' ':
	if(ctrl || alt)
	    break;
	if(!shift) {
	    if(!GUI_ENABLED) {
		gui_play_stop();
	    } else {
		gui_play_stop();
		/* toggle editing mode */
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(editing_toggle), !GUI_EDITING);
	    }
	    handled = TRUE;
	}
	break;
    }

    return handled;
}

static int
keyevent (GtkWidget *widget,
	  GdkEventKey *event,
	  gpointer data)
{
    static gboolean (*handle_page_keys[])(int,int,int,guint32,gboolean) = {
	fileops_page_handle_keys,
	track_editor_handle_keys,
	instrument_editor_handle_keys,
	sample_editor_handle_keys,
	modinfo_page_handle_keys,
    };
    gboolean pressed = (gboolean)data;
    gboolean handled = FALSE;

    if(capture_keys && GTK_WIDGET_VISIBLE(notebook)) {
	int shift = event->state & GDK_SHIFT_MASK;
	int ctrl = event->state & GDK_CONTROL_MASK;
	int alt = event->state & GDK_MOD1_MASK;

	if(pressed)
	    handled = gui_handle_standard_keys(shift, ctrl, alt, event->keyval);
	handled = handled || handle_page_keys[notebook_current_page](shift, ctrl, alt, event->keyval, pressed);

	if(handled) {
	    if(pressed)
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
	    else
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_release_event");
	}
    } else {
	if(pressed) {
	    switch(event->keyval) {
	    case GDK_Tab:
	    case GDK_Return:
		gtk_window_set_focus(GTK_WINDOW(mainwindow), NULL);
		capture_keys = 1;
		break;
	    }
	}
    }

    return 1;
}

static void
playlist_position_changed (Playlist *p,
			   int newpos)
{
    if(newpos == songpos)
	return;

    songpos = newpos;

    if(gui_playing_mode) {
	// This will only be executed when the user changes the song position manually
	gui_mixer_set_songpos(newpos);
	// See comment in audio.c::audio_mix() for why we do this
	set_songpos_count++;
	gui_set_current_pattern(playlist_get_nth_pattern(p, newpos));
    }
}

static void
playlist_restart_position_changed (Playlist *p,
				   int pos)
{
    xm->restart_position = pos;
    xm_set_modified(1);
}

static void
playlist_entry_changed (Playlist *p,
			int pos,
			int pat)
{
    int i;

    if(pos != -1) {
	xm->pattern_order_table[pos] = pat;
    } else {
	for(i = 0; i < xm->song_length; i++) {
	    xm->pattern_order_table[i] = playlist_get_nth_pattern(p, i);
	}
    }

    xm_set_modified(1);
}

static void
playlist_song_length_changed (Playlist *p,
			      int length)
{
    xm->song_length = length;
    playlist_entry_changed(p, -1, -1);
}

static void
spin_editpat_changed (GtkSpinButton *spin)
{
    int n = gtk_spin_button_get_value_as_int(spin);

    if(n != editing_pat)
	gui_set_current_pattern(n);
}

static void
spin_patlen_changed (GtkSpinButton *spin)
{
    int n = gtk_spin_button_get_value_as_int(spin);
    XMPattern *pat = &xm->patterns[editing_pat];

    if(n != pat->length) {
	st_set_pattern_length(pat, n);
	tracker_set_pattern(tracker, NULL);
	tracker_set_pattern(tracker, pat);
	xm_set_modified(1);
    }
}

static void
spin_numchans_changed (GtkSpinButton *spin)
{
    int n = gtk_spin_button_get_value_as_int(spin);

    g_assert((n & 1) == 0);

    if(xm->num_channels != n) {
	gui_play_stop();
	tracker_set_pattern(tracker, NULL);
	st_set_num_channels(xm, n);
	gui_init_xm(0);
	xm_set_modified(1);
    }
}

static void
tempo_slider_changed (int value)
{
    audio_ctlpipe_id i;
    xm->tempo = value;
    xm_set_modified(1);
    i = AUDIO_CTLPIPE_INIT_PLAYER;
    write(audio_ctlpipe, &i, sizeof(i));
}

static void
bpm_slider_changed (int value)
{
    audio_ctlpipe_id i;
    xm->bpm = value;
    xm_set_modified(1);
    i = AUDIO_CTLPIPE_INIT_PLAYER;
    write(audio_ctlpipe, &i, sizeof(i));
}

static void
gui_adj_amplification_changed (GtkAdjustment *adj)
{
    audio_ctlpipe_id a = AUDIO_CTLPIPE_SET_AMPLIFICATION;
    float b = 8.0 - adj->value;

    write(audio_ctlpipe, &a, sizeof(a));
    write(audio_ctlpipe, &b, sizeof(b));
}

static void
gui_adj_pitchbend_changed (GtkAdjustment *adj)
{
    audio_ctlpipe_id a = AUDIO_CTLPIPE_SET_PITCHBEND;
    float b = adj->value;

    write(audio_ctlpipe, &a, sizeof(a));
    write(audio_ctlpipe, &b, sizeof(b));
}

static void
gui_reset_pitch_bender (void)
{
    gtk_adjustment_set_value(adj_pitchbend, 0.0);
}

static void
notebook_page_switched (GtkNotebook *notebook,
			GtkNotebookPage *page,
			int page_num)
{
    notebook_current_page = page_num;
}

void
gui_update_player_pos (double songtime,
		       int nsongpos,
		       int npatpos)
{
    int m = xm_get_modified();

    if(gui_playing_mode == PLAYING_NOTE)
	return;

    /* This test prevents feedback which occurs when manually changing the song position;
       if the audio thread doesn't follow immediately, the song position would be set back
       to its old position some lines below... */
    if(set_songpos_count != 0 || (set_songpos_wait_time != -1.0 && songtime < set_songpos_wait_time))
	return;

    set_songpos_wait_time = -1.0;

    if(gui_playing_mode == PLAYING_SONG && nsongpos != songpos) {
	// This will not be executed if the user has manually changed the song position
	songpos = nsongpos;
	playlist_set_position(playlist, songpos);
	gui_set_current_pattern(xm->pattern_order_table[songpos]);
    }
    if(gui_playing_mode != PLAYING_NOTE) {
	tracker_set_patpos(tracker, npatpos);
    }

    if(notebook_current_page == 0) {
	gdk_flush(); /* X drawing accumulates otherwise and makes pattern scrolling rather non-realtime :) */
    }

    xm_set_modified(m);
}

void
gui_clipping_indicator_update (double songtime)
{
    if(songtime < 0.0) {
	gui_clipping_led_status = 0;
	gtk_widget_draw(gui_clipping_led, NULL);
    } else {
	audio_clipping_indicator *c = time_buffer_get(audio_clipping_indicator_tb, songtime);
	gui_clipping_led_status = c && c->clipping;
	gtk_widget_draw(gui_clipping_led, NULL);
    }
}

static void
read_mixer_pipe (gpointer data,
		 gint source,
		 GdkInputCondition condition)
{
    audio_backpipe_id a;
    struct pollfd pfd = { source, POLLIN, 0 };
    int x;

    static char *msgbuf = NULL;
    static int msgbuflen = 0;

  loop:
    if(poll(&pfd, 1, 0) <= 0)
	return;

    read(source, &a, sizeof(a));
//    printf("read %d\n", a);

    switch(a) {
    case AUDIO_BACKPIPE_PLAYING_STOPPED:
	if(startstop_count > 0) {
	    /* can be equal to zero when the audio subsystem decides to stop playing on its own. */
	    startstop_count--;
	}
	gui_playing_mode = 0;
	scope_group_stop_updating(scopegroup);
	tracker_stop_updating();
	gui_enable(1);
	break;

    case AUDIO_BACKPIPE_PLAYING_STARTED:
    case AUDIO_BACKPIPE_PLAYING_PATTERN_STARTED:
	startstop_count--;
	gui_playing_mode = (a == AUDIO_BACKPIPE_PLAYING_STARTED) ? PLAYING_SONG : PLAYING_PATTERN;
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(editing_toggle), 0);
	gui_enable(0);
	scope_group_start_updating(scopegroup);
	tracker_start_updating();
	break;

    case AUDIO_BACKPIPE_PLAYING_NOTE_STARTED:
	startstop_count--;
	if(!gui_playing_mode) {
	    gui_playing_mode = PLAYING_NOTE;
	    scope_group_start_updating(scopegroup);
	    tracker_start_updating();
	}
	break;

    case AUDIO_BACKPIPE_DRIVER_OPEN_FAILED:
	startstop_count--;
	break;

    case AUDIO_BACKPIPE_SONGPOS_CONFIRM:
	set_songpos_count--;
	readpipe(source, &set_songpos_wait_time, sizeof(set_songpos_wait_time));
	g_assert(set_songpos_count >= 0);
	break;

    case AUDIO_BACKPIPE_ERROR_MESSAGE:
    case AUDIO_BACKPIPE_WARNING_MESSAGE:
	readpipe(source, &x, sizeof(x));
	if(msgbuflen < x + 1) {
	    g_free(msgbuf);
	    msgbuf = g_new(char, x + 1);
	    msgbuflen = x + 1;
	}
	readpipe(source, msgbuf, x + 1);
	if(a == AUDIO_BACKPIPE_ERROR_MESSAGE)
	    gnome_error_dialog(msgbuf);
	else
	    gnome_warning_dialog(msgbuf);
	break;

    default:
	fprintf(stderr, "\n\n*** read_mixer_pipe: unexpected backpipe id %d\n\n\n", a);
	break;
    }

    goto loop;
}

static void
wait_for_player (void)
{
    struct pollfd pfd = { audio_backpipe, POLLIN, 0 };

    startstop_count++;
    while(startstop_count != 0) {
	g_return_if_fail(poll(&pfd, 1, -1) > 0);
	read_mixer_pipe(NULL, audio_backpipe, 0);
    }
}

static void
play_song (void)
{
    int sp = songpos;
    int pp = 0;
    audio_ctlpipe_id i = AUDIO_CTLPIPE_PLAY_SONG;

    gui_play_stop();

    write(audio_ctlpipe, &i, sizeof(i));
    write(audio_ctlpipe, &sp, sizeof(sp));
    write(audio_ctlpipe, &pp, sizeof(pp));

    gui_set_current_pattern(xm->pattern_order_table[sp]);
    wait_for_player();
}

static void
play_pattern (void)
{
    gui_play_stop();
    gui_mixer_play_pattern(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_editpat)), 0);
    wait_for_player();
}

void
gui_play_stop (void)
{
    gui_mixer_stop_playing();
    wait_for_player();
}

void
gui_init_xm (int new_xm)
{
    int m = xm_get_modified();
    audio_ctlpipe_id i;

    i = AUDIO_CTLPIPE_INIT_PLAYER;
    write(audio_ctlpipe, &i, sizeof(i));
    tracker_reset(tracker);
    if(new_xm) {
	programlist_initialize();
	editing_pat = -1;
	gui_set_current_pattern(xm->pattern_order_table[0]);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(curins_spin), 1);
	current_instrument_changed(GTK_SPIN_BUTTON(curins_spin));
	modinfo_set_current_instrument(0);
	modinfo_set_current_sample(0);
	modinfo_update_all();
    } else {
	i = editing_pat;
	editing_pat = -1;
	gui_set_current_pattern(i);
    }
    gui_subs_set_slider_value(&tempo_slider, xm->tempo);
    gui_subs_set_slider_value(&bpm_slider, xm->bpm);
    tracker_set_num_channels(tracker, xm->num_channels);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_numchans), xm->num_channels);
    scope_group_set_num_channels(scopegroup, xm->num_channels);
    xm_set_modified(m);
}

void
gui_free_xm (void)
{
    gui_play_stop();
    instrument_editor_set_instrument(NULL);
    sample_editor_set_sample(NULL);
    tracker_set_pattern(tracker, NULL);
    XM_Free(xm);
}

void
gui_new_xm (void)
{
    xm = XM_New();

    if(!xm) {
	fprintf(stderr, "Whooops, having memory problems?\n");
	exit(1);
    }
    gui_init_xm(1);
}

void
gui_load_xm (const char *filename)
{
    gui_free_xm();
    xm = XM_Load(filename);

    if(!xm) {
	gui_new_xm();
    } else
	gui_init_xm(1);
}

void
gui_play_note (int channel,
	       int note)
{
    int instrument = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin));
    audio_ctlpipe_id a = AUDIO_CTLPIPE_PLAY_NOTE;

    write(audio_ctlpipe, &a, sizeof(a));
    write(audio_ctlpipe, &channel, sizeof(channel));
    write(audio_ctlpipe, &note, sizeof(note));
    write(audio_ctlpipe, &instrument, sizeof(instrument));
    startstop_count++;
}

void
gui_play_note_full (int channel,
		    int note,
		    STSample *sample,
		    int offset)
{
    audio_ctlpipe_id a = AUDIO_CTLPIPE_PLAY_NOTE_FULL;
    write(audio_ctlpipe, &a, sizeof(a));
    write(audio_ctlpipe, &channel, sizeof(channel));
    write(audio_ctlpipe, &note, sizeof(note));
    write(audio_ctlpipe, &sample, sizeof(sample));
    write(audio_ctlpipe, &offset, sizeof(offset));
    startstop_count++;
}

void
gui_play_note_keyoff (int channel)
{
    audio_ctlpipe_id a = AUDIO_CTLPIPE_PLAY_NOTE_KEYOFF;
    write(audio_ctlpipe, &a, sizeof(a));
    write(audio_ctlpipe, &channel, sizeof(channel));
}

static void
gui_enable (int enable)
{
    gtk_widget_set_sensitive(vscrollbar, enable);
    gtk_widget_set_sensitive(editing_toggle, enable);
    gtk_widget_set_sensitive(spin_patlen, enable);
    playlist_enable(playlist, enable);
}

void
gui_set_current_pattern (int p)
{
    int m = xm_get_modified();

    if(editing_pat == p)
	return;

    editing_pat = p;
    tracker_set_pattern(tracker, &xm->patterns[p]);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_editpat), p);
    gui_update_pattern_data();
    if(!GUI_ENABLED)
	gui_mixer_set_pattern(p);
    xm_set_modified(m);
}

void
gui_update_pattern_data (void)
{
    int p = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_editpat));

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_patlen), xm->patterns[p].length);
}

int
gui_get_current_pattern (void)
{
    return editing_pat;
}

static void
offset_current_pattern (int offset)
{
    int nv;

    nv = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_editpat)) + offset;

    if(nv < 0)
	nv = 0;
    else if(nv > 255)
	nv = 255;

    gui_set_current_pattern(nv);
}

void
gui_set_current_instrument (int n)
{
    int m = xm_get_modified();
    g_return_if_fail(n >= 1 && n <= 128);
    if(n != gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin))) {
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(curins_spin), n);
    }
    xm_set_modified(m);
}

void
gui_set_current_sample (int n)
{
    int m = xm_get_modified();
    g_return_if_fail(n >= 0 && n <= 15);
    if(n != gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin))) {
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(cursmpl_spin), n);
    }
    xm_set_modified(m);
}

int
gui_get_current_sample (void)
{
    return gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin));
}

int
gui_get_current_octave_value (void)
{
    return gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_octave));
}

int
gui_get_current_jump_value (void)
{
    return gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_jump));
}

int
gui_get_current_instrument (void)
{
    return gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin));
}

static void
offset_current_instrument (int offset)
{
    int nv, v;

    v = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin));
    nv = v + offset;

    if(nv < 1)
	nv = 1;
    else if(nv > 128)
	nv = 128;

    gui_set_current_instrument(nv);
}

static void
offset_current_sample (int offset)
{
    int nv, v;

    v = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin));
    nv = v + offset;

    if(nv < 0)
	nv = 0;
    else if(nv > 15)
	nv = 15;

    gui_set_current_sample(nv);
}

static void
text_entry_selected (void)
{
    GSList *i;
    GtkWidget *f;

    if(capture_keys == 1) {
	for(i = text_entries, f = GTK_WINDOW(mainwindow)->focus_widget; i; i = g_slist_next(i)) {
	    if(f == i->data) {
		capture_keys = 0;
		break;
	    }
	}
    }
}

static void
text_entry_unselected (void)
{
    if(capture_keys == 0) {
	capture_keys = 1;
    }
}

void
gui_add_text_entry_capture (GtkWidget *widget)
{
    gtk_signal_connect_after(GTK_OBJECT(widget), "focus_in_event",
			     GTK_SIGNAL_FUNC(text_entry_selected), NULL);
    gtk_signal_connect_after(GTK_OBJECT(widget), "focus_out_event",
			     GTK_SIGNAL_FUNC(text_entry_unselected), NULL);

    text_entries = g_slist_prepend(text_entries, widget);
}

void
gui_get_text_entry (int length,
		    void(*changedfunc)(),
		    GtkWidget **widget)
{
    GtkWidget *thing;

    thing = gtk_entry_new_with_max_length(length);

    gtk_signal_connect_after(GTK_OBJECT(thing), "changed",
			     GTK_SIGNAL_FUNC(changedfunc), NULL);

    gui_add_text_entry_capture(thing);

    *widget = thing;
}

static void
programlist_initialize (void)
{
    int i;

    playlist_freeze_signals(playlist);
    playlist_freeze(playlist);
    songpos = 0;
    playlist_set_position(playlist, 0);
    playlist_set_length(playlist, xm->song_length);
    for(i = 0; i < xm->song_length; i++) {
	playlist_set_nth_pattern(playlist, i, xm->pattern_order_table[i]);
    }
    playlist_set_restart_position(playlist, xm->restart_position);
    playlist_thaw(playlist);
    playlist_thaw_signals(playlist);
}

static gint
gui_clipping_led_event (GtkWidget *thing,
			GdkEvent *event)
{
    static GdkGC *clipping_led_gc = NULL;

    if(!clipping_led_gc)
	clipping_led_gc = gdk_gc_new(thing->window);

    switch (event->type) {
    case GDK_MAP:
    case GDK_EXPOSE:
	gdk_gc_set_foreground(clipping_led_gc, gui_clipping_led_status ? &gui_clipping_led_on : &gui_clipping_led_off);
	gdk_draw_rectangle(thing->window, clipping_led_gc, 1, 0, 0, -1, -1);
    default:
	break;
    }
    
    return 0;
}

void
gui_go_to_fileops_page (void)
{
    gtk_notebook_set_page(GTK_NOTEBOOK(notebook),
			  0);
}

int
gui_init (int argc,
	  char *argv[])
{
    GtkWidget *thing, *mainvbox, *table, *hbox, *frame, *mainvbox0;
    GdkColormap *colormap;

#ifdef USE_GNOME
    gnome_init("SoundTracker", VERSION, argc, argv);
#else
    gtk_init(&argc, &argv);
#endif

    pipetag = gdk_input_add(audio_backpipe, GDK_INPUT_READ, read_mixer_pipe, NULL);

    text_entries = g_slist_alloc();

#ifdef USE_GNOME
    mainwindow = gnome_app_new("SoundTracker", "SoundTracker " VERSION);
#else
    mainwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (mainwindow), "SoundTracker " VERSION);
#endif
    gtk_signal_connect (GTK_OBJECT (mainwindow), "delete_event",
			GTK_SIGNAL_FUNC (menubar_quit_requested), NULL);

    fileops_dialogs[DIALOG_LOAD_MOD] = file_selection_create(_("Load XM..."), file_selected);
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileops_dialogs[DIALOG_LOAD_MOD]), gui_settings.loadmod_path);
    fileops_dialogs[DIALOG_SAVE_MOD] = file_selection_create(_("Save XM..."), file_selected);
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileops_dialogs[DIALOG_SAVE_MOD]), gui_settings.savemod_path);
#ifndef NO_AUDIOFILE
    fileops_dialogs[DIALOG_SAVE_MOD_AS_WAV] = file_selection_create(_("Render module as WAV..."), file_selected);
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileops_dialogs[DIALOG_SAVE_MOD_AS_WAV]), gui_settings.savemodaswav_path);
#endif

    mainvbox0 = gtk_vbox_new(FALSE, 0);
    gtk_container_border_width(GTK_CONTAINER(mainvbox0), 0);
#ifdef USE_GNOME
    gnome_app_set_contents(GNOME_APP(mainwindow), mainvbox0);
#else
    gtk_container_add(GTK_CONTAINER(mainwindow), mainvbox0);
#endif
    gtk_widget_show(mainvbox0);

    menubar_create(mainwindow, mainvbox0);

    mainvbox = gtk_vbox_new(FALSE, 4);
    gtk_container_border_width(GTK_CONTAINER(mainvbox), 4);
    gtk_box_pack_start(GTK_BOX(mainvbox0), mainvbox, TRUE, TRUE, 0);
    gtk_widget_show(mainvbox);

    /* The upper part of the window */
    mainwindow_upper_hbox = gtk_hbox_new(FALSE, 4);
    gtk_box_pack_start(GTK_BOX(mainvbox), mainwindow_upper_hbox, FALSE, TRUE, 0);
    gtk_widget_show(mainwindow_upper_hbox);

    /* Program List */
    thing = playlist_new();
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    playlist = PLAYLIST(thing);
    gtk_signal_connect (GTK_OBJECT (playlist), "current_position_changed",
			GTK_SIGNAL_FUNC (playlist_position_changed), NULL);
    gtk_signal_connect (GTK_OBJECT (playlist), "restart_position_changed",
			GTK_SIGNAL_FUNC (playlist_restart_position_changed), NULL);
    gtk_signal_connect (GTK_OBJECT (playlist), "song_length_changed",
			GTK_SIGNAL_FUNC (playlist_song_length_changed), NULL);
    gtk_signal_connect (GTK_OBJECT (playlist), "entry_changed",
			GTK_SIGNAL_FUNC (playlist_entry_changed), NULL);

    thing = gtk_vseparator_new();
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    /* Basic editing commands and properties */
    
    table = gtk_table_new(5, 2, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 4);
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), table, FALSE, TRUE, 0);
    gtk_widget_show(table);

    thing = gtk_button_new_with_label (_("Play Song"));
    gtk_signal_connect (GTK_OBJECT (thing), "clicked",
			GTK_SIGNAL_FUNC(play_song), NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), thing, 0, 1, 0, 1);
    gtk_widget_show(thing);

    thing = gtk_button_new_with_label (_("Play Pattern"));
    gtk_signal_connect (GTK_OBJECT (thing), "clicked",
			GTK_SIGNAL_FUNC(play_pattern), NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), thing, 1, 2, 0, 1);
    gtk_widget_show(thing);

    thing = gtk_button_new_with_label (_("Stop"));
    gtk_signal_connect (GTK_OBJECT (thing), "clicked",
			GTK_SIGNAL_FUNC(gui_play_stop), NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), thing, 0, 2, 1, 2);
    gtk_widget_show(thing);

    thing = gui_subs_create_slider(&tempo_slider);
    gtk_table_attach_defaults(GTK_TABLE(table), thing, 0, 1, 3, 4);

    gtk_widget_show(thing);

    thing = gui_subs_create_slider(&bpm_slider);
    gtk_table_attach_defaults(GTK_TABLE(table), thing, 0, 1, 4, 5);
    gtk_widget_show(thing);

    hbox = gtk_hbox_new(FALSE, 4);
    gtk_table_attach_defaults(GTK_TABLE(table), hbox, 0, 2, 2, 3);
    gtk_widget_show(hbox);

    thing = gtk_label_new(_("Number of Channels:"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    add_empty_hbox(hbox);

    spin_numchans = gtk_spin_button_new(GTK_ADJUSTMENT(gtk_adjustment_new(8, 2, 32, 2.0, 8.0, 0.0)), 0, 0);
    gtk_box_pack_start(GTK_BOX(hbox), spin_numchans, FALSE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(spin_numchans), "changed",
		       GTK_SIGNAL_FUNC(spin_numchans_changed), NULL);
    gtk_widget_show(spin_numchans);

    hbox = gtk_hbox_new(FALSE, 4);
    gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 2, 3, 4);
    gtk_widget_show(hbox);

    thing = gtk_label_new(_("Pattern"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    add_empty_hbox(hbox);

    spin_editpat = gtk_spin_button_new(GTK_ADJUSTMENT(gtk_adjustment_new(0, 0, 255, 1.0, 10.0, 0.0)), 0, 0);
    gtk_box_pack_start(GTK_BOX(hbox), spin_editpat, FALSE, TRUE, 0);
    gtk_widget_show(spin_editpat);
    gtk_signal_connect(GTK_OBJECT(spin_editpat), "changed",
		       GTK_SIGNAL_FUNC(spin_editpat_changed), NULL);
    
    hbox = gtk_hbox_new(FALSE, 4);
    gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 2, 4, 5);
    gtk_widget_show(hbox);

    thing = gtk_label_new(_("PatLength"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    add_empty_hbox(hbox);

    spin_patlen = gtk_spin_button_new(GTK_ADJUSTMENT(gtk_adjustment_new(64, 1, 256, 1.0, 16.0, 0.0)), 0, 0);
    gtk_box_pack_start(GTK_BOX(hbox), spin_patlen, FALSE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(spin_patlen), "changed",
		       GTK_SIGNAL_FUNC(spin_patlen_changed), NULL);
    gtk_widget_show(spin_patlen);

    /* Scopes Group or Instrument / Sample Listing */

    thing = gtk_vseparator_new();
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
    
    scopegroup = SCOPE_GROUP(scope_group_new());
    gtk_widget_show(GTK_WIDGET(scopegroup));
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), GTK_WIDGET(scopegroup), TRUE, TRUE, 0);

    /* Amplification and Pitchbender */

    thing = gtk_vseparator_new();
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
    
    hbox = gtk_vbox_new(FALSE, 2);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), hbox, FALSE, TRUE, 0);

    adj_amplification = GTK_ADJUSTMENT(gtk_adjustment_new(7.0, 0, 8.0, 0.1, 0.1, 0.1));
    thing = gtk_vscale_new(adj_amplification);
    gtk_scale_set_draw_value(GTK_SCALE(thing), FALSE);
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(hbox), thing, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(adj_amplification), "value_changed",
			GTK_SIGNAL_FUNC(gui_adj_amplification_changed), NULL);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
    gtk_box_pack_start(GTK_BOX(hbox), frame, FALSE, TRUE, 0);
    gtk_widget_show(frame);

    gui_clipping_led = thing = gtk_drawing_area_new();
    gtk_drawing_area_size(GTK_DRAWING_AREA(thing), 15, 15);
    gtk_widget_set_events(thing, GDK_EXPOSURE_MASK);
    gtk_container_add (GTK_CONTAINER(frame), thing);
    colormap = gtk_widget_get_colormap(thing);
    gui_clipping_led_on.red = 0xffff;
    gui_clipping_led_on.green = 0;
    gui_clipping_led_on.blue = 0;
    gui_clipping_led_on.pixel = 0;
    gui_clipping_led_off.red = 0;
    gui_clipping_led_off.green = 0;
    gui_clipping_led_off.blue = 0;
    gui_clipping_led_off.pixel = 0;
    gdk_color_alloc(colormap, &gui_clipping_led_on);
    gdk_color_alloc(colormap, &gui_clipping_led_off);
    gtk_signal_connect(GTK_OBJECT(thing), "event", GTK_SIGNAL_FUNC(gui_clipping_led_event), thing);
    gtk_widget_show (thing);

    hbox = gtk_vbox_new(FALSE, 2);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), hbox, FALSE, TRUE, 0);

    adj_pitchbend = GTK_ADJUSTMENT(gtk_adjustment_new(0.0, -20.0, +20.0, 1, 1, 1));
    thing = gtk_vscale_new(adj_pitchbend);
    gtk_scale_set_draw_value(GTK_SCALE(thing), FALSE);
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(hbox), thing, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(adj_pitchbend), "value_changed",
			GTK_SIGNAL_FUNC(gui_adj_pitchbend_changed), NULL);

    thing = gtk_button_new_with_label("R");
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (thing), "clicked",
			GTK_SIGNAL_FUNC(gui_reset_pitch_bender), NULL);

    /* Instrument, sample, editing status */

    mainwindow_second_hbox = hbox = gtk_hbox_new(FALSE, 4);
    gtk_box_pack_start(GTK_BOX(mainvbox), hbox, FALSE, TRUE, 0);
    gtk_widget_show(hbox);

    editing_toggle = thing = gtk_check_button_new_with_label("Editing");
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(thing), 0);
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gtk_label_new(_("Octave"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    spin_octave = gtk_spin_button_new(GTK_ADJUSTMENT(gtk_adjustment_new(5.0, 0.0, 6.0, 1.0, 1.0, 0.0)), 0, 0);
    gtk_box_pack_start(GTK_BOX(hbox), spin_octave, FALSE, TRUE, 0);
    gtk_widget_show(spin_octave);

    thing = gtk_label_new(_("Jump"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    spin_jump = gtk_spin_button_new(GTK_ADJUSTMENT(gtk_adjustment_new(1.0, 0.0, 8.0, 1.0, 1.0, 0.0)), 0, 0);
    gtk_box_pack_start(GTK_BOX(hbox), spin_jump, FALSE, TRUE, 0);
    gtk_widget_show(spin_jump);

    thing = gtk_label_new(_("Instr"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    curins_spin = gtk_spin_button_new(GTK_ADJUSTMENT(gtk_adjustment_new(1.0, 1.0, 128.0, 1.0, 16.0, 0.0)), 0, 0);
    gtk_box_pack_start(GTK_BOX(hbox), curins_spin, FALSE, TRUE, 0);
    gtk_widget_show(curins_spin);
    gtk_signal_connect (GTK_OBJECT(curins_spin), "changed",
			GTK_SIGNAL_FUNC(current_instrument_changed), NULL);

    gui_get_text_entry(22, current_instrument_name_changed, &gui_curins_name);
    gtk_box_pack_start(GTK_BOX(hbox), gui_curins_name, TRUE, TRUE, 0);
    gtk_widget_show(gui_curins_name);
    gtk_widget_set_usize(gui_curins_name, 100, gui_curins_name->requisition.height);

    thing = gtk_label_new(_("Sample"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    cursmpl_spin = gtk_spin_button_new(GTK_ADJUSTMENT(gtk_adjustment_new(0.0, 0.0, 15.0, 1.0, 4.0, 0.0)), 0, 0);
    gtk_box_pack_start(GTK_BOX(hbox), cursmpl_spin, FALSE, TRUE, 0);
    gtk_widget_show(cursmpl_spin);
    gtk_signal_connect (GTK_OBJECT(cursmpl_spin), "changed",
			GTK_SIGNAL_FUNC(current_sample_changed), NULL);

    gui_get_text_entry(22, current_sample_name_changed, &gui_cursmpl_name);
    gtk_box_pack_start(GTK_BOX(hbox), gui_cursmpl_name, TRUE, TRUE, 0);
    gtk_widget_show(gui_cursmpl_name);
    gtk_widget_set_usize(gui_cursmpl_name, 100, gui_cursmpl_name->requisition.height);

    /* The notebook */

    notebook = gtk_notebook_new();
    gtk_box_pack_start(GTK_BOX(mainvbox), notebook, TRUE, TRUE, 0);
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
    gtk_widget_show(notebook);
    gtk_container_border_width(GTK_CONTAINER(notebook), 0);
    gtk_signal_connect(GTK_OBJECT(notebook), "switch_page",
		       GTK_SIGNAL_FUNC(notebook_page_switched), NULL);

    fileops_page_create(GTK_NOTEBOOK(notebook));
    tracker_page_create(GTK_NOTEBOOK(notebook));
    instrument_page_create(GTK_NOTEBOOK(notebook));
    sample_editor_page_create(GTK_NOTEBOOK(notebook));
    modinfo_page_create(GTK_NOTEBOOK(notebook));

    // Activate tracker page
    gtk_notebook_set_page(GTK_NOTEBOOK(notebook),
			  1);
    notebook_current_page = 1;

    /* status bar */
/*
    hbox = gtk_hbox_new(FALSE, 4);
    gtk_box_pack_start(GTK_BOX(mainvbox), hbox, FALSE, TRUE, 0);
    gtk_widget_show(hbox);
    
    add_empty_hbox(hbox);
    thing = gtk_vseparator_new();
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gui_button (mainwindow, GNOME_STOCK_BUTTON_UP,
			0, NULL, NULL, 0);
    gtk_box_pack_start (GTK_BOX (hbox), thing, 0, 0, 0);
    gtk_widget_show (thing);
//    add_tip (wid, "Link/DeLink this tab");
*/
    /* capture all key presses */
    gtk_widget_add_events(GTK_WIDGET(mainwindow), GDK_KEY_RELEASE_MASK);
    gtk_signal_connect(GTK_OBJECT(mainwindow), "key_press_event", GTK_SIGNAL_FUNC(keyevent), (gpointer)1);
    gtk_signal_connect(GTK_OBJECT(mainwindow), "key_release_event", GTK_SIGNAL_FUNC(keyevent), (gpointer)0);

    if(argc == 2) {
	gui_load_xm(argv[1]);
    } else {
	gui_new_xm();
    }

    menubar_init_prefs();

    gtk_widget_show (mainwindow);

    if(show_tips) {
	tips_dialog_create();
    }

    if(!keys_init()) {
	return 0;
    }

    return 1;
}
