// filename.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "localdefs.h"
#include "filename.h"
#include <String.h>

#ifdef String
#undef String
#endif

int FileName::tmpCount = 0;
int FileName::untitledCount = 0;

// all of these return pointer to single copies of static data, and so
// routines should *always* copy these strings and not use them directly

const char*
FileName::tmpName(const char* suffix) {
	static char tmpname[128];
	sprintf(tmpname, "tmp_%d%s", tmpCount++, suffix);
	return tmpname;
}

const char*
FileName::untitledName(const char* suffix) {
	static char uname[128];
	sprintf(uname, "untitled_%d%s", untitledCount++, suffix);
	return uname;
}

const char*
FileName::segmentName(const char* name, double skip, double dur) {
	static char sname[256];
	if(skip > 0.0 || dur != 0.0)
		sprintf(sname, "tmp_s%g_d%g_%s", skip, dur, stripPath(name));
	else
		strcpy(sname, name);
	return sname;
}

boolean
FileName::isTempName(const char* name) {
	return (!strncmp(name, "untitled_", 9) || !strncmp(name, "tmp_", 4));
}

boolean
FileName::isFullPathName(const char* name) {
	return (index(name, '/') != NULL);
}

const char*
FileName::sfName(const char* filename) {
	static char sfname[1024];
	if(isFullPathName(filename))
		return filename;
	const char* sfdir = getenv("SFDIR");
	if(sfdir == NULL) sfdir = ".";
	sprintf(sfname, "%s/%s", sfdir, filename);
	return sfname;
}

const char*
FileName::stripPath(const char* filename) {
	static char name[1024];
	strcpy(name, filename);	// to avoid nonwritable strings
	const char* newName = rindex(name, '/');
	return (newName != NULL) ? newName + 1 : name;
}

const char *
FileName::changeSuffix(const char* filename, const char* newSuffix) {
	String name(filename);
	static String base;
	static Regex oldSuffix("[.]snd\\|[.]au\\|[.][wW][aA][vV]\\|[.]aif[fc]\\|[.]fp\\|[.]sound\\|[.]short\\|[.]float");
	base = name.before(oldSuffix, -1);
	if(base == "") base = name;
	base += newSuffix;
	return base;
}

FileType
FileName::fileType(const char *filename) {
	String s(filename);
	int len = strlen(filename);
	return(s.contains(".lpc", len - 4) ? LPC_Data
		: (s.contains(".pt", len - 3) || 
		   s.contains(".pch", len - 4)) ? PCH_Data
		: s.contains(".fft", len - 4) ? FFT_Data
		: s.contains(".evp", len - 4) ? EVP_Data
		: (s.contains(".pv", len - 3) ||
			s.contains(".i", len - 2)) ? Pvoc_Data
		: Sound_Data
	);
}

