// editor.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// The DataEditor and its subclasses manage all the operations upon Data
// objects.  It keeps track of the current region to be edited, and invokes
// the cloning methods upon the data to create the selection to be modified.
// DataEditor instances are members of their associated Controller, and never
// exist on their own.

#ifndef EDITOR_H
#ifdef __GNUG__
#pragma interface
#endif
#define EDITOR_H

#include "range.h"

class Controller;
class Data;
class Sound;
class FileAccessCommand;
class LPCData;
class FFTData;
class DataFile;
struct MenuBarInfo;
class Modifier;
class DataModifier;

extern const char *scat(const char *, const char *);
extern const char *scat(const char *, const char *, const char *);

class DataEditor {
public:
	virtual Status displayInternalBuffer();
	virtual Status setDefaultDir();
	virtual Status newFile();	// new file of same type as editor
	Status newLPCFile();		// specific types
	Status newSoundFile();
	Status newEnvelopeFile();
	Status newPvocFile();
	virtual Status openNewFile();
	virtual Response closeFile();
	virtual Status saveFile();
	virtual Status saveToFile();
	virtual Status revertToSaved();
	virtual Status changeComment();
	virtual Status information();
	virtual Status dataDump();
	virtual Status copy();
	virtual Status copyToNew();
	virtual Status remove();
	virtual Status removeToNew();
	virtual Status erase();
	virtual Status spliceOut();
	virtual Status spliceOutToNew();
	virtual Status zap();
	virtual Status mix();
	virtual Status replace();
	virtual Status crossfade();
	virtual Status spliceIn();
	virtual Status changeLength();
	virtual Status normalize();
	virtual Status lowPassFilter();
	virtual Status scaleValues();
	virtual Status vertOffset();
	virtual Status rescale();
	virtual Status delay();
	virtual Status applyEnvelope();
	virtual Status reverse();
	virtual Status insertSpace();
	virtual Status interpolate();
	virtual Status changeSampleRate();
	virtual Status showPeakFrame();
	virtual Status extractEnvelope();
	virtual Status findZeroCrossing();
	virtual Status findSlopeChange();
	virtual Status setDataOptions() { return Succeed; }

	Controller* openNewFile(const char* windowtitle);

	virtual const MenuBarInfo *menuBarInfo() const;
	Controller *controller;
	Data* model() const { return data; }	// pointer to underlying data
	Data* currentSelection();		// portion of displayed data to be modified
	Data* currentSource();			// selected data for binary operations
	Data* copyBuffer() { return internalBuffer; }	// pasteboard buffer
protected:
	DataEditor(Controller *c, const Data *d=nil);
	static DataEditor* create(Controller *, const char *);
	static DataEditor* create(Controller *, const Data *);
	virtual ~DataEditor();
	virtual const char* defaultDir();
	boolean readFile(DataFile *);
	Data* createModel(DataFile *);
	static Status setCopyBuffer(Data *);
	boolean selectionMade() { return isSelected; }
	void reselect();
	void setEditRegion(const Range &region, int chan);
	void setInsertPoint(int point, int chan);
	void setLocation(const Range &);
	void addChannel(int chan);
	void setChannels(int first, int last) { channels.set(first, last); }
	void setChannels(const Range& chans) { channels = chans; }
	int currentInsert() { return location.intMin(); }
	Range currentRegion() { return location; }
	int nchans() { return int(channels.spread()) + 1; }
	virtual Range currentChannels();
	long timeStamp() { return timestamp; }
	static void freeSource(Data *s);
	void markEditRegion(int);
	
	virtual boolean keyCommand(unsigned long);
	static boolean applyModifierUsing(Modifier &, Controller *);
	virtual boolean applyModifier(Modifier &);
	virtual boolean applyModifierToNew(Modifier &);
	
	void setTimeStamp();
	Range channels;
private:
	virtual Data *newModel() = 0;
	void setSelection(Data *);
	Data* getSelection(long &timestamp);
	void reset() {
		channels.set(-1, -1); timestamp = 0; isSelected = false;
	}
	void init(Data *);
	void setData(Data *);
private:
	friend Controller;
	typedef class DataEditor* (*C_Constructor)(Controller *);
	typedef class DataEditor* (*C_D_Constructor)(Controller *, const Data *);
	static Data *internalBuffer;
	Data *data;
	Data *selection;
	long timestamp;
	boolean isSelected;
	Range location;
	static C_Constructor ctor_table1[];
	static C_D_Constructor ctor_table2[];
};

class LPCEditor : public DataEditor {
	typedef DataEditor Super;
public:
	static Controller* new_File(Controller *);
	redefined Status newFile();
	redefined Status saveFile();
	redefined Status saveToFile();
	Status stabilizeFrames();
	Status mergePitchData();
	Status displayFrameAmplitudes();
	Status displayFrameFormants();
	Status adjustPitchDeviation();
	redefined Status setDataOptions();
protected:
	LPCEditor(Controller *c) : DataEditor(c) {}
	LPCEditor(Controller *c, const Data *d) : DataEditor(c, d) {}

	redefined boolean keyCommand(unsigned long);
	redefined Range currentChannels();

public:
	redefined const MenuBarInfo *menuBarInfo() const;
	static DataEditor* new_DataEditor1(Controller *);
	static DataEditor* new_DataEditor2(Controller *, const Data *);
private:	
	friend Controller;
	redefined Data *newModel();
};

class PCHEditor : public DataEditor {
	typedef DataEditor Super;
protected:
	PCHEditor(Controller *c) : DataEditor(c) {}
	PCHEditor(Controller *c, const Data *d) : DataEditor(c, d) {}
public:
	redefined const MenuBarInfo *menuBarInfo() const;
	static DataEditor* new_DataEditor1(Controller *);
	static DataEditor* new_DataEditor2(Controller *, const Data *);
private:	
	friend Controller;
	redefined Data *newModel();
};

class FFTEditor : public DataEditor {
protected:
	FFTEditor(Controller *c) : DataEditor(c) {}
	FFTEditor(Controller *c, const Data *d) : DataEditor(c, d) {}
public:
	redefined Status setDataOptions();
	redefined const MenuBarInfo *menuBarInfo() const;
	static DataEditor* new_DataEditor1(Controller *);
	static DataEditor* new_DataEditor2(Controller *, const Data *);
private:	
	friend Controller;
	redefined Data *newModel();
};

class EnvelopeEditor : public DataEditor {
	typedef DataEditor Super;
public:
	static Controller* new_File(Controller *);
	redefined Status newFile();
	Status createLinearCurve();
	Status createExponentialCurve();
	redefined Status setDataOptions();
protected:
	EnvelopeEditor(Controller *c) : DataEditor(c) {}
	EnvelopeEditor(Controller *c, const Data *d) : DataEditor(c, d) {}

	redefined boolean keyCommand(unsigned long);
public:
	redefined const MenuBarInfo *menuBarInfo() const;
	static DataEditor* new_DataEditor1(Controller *);
	static DataEditor* new_DataEditor2(Controller *, const Data *);
private:
	friend Controller;
	redefined Data *newModel();
};

#endif
