/*----------------------------------------------------------------
 * midievent.h
 *	read midi file and parse events
 *
 * Copyright (C) 1996-1998 Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *----------------------------------------------------------------*/

#ifndef MIDIEVENT_H_DEF
#define MIDIEVENT_H_DEF

typedef struct _MidiEvent {
	int time;	/* tempo counter */
	int csec;	/* realtime csec counter */
	unsigned char channel, type;
	union {
		short par[2]; /* RPN events */
		int val; /* others */
		char *text; /* text message */
	} p;
} MidiEvent;

#define KeyOf(ev) ((ev)->p.par[0])
#define VelOf(ev) ((ev)->p.par[1])
#define CtrlOf(ev) ((ev)->p.par[0])
#define CValOf(ev) ((ev)->p.par[1])
#define BankOf(ev) ((ev)->p.par[0])
#define IsMSB(ev)  ((ev)->p.par[1])

#define FX_CUTOFF	0
#define FX_RESONANCE	1
#define FX_ATTACK	2
#define FX_RELEASE	3
#define FX_VIBRATE	4
#define FX_VIBDEPTH	5
#define FX_VIBDELAY	6
#define FX_NUMS		7

typedef struct _MidiInfo {
	int format;
	int tracks;
	int divisions;

	/* variable data */
	int startcs, endidx;	/* start csec, end index */
	int prevcs, curcs;	/* prev/current csec */
	int updatecs, echocs;	/* timer for control update & echo back */
	int prevev, curev;	/* event counter */
	int tempo;		/* current tempo */

	/* midi data */
	int nlists;
	MidiEvent *list;

	/* other control parameters */
	int chorus, reverb;
	int chorusdepth, reverbdepth;
	int volume_base, master_volume;
	int bass_level, treble_level;
	int accept_all_off;
	int realtime_pan;
	int new_volume_mode;
	int multi_part;
	int check_same_csec;
	int midi_mode;	/* GM=0, MT32=1, MT32(conv)=2, GS=3, XG=4, CMF=5/6 */
	int check_gs_macro;
	int check_xg_macro, xg_mapping;
	int chn_prior;
	int do_tuning;
	unsigned int drumflag;
	int track_nums;
	int volscale;
	int skip_blank_head;
	int chn_volflag;
	int base_offset; /* in semitones */
	int parse_title;
	int seq_buffered, use_echoback;
	int incremental;
	int use_fx;
	int fx_sense[FX_NUMS];
	
	char *filename;
	char *dynamicload;
	char *xgload;
	int dynamic_loaded;
} MidiInfo;

enum {
	MODE_GM = 0, MODE_MT32, MODE_MT32_INT, MODE_GS, MODE_XG,
	MODE_CMF, MODE_CMF2
};

#define DRUMBIT(ch)	(1 << (ch))	/* ch: 0 - 31 */

#define MidiTempo(a,b,c) ((int)(a) * 65536 + (int)(b) * 256 + (int)(c))


/* Midi events */
enum {
	ME_NONE,
	ME_NOTEON, ME_NOTEOFF, ME_KEYPRESSURE, ME_CONTROL, 
	ME_PROGRAM, ME_CHNPRESSURE, ME_PITCHWHEEL,
	ME_TEMPO, ME_PITCH_SENS, ME_TONE_BANK,
	ME_ALL_SOUNDS_OFF, ME_RESET_CONTROLLERS, ME_ALL_NOTES_OFF,

	ME_LYRIC,

	ME_MASTER_VOLUME,

	ME_FINETUNE, ME_COARSETUNE,
	ME_AWE_FX, ME_GS_FX,
	
	ME_EOT,
};

/* default values */
#define MIDI_DEFAULT_TEMPO	500000

enum {
	EV_PLAY=0, EV_CONTROL, EV_SETSTAT, EV_STARTUP, EV_PRELOAD,
};

enum {
	SY_CHORUS_MODE, SY_REVERB_MODE,
};

/* prototypes */
MidiEvent *ReadMidiFile(FILE *fp, MidiInfo *infop);
void FreeMidiFile(MidiInfo *infop);

int do_midi_event(MidiEvent *ev, MidiInfo *minfo, int mode);

extern MidiInfo glinfo, gl_default;
int play_midi_file(MidiInfo *minfo);
int midi_open(MidiInfo *mp);
void midi_close(MidiInfo *mp);
void init_preload(MidiInfo *mp);
void preload_sample(MidiInfo *mp, char *fontfile);
void add_preload(int chn, int preset, int bank, int keynote);
void load_sub_sf(MidiInfo *mp, char *args);
void load_partial_fonts(MidiInfo *mp);
int find_midi_index(MidiInfo *mp, int before_reading);

void copy_midi_info(MidiInfo *dst, MidiInfo *src, char *fname);

void channel_init(MidiInfo *mp);
void channel_set(MidiInfo *mp);

#endif
