/*  PlayList.cpp - playlist window 'n stuff
 *  Copyright (C) 1998 Andy Lo A Foe <andy@alsa-project.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <pthread.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include "Main.h"
#include "Playlist.h"
#include "CorePlayer.h"
#include "gladesrc.h"

extern int global_playlist_show;
extern gint global_playing;

void destroy_notify(gpointer);
int new_list_item(const char *path, gchar **list_item);

int playlist_not_empty(GtkWidget *widget)
{
	GtkWidget *list;

	list =  (GtkWidget *) gtk_object_get_data(GTK_OBJECT(widget), "list");

	if (list)
		return GTK_CLIST(list)->rows;
	else
		return 0;
}

gboolean playlist_delete_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
        gint x, y;

		if (widget->window) {
        	gdk_window_get_origin(widget->window, &x, &y);
        	if (windows_x_offset >= 0) {
				x -= windows_x_offset;
				y -= windows_y_offset;
			}
			gtk_widget_hide(widget);
			gtk_widget_set_uposition(widget, x, y);
			global_playlist_show = 0;
		}
		return TRUE;
}


void dialog_cancel(GtkWidget *widget, gpointer data)
{
	GtkWidget *dialog = (GtkWidget *)data;

	gtk_widget_hide(dialog);
}


void dialog_delete(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gtk_widget_hide(widget);
}


void list_add(gpointer data);

typedef struct _list_node {
	GtkWidget *the_list;
	GtkLabel *status;
	gchar start_path[1024];
	// Accounting stuff
	gint count;
	gint added;
	gint last_reported;
} list_node;

void load_list_ok(GtkWidget *widget, gpointer data)
{
	gpointer list = NULL;
	char path[1024];
	char *list_item[3];

	list =  gtk_object_get_data(GTK_OBJECT(data), "list");

	if (list) {
		gtk_widget_hide(GTK_WIDGET(data));	
		strcpy(path, gtk_file_selection_get_filename(GTK_FILE_SELECTION(data)));
		printf("Loading from: %s\n", path);

		FILE *in_list = fopen(path, "r");

		if (in_list) {
			while (fscanf(in_list, "%[^\n]\n", path) == 1) {
				printf("found %s\n", path);
				if (new_list_item(path, list_item)) {
					int index = gtk_clist_append(GTK_CLIST(list),
						list_item);
					gtk_clist_set_row_data_full(GTK_CLIST(list),
						index, list_item[2], destroy_notify);	
					gtk_clist_set_shift(GTK_CLIST(list),
						index, 1, 2, 2);
				}
			}				
						
		}
	}
}

void save_list_ok(GtkWidget *widget, gpointer data)
{
	gpointer list = NULL;
	char path[1024];

	list =  gtk_object_get_data(GTK_OBJECT(data), "list");

	if (list) {
		gtk_widget_hide(GTK_WIDGET(data));
		strcpy(path, gtk_file_selection_get_filename(GTK_FILE_SELECTION(data)));
		printf("Saving to: %s\n", path);
	
		FILE *out_list = fopen(path, "w");
	
		if (out_list) {
			int count = 0;
			gchar *bla;

			//fprintf(out_list, "%s\n", MAGIC_ID);
			while ((bla = (gchar *)gtk_clist_get_row_data(GTK_CLIST(list), count++))) {
				fprintf(out_list, "%s\n", bla);
			}	
			fclose(out_list);
		}
	}	
	
}

void add_file_ok(GtkWidget *widget, gpointer data)
{
	gpointer list = NULL;
	pthread_t adder;
	list_node *start_node = new list_node;
	list = 	gtk_object_get_data(GTK_OBJECT(data), "list");	
	strcpy(start_node->start_path, gtk_file_selection_get_filename(GTK_FILE_SELECTION(data)));
	start_node->the_list = (GtkWidget *)list;
	start_node->count = 0;
	start_node->added = 0;
	start_node->last_reported = 0;
	start_node->status = (GtkLabel *)gtk_object_get_data(
		GTK_OBJECT(list), "status");
	gtk_label_set_text(start_node->status, "Adding files...");
	gtk_clist_freeze(GTK_CLIST(list));
	pthread_create(&adder, NULL, (void * (*)(void *))list_add, start_node);
	pthread_detach(adder);
}


int new_list_item(const char *path, gchar **list_item)
{
  struct stat buf;
  
  lstat(path, &buf);
  if (S_ISREG(buf.st_mode)) 
    {
      input_plugin *plugin = NULL;
      
      for (int i = 0; i < CorePlayer::plugin_count; i++) 
	{
	  input_plugin *p = CorePlayer::plugins + i;
	  if (p->can_handle ((char *)path))
	    {
	      plugin = p;
	      break;
	    }
	}
      if (!plugin)
	return 0;

      gchar *filename = strrchr(path, '/');
      filename++; // skip /

      gchar *new_path = (gchar *)g_malloc(256);
      if (new_path)
	strcpy(new_path, path);
      list_item[0] = NULL;
      list_item[1] = filename;
      list_item[2] = new_path;
      return 1;
    }
  return 0;
}

void list_add(gpointer data)
{
	list_node *node = (list_node *)data;
	struct stat buf;
	gchar *list_item[3];
	
	if (node->added - node->last_reported > 3) {
		gchar buf[30];
		sprintf(buf, "%d files added", node->added);
		gdk_threads_enter();
	        gtk_label_set_text(node->status, buf);
		gdk_threads_leave();
		node->last_reported = node->added;
	}	

	lstat(node->start_path, &buf);
	if (S_ISREG(buf.st_mode)) {
		if (new_list_item(node->start_path, list_item)) {
			gdk_threads_enter();
			int index = gtk_clist_append(GTK_CLIST(node->the_list), list_item);
			gtk_clist_set_row_data_full(GTK_CLIST(node->the_list),
				index, list_item[2], destroy_notify);
			gtk_clist_set_shift(GTK_CLIST(node->the_list),
				index, 1, 2, 2);
			gdk_threads_leave();
			node->added++;
		}
	} else if (S_ISDIR(buf.st_mode)) {
		//printf("directory %s...\n", node->start_path);
		dirent *entry;
		DIR *dir = opendir(node->start_path);
		if (dir) {
			while ((entry = readdir(dir)) != NULL) {
				if (strcmp(entry->d_name, ".") == 0 ||
					strcmp(entry->d_name, "..") == 0) 
					continue;
				list_node *new_node = new list_node;
				new_node->the_list = node->the_list;
				new_node->status = node->status;
				new_node->count = node->count+1;
				new_node->added = node->added;
				new_node->last_reported = node->last_reported;
				sprintf(new_node->start_path, "%s/%s", 
					node->start_path, entry->d_name);
				list_add(new_node); 
				node->added = new_node->added;
				node->last_reported = new_node->last_reported;
				delete new_node;
			}
			closedir(dir);	
		}
	}	
	if (!node->count) {
		char buf[30];
		sprintf(buf, "%d file%s added", node->added, 
			node->added == 1 ? " was" : "s were");
		gdk_threads_enter();
		gtk_label_set_text(node->status, buf);	
		gtk_clist_thaw(GTK_CLIST(node->the_list));
		gdk_threads_leave();
		delete node;
	}
}

void dialog_popup(GtkWidget *widget, gpointer data)
{
        GtkWidget *dialog = (GtkWidget *)data;
        gtk_widget_show(dialog);
} 


void destroy_notify(gpointer data)
{
	gchar *path = (gchar *)data;
	g_free(path);
}


void clear_cb(GtkWidget *widget, gpointer data)
{
	GtkCList *list = GTK_CLIST(data);
	GtkLabel *status = (GtkLabel *)gtk_object_get_data(GTK_OBJECT(data), "status");
	gtk_clist_clear(list);
	gtk_label_set_text(status, "List was cleared");
}


void key_press_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	printf("Key down\n");
}


void button_press_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gtk_widget_grab_focus(widget);
}


void playlist_remove(GtkWidget *widget, gpointer data)
{
	GtkWidget *list = (GtkWidget *)data;

	if (list) {
		int selected;
		if (GTK_CLIST(list)->selection) {
			selected = GPOINTER_TO_INT(GTK_CLIST(list)->selection->data);
			gtk_clist_remove(GTK_CLIST(list), selected);
			gtk_clist_select_row(GTK_CLIST(list), selected, 1);
		} else {
			printf("Nothing selected\n");
		}      
	}
}

void playlist_click(GtkWidget *widget, gint row, gint column, 
	GdkEvent *bevent, gpointer data)
{
	GtkWidget *win = (GtkWidget *)gtk_object_get_data(GTK_OBJECT(widget), "window");
	if (win && (bevent && bevent->type == GDK_2BUTTON_PRESS)) {
		gchar *path;	
		int selected;
		CorePlayer *p = (CorePlayer *)gtk_object_get_data(
				GTK_OBJECT(win), "target");
		selected = GPOINTER_TO_INT(GTK_CLIST(widget)->selection->data);
		path = (gchar *)gtk_clist_get_row_data(GTK_CLIST(widget),
			selected);
		if (p) {
			global_playing = 0;
			p->Stop();
			p->SetFile(path);
			p->Start();
			global_playing = 1;
		} else {
			printf("target not found\n");
		}	
	}
}


void playlist_prev(GtkWidget *widget, gpointer data)
{
	GtkWidget *list = NULL;

        if (data) {
                list = (GtkWidget *)gtk_object_get_data(
                        GTK_OBJECT(data), "list");
        }
        if (list) {
                int selected;
                gchar *path;
                GtkWidget *win = (GtkWidget *)gtk_object_get_data(
                        GTK_OBJECT(list), "window");
                if (win) {
                        CorePlayer *p = (CorePlayer *)gtk_object_get_data(
                                GTK_OBJECT(win), "target");
			if (GTK_CLIST(list)->selection) {
                        	selected = GPOINTER_TO_INT(
                               		 GTK_CLIST(list)->selection->data);
			} else {
				selected = 1;
			}
                        path = (gchar *)gtk_clist_get_row_data(
                                GTK_CLIST(list), selected-1);
#if 1 
                        if (path && p) {
				global_playing = 0;
                                p->Stop();
                                p->SetFile(path);
                                p->Start();
				global_playing = 1;
                        }
#endif
                        gtk_clist_select_row(GTK_CLIST(list), selected-1, 1);
                }
        }
}


void playlist_next(GtkWidget *widget, gpointer data)
{
	GtkWidget *list = NULL;

	if (data) {
		list = (GtkWidget *)gtk_object_get_data(
			GTK_OBJECT(data), "list");
	}
	if (list) {
		int selected;
		gchar *path;
		GtkWidget *win = (GtkWidget *)gtk_object_get_data(
			GTK_OBJECT(list), "window");
		if (win) {
			CorePlayer *p = (CorePlayer *)gtk_object_get_data(
				GTK_OBJECT(win), "target");
			//printf("Going to ask for select\n");
			if (GTK_CLIST(list)->selection) {
				selected = GPOINTER_TO_INT(
					GTK_CLIST(list)->selection->data);
			} else {
				//printf("Nothing selected\n");
				selected = -1;
			}
			//printf("selected = %d\n", selected);
			path = (gchar *)gtk_clist_get_row_data(
				GTK_CLIST(list), selected+1);
#if 1		// Hmm, this is weird
			if (path && p) {
				//printf("Found path: %s\n", path);
				global_playing = 0;
				p->Stop();
				p->SetFile(path);
				p->Start();
				global_playing = 1;
			}
#endif
			gtk_clist_select_row(GTK_CLIST(list), selected+1, 1);
		}
	}
}


GtkWidget *init_playlist_window()
{
	GtkWidget *playlist_window;
	GtkWidget *working;
	GtkWidget *add_file;
	GtkWidget *save_list;
	GtkWidget *load_list;
	GtkWidget *list;
	GtkWidget *list_status;
	GtkWidget *status;
	GtkStyle *style;
	GdkFont *bold_font;

	bold_font =
        gdk_font_load("-adobe-helvetica-bold-r-normal--12-*-*-*-*-*-*-*");

        playlist_window = create_playlist_window();

	list = get_widget(playlist_window, "playlist");
	gtk_object_set_data(GTK_OBJECT(list), "window", playlist_window);
	gtk_object_set_data(GTK_OBJECT(playlist_window), "list", list);
	list_status = get_widget(playlist_window, "playlist_status");
	status = gtk_label_new("Nothing here yet");
	style = gtk_style_copy(gtk_widget_get_style(status));
	gdk_font_unref(style->font);
	style->font = bold_font;
        gdk_font_ref(style->font);
        gtk_widget_set_style(GTK_WIDGET(status), style);
	
	gtk_widget_show(status);
	gtk_box_pack_start(GTK_BOX(list_status), status, true, false, 1);
	
	gtk_object_set_data(GTK_OBJECT(list), "status", status);
	
	style = gtk_style_copy(gtk_widget_get_style(list));
	gtk_widget_set_style(GTK_WIDGET(list), style);	
	gtk_clist_set_column_width(GTK_CLIST(list), 0, 16);
	gtk_clist_set_row_height(GTK_CLIST(list), 20);

	
	gtk_signal_connect(GTK_OBJECT(playlist_window), "destroy",
		GTK_SIGNAL_FUNC(playlist_delete_event), NULL);
	gtk_signal_connect(GTK_OBJECT(playlist_window), "delete_event",
		GTK_SIGNAL_FUNC(playlist_delete_event), NULL);

	add_file = gtk_file_selection_new("Add file(s)");
	load_list = gtk_file_selection_new("Load Playlist");
	save_list = gtk_file_selection_new("Save Playlist");

	gtk_object_set_data(GTK_OBJECT(add_file), "list", list);
	gtk_object_set_data(GTK_OBJECT(load_list), "list", list);
	gtk_object_set_data(GTK_OBJECT(save_list), "list", list);
	gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(add_file));
	gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(load_list));
	gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(save_list));

	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(add_file)->cancel_button),
		"clicked", GTK_SIGNAL_FUNC(dialog_cancel), add_file);
 	gtk_signal_connect(GTK_OBJECT(add_file), "delete_event",
                GTK_SIGNAL_FUNC(dialog_delete), NULL);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(add_file)->ok_button),
    	"clicked", GTK_SIGNAL_FUNC(add_file_ok), add_file);

	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(load_list)->cancel_button),
		"clicked", GTK_SIGNAL_FUNC(dialog_cancel), load_list);
	gtk_signal_connect(GTK_OBJECT(load_list), "delete_event",
		GTK_SIGNAL_FUNC(dialog_delete), NULL);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(load_list)->ok_button),
		"clicked", GTK_SIGNAL_FUNC(load_list_ok), load_list);

	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(save_list)->cancel_button),
		"clicked", GTK_SIGNAL_FUNC(dialog_cancel), save_list);
	gtk_signal_connect(GTK_OBJECT(save_list), "delete_event",
		GTK_SIGNAL_FUNC(dialog_delete), NULL);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(save_list)->ok_button),
		"clicked", GTK_SIGNAL_FUNC(save_list_ok), save_list);

	working = get_widget(playlist_window, "add_button");
	gtk_signal_connect(GTK_OBJECT(working), "clicked",
		GTK_SIGNAL_FUNC(dialog_popup), add_file);
	working = get_widget(playlist_window, "clear_button");
	gtk_signal_connect(GTK_OBJECT(working), "clicked",
		GTK_SIGNAL_FUNC(clear_cb), list);
	gtk_signal_connect(GTK_OBJECT(list), "key_press_event",
		GTK_SIGNAL_FUNC(key_press_event), list);
	gtk_signal_connect(GTK_OBJECT(list), "button_press_event",
		GTK_SIGNAL_FUNC(button_press_event), list);
	gtk_signal_connect(GTK_OBJECT(list), "select_row",
		GTK_SIGNAL_FUNC(playlist_click), playlist_window);
	working = get_widget(playlist_window, "del_button");
	gtk_signal_connect(GTK_OBJECT(working), "clicked",
                GTK_SIGNAL_FUNC(playlist_remove), list);

	working = get_widget(playlist_window, "save_button");
	gtk_signal_connect(GTK_OBJECT(working), "clicked",
			GTK_SIGNAL_FUNC(dialog_popup), save_list);
	working = get_widget(playlist_window, "load_button");
	gtk_signal_connect(GTK_OBJECT(working), "clicked",
			GTK_SIGNAL_FUNC(dialog_popup), load_list);

	gtk_widget_grab_focus(GTK_WIDGET(list));

	return playlist_window;
}
