/* Recording controls */

#include <gtk/gtk.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <stdlib.h>

#include "int.h"

#include "preferences.h"
#include "main.h"
#include "colors.h"
#include "record.h"
#include "piping.h"
#include "tracks.h"
#include "tracklist.h"
#include "trackedit.h"
#include "rterm.h"
#include "datacopydlg.h"
#include "varman.h"
#include "varmanwidgets.h"
#include "cddrives.h"
#include "preview.h"
#include "layoutconfig.h"
#include "fileman.h"
#include "updatehandlers.h"
#include "fsedit.h"
#include "dircache.h"

/* uncomment for debugging */
/* #define DEBUG */

int record_active;  /* becomes true whenever access to the cdwriter is being performed*/

/* The following vars are not being exported to other objects,
 * so they do not have the leading "record_" ... */

int inpipe;
int outpipe;
int errpipe;
	
rterm_pipeinfo *outtag;
rterm_pipeinfo *errtag;

int trackpipe;

int cdrecordpid;
int cdrecordstatus;
int cdrecordinpipetag;

char readbuf[RECORD_BUFSIZE];
int  readcount;

int  currenttrack;
datacopydlg_dlginfo *dlg;

int  measure_timeout_handle;
int  measure_size;

/* End of local var list */

void record_unexpected_childexit()
{
	rterm_printcln(Blue,_("\nChild exited unexpectedly."));
}
;

/* eject the cd if the user chose to do so */
void record_eject()
{
	char *call;
	/* if eject is enabled */
	if (!strcmp(varman_getvar(global_defs,"rec_doeject"),"true"))
	     {
		     call=varman_getvar_copy(global_defs,"rec_ejectdisc");
		     varman_replacevars(global_defs,call);
		     inpipe=-1;
		     outpipe=-1;
		     errpipe=-1;
		     cdrecordpid=piping_create(call,&inpipe,&outpipe,&errpipe);
		     free(call);
		     outtag=rterm_connectpipe(&Black,outpipe);
		     errtag=rterm_connectpipe(&Red,errpipe);
		     
		     /* wait for the eject client to finish */
		     waitpid(cdrecordpid,&cdrecordstatus,0);
		     
		     rterm_disconnectpipe(outtag);
		     rterm_disconnectpipe(errtag);
		     close(inpipe);
		     close(outpipe);
		     close(errpipe);
	     }
	    ;
}
;
   
void record_complete()
{   
	record_eject();
	
	rterm_printcln(Blue,_("CD recording process finished."));

        layoutconfig_widget_hide(dlg->messagebox,"recordprogressdialog");
   	datacopydlg_destroy(dlg);
	record_active=0;
	cddrives_updateenable();
}
;

int record_fixating_finished(int status,gpointer data)
{	
	rterm_disconnectpipe(outtag);
	rterm_disconnectpipe(errtag);
	close(inpipe);
	close(outpipe);
	close(errpipe);
        
        gtk_timeout_remove(measure_timeout_handle);
        varman_setvar_value(global_defs,
			    "statistics_leadoutsize",
			    measure_size);
   
   	record_complete();
   
   	return 0;
};

gint record_measure_updateprogress(gpointer data)
{
   int readsize;
   
   readsize=(int)varman_getvar_value(global_defs,"rec_speed")*170*1024;
   datacopydlg_newdatablock(dlg,readsize);   
   measure_size+=readsize;
   
#ifdef DEBUG
   printf ("record_fixating_updateprogress: added %i bytes of \"data\"\n",readsize);
#endif

   return 1;
};

void record_finishrecord()
{
   piping_watch *fixrom;
   char *call;
	
   /* only if fixating is enabled */
   if (!strcmp(varman_getvar(global_defs,"rec_dofixate"),"true"))
     {
	rterm_printcln (Blue,_("\nfixating CDROM."));
	call=varman_getvar_copy(global_defs,"rec_fixate");
	varman_replacevars(global_defs,call);
	inpipe=-1;errpipe=-1;outpipe=-1;
	cdrecordpid=piping_create(call,&inpipe,&outpipe,&errpipe);
	free(call);
	outtag=rterm_connectpipe(&Black,outpipe);
	errtag=rterm_connectpipe(&Red,errpipe);
	fixrom=piping_pidwatch_add(cdrecordpid,
				   record_fixating_finished,
				   NULL);
	measure_size=0;
	measure_timeout_handle=gtk_timeout_add(1000,
					       record_measure_updateprogress,
					       NULL);
	
	/* update progress dialog */
	/* this is all about updating the progress dialog */
	datacopydlg_restartthread(dlg,
				  1,
				  (int)varman_getvar_value(global_defs,
							   "statistics_leadoutsize")
				  );
	datacopydlg_configurethread(dlg,
				    1,				       
				    DATACOPYDLG_SHOWFRAME|
				    DATACOPYDLG_SHOWLABEL|
				    DATACOPYDLG_SHOWTIME_REMAINING|
				    DATACOPYDLG_SHOWTIME_ELAPSED|
				    DATACOPYDLG_SHOWPROGRESS,
				    NULL,
				    _("fixating cdrom"),
				    -1);
     }
   else
     record_complete();
}
;

void record_nexttrack();

void record_writetrack(gpointer data,
		       gint source,
		       GdkInputCondition condition)
{	
	readcount=read(trackpipe,(char*)&readbuf,RECORD_BUFSIZE);
	if (waitpid(cdrecordpid,&cdrecordstatus,WNOHANG)!=0) 
	  {
		  record_unexpected_childexit();
		  readcount=0; // exit writing process
	  }
	;
	if (readcount>0)
	  {
		  write(inpipe,(char*)&readbuf,readcount);
		  datacopydlg_newdatablock(dlg,readcount);
	  }
	else  // current track is done
	  {
		  gdk_input_remove(cdrecordinpipetag);
		  
		  tracks_closepipe(tracklist_gettrack(trackedit,currenttrack));
		  close(inpipe);
		  waitpid(cdrecordpid,&cdrecordstatus,0); // wait for cdrecord to exit.
		  rterm_disconnectpipe(outtag);
		  rterm_disconnectpipe(errtag);
		  close(outpipe);
		  close(errpipe);

		  
		  currenttrack++;
		  if (currenttrack<trackedit->entries)
		    {
   			    record_nexttrack();
		    }
		  else 
		    {
			    record_finishrecord();
		    }
		  ;
	  }
	;
}
;
	
void record_nexttrack()
{     
   char *call;
   
   /* do record only tracks that are available */
   if (tracks_valid(tracklist_gettrack(trackedit,currenttrack)))
     {		  	
	trackpipe=tracks_openpipe(tracklist_gettrack(trackedit,currenttrack));
	
	call=varman_getvar_copy(global_defs,"rec_writetrack");
	varman_replacevars(global_defs,call);
	varman_replacestring(call,"$tracktype",
			     tracklist_gettrack(trackedit,currenttrack)->tracktype);
	inpipe=-1;errpipe=-1;outpipe=-1;
	cdrecordpid=piping_create(call,&inpipe,&outpipe,&errpipe);
	outtag=rterm_connectpipe(&Black,outpipe);
	errtag=rterm_connectpipe(&Red,errpipe);
	
	/* this is all about updating the progress dialog */
	datacopydlg_restartthread(dlg,
				  1,
				  tracks_tracksize(tracklist_gettrack(trackedit,currenttrack)));
	sprintf(call,
		_("Track %i/%i"),
		currenttrack+1,
		trackedit->entries);
	datacopydlg_configurethread(dlg,
				    0,
				    0,
				    NULL,				    
				    call,
				    -1);
	free(call);
			  
	cdrecordinpipetag=gdk_input_add (inpipe,
					 GDK_INPUT_WRITE,
					 record_writetrack,
					 NULL);
     }
   else
     {
	currenttrack++;
	if (currenttrack<trackedit->entries)
	  {
	     record_nexttrack();
	  }
	else 
	  {
	     record_finishrecord();
	  }
	;
     }
   ;
}
;

void record_afterprecache()
{
   int x,discsize;
   char buf[256];
	
   record_active=1;
   cddrives_updatedisable();
   
   discsize=0;
   for (x=0;x<trackedit->entries;x++)
     {
	discsize+=tracks_tracksize(tracklist_gettrack(trackedit,x));
     }
   ;
   
   sprintf((char*)&buf,_("Recording %i bytes to CD"),discsize);
   rterm_printcln(Blue,(char*)&buf);
   
   currenttrack=0;
   
   if (!preview_player_active)
     {
	if (trackedit->entries>0)
	  {		
	     dlg=datacopydlg_create(_("Writing CD"),
				    NULL,NULL,
				    2,
				    
				    DATACOPYDLG_SHOWFRAME|
				    DATACOPYDLG_SHOWLABEL|
				    DATACOPYDLG_SHOWTIME_REMAINING|
				    DATACOPYDLG_SHOWTIME_ELAPSED|
				    DATACOPYDLG_SHOWPROGRESS|
				    DATACOPYDLG_SHOWPROGRESSINTITLE,
				    _("CD status"),
				    _("Please wait"),
				    discsize+
				    (!strcmp(varman_getvar(global_defs,"rec_dofixate"),"true"))*
				    (int)varman_getvar_value(global_defs,
							     "statistics_leadoutsize")
				    ,
				    
				    DATACOPYDLG_SHOWFRAME|
				    DATACOPYDLG_SHOWTHROUGHPUT|
				    DATACOPYDLG_SHOWTIME_REMAINING|
				    DATACOPYDLG_SHOWTIME_ELAPSED|			       
				    DATACOPYDLG_SHOWPROGRESS,
				    _("Track Status"),
				       "",
				    0
				    
				    );
	     layoutconfig_widget_show(dlg->messagebox,
				      "recordprogressdialog");
	     record_nexttrack();
	  }
	else
	  {		  
	     rterm_printcln(Blue,_("No recordable Tracks found."));
	     record_active=0;
	     cddrives_updateenable();
	  }
	;
     }
   else
     {
	rterm_printcln(Blue,_("Preview Player active,\nrecording process cancelled."));
	record_active=0;
	cddrives_updateenable();
     }
        ;
}
;

void record_startrecord(GtkWidget *w)
{
   if (record_active) return;
   
   gtk_notebook_set_page(GTK_NOTEBOOK(destination),2);
   /*So it can be called 
    *from menubar or toolbar*/
   
   trackedit_doprereading(atwrite,
			  record_afterprecache);
};

int record_clear_finished(int status,gpointer data)
{
   rterm_disconnectpipe(outtag);
   rterm_disconnectpipe(errtag);
   
   close(inpipe);
   close(outpipe);
   close(errpipe);
	
   record_eject();
   
   rterm_printcln(Blue,_("CDRW blanking complete."));
   
   record_active=0;
   cddrives_updateenable();
   
   layoutconfig_widget_hide(dlg->messagebox,
			    "blankingprogressdialog");
   datacopydlg_destroy(dlg);
   gtk_timeout_remove(measure_timeout_handle);
   varman_setvar_value(global_defs,
		       (!strcmp(varman_getvar(global_defs,"rec_blankingmode"),
				"fast"))
		       ?"statistics_fastblank"
		       :"statistics_allblank",
		       measure_size);
   
   return 0;
}
;

void record_clear(GtkWidget *w)
{
   piping_watch *cdrecordwatch;
   char *call;
   
   if (record_active) return; // dont run 2 cdwrite processes simultaneously	
   gtk_notebook_set_page(GTK_NOTEBOOK(destination),2);
   /*So it can be called 
    *from menubar or toolbar*/
   
   record_active=1;
   cddrives_updatedisable();
   
   call=varman_getvar_copy(global_defs,"rec_cleardisc");
   varman_replacevars(global_defs,call);	
   rterm_printcln (Blue,_("blanking CDRW."));
   inpipe=-1;errpipe=-1;outpipe=-1;
   cdrecordpid=piping_create(call,&inpipe,&outpipe,&errpipe);
   free(call);
   outtag=rterm_connectpipe(&Black,outpipe);
   errtag=rterm_connectpipe(&Red,errpipe);
   
   cdrecordwatch=piping_pidwatch_add(cdrecordpid,
				     record_clear_finished,
				     NULL);

   dlg=datacopydlg_create(_("Blanking CD/RW"),
			  NULL,NULL,
			  1,
			  
			  DATACOPYDLG_SHOWTIME_REMAINING|
			  DATACOPYDLG_SHOWTIME_ELAPSED|
			  DATACOPYDLG_SHOWPROGRESS|
			  DATACOPYDLG_SHOWPROGRESSINTITLE,
			  "",
			  "",
			  /* different values for fast and normal blank */
			  (!strcmp(varman_getvar(global_defs,"rec_blankingmode"),
				   "fast"))
			  ?
			  (int)varman_getvar_value(global_defs,
						   "statistics_fastblank")
			  :
			  (int)varman_getvar_value(global_defs,
						   "statistics_allblank")
			  );   
   measure_size=0;
   measure_timeout_handle=gtk_timeout_add(1000,
					  record_measure_updateprogress,
					  NULL);
   layoutconfig_widget_show(dlg->messagebox,
			    "blankingprogressdialog");
   
}
;

varmanwidgets_widget *makebootable;

void record_bootimage_changed(VARMAN_VAR_ENTRY *var,gpointer data)
{
   char *filename;
   char *bootfile;
   
   int ok=0;
   
   filename=(char*)malloc(MAXPATHLENGTH);
   strcpy(filename,FSEDIT_DESTPATH);
   if (filename[strlen(filename)-1]=='/')
     filename[strlen(filename)-1]=0;
   bootfile=varman_getvar(global_defs,"isotrack_bootimage");
   strcat (filename,"/");
   strcat(filename,bootfile);
   
   ok=((fileman_file_exist(filename))
       &&(bootfile[0]!='/')
       &&(!dircache_isdir(filename))
       );
   gtk_widget_set_sensitive(makebootable->edit,
			    ok);
   if (!ok)
     varman_setvar(global_defs,"isotrack_makebootable","false");
   
   free(filename);
};

GtkWidget *record_create()
{
	GtkWidget *options;
   	GtkWidget *rec;
 	
	GtkWidget *hbox_record;
	GtkWidget *frame_record;
	GtkWidget *button_record;
	varmanwidgets_widget *button_multisession;
	varmanwidgets_widget *button_fixate;

	GtkWidget *hbox_cdrw;
	GtkWidget *frame_cdrw;
	GtkWidget *button_clear;
	varmanwidgets_widget *button_cleartype;
   
   	GtkWidget *hbox_general;
        GtkWidget *frame_general;
   	varmanwidgets_widget *speed;
	varmanwidgets_widget *button_doeject;
   	varmanwidgets_widget *button_dummywrite;

   	GtkWidget *hbox_iso;
        GtkWidget *frame_iso;
   	varmanwidgets_widget *bootimage;
   
	GtkWidget *terminal;

	GtkWidget *label;
	GtkWidget *record_prefs;
	
	varmanwidgets_widget *edit;

   	rec=gtk_hbox_new(0,0);
        gtk_widget_show(rec);
   

	/* General recording options */

	frame_general=gtk_frame_new(_("General"));
	hbox_general=gtk_vbox_new(0,0);
	
	speed=varmanwidgets_spin_new(_("Recorder Speed"),
				     "rec_speed",
				     global_defs,
				     APPLYMODE_ALWAYS,
				     0,
				     GTK_ADJUSTMENT(gtk_adjustment_new(0,1,24,1,10,0))
				    );
	gtk_box_pack_start(GTK_BOX(hbox_general),
			   speed->visual,0,0,0);
	gtk_widget_show(speed->visual);
	
	button_doeject=varmanwidgets_checkbox_new(_("Eject disc"),
						  "rec_doeject",
						  global_defs,
						  APPLYMODE_ALWAYS,
						  0,
						  "true",
						  "false"
						  );
	gtk_widget_show(button_doeject->visual);
	gtk_box_pack_start(GTK_BOX(hbox_general),button_doeject->visual,0,0,0);

	button_dummywrite=varmanwidgets_checkbox_new(_("Dummy Write"),
						     "rec_dummymode",
						     global_defs,
						     APPLYMODE_ALWAYS,
						     0,
						     " -dummy",
						     ""
						    );
	gtk_widget_show(button_dummywrite->visual);
	gtk_box_pack_start(GTK_BOX(hbox_general),button_dummywrite->visual,0,0,0);
   
   
	gtk_container_add(GTK_CONTAINER(frame_general),hbox_general);
	gtk_widget_show(hbox_general);
	gtk_widget_show(frame_general);
   
        gtk_box_pack_start(GTK_BOX(rec),
		      frame_general,
		      0,0,0);
   
   
	options=gtk_vbox_new(0,0);
   	gtk_widget_show(options);
        gtk_box_pack_start(GTK_BOX(rec),
			   options,
			   0,0,0);
   	
   	/* recording frame */
   	
	frame_record=gtk_frame_new(_("Recording"));
	hbox_record=gtk_hbox_new(0,0);
	
	button_record=gtk_button_new_with_label(_("Record"));
	gtk_signal_connect (GTK_OBJECT(button_record),"clicked",
			    GTK_SIGNAL_FUNC (record_startrecord),NULL);
	gtk_box_pack_start(GTK_BOX(hbox_record),button_record,0,0,0);
	gtk_widget_show(button_record);

	button_multisession=varmanwidgets_checkbox_new(_("Multisession"),
						 "rec_multisession",
						 global_defs,
						 APPLYMODE_ALWAYS,
						 0,
						 " -multi",
						 ""
						 );
	gtk_widget_show(button_multisession->visual);
	gtk_box_pack_start(GTK_BOX(hbox_record),button_multisession->visual,0,0,0);
   
	button_fixate=varmanwidgets_checkbox_new(_("Fixate"),
						 "rec_dofixate",
						 global_defs,
						 APPLYMODE_ALWAYS,
						 0,
						 "true",
						 "false"
						 );
	gtk_widget_show(button_fixate->visual);
	gtk_box_pack_start(GTK_BOX(hbox_record),button_fixate->visual,0,0,0);
   
	gtk_container_add(GTK_CONTAINER(frame_record),hbox_record);
	gtk_widget_show(hbox_record);
	gtk_widget_show(frame_record);
   
        gtk_box_pack_start(GTK_BOX(options),
			   frame_record,
			   0,0,0);
   	
	/* cd rw commands frame */   

	frame_cdrw=gtk_frame_new("CD/RW");
	hbox_cdrw=gtk_hbox_new(0,0);
	
	button_clear=gtk_button_new_with_label(_("Clear Disc"));
	gtk_signal_connect (GTK_OBJECT(button_clear),"clicked",
			    GTK_SIGNAL_FUNC (record_clear),NULL);
	gtk_box_pack_start(GTK_BOX(hbox_cdrw),button_clear,0,0,0);
	gtk_widget_show(button_clear);

	button_cleartype=varmanwidgets_checkbox_new(_("TOC only"),
						    "rec_blankingmode",
						    global_defs,
						    APPLYMODE_ALWAYS,
						    0,
						    "fast",
						    "all"
						    );
	gtk_box_pack_start(GTK_BOX(hbox_cdrw),button_cleartype->visual,0,0,0);
	gtk_widget_show(button_cleartype->visual);
	
	gtk_container_add(GTK_CONTAINER(frame_cdrw),hbox_cdrw);
	gtk_widget_show(hbox_cdrw);
	gtk_widget_show(frame_cdrw);

        gtk_box_pack_start(GTK_BOX(options),
			   frame_cdrw,
			   0,0,0);
   	
   /* common iso options */
   
   frame_iso=gtk_frame_new(_("ISO Options"));
   hbox_iso=gtk_hbox_new(0,0);
   
   bootimage=varmanwidgets_entry_new(_("Boot Image"),
				     "isotrack_bootimage",
				     global_defs,
				     APPLYMODE_ALWAYS,
				     0,64);
   gtk_box_pack_start(GTK_BOX(hbox_iso),bootimage->visual,0,0,0);
   gtk_widget_show(bootimage->visual);
   /* make checkbox below insensitive if path to bootimage is invalid */
   varman_install_handler(global_defs,
			  "isotrack_bootimage",
			  record_bootimage_changed,
			  NULL);
   updatehandlers_register(&fsedit_updatehandlers,
			   (updatehandlers_handler)record_bootimage_changed,
			   NULL);
   
   makebootable=varmanwidgets_checkbox_new(_("Make Bootable"),
					   "isotrack_makebootable",
					   global_defs,
					   APPLYMODE_ALWAYS,
					   0,
					   "true",
					   "false");
   gtk_box_pack_start(GTK_BOX(hbox_iso),makebootable->visual,0,0,0);
   gtk_widget_show(makebootable->visual);
   	
   gtk_container_add(GTK_CONTAINER(frame_iso),hbox_iso);
   gtk_widget_show(hbox_iso);
   gtk_widget_show(frame_iso);

   gtk_box_pack_start(GTK_BOX(options),
		      frame_iso,
		      0,0,0);
   
   /* recording terminal */	
   
   terminal=rterm_create();
   gtk_widget_show(terminal);
   
   gtk_box_pack_end(GTK_BOX(rec),
		    terminal,
		    1,1,0);
   
   /* Create Prefernces page for recording options */
   record_prefs=gtk_vbox_new(0,0);
   gtk_widget_show(record_prefs);
   label=gtk_label_new(_("Recorder"));
   gtk_widget_show(label);
   gtk_notebook_append_page(GTK_NOTEBOOK(preferences_editpad),
				 record_prefs,
			    label);
   
   edit=varmanwidgets_entry_new(_("Recorder device"),
				"recorder",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);
   
   edit=varmanwidgets_entry_new(_("Recorder Mounting point"),
				"rec_mountpoint",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);
   
   edit=varmanwidgets_entry_new(_("SCSI ID"),
				"rec_scsiid",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);
   
   edit=varmanwidgets_entry_new(_("recording client"),
				"rec_client",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);
   
   edit=varmanwidgets_entry_new(_("Fixating command"),
				"rec_fixate",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);
   
   edit=varmanwidgets_entry_new(_("Trackwrite command"),
				"rec_writetrack",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);
   
   edit=varmanwidgets_entry_new(_("Blank disc command"),
				"rec_cleardisc",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);	
   
   edit=varmanwidgets_entry_new(_("Eject disc command"),
				"rec_ejectdisc",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);	
   
   edit=varmanwidgets_entry_new(_("Multisession Info"),
				"rec_getmsinfo",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);	
   
   /* call record_bootimage_changed callback to set the initial state of
    * the "Make bootable" Checkbox */
   record_bootimage_changed(NULL,NULL);

   record_active=0; // cdwriter is initially not active	
   return rec;
}
;
	
	
	
	
	
