#
# Copyright (c) 1997 Loughborough University
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. All advertising materials mentioning features or use of this software
#    must display the following acknowledgement:
#      This product includes software developed by the LUTCHI Research
#      Centre at Loughborough University.
# 4. Neither the name of the University nor of the Centre may be used
#    to endorse or promote products derived from this software without
#    specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
#

set charset iso8859-2

set version "WBD\n1.0ucl4"

################################################################
#
# Whiteboard, (a 'wb' clone) Tcl initialisation file.
#
# Julian Highfield, March 1997.
#
#################################################################

#
# Global interface variables.
#

if {$tcl_platform(platform)=="windows"} {
	set tmpdir $env(TEMP)
	file delete -force $tmpdir/wbdtmp
	file mkdir $tmpdir/wbdtmp
}

if {$tcl_platform(platform)=="unix"} {
	option add *borderWidth 		1 
	option add *font -*-helvetica-medium-r-normal--12-*-p-*-iso8859-1
	. configure -background gray80

	option add *background gray80 startupFile
	option add *foreground black startupFile
	option add *activeBackground gray85 startupFile
	option add *selectBackground gray85 startupFile

	option add *scrollbarBackground gray50 startupFile
	option add *scrollbarForeground gray80 startupFile
	option add *scrollbarActiveForeground gray95 startupFile

	option add *Checkbutton.anchor w startupFile
	option add *Radiobutton.anchor w startupFile
	option add *Radiobutton.relief flat startupFile
	option add *Scale.sliderForeground gray80 startupFile
	option add *Scale.activeForeground gray85 startupFile
	option add *Scale.background gray80 startupFile
	option add *selector forestgreen startupFile
} 

set GWbdFont -*-helvetica-medium-r-normal--12-*-p-*-iso8859-1

#wm title . $W(confname)
tk appname $W(confname)

set PageCurrent 1
set PageTotal   1

set tk_strictMotif 1

proc pserror {file} {
	global erroron

	if {[expr $erroron==0]} {return} else {set erroron 0}
	toplevel .error
	wm transient .error

	set error 2

	label .error.1 -text [concat "Ghostscript failed to convert the Postscript file: " $file]
	label .error.2 -text "There may be several reasons for this:"
	label .error.3 -text "- The file you tried to import was not a valid Postscript file."
	label .error.4 -text "- Ghostscript has not been installed."
	label .error.5 -text "- the PATH and GS_LIB environment variables are not set to the Ghostscript installation path."
	button .error.ok -text OK -command {set error 1}

	pack .error.1 .error.2 .error.3 .error.4 .error.5 -side top -anchor w
	pack .error.ok     -side bottom

	grab set .error
	tkwait variable error
	destroy .error
}

#################################################################

#
# Drawing.
#

proc draw_rect {style colour fill from_x from_y to_x to_y tags} {

	.paper create rect $from_x $from_y $to_x $to_y \
		-outline $colour -width $style -tags $tags
}

set last_x 0
set last_y 0
set line_colour 0
set line_style 0
set first_xy ""

proc draw_line {style colour from_x from_y to_x to_y tags} {
	global first_xy last_x last_y line_colour line_style
	set line_style $style
	set line_colour $colour
	if {[expr $from_x==$last_x && $from_y==$last_y] || ![string compare $first_xy ""]} {
		lappend first_xy $from_x
		lappend first_xy $from_y
		set last_x $to_x
		set last_y $to_y
	} else {
		# maybe draw what we've received????
	}
}


# Relabel the items as group members.
# Only locally drawn items are grouped.
proc draw_group_lines {first last start end tags} {
	global first_xy last_x last_y line_colour line_style

	if {![string compare $first_xy ""]} {
		return
	}
	eval {.paper create line} $first_xy $last_x $last_y \
		{-fill $line_colour -width $line_style -joinstyle round -tags $start$tags$end}
	.paper delete temp
	set first_xy ""
}

proc draw_group_chrs {first last start end tags} {
	set i $first
	while {$i <= $last} {
		.paper addtag $start$tags$end withtag $start$i$end
		.paper dtag $start$i$end
		set i [expr $i + 1]
	}
}

proc draw_oval {style colour fill from_x from_y to_x to_y tags} {

	.paper create oval $from_x $from_y $to_x $to_y \
		-outline $colour -width $style -tags $tags
}

proc draw_oval2 {style colour fill from_x from_y to_x to_y tags} {

	.paper create oval $from_x $from_y $to_x $to_y \
		-outline $colour -width $style -tags $tags
}

## chr orientation colour where_x where_y font size tags 
proc draw_chr {chr orientation colour where_x where_y family weight slant size tags} {
	global charset
	set xfont -*-$family-$weight-$slant-*-*-$size-*-*-*-*-*-$charset
	set r [catch {.paper create text $where_x $where_y -fill $colour \
			-text [subst $chr] -anchor sw -font $xfont \
			-tags $tags}	]

	if {$r == 1} {

		# Failed. Probably a bad font. Try wildcards for most things.
		set xfont -*-$family-*-*-*-*-$size-*-*-*-*-*-*-*
		set r [catch {.paper create text $where_x $where_y -fill $colour \
				-text [subst $chr] -anchor sw -font $xfont \
				-tags $tags}	]

		if {$r == 1} {
			# Failed. Probably a bad font. Use a known good one.
			set xfont -*-helvetica-bold-r-*-*-$size-*-*-*-*-*-iso8859-1
			catch {.paper create text $where_x $where_y -fill $colour \
				-text [subst $chr] -anchor sw -font $xfont \
				-tags $tags}
		}
	}
}

proc draw_chr2 {chr orientation colour where_x where_y xfont tags} {

	.paper create text $where_x $where_y -fill $colour \
		-text $chr -anchor sw -font $xfont -tags $tags
}

proc draw_postscript {file} {
	global GOrientation PageCurrent tcl_platform tmpdir erroron

	#puts "Postscipt is in file $file"

	# What size is needed?
	set w [.paper cget -width]
	set h [.paper cget -height]
	set hw [expr $w / 2]
	set hh [expr $h / 2]
	set dpih [expr $h / 11.7]
	set dpiw [expr $w / 8.2]
	set dpih2 [expr $h / 8.2]
	set dpiw2 [expr $w / 11.7]
	set size [join [list $w x $h] {}]
	set hsize [join [list $hw x $hh] {}]
	set dpisize [join [list $dpiw x $dpih] {}]
	set dpi2size [join [list $dpiw2 x $dpih2] {}]

	# Turn the postscript into ppm.
	# under windows we cache the ppm files in a directory called wbdtmp.....
	if {$tcl_platform(platform)=="windows"} {
		if {[file exists $tmpdir/wbdtmp/$PageCurrent.$GOrientation.ppm]==0} {
			switch $GOrientation {
				0 {exec gswin32c -dNOPAUSE -q -g$size -r$dpisize -sDEVICE=ppmraw \
					-sOutputFile=$tmpdir\\wbdtmp\\$PageCurrent.$GOrientation.ppm \
					$file -c quit}
				1 {exec gswin32c -dNOPAUSE -q -g$size -r$dpi2size -sDEVICE=ppmraw \
					-sOutputFile=$tmpdir\\wbdtmp\\$PageCurrent.$GOrientation.ppm landscap.ps \
					landscap.ps landscap.ps \
					$file -c quit}
				2 {exec gswin32c -dNOPAUSE -q -g$size -r$dpisize -sDEVICE=ppmraw \
					-sOutputFile=$tmpdir\\wbdtmp\\$PageCurrent.$GOrientation.ppm \
					landscap.ps landscap.ps \
					$file -c quit}
				3 {exec gswin32c -dNOPAUSE -q -g$size -r$dpi2size -sDEVICE=ppmraw \
					-sOutputFile=$tmpdir\\wbdtmp\\$PageCurrent.$GOrientation.ppm landscap.ps \
					$file -c quit}
			}
		}
	# Check to see if ghostscript created ppm file.
		if {[file exists $tmpdir/wbdtmp/$PageCurrent.$GOrientation.ppm]==0} {
			pserror $file
		} else {
			catch {image delete bkgrnd}
			image create photo bkgrnd -file $tmpdir/wbdtmp/$PageCurrent.$GOrientation.ppm
			.paper create image $hw $hh -image bkgrnd
		}
		set erroron 0
	} else {
		switch $GOrientation {
			0 {exec gs -dNOPAUSE -q -g$size -r$dpisize -sDEVICE=ppmraw \
				-sOutputFile=$file.%d.ppm \
				$file -c quit}
			1 {exec gs -dNOPAUSE -q -g$size -r$dpi2size -sDEVICE=ppmraw \
				-sOutputFile=$file.%d.ppm landscap.ps \
				landscap.ps landscap.ps \
				$file -c quit}
			2 {exec gs -dNOPAUSE -q -g$size -r$dpisize -sDEVICE=ppmraw \
				-sOutputFile=$file.%d.ppm \
				landscap.ps landscap.ps \
				$file -c quit}
			3 {exec gs -dNOPAUSE -q -g$size -r$dpi2size -sDEVICE=ppmraw \
				-sOutputFile=$file.%d.ppm landscap.ps \
				$file -c quit}
		}
		catch {image delete bkgrnd}
		image create photo bkgrnd -file $file.1.ppm
		.paper create image $hw $hh -image bkgrnd
		file delete $file.1.ppm
	}
}

proc draw_delete {tag} {
	.paper delete $tag
	.paper delete temp
}

proc draw_move {tag dx dy} {

	.paper move $tag $dx $dy
}

proc draw_copy {fromtag totag} {

	set copied [.paper find withtag $fromtag]

	foreach i $copied {

		# Copy the item...
		set type [.paper type $i]
		if {$type == "rectangle"} {
			set style  [.paper itemcget $i -width]
			set colour [.paper itemcget $i -outline]
			set coords [.paper coords $i]
			draw_rect $style $colour fill \
				[lindex $coords 0] \
				[lindex $coords 1] \
				[lindex $coords 2] \
				[lindex $coords 3] \
				$totag
		}
		if {$type == "line"} {
			global first_xy line_colour line_style last_x last_y
			set style  [.paper itemcget $i -width]
			set colour [.paper itemcget $i -fill]
			set coords [.paper coords $i]
			set first_xy [lrange $coords 0 [expr [llength $coords]-3]]
			set last_x [lindex $coords [expr [llength $coords]-2]]
			set last_y [lindex $coords end]
			set line_colour $colour
			set line_style $style
			draw_group_lines "" "" "" "" $totag
		}
		if {$type == "oval"} {
			set style  [.paper itemcget $i -width]
			set colour [.paper itemcget $i -outline]
			set coords [.paper coords $i]
			draw_oval2 $style $colour fill \
				[lindex $coords 0] \
				[lindex $coords 1] \
				[lindex $coords 2] \
				[lindex $coords 3] \
				$totag
		}
		if {$type == "text"} {
			set style  [.paper itemcget $i -width]
			set colour [.paper itemcget $i -fill]
			set font   [.paper itemcget $i -font]
			set text   [.paper itemcget $i -text]
			set coords [.paper coords $i]
			draw_chr2 $text 90 $colour \
				[lindex $coords 0] \
				[lindex $coords 1] \
				$font $totag
		}
	}
}

proc clear_page {} {

	.paper delete all
}

proc set_page {this total} {

	global PageCurrent
	global PageTotal

	set PageCurrent $this
	set PageTotal $total

	set begin [expr ($this.0 - 1.0) / $total.0]
	set end   [expr $this.0 / $total.0]
	.info.pages set $begin $end
}

proc scroll_page {action what args} {

	global PageCurrent
	global PageTotal

	if {$action == "moveto"} {

		# $what is a fraction.
		. config -cursor {watch}
		PageSet [expr $what * $PageTotal]
		. config -cursor {}
	}

	if {$action == "scroll"} {

		# $what is a page or unit count.
		. config -cursor {watch}
		PageSet [expr $PageCurrent + $what]
		. config -cursor {}
	}

}


proc draw_band {type style colour fill from_x from_y to_x to_y} {

	if {$type == "line"} {
		.paper create line $from_x $from_y $to_x $to_y \
			-width $style -tag band
	}

	if {$type == "arrow"} {
		# Arrows are built out of lines.
		# Arrowheads are 4x$LineWidth long and 3x wide.
		set w1 [expr 2 * $style]
		set w2 [expr 4 * $style]
		.paper create line $from_x $from_y $to_x $to_y \
			-arrow last -arrowshape "$w2 $w2 $style" \
			-width $style -tag band
	}

	if {$type == "rect"} {
		.paper create rect $from_x $from_y $to_x $to_y \
			-outline $colour -width $style -tag band
	}

}

proc clear_band {} {

	.paper delete band
}


#################################################################

#
# Interaction.
#


proc ToolSet {w tool} {

	global GTool

#	$w.select    configure -relief raised
	$w.text      configure -relief raised
	$w.pencil    configure -relief raised
	$w.line      configure -relief raised
	$w.arrow     configure -relief raised
	$w.rectangle configure -relief raised
	$w.oval      configure -relief raised
	$w.copy      configure -relief raised
	$w.move      configure -relief raised

	if {$tool != "none"} {
		$w.$tool     configure -relief sunken
	}

	set GTool $tool
}


proc DoDraw {w x y} {

	global GMouseState

	while {$GMouseState == "Down"} {
		set xy [winfo pointerxy $w]
		set x [lindex $xy 0]
		set y [lindex $xy 1]
#		puts "   ...at $x $y"
	}
}

#
# User input handling.
#

# Convert to C format hexadecimal.
proc NumericColour {c} {

	# Returns a list of decimal r,g,b in 0-65535.
	set rgb [winfo rgb . $c]

	set r [expr [lindex $rgb 0] / 257]
	set g [expr [lindex $rgb 1] / 257]
	set b [expr [lindex $rgb 2] / 257]

	set result [format "0x%02X%02X%02X00" $r $g $b]

	return $result
}

# Find unit (linewidth) vector.
proc ArrowVector {w y1 x1 y2 x2} {

	set dx [expr $x2 - $x1]
	set dy [expr $y2 - $y1]
	set l [expr sqrt (($dx * $dx) + ($dy * $dy))]
	if {$l != 0} {
		set l  [expr $l / $w]
		set dx [expr $dx / $l]
		set dy [expr $dy / $l]
	} else {
		set dx 0
		set dy 0
	}

	set p1x [expr $x2 - (3 * $dx) + $dy]
	set p1y [expr $y2 - (3 * $dy) - $dx]
	set p2x [expr $x2 - (3 * $dx) - $dy]
	set p2y [expr $y2 - (3 * $dy) + $dx]

	set r [list $dy $dx $p1y $p1x $p2y $p2x]
	return $r
}

proc PLineWidth {w l} {

	global GTool

	set r $l
#	set p [winfo pointerp $w]
#	if {($p > 0) && ($GTool == "pencil")} {
#		set r [expr 1 + ((($l - 1) * $p) / 255)]
#	}

	return $r
}

proc DoMouseDown {w x y} {

	global GMouseState 
	global GMouseOldPosX GMouseOldPosY
	global GTool GBBox GSelTag

	set GMouseState "Down"

	#puts "MouseDown at $x $y"
	#DoDraw $w $x $y
	set GMouseOldPosX $x
	set GMouseOldPosY $y

	if {($GTool == "copy") || ($GTool == "move")} {
		# Select the object.
		set list1 [.paper find closest $x $y]
		set list2 [.paper gettags $list1]
		set GSelTag "seqno:"
		foreach i $list2 {
			if {$i != "current"} {
				set GBBox [.paper bbox $i]
				set GSelTag $i
			}
		}

		if {[string first "local:" $GSelTag] == 0} {

			# Draw box/band.
			set x1 [lindex $GBBox 0]
			set y1 [lindex $GBBox 1]
			set x2 [lindex $GBBox 2]
			set y2 [lindex $GBBox 3]
			draw_band rect 1 black fill $x1 $y1 $x2 $y2
		}
	}
}

proc DoMouseMotion {w x y} {

	global GMouseState
	global GMouseOldPosX GMouseOldPosY
	global GLineColour GLineWidth
	global GTool GFirstCommand GLastCommand
	global GBBox GSelTag GDeleteList

	set LineWidth [PLineWidth $w $GLineWidth]

	if {$GMouseState == "Down"} {

	    if {$GTool == "pencil"} {
		# puts "Mouse     at $x $y"
		set first [	Draw line $LineWidth \
				[NumericColour $GLineColour] \
				$GMouseOldPosX $GMouseOldPosY $x $y	]
		.paper create line $GMouseOldPosX $GMouseOldPosY $x $y \
			-fill $GLineColour -width $GLineWidth -joinstyle round -tags temp
		set GMouseOldPosX $x
		set GMouseOldPosY $y
		if {$GFirstCommand == 0} {
			set GFirstCommand $first
		}
	    }

	    if {$GTool == "line"} {

		# Clear old rubber band.
		clear_band

		# Draw rubber band.
		draw_band line $LineWidth black fill $GMouseOldPosX \
			$GMouseOldPosY $x $y
	    }

	    if {$GTool == "arrow"} {

		# Clear old rubber band.
		clear_band

		# Draw rubber band.
		draw_band arrow $LineWidth black fill $GMouseOldPosX \
			$GMouseOldPosY $x $y
	    }

	    if {$GTool == "oval"} {

		# Clear old rubber band.
		clear_band

		# Draw rubber band.
		draw_band rect 1 black fill $GMouseOldPosX \
			$GMouseOldPosY $x $y
	    }

	    if {$GTool == "rectangle"} {

		# Clear old rubber band.
		clear_band

		# Draw rubber band.
		draw_band rect 1 black fill $GMouseOldPosX \
			$GMouseOldPosY $x $y
	    }

	    if {$GTool == "select"} {

		# Clear old rubber band.
		clear_band

		# Draw rubber band.
		draw_band rect 1 black fill $GMouseOldPosX \
			$GMouseOldPosY $x $y
	    }

	    if {($GTool == "copy") || ($GTool == "move")} {

		# Clear old box/band.
		clear_band

		if {[string first "local:" $GSelTag] == 0} {
			# Draw box/band.
			set dy [expr $y - $GMouseOldPosY]
			set dx [expr $x - $GMouseOldPosX]
			set x1 [expr $dx + [lindex $GBBox 0]]
			set y1 [expr $dy + [lindex $GBBox 1]]
			set x2 [expr $dx + [lindex $GBBox 2]]
			set y2 [expr $dy + [lindex $GBBox 3]]
			draw_band rect 1 black fill $x1 $y1 $x2 $y2
		}
	    }
	}

	if {$GTool == "text"} {

		set GMouseOldPosX $x
		set GMouseOldPosY $y

		if {$GFirstCommand != 0} {
			set first $GFirstCommand

			# Do text end... group text.
			Draw group_text $first $GLastCommand

			set GFirstCommand 0
			set GDeleteList {}
		}
	}
}

proc DoDeleteMotion {W x y} {

	# Select the object.
	set list1 [.paper find closest $x $y 1]
	set list2 [.paper gettags $list1]
	set GSelTag "seqno:"
	foreach i $list2 {
		if {$i != "current"} {
			set GSelTag $i
		}
	}

	if {[string first "local:" $GSelTag] == 0} {

		# Delete the object.
		Draw delete $GSelTag
	}
}

proc DoMouseUp {w x y} {

	global GMouseState
	global GMouseOldPosX GMouseOldPosY
	global GLineColour GLineWidth
	global GTool GFirstCommand GSelTag

	set LineWidth [PLineWidth $w $GLineWidth]

	# puts "MouseUp   at $x $y"

	if {$GMouseState == "Down"} {
	    if {$GTool == "pencil"} {

		set last [	Draw line $LineWidth \
				[NumericColour $GLineColour] \
				$GMouseOldPosX $GMouseOldPosY $x $y	]
		set GMouseOldPosX $x
		set GMouseOldPosY $y

		set first $last
		if {$GFirstCommand != 0} {
			set first $GFirstCommand
		}

		# Do mouse end... group lines.
		Draw group_lines $first $last

		set GFirstCommand 0
	    }

	    if {$GTool == "line"} {


		# Clear old rubber band.
		clear_band

		set first [	Draw line $LineWidth \
				[NumericColour $GLineColour] \
				$GMouseOldPosX $GMouseOldPosY $x $y	]
		Draw group_lines $first $first
	    }

	    if {$GTool == "arrow"} {


		# Clear old rubber band.
		clear_band

		# Arrows are built out of lines.
		set v [ArrowVector $LineWidth $GMouseOldPosY $GMouseOldPosX $y $x]
		set dy [lindex $v 0]
		set dx [lindex $v 1]

		# Arrowheads are 4x$LineWidth long and 3x wide.
		set first [	Draw line $LineWidth \
				[NumericColour $GLineColour] \
				$GMouseOldPosX $GMouseOldPosY $x $y	]
		Draw line $LineWidth [NumericColour $GLineColour] \
			$x $y [lindex $v 3] [lindex $v 2]
		Draw line $LineWidth [NumericColour $GLineColour] \
			[lindex $v 3] [lindex $v 2] \
			[lindex $v 5] [lindex $v 4]
		set last [	Draw line $LineWidth \
				[NumericColour $GLineColour] \
				[lindex $v 5] [lindex $v 4] $x $y	]
		Draw group_lines $first $last
	    }

	    if {$GTool == "oval"} {

		# Clear old rubber band.
		clear_band

		# Draw the oval.
		Draw oval $LineWidth [NumericColour $GLineColour] \
			$GMouseOldPosX $GMouseOldPosY $x $y
	    }

	    if {$GTool == "rectangle"} {

		# Clear old rubber band.
		clear_band

		# Draw the rect.
		if {$GMouseOldPosY > $y} {
			set from_y $y
			set to_y   $GMouseOldPosY
		} else {
			set from_y $GMouseOldPosY
			set to_y   $y
		}
		if {$GMouseOldPosX > $x} {
			set from_x $x
			set to_x   $GMouseOldPosX
		} else {
			set from_x $GMouseOldPosX
			set to_x   $x
		}

		Draw rect $LineWidth [NumericColour $GLineColour] \
			$from_x $from_y $to_x $to_y
	    }

	    if {$GTool == "select"} {

		# Clear old rubber band.
		clear_band
	    }

	    if {$GTool == "copy"} {

		# Clear old box/band.
		clear_band

		# Do the copy.
		if {[string first "local:" $GSelTag] == 0} {

			Draw copy $GSelTag \
				[expr $x - $GMouseOldPosX] \
				[expr $y - $GMouseOldPosY]
		}
	    }

	    if {$GTool == "move"} {

		# Clear old box/band.
		clear_band

		# Do the copy.
		if {[string first "local:" $GSelTag] == 0} {

			Draw copy $GSelTag \
				[expr $x - $GMouseOldPosX] \
				[expr $y - $GMouseOldPosY]

			# Delete the old item.
			Draw delete $GSelTag
		}
	    }
	}

	set GMouseState "Up"
}


proc DoKeyPress {w x y A} {

	global GTool
	global GTextSize GTextFont
	global GLineColour GLineWidth
	global GMouseOldPosX GMouseOldPosY
	global GTextFirstPosX GTextFirstPosY
	global GFirstCommand GLastCommand GDeleteList
	global GOrientation W


	if {$GTool == "text"} {

		# Calculate the distance between lines.
		if {($GOrientation == 0) || ($GOrientation == 2)} {
			if {$W(relate)} {set yStep [expr $GTextSize * [.paper cget -height] / 560]} else {set yStep [expr $GTextSize * [.paper cget -height] / 792]}
		} else {
			if {$W(relate)} {set yStep [expr $GTextSize * [.paper cget -width] / 609]} else {set yStep [expr $GTextSize * [.paper cget -width] / 612]}

		}

		# New line.
		if {($A == 65293) || ($A == 10)} {

			set xx [list 0 $GMouseOldPosY $GMouseOldPosX]
			lappend GDeleteList $xx

			set GMouseOldPosY \
			[expr $GMouseOldPosY + $yStep]
			set GMouseOldPosX $GTextFirstPosX

			return
		}

		# Delete.
		if {$A == 65288} {

		    if {[llength $GDeleteList] > 0} {

			# Move the insert position.
			set delete [lindex $GDeleteList end]
			set GMouseOldPosY [lindex $delete 1]
			set GMouseOldPosX [lindex $delete 2]

			set what [lindex $delete 0]
			# Send delete message for command.
			if {$what != 0} {

				set item [Draw delete local:$what]
				set GLastCommand $item
			}

			# Shorten the history.
			set GDeleteList [lreplace $GDeleteList end end]

			if {[llength $GDeleteList] == 0} {
				set GFirstCommand 0
			}
		    }

		    return
		}

		# Reject other non-printable characters.
		if {($A > 128) || ($A < 32)} {
			return
		}

		set value [expr $GOrientation * 90]
		set item [	Draw chr $A $value \
				[NumericColour $GLineColour] \
			 	$GMouseOldPosX $GMouseOldPosY \
				$GTextFont $GTextSize	]

		if {$GFirstCommand == 0} {
			set GFirstCommand $item
			set GTextFirstPosX $GMouseOldPosX
			set GTextFirstPosY $GMouseOldPosY
		}

		set xx [list $item $GMouseOldPosY $GMouseOldPosX]
		lappend GDeleteList $xx

		# Offset the position of the next character.
		#set width $GTextSize
		# !Isn't drawn until wbd receives the packet!
		set box [.paper bbox local:$item]
		if {$box != ""} {
			set width [expr [lindex $box 2] - [lindex $box 0]]
			set GMouseOldPosX [expr $GMouseOldPosX + $width]
		}
		set GLastCommand $item
	}
}


#################################################################


#
# Dumping and Loading the whiteboard.
#

proc DumpWB {} {

	global GFileDir

	# Select the file.
	set types {
		{{Whiteboard Dump} {.dmp}}
	}
	set filename [tk_getSaveFile -defaultextension .dmp \
		-filetypes $types \
		-parent . -title {Dump to...}]

#	set filename [FileSave {Dump to...} $GFileDir *.dmp]

	if {$filename == ""} return

	WbDump $filename
}

proc LoadWB {} {

	global PageTotal GFileDir

#	 Select the file.
	set types {
		{{Whiteboard Dump} {.dmp}}
	}
	set filename [tk_getOpenFile -defaultextension .dmp \
		-filetypes $types \
		-parent . -title {Load from...}]

#	set filename [FileOpen {Load from...} $GFileDir *.dmp]

	if {$filename == ""} return

	WbReload $filename
	PageSet $PageTotal
}


#
# Importing and Exporting PostScript.
#

proc PrintFileWB {} {

	global GFileDir

	 #Select the file.
	set types {
		{{Postscript} {.ps}}
	}
	set filename [tk_getSaveFile -defaultextension .ps \
		-filetypes $types \
		-parent . -title {Print to file...}]

#	set filename [FileSave {Print to file...} $GFileDir *.ps]

	if {$filename == ""} return

	.paper postscript -file $filename -colormode color -rotate yes
	catch {exec chmod 755 $filename}
}

proc PrintWB {} {

	global GFileDir

	# Select the file.
	set filename /var/tmp/temp.ps

	# Delete the file.
	file delete $filename

	.paper postscript -file $filename -colormode color -rotate yes
	catch {exec chmod 755 $filename}

	toplevel .print
	wm transient .print
	wm geometry .print [FilePosition 150 100]

	label .print.l -text {Print command:}
	entry .print.e
	.print.e insert end {lp}
	button .print.ok -text OK -command {
		set cmd [.print.e get]
		set cmd [list exec $cmd /var/tmp/temp.ps]
		set cmd [join $cmd { }]
		set cmd [split $cmd { }]
		destroy .print
		eval $cmd
	}
	button .print.cancel -text Cancel -command {destroy .print}

	pack .print.l -side top -fill x -expand 1
	pack .print.e -side top -fill x -expand 1
	pack .print.ok     -side left
	pack .print.cancel -side right

	# Delete the file.
	file delete $filename
}

proc FilePosition {nw nh} {

	set list [split [wm geometry .] x+]

	set scrh [winfo screenheight .]
	set scrw [winfo screenwidth  .]

	set w [lindex $list 0]
	set h [lindex $list 1]
	set x [lindex $list 2]
	set y [lindex $list 3]

	set midx [expr $x + [expr $w / 2]]
	set midy [expr $y + [expr $h / 2]]

	set rx [expr $midx - $nw]
	set ry [expr $midy - $nh]

#	Make sure it fits on the screen!
	if {[expr $rx + $nw] > $scrw} { set rx [expr $scrw - $nw] }
	if {[expr $ry + $nh] > $scrh} { set ry [expr $scrh - $nh] }
	if {$rx < 10} { set rx 10 }
	if {$ry < 10} { set ry 10 }

	return +$rx+$ry
}
proc ImportConfirm {filename} {

	global GConfirm W

	if {[file exists $filename] == 0} {return 1}

	set size [file size $filename]
	if {$size < $W(pssize)} {return 1}

	toplevel .confirm
	wm transient .confirm
	wm geometry .confirm [FilePosition 150 50]

	set GConfirm 2

	label .confirm.l -text "This file is $size bytes\nImport it anyway?"
	button .confirm.ok -text Import -command {set GConfirm 1}
	button .confirm.cancel -text Cancel -command {set GConfirm 0}

	pack .confirm.l      -side top -fill x
	pack .confirm.ok     -side left
	pack .confirm.cancel -side right

	grab set .confirm

	tkwait variable GConfirm

	destroy .confirm

	return $GConfirm
}

proc ImportPS {} {

	global GFileDir erroron

#	 Select the file.
	set types {
		{{Postscript} {.ps}}
	}
		set filename [tk_getOpenFile -defaultextension .ps \
		-filetypes $types \
		-parent . -title {Select a postscript file...}]

#	set filename [FileOpen {Select a postscript file...} $GFileDir *.ps]

	if {$filename == ""} return

	if {[ImportConfirm $filename]} {

		# New page.
		PageNew
	
		# Read it in.
		set erroron 1
		draw_postscript $filename
		SendPS $filename
	}
}

#
# Grabbing part of the screen.
#

proc ImportGrab {} {

	set filename /var/tmp/temp.ps

	# Grab as postscript.
	exec xgrabsc -cps -eps -bin > $filename

	if {[ImportConfirm $filename]} {

		# New page.
		PageNew

		# Read it in.
		SendPS $filename
	}

	# Delete the file.
	file delete $filename
}


#
# Importing text.
#


proc WriteText {x y txt} {

	global GTool

	set lTool $GTool
	ToolSet .edge.tools text

	# Ensure it will be at the current location.
	DoMouseMotion .paper $x $y

	set ll [string length $txt]
	for {set i 0} {$i < $ll} {set i [expr $i + 1]} {
		set c [string index $txt $i]
		scan $c "%c" v
		DoKeyPress .paper $x $y $v
	}

	# always send the group text call - so that other wb's know data is missing.
	SetSend 1
	# Ensure the group_text call gets done.
	DoMouseMotion .paper $x $y

	ToolSet .edge.tools $lTool
}

proc ImportText {} {

	global GFileDir

	# Select the file.
	set filename [tk_getOpenFile -defaultextension .txt \
		-parent . -title {Select a text file...}]

	#set filename [FileOpen {Select a text file...} $GFileDir *.txt]

	# Read it in.
	if {$filename == ""} {return TCL_OK}
	set chan [open $filename]
	if {$chan == 0} {return TCL_OK}

	set this_line {}
	set going 1

	while {$going} {
		# New page.
		PageNew

		set this_page 0
		set page ""
		while {$this_page < 26} {
			set count [gets $chan this_line]
			if {$count < 0}  {set going 0;break}
			if {$count >= 1} {
				#regsub -all {\\} $this_line {\\\\} this_line
				append page $this_line
			}
			append page "\n"

			incr this_page
		}
		# if this is the last page then send it, otherwise don't bother unless
		# we get a request for the page (same as wb...)
		if {$going!=0} {SetSend 0}
		WriteText 30 30 $page
	}

	close $chan
}

proc clean_up {} {
	global tmpdir
	catch {file delete -force $tmpdir/wbdtmp}
	exit
}

#################################################################

#
# Choose an orientation for displaying the page.
#

proc Orient {which} {

	global PageCurrent
	global GOrientation W

	#set w [.paper cget -width]
	#set h [.paper cget -height]
	#puts "Orient $which (from $GOrientation) ($w by $h)"

	# Choose the canvas size.
	if {($GOrientation == 0) || ($GOrientation == 2)} {
		if {$W(relate)} {set h [.paper cget -width]} else {set w [.paper cget -width]}
		if {$W(relate)} {set w [.paper cget -height]} else {set h [.paper cget -height]}
	} else {
		set h [.paper cget -width]
		set w [.paper cget -height]
	}

	set GOrientation $which

	# Tell the rest of the system.
	if {($which == 0) || ($which == 2)} {
		if {$W(relate)} {CanvasResize $h $w} else {CanvasResize $w $h}
	} else {
		CanvasResize $h $w
	}

	wm geometry . {}

	update
}

proc CanvasResize {w h} {

	global PageCurrent
	global GOrientation W

	.paper config -width $w -height $h
	update

	if {($GOrientation == 0) || ($GOrientation == 2)} {
		if {$W(relate)} {PageCanvasSize $h $w $GOrientation} else {PageCanvasSize $w $h $GOrientation}
	} else {
		PageCanvasSize $h $w $GOrientation
	}

	# Clear and redraw...
	clear_page
	PageSet $PageCurrent

	update
}

proc WbSizeForce {w h} {

	wm geometry . [join [list $w x $h] {}]
}

proc StateWB {state} {

	if {$state == "0"} {set state normal}
	if {$state == "1"} {set state disabled}

	set w1 .edge
	set w2 $w1.tools
	set w3 .info

        $w2.move configure -state $state
        $w2.copy configure -state $state
        $w2.text configure -state $state
        $w2.pencil configure -state $state
        $w2.line configure -state $state
        $w2.arrow configure -state $state
        $w2.rectangle configure -state $state
        $w2.oval configure -state $state
        $w2.colour configure -state $state
        $w2.width configure -state $state
        $w2.font configure -state $state
#       $w1.quit configure -state $state
        $w3.newpage configure -state $state
        $w3.impps configure -state $state
        $w3.imptext configure -state $state
#	$w3.dump configure -state $state
#	$w3.load configure -state $state
	$w3.grab configure -state $state
#        $w3.ps configure -state $state

	# Turn the drawing tools on/off.
	if {$state == "disabled"} {ToolSet $w2 none}
	if {$state == "normal"}  {ToolSet $w2 pencil}
}

#
# Building the control panel.
#

proc BuildPanel {w} {

	global W

	set PanelFont -Adobe-Helvetica-*-R-Normal-*-*-120-*-*-*-*-*-*

	listbox $w.members -width 25 -height 6 -borderwidth 2 \
		 -font $PanelFont
	frame $w.info -relief groove -borderwidth 2
	frame $w.controls -relief groove -borderwidth 2
	button $w.dismiss -text Dismiss \
		-command "destroy $w; set GPanel {}"

	label $w.info.left  -width 8  -justify left  -font $PanelFont
	label $w.info.right -width 16 -justify right -font $PanelFont

	label $w.controls.conference  -font $PanelFont \
		-text "$W(confGroup)/$W(confPort)/$W(confTTL)"
	entry $w.controls.username
	$w.controls.username insert end $W(username)
	entry $w.controls.confname
	$w.controls.confname insert end $W(confname)
	frame $w.controls.left
	frame $w.controls.right
	checkbutton $w.controls.readonly -indicatoron 1 \
		-command {StateWB $W(readonly)} \
		-variable W(readonly) \
		-text {Read only}

	button $w.controls.left.dump -width 8 -text {Dump} \
		-command DumpWB -font $PanelFont
	button $w.controls.right.load -width 8 -text {Reload} \
		-command LoadWB -font $PanelFont
#	button $w.controls.left.ps -width 8 -text {Print Page} \
#		-command PrintWB -font $PanelFont
#	button $w.controls.right.ps -width 8 -text {Print to file} \
#		-command PrintFileWB -font $PanelFont

	pack $w.members -side top -fill x
	pack $w.info -side top -fill x -expand 1
	pack $w.controls -side top -fill x
	pack $w.dismiss -side bottom -fill x

	pack $w.info.left  -side left
	pack $w.info.right -side right

	pack $w.controls.conference -fill x
	pack $w.controls.username -fill x
	pack $w.controls.confname -fill x
	pack $w.controls.readonly -expand 1 -fill x

	pack $w.controls.left  -side left
	pack $w.controls.right -side right

	pack $w.controls.left.dump  -side top -fill x
	pack $w.controls.right.load -side top -fill x
#	pack $w.controls.left.ps    -side top -fill x
#	pack $w.controls.right.ps   -side top -fill x

	bind $w.controls.username <KeyPress-Return> {
		set W(username) [.panel.controls.username get]
		set r [.panel.controls.username get]
		Sender set username $W(username)
	}

	bind $w.controls.confname <KeyPress-Return> {
		set W(confname) [.panel.controls.confname get]
		tk appname $W(confname)
		wm title . $W(confname)
		Sender set confname $W(confname)
	}
}

proc UpdatePanelInfo {idx} {

	set desc [Sender info $idx]

	set addr  [lindex $desc 0]
	set name  [lindex $desc 1]
	set quiet [lindex $desc 2]
	set delay [lindex $desc 3]
	set idle  [lindex $desc 4]
	set drawn [lindex $desc 5]

	set txtl "Address:\nIdle:\nQuiet\nDistance:\nDrawOps:"
	set txtr "$addr\n$idle ms\n$quiet ms\n$delay ms\n$drawn"
	.panel.info.left  config -text $txtl
	.panel.info.right config -text $txtr
}

proc UpdatePanel {count} {

	global GPanel

	if {$GPanel == ""} {return}

	.panel.members delete 0 end

	for {set i 0} {$i < $count} {incr i} {

		set desc [Sender info $i]
		set name [lindex $desc 1]
		.panel.members insert end $name

		UpdatePanelInfo $i
	}

	bind .panel.members <Button-1> {

		set idx [.panel.members nearest %y]
		set listsel $idx
		UpdatePanelInfo $listsel
	}
}

proc ShowPanel {} {

	global GPanel

	if {$GPanel != ""} {return}

	set GPanel .panel

	toplevel .panel
	wm transient .panel
	wm title .panel "Controls"

	BuildPanel .panel

	UpdatePanel [Sender count]
}


#
# Building the interface.
#

proc BuildTools {w} {

	global GWbdFont charset

	button $w.move -text "Move" -font $GWbdFont \
		-command "ToolSet $w move" -borderwidth 3
	pack $w.move -side top -fill x

	button $w.copy -text "Copy" -font $GWbdFont \
		-command "ToolSet $w copy" -borderwidth 3
	pack $w.copy -side top -fill x

	button $w.text -text "T" \
		-command "ToolSet $w text" -borderwidth 3 \
		-font -Adobe-Times-*-R-Normal-*-*-180-*-*-*-*-*-*
	pack $w.text -side top -fill x

	button $w.line -image bitmap_line -text "Line" \
		-command "ToolSet $w line" -borderwidth 3
	pack $w.line -side top -fill x

	button $w.arrow -image bitmap_arrow -text "Arrow" \
		-command "ToolSet $w arrow" -borderwidth 3
	pack $w.arrow -side top -fill x

	button $w.rectangle -image bitmap_rectangle -text "Rectangle" \
		-command "ToolSet $w rectangle" -borderwidth 3
	pack $w.rectangle -side top -fill x

	button $w.oval -image bitmap_oval -text "Oval" \
		-command "ToolSet $w oval" -borderwidth 3
	pack $w.oval -side top -fill x

	button $w.pencil -image bitmap_pencil -text "Pencil" \
		-command "ToolSet $w pencil" -borderwidth 3
	pack $w.pencil -side top -fill x

	ToolSet $w line

	##### Colour menu. #####
	menubutton $w.colour -text "Colour" -menu $w.colour.menu \
		-relief raised -font $GWbdFont
	menu $w.colour.menu -tearoff 0
	bitmap_colour1 configure -foreground black
	$w.colour.menu add command -label "black" \
		-command {set GLineColour black} \
		-image bitmap_colour1 -foreground black \
		-activebackground black -activeforeground lightgrey
	bitmap_colour2 configure -foreground grey
	$w.colour.menu add command -label "grey" \
		-command {set GLineColour grey} \
		-image bitmap_colour2 -foreground grey \
		-activebackground grey -activeforeground lightgrey
	bitmap_colour3 configure -foreground white
	$w.colour.menu add command -label "white" \
		-command {set GLineColour white} \
		-image bitmap_colour3 -foreground white \
		-activebackground white -activeforeground lightgrey
	bitmap_colour4 configure -foreground red
	$w.colour.menu add command -label "red" \
		-command {set GLineColour red} \
		-image bitmap_colour4 -foreground red \
		-activebackground red -activeforeground lightgrey
	bitmap_colour5 configure -foreground orange
	$w.colour.menu add command -label "orange" \
		-command {set GLineColour orange} \
		-image bitmap_colour5 -foreground orange \
		-activebackground orange -activeforeground lightgrey
	bitmap_colour6 configure -foreground yellow
	$w.colour.menu add command -label "yellow" \
		-command {set GLineColour yellow} \
		-image bitmap_colour6 -foreground yellow \
		-activebackground yellow -activeforeground lightgrey
	bitmap_colour7 configure -foreground green
	$w.colour.menu add command -label "green" \
		-command {set GLineColour green} \
		-image bitmap_colour7 -foreground green \
		-activebackground green -activeforeground lightgrey
	bitmap_colour8 configure -foreground blue
	$w.colour.menu add command -label "blue" \
		-command {set GLineColour blue} \
		-image bitmap_colour8 -foreground blue \
		-activebackground blue -activeforeground lightgrey
	bitmap_colour9 configure -foreground purple
	$w.colour.menu add command -label "purple" \
		-command {set GLineColour purple} \
		-image bitmap_colour9 -foreground purple \
		-activebackground purple -activeforeground lightgrey
	$w.colour.menu add command -label "Other" \
		-command {set GLineColour [tk_chooseColor]}
	pack $w.colour -side top -fill x

	##### Line width menu. #####
	menubutton $w.width -text "Width" -menu $w.width.menu \
		-relief raised -font $GWbdFont
	menu $w.width.menu -tearoff 0
	$w.width.menu add command -label "1" \
		-command {set GLineWidth 1} -image bitmap_line01
	$w.width.menu add command -label "2" \
		-command {set GLineWidth 2} -image bitmap_line02
	$w.width.menu add command -label "4" \
		-command {set GLineWidth 4} -image bitmap_line04
	$w.width.menu add command -label "8" \
		-command {set GLineWidth 8} -image bitmap_line08
	$w.width.menu add command -label "12" \
		-command {set GLineWidth 12} -image bitmap_line12
	$w.width.menu add command -label "16" \
		-command {set GLineWidth 16} -image bitmap_line16
	pack $w.width -side top -fill x

	##### Font menu. #####
	menubutton $w.font -text "Font" -menu $w.font.menu \
		-relief raised -font $GWbdFont
	menu $w.font.menu -tearoff 0
	$w.font.menu add command -label "Abc" \
		-command {set GTextFont 0x000600; set GTextSize 14} \
		-font -*-helvetica-bold-r-*-*-12-*-*-*-*-*-$charset
	$w.font.menu add command -label "Abc" \
		-command {set GTextFont 0x000600; set GTextSize 18} \
		-font -*-helvetica-bold-r-*-*-18-*-*-*-*-*-$charset
	$w.font.menu add command -label "Abc" \
		-command {set GTextFont 0x000600; set GTextSize 24} \
		-font -*-helvetica-bold-r-*-*-24-*-*-*-*-*-$charset
	$w.font.menu add command -label "Abc" \
		-command {set GTextFont 0x001600; set GTextSize 14} \
		-font -*-helvetica-bold-o-*-*-12-*-*-*-*-*-$charset
	$w.font.menu add command -label "Abc" \
		-command {set GTextFont 0x001600; set GTextSize 18} \
		-font -*-helvetica-bold-o-*-*-18-*-*-*-*-*-$charset
	$w.font.menu add command -label "Abc" \
		-command {set GTextFont 0x001600; set GTextSize 24} \
		-font -*-helvetica-bold-o-*-*-24-*-*-*-*-*-$charset
	pack $w.font -side top -fill x

	button $w.o0 -text "Port." -image bitmap_port \
		-command "Orient 0" -borderwidth 3 
	button $w.o1 -text "Land." -image bitmap_land \
		-command "Orient 1" -borderwidth 3 
	button $w.o2 -text "ALand" -image bitmap_aland \
		-command "Orient 3" -borderwidth 3 
	button $w.o3 -text "APort" -image bitmap_aport \
		-command "Orient 2" -borderwidth 3 
	pack $w.o1 -side top -fill x
	pack $w.o0 -side top -fill x
	pack $w.o2 -side top -fill x
	pack $w.o3 -side top -fill x
}

proc BuildEdge {w} {

	global GWbdFont W version

	label $w.version -text $version -font $GWbdFont
	if {$W(relate)==0} {
		button $w.quit -text "Quit" -command clean_up \
			 -font $GWbdFont
		pack $w.quit -side bottom -fill x
	}
    	pack $w.version -side bottom -fill x

	frame $w.tools
	pack $w.tools -side top

	BuildTools $w.tools
}

proc BuildInfo {w} {

	global GWbdFont

	button $w.newpage -width 8 -text {New Page} \
		 -font $GWbdFont -command PageNew
	pack $w.newpage -side left

	button $w.impps -width 8 -text {Import PS} \
		 -font $GWbdFont -command ImportPS
	pack $w.impps -side left

	button $w.imptext -width 8 -text {Import Text} \
		 -font $GWbdFont -command ImportText
	pack $w.imptext -side left

#	button $w.ps -width 8 -text {Print} -command PrintWB \
#		 -font $GWbdFont
#	pack $w.ps -side left
#	button $w.dump -width 8 -text {Dump} -command DumpWB \
#		 -font $GWbdFont
#	pack $w.dump -side left
#	button $w.load -width 8 -text {Reload} -command LoadWB \
#		 -font $GWbdFont
#	pack $w.load -side left
	button $w.grab -width 8 -text {Grab} -command ImportGrab \
		 -font $GWbdFont -state disabled
	pack $w.grab -side left

	scrollbar $w.pages -orient horizontal -command scroll_page
	pack $w.pages -side bottom -fill x

	$w.pages set 0 1
}

proc BuildWB {} {

	global GWbdFont W

	frame .edge
	pack .edge -side right -fill y

	BuildEdge .edge

	frame .info
	pack .info -side bottom -fill x

	BuildInfo .info

	if {$W(relate)} {canvas .paper -background white -width 609 -height 560} else {canvas .paper -background white -width 792 -height 612}

	pack .paper -side left

	# MouseDown in the canvas requires an action.
	bind .paper <ButtonPress-1>   {focus -force .paper; DoMouseDown .paper %x %y}
	bind .paper <ButtonRelease-1> {DoMouseUp .paper %x %y}
	bind .paper <Motion>          {DoMouseMotion .paper %x %y}

	# Panel
	bind .paper <Shift-ButtonPress-1>   {ShowPanel}

	# Delete
	bind .paper <Control-ButtonPress-1>   {

		if {$GMouseState == "Down"} {return}

		DoDeleteMotion .paper %x %y
	}
#	bind .paper <Button3-Motion>   {
#
#		if {$GMouseState == "Down"} {return}
#
#		DoDeleteMotion .paper %x %y
#	}

	# Undelete
	bind .paper <Control-ButtonPress-2>   {

		if {$GMouseState == "Down"} {return}

		set u [UnDelete]
		if {$u > 0} {Draw delete local:$u}
	}

	# KeyPress in the canvas requires an action.
	bind . <KeyPress>             {DoKeyPress .paper %x %y %N}
	bind .paper <ButtonPress-2>   {

		if {$GMouseState == "Down"} {return}

		# Get selected text.
		set v [catch {
			set txt [selection get -displayof %W]
		}]

		if {$v == 0} {
			# Paste text.
			if {$txt != ""} {
				WriteText %x %y $txt
			}
		}
	}
}

bitmaps_init
BuildWB

set GMouseState "Up"
set GMouseOldPosX {}
set GMouseOldPosY {}
set GLineColour black
set GLineWidth 4
set GTool line
set GTextSize 18
set GTextFont 0x000602
set GTextFirstPosX {}
set GTextFirstPosY {}
set GFirstCommand 0
set GDeleteList {}
set GLastCommand 0
set GFileDir ""

set GFileName ""
set GFilePattern {~}

set GOrientation $W(orientation)

set GPanel ""

if {($GOrientation == 0) || ($GOrientation == 2)} {
	if {$W(relate)} {PageCanvasSize 609 560 $GOrientation} else {PageCanvasSize 612 792  $GOrientation}
} else {
	if {$W(relate)} {PageCanvasSize 609 560 $GOrientation} else {PageCanvasSize 792 612 $GOrientation}
}

if {$W(readonly) == 1} {StateWB disabled}

wm minsize . 100 100
wm protocol . WM_DELETE_WINDOW clean_up

if {$W(geometry) != ""} {
	wm geometry . $W(geometry)
}
