/*****************************************************************************
* wanconfig.c	WAN Multiprotocol Router Configuration Utility.
*
* Author:	Nenad Corbic	<ncorbic@sangoma.com>
*               Gideon Hack
*
* Copyright:	(c) 1995-1999 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ----------------------------------------------------------------------------
* Sep 23, 1999  Nenad Corbic    Added support for HDLC STREAMING, Primary
*                               and secondary ports. 
* Jun 02, 1999 	Gideon Hack	Added support for the S514 PCI adapter.
* Jan 07, 1998	Jaspreet Singh	Made changes for 2.1.X.
*				Added support for WANPIPE and API integration.
* Jul 20, 1998	David Fong	Added Inverse ARP option to channel config.
* Jun 26, 1998	David Fong	Added IP_MODE to PPP Configuration.
*				Used for Dynamic IP Assignment.
* Jun 18, 1998	David Fong	Added Cisco HDLC definitions and structures
* Dec 16, 1997	Jaspreet Singh 	Moved IPX and NETWORK to 'chan_conftab'
* Dec 08, 1997	Jaspreet Singh	Added USERID, PASSWD and SYSNAME in 
*				'chan_conftab' 
* Dec 05, 1997	Jaspreet Singh  Added PAP and CHAP in 'chan_conftab'
*				Added AUTHENTICATOR in 'ppp_conftab' 
* Oct 12, 1997	Jaspreet Singh	Added IPX and NETWORK to 'common_conftab'
*				Added MULTICAST to 'chan_conftab'
* Oct 02, 1997  Jaspreet Singh	Took out DLCI from 'fr_conftab'. Made changes 
*				so that a list of DLCI is prepared to 
*				configuring them when emulating a NODE 
* Jul 07, 1997	Jaspreet Singh	Added 'ttl' to 'common_conftab'
* Apr 25, 1997  Farhan Thawar   Added 'udp_port' to 'common_conftab'
* Jan 06, 1997	Gene Kozin	Initial version based on WANPIPE configurator.
*****************************************************************************/

/*****************************************************************************
* Usage:
*   wanconfig [-f {conf_file}]	Configure WAN links and interfaces
*   wanconfig -d {device}	Shut down WAN link
*   wanconfig -h|?		Display help screen
*
* Where:
*   {conf_file}		configuration file name. Default is /etc/wanpipe1.conf
*   {device}		name of the WAN device in /proc/net/wanrouter directory
*
* Optional switches:
*   -v			verbose mode
*   -h or -?		display help screen
*****************************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <linux/wanrouter.h>	/* WAN router API definitions */

/****** Defines *************************************************************/

#ifndef	min
#define	min(a,b)	(((a)<(b))?(a):(b))
#endif

#define	is_digit(ch) (((ch)>=(unsigned)'0'&&(ch)<=(unsigned)'9')?1:0)

enum ErrCodes			/* Error codes */
{
	ERR_SYSTEM = 1,		/* system error */
	ERR_SYNTAX,		/* command line syntax error */
	ERR_CONFIG,		/* configuration file syntax error */
	ERR_LIMIT
};

/* Command line parsing stuff */
#define ARG_SWITCH	'-'	/* switch follows */
#define	SWITCH_CONFFILE	'f'	/* use alternative configuration file */
#define	SWITCH_SHUTDOWN	'd'	/* shut down link */
#define	SWITCH_VERBOSE	'v'	/* enable verbose output */
#define	SWITCH_GETHELP	'h'	/* show help screen */
#define	SWITCH_ALTHELP	'?'	/* same */

/* Configuration stuff */
#define	MAX_CFGLINE	256
#define	MAX_CMDLINE	256
#define	MAX_TOKENS	32

/****** Data Types **********************************************************/

typedef struct look_up		/* Look-up table entry */
{
	uint	val;		/* look-up value */
	void*	ptr;		/* pointer */
} look_up_t;

typedef struct key_word		/* Keyword table entry */
{
	char*	keyword;	/* -> keyword */
	uint	offset;		/* offset of the related parameter */
	int	dtype;		/* data type */
} key_word_t;

typedef struct data_buf		/* General data buffer */
{
	unsigned size;
	void* data;
} data_buf_t;

/*
 * Data types for configuration structure description tables.
 */
#define	DTYPE_INT	1
#define	DTYPE_UINT	2
#define	DTYPE_LONG	3
#define	DTYPE_ULONG	4
#define	DTYPE_SHORT	5
#define	DTYPE_USHORT	6
#define	DTYPE_CHAR	7
#define	DTYPE_UCHAR	8
#define	DTYPE_PTR	9
#define	DTYPE_STR	10
#define	DTYPE_FILENAME	11

typedef struct chan_def			/* Logical Channel definition */
{
	char name[WAN_IFNAME_SZ+1];	/* interface name for this channel */
	char* addr;			/* -> media address */
	char* conf;			/* -> configuration data */
	char* usedby;			/* used by WANPIPE or API */
	char* descr;			/* -> description */
	struct chan_def* next;		/* -> next channel definition */
} chan_def_t;

typedef struct link_def			/* WAN Link definition */
{
	char name[WAN_DRVNAME_SZ+1];	/* link device name */
	int config_id;			/* configuration ID */
	char* conf;			/* -> configuration data */
	char* descr;			/* -> description */
	chan_def_t* chan;		/* list of channel definitions */
	struct link_def* next;		/* -> next link definition */
} link_def_t;

/****** Function Prototypes *************************************************/

int arg_proc (int argc, char* argv[]);
void show_error	(int err);
int shutdown (void);
int configure (void);
int parse_conf_file (char* fname);
int build_linkdef_list (FILE* file);
int build_chandef_list (FILE* file);
char* read_conf_section (FILE* file, char* section);
int read_conf_record (FILE* file, char* key);
int configure_link (link_def_t* def);
int configure_chan (int dev, chan_def_t* def);
int set_conf_param (char* key, char* val, key_word_t* dtab, void* conf);
int tokenize (char* str, char **tokens);
char* strstrip (char* str, char* s);
char* strupcase	(char* str);
void* lookup (int val, look_up_t* table);
int name2val (char* name, look_up_t* table);
int read_data_file (char* name, data_buf_t* databuf);
unsigned long filesize (FILE* file);
unsigned int dec_to_uint (unsigned char* str, int len);

extern	int close (int);
/****** Global Data *********************************************************/

char progname[] =	"wanconfig";
char def_conf_file[] =	"/etc/wanpipe1.conf";	/* default name */
char router_dir[] =	"/proc/net/wanrouter";	/* location of WAN devices */
char banner[] =		"WAN Router Configurator. v2.1.0 "
			"(c) 1995-1999 Sangoma Technologies Inc."
;
char helptext[] =
	"Usage:"
	"\twanconfig [-f {conf_file}]   Configure WAN links and interfaces\n"
	"\twanconfig -d {device}        Shut down WAN link\n"
	"\nWhere:"
	"\t{conf_file}\tconfiguration file (default is /etc/wanpipe#.conf)\n"
	"\t{device}\tname of WAN device in /proc/net/wanrouter directory\n"
	"\nOptional switches:\n"
	"\t-v\tverbose mode\n"
	"\t-h|?\tshow this help\n"
;
char* err_messages[] =				/* Error messages */
{
	"Invalid command line syntax",		/* ERR_SYNTAX */
	"Invalid configuration file syntax",	/* ERR_CONFIG */
	"Unknown error code",			/* ERR_LIMIT */
};
enum	/* modes */
{
	DO_CONFIGURE,
	DO_SHUTDOWN,
	DO_HELP
} action;				/* what to do */
int verbose;				/* verbosity level */
char* conf_file = def_conf_file;	/* configuration file */
link_def_t* link_defs;			/* list of WAN link definitions */
union
{
	wandev_conf_t linkconf;		/* link configuration structure */
	wanif_conf_t chanconf;		/* channel configuration structure */
} u;

/*
 * Configuration structure description tables.
 * WARNING:	These tables MUST be kept in sync with corresponding data
 *		structures defined in linux/wanrouter.h
 */
key_word_t common_conftab[] =	/* Common configuration parameters */
{
  { "IOPORT",     offsetof(wandev_conf_t, ioport),     DTYPE_UINT },
  { "MEMADDR",    offsetof(wandev_conf_t, maddr),       DTYPE_UINT },
  { "MEMSIZE",    offsetof(wandev_conf_t, msize),       DTYPE_UINT },
  { "IRQ",        offsetof(wandev_conf_t, irq),         DTYPE_UINT },
  { "DMA",        offsetof(wandev_conf_t, dma),         DTYPE_UINT },
  { "S514CPU",    offsetof(wandev_conf_t, S514_CPU_no), DTYPE_STR },
  { "PCISLOT",    offsetof(wandev_conf_t, PCI_slot_no), DTYPE_UINT },
  { "COMMPORT",   offsetof(wandev_conf_t, comm_port),   DTYPE_UCHAR },
  { "BAUDRATE",   offsetof(wandev_conf_t, bps),         DTYPE_UINT },
  { "MTU",        offsetof(wandev_conf_t, mtu),         DTYPE_UINT },
  { "UDPPORT",    offsetof(wandev_conf_t, udp_port),    DTYPE_UINT },
  { "TTL",	  offsetof(wandev_conf_t, ttl),		DTYPE_UCHAR },
  { "INTERFACE",  offsetof(wandev_conf_t, interface),   DTYPE_UCHAR },
  { "CLOCKING",   offsetof(wandev_conf_t, clocking),    DTYPE_UCHAR },
  { "LINECODING", offsetof(wandev_conf_t, line_coding), DTYPE_UCHAR },
  { "STATION",    offsetof(wandev_conf_t, station),     DTYPE_UCHAR },
  { "CONNECTION", offsetof(wandev_conf_t, connection),  DTYPE_UCHAR },
  { "OPTION1",    offsetof(wandev_conf_t, hw_opt[0]),   DTYPE_UINT },
  { "OPTION2",    offsetof(wandev_conf_t, hw_opt[1]),   DTYPE_UINT },
  { "OPTION3",    offsetof(wandev_conf_t, hw_opt[2]),   DTYPE_UINT },
  { "OPTION4",    offsetof(wandev_conf_t, hw_opt[3]),   DTYPE_UINT },
  { "FIRMWARE",   offsetof(wandev_conf_t, data_size),   DTYPE_FILENAME },
  { "RXMODE",     offsetof(wandev_conf_t, read_mode),   DTYPE_CHAR },
  { NULL, 0, 0 }
 };
 
key_word_t ppp_conftab[] =	/* PPP-specific configuration */
 {
  { "RESTARTTIMER",   offsetof(wan_ppp_conf_t, restart_tmr),   DTYPE_UINT },
  { "AUTHRESTART",    offsetof(wan_ppp_conf_t, auth_rsrt_tmr), DTYPE_UINT },
  { "AUTHWAIT",       offsetof(wan_ppp_conf_t, auth_wait_tmr), DTYPE_UINT },
  
  { "DTRDROPTIMER",   offsetof(wan_ppp_conf_t, dtr_drop_tmr),  DTYPE_UINT },
  { "CONNECTTIMEOUT", offsetof(wan_ppp_conf_t, connect_tmout), DTYPE_UINT },
  { "CONFIGURERETRY", offsetof(wan_ppp_conf_t, conf_retry),    DTYPE_UINT },
  { "TERMINATERETRY", offsetof(wan_ppp_conf_t, term_retry),    DTYPE_UINT },
  { "MAXCONFREJECT",  offsetof(wan_ppp_conf_t, fail_retry),    DTYPE_UINT },
  { "AUTHRETRY",      offsetof(wan_ppp_conf_t, auth_retry),    DTYPE_UINT },
  { "AUTHENTICATOR",  offsetof(wan_ppp_conf_t, authenticator), DTYPE_UCHAR},
  { "IP_MODE",        offsetof(wan_ppp_conf_t, ip_mode),       DTYPE_UCHAR},
  { NULL, 0, 0 }
};

key_word_t chdlc_conftab[] =	/* Cisco HDLC-specific configuration */
 {
  { "IGNORE_DCD", offsetof(wan_chdlc_conf_t, ignore_dcd), DTYPE_UCHAR},
  { "IGNORE_CTS", offsetof(wan_chdlc_conf_t, ignore_cts), DTYPE_UCHAR},
  { "IGNORE_KEEPALIVE", offsetof(wan_chdlc_conf_t, ignore_keepalive), DTYPE_UCHAR},
  { "HDLC_STREAMING", offsetof(wan_chdlc_conf_t, hdlc_streaming), DTYPE_UCHAR},
  { "KEEPALIVE_TX_TIMER", offsetof(wan_chdlc_conf_t, keepalive_tx_tmr), DTYPE_UINT },
  { "KEEPALIVE_RX_TIMER", offsetof(wan_chdlc_conf_t, keepalive_rx_tmr), DTYPE_UINT },
  { "KEEPALIVE_ERR_MARGIN", offsetof(wan_chdlc_conf_t, keepalive_err_margin),    DTYPE_UINT },
  { "SLARP_TIMER", offsetof(wan_chdlc_conf_t, slarp_timer),    DTYPE_UINT },
  { NULL, 0, 0 }
};

key_word_t fr_conftab[] =	/* Frame relay-specific configuration */
{
  { "SIGNALLING",    offsetof(wan_fr_conf_t, signalling),     DTYPE_UINT },
  { "T391", 	     offsetof(wan_fr_conf_t, t391),           DTYPE_UINT },
  { "T392",          offsetof(wan_fr_conf_t, t392),           DTYPE_UINT },
  { "N391",	     offsetof(wan_fr_conf_t, n391),           DTYPE_UINT },
  { "N392",          offsetof(wan_fr_conf_t, n392),           DTYPE_UINT },
  { "N393",          offsetof(wan_fr_conf_t, n393),           DTYPE_UINT },
  { "NUMBER_OF_DLCI",    offsetof(wan_fr_conf_t, dlci_num),       DTYPE_UINT },
  { NULL, 0, 0 }
};

key_word_t x25_conftab[] =	/* X.25-specific configuration */
{
  { "LOWESTPVC",    offsetof(wan_x25_conf_t, lo_pvc),       DTYPE_UINT },
  { "HIGHESTPVC",   offsetof(wan_x25_conf_t, hi_pvc),       DTYPE_UINT },
  { "LOWESTSVC",    offsetof(wan_x25_conf_t, lo_svc),       DTYPE_UINT },
  { "HIGHESTSVC",   offsetof(wan_x25_conf_t, hi_svc),       DTYPE_UINT },
  { "HDLCWINDOW",   offsetof(wan_x25_conf_t, hdlc_window),  DTYPE_UINT },
  { "PACKETWINDOW", offsetof(wan_x25_conf_t, pkt_window),   DTYPE_UINT },
  { "CCITTCOMPAT",  offsetof(wan_x25_conf_t, ccitt_compat), DTYPE_UINT },
  { NULL, 0, 0 }
};

key_word_t chan_conftab[] =	/* Channel configuration parameters */
{
  { "IDLETIMEOUT",   	offsetof(wanif_conf_t, idle_timeout), 	DTYPE_UINT },
  { "HOLDTIMEOUT",   	offsetof(wanif_conf_t, hold_timeout), 	DTYPE_UINT },
  { "CIR",           	offsetof(wanif_conf_t, cir), 	   	DTYPE_UINT },
  { "BC",            	offsetof(wanif_conf_t, bc),		DTYPE_UINT },
  { "BE", 	     	offsetof(wanif_conf_t, be),		DTYPE_UINT },
  { "MULTICAST",     	offsetof(wanif_conf_t, mc),		DTYPE_UCHAR},
  { "IPX",	     	offsetof(wanif_conf_t, enable_IPX),	DTYPE_UCHAR},
  { "NETWORK",       	offsetof(wanif_conf_t, network_number),	DTYPE_ULONG}, 
  { "PAP",     	     	offsetof(wanif_conf_t, pap),		DTYPE_UCHAR},
  { "CHAP",          	offsetof(wanif_conf_t, chap),		DTYPE_UCHAR},
  { "USERID",        	offsetof(wanif_conf_t, userid),	 	DTYPE_STR},
  { "PASSWD",        	offsetof(wanif_conf_t, passwd),		DTYPE_STR},
  { "SYSNAME",       	offsetof(wanif_conf_t, sysname),		DTYPE_STR},
  { "INARP", 	     	offsetof(wanif_conf_t, inarp),          	DTYPE_UCHAR},
  { "INARPINTERVAL", 	offsetof(wanif_conf_t, inarp_interval), 	DTYPE_UINT },
  { "IGNORE_DCD",  	offsetof(wanif_conf_t, ignore_dcd),        	DTYPE_UCHAR},
  { "IGNORE_CTS",    	offsetof(wanif_conf_t, ignore_cts),        	DTYPE_UCHAR},
  { "IGNORE_KEEPALIVE", offsetof(wanif_conf_t, ignore_keepalive), 	DTYPE_UCHAR},
  { "HDLC_STREAMING", 	offsetof(wanif_conf_t, hdlc_streaming), 	DTYPE_UCHAR},
  { "KEEPALIVE_TX_TIMER",	offsetof(wanif_conf_t, keepalive_tx_tmr), 	DTYPE_UINT },
  { "KEEPALIVE_RX_TIMER",	offsetof(wanif_conf_t, keepalive_rx_tmr), 	DTYPE_UINT },
  { "KEEPALIVE_ERR_MARGIN",	offsetof(wanif_conf_t, keepalive_err_margin),	DTYPE_UINT },
  { "SLARP_TIMER", 	offsetof(wanif_conf_t, slarp_timer),    DTYPE_UINT },
  { "TTL",        	offsetof(wanif_conf_t, ttl),         DTYPE_UCHAR },
  { "INTERFACE",  	offsetof(wanif_conf_t, interface),   DTYPE_UCHAR },
  { "CLOCKING",   	offsetof(wanif_conf_t, clocking),    DTYPE_UCHAR },
  { "BAUDRATE",   	offsetof(wanif_conf_t, bps),         DTYPE_UINT },
  { "MTU",        	offsetof(wanif_conf_t, mtu),         DTYPE_UINT },
  { NULL, 0, 0 }
};

look_up_t conf_def_tables[] =
{
	{ WANCONFIG_PPP,	ppp_conftab	},
	{ WANCONFIG_FR,		fr_conftab	},
	{ WANCONFIG_X25,	x25_conftab	},
	{ WANCONFIG_CHDLC,	chdlc_conftab	},
	{ 0,			NULL		}
};

look_up_t	config_id_str[] =
{
	{ WANCONFIG_PPP,	"WAN_PPP"	},
	{ WANCONFIG_FR,		"WAN_FR"	},
	{ WANCONFIG_X25,	"WAN_X25"	},
	{ WANCONFIG_CHDLC,	"WAN_CHDLC"	},
        { WANCONFIG_BSC,        "WAN_BSC"       },
        { WANCONFIG_HDLC,       "WAN_HDLC"      },
	{ 0,			NULL,		}
};

/*
 * Configuration options values and their symbolic names.
 */
look_up_t	sym_table[] =
{
	/*----- General -----------------------*/
	{ WANOPT_OFF,		"OFF"		}, 
	{ WANOPT_ON,		"ON"		}, 
	{ WANOPT_NO,		"NO"		}, 
	{ WANOPT_YES,		"YES"		}, 
	/*----- Interface type ----------------*/
	{ WANOPT_RS232,		"RS232"		},
	{ WANOPT_V35,		"V35"		},
	/*----- Data encoding -----------------*/
	{ WANOPT_NRZ,		"NRZ"		}, 
	{ WANOPT_NRZI,		"NRZI"		}, 
	{ WANOPT_FM0,		"FM0"		}, 
	{ WANOPT_FM1,		"FM1"		}, 
	/*----- Link type ---------------------*/
	{ WANOPT_POINTTOPOINT,	"POINTTOPOINT"	},
	{ WANOPT_MULTIDROP,	"MULTIDROP"	},
	/*----- Clocking ----------------------*/
	{ WANOPT_EXTERNAL,	"EXTERNAL"	}, 
	{ WANOPT_INTERNAL,	"INTERNAL"	}, 
	/*----- Station -----------------------*/
	{ WANOPT_DTE,		"DTE"		}, 
	{ WANOPT_DCE,		"DCE"		}, 
	{ WANOPT_CPE,		"CPE"		}, 
	{ WANOPT_NODE,		"NODE"		}, 
	{ WANOPT_SECONDARY,	"SECONDARY"	}, 
	{ WANOPT_PRIMARY,	"PRIMARY"	}, 
	/*----- Connection options ------------*/
	{ WANOPT_PERMANENT,	"PERMANENT"	}, 
	{ WANOPT_SWITCHED,	"SWITCHED"	}, 
	{ WANOPT_ONDEMAND,	"ONDEMAND"	}, 
	/*----- Frame relay in-channel signalling */
	{ WANOPT_FR_ANSI,	"ANSI"		}, 
	{ WANOPT_FR_Q933,	"Q933"		}, 
	{ WANOPT_FR_LMI,	"LMI"		}, 
	/*----- PPP IP Mode Options -----------*/
	{ WANOPT_PPP_STATIC,	"STATIC"	}, 
	{ WANOPT_PPP_HOST,	"HOST"		}, 
	{ WANOPT_PPP_PEER,	"PEER"		}, 
	/*----- CHDLC Protocol Options --------*/
/* DF for now	{ WANOPT_CHDLC_NO_DCD,	"IGNORE_DCD"	},
	{ WANOPT_CHDLC_NO_CTS,	"IGNORE_CTS"	}, 
	{ WANOPT_CHDLC_NO_KEEP,	"IGNORE_KEEPALIVE"}, 
*/
	{ WANOPT_PRI,           "PRI"           },
	{ WANOPT_SEC,           "SEC"           },
	
        /*------Read Mode ---------------------*/
        { WANOPT_INTR,          "INT"           },
        { WANOPT_POLL,          "POLL"          },
	/*----- End ---------------------------*/
	{ 0,			NULL		}, 
};

/****** Entry Point *********************************************************/

int main (int argc, char *argv[])
{
	int err = 0;	/* return code */
	int skip;

	/* Process command line switches */
	for (skip = 0, --argc, ++argv;
	     argc && (**argv == ARG_SWITCH);
	     argc -= skip, argv += skip) {
		skip = arg_proc(argc, argv);
		if (skip == 0) {
			err = ERR_SYNTAX;
			show_error(err);
			break;
		}
	}

	/* Perform requested action */
	if (verbose) puts(banner);
	if (!err)
	switch (action) {
	case DO_CONFIGURE:
		err = configure();
		break;

	case DO_SHUTDOWN:
		//for (; argc && !err; --argc, ++argv)
		shutdown();
		break;

	default:
		err = ERR_SYNTAX;
	}
	if (err == ERR_SYNTAX) puts(helptext);
	return err;
}

/*============================================================================
 * Process command line.
 *	Return number of successfully processed arguments, or 0 in case of
 *	syntax error.
 */
int arg_proc (int argc, char *argv[])
{
	int cnt = 0;

	switch (argv[0][1])
	{
	case SWITCH_CONFFILE:
		action = DO_CONFIGURE;
		if ((argc) > 1 && (*argv[1] != ARG_SWITCH)) {
			conf_file = argv[1];
			cnt = 2;
		}else 
			cnt = 1;
		break;

	case SWITCH_SHUTDOWN:
		action = DO_SHUTDOWN;
		if ((argc) > 1 && (*argv[1] != ARG_SWITCH)) {
			conf_file = argv[1];
			cnt = 2;
		}else 
			cnt = 1;
		break;

	case SWITCH_GETHELP:
	case SWITCH_ALTHELP:
		action = DO_HELP;
		cnt = 1;
		break;

	case SWITCH_VERBOSE:
	  	verbose = 1;
		cnt = 1;
		break;
	}
	return cnt;
}

/*============================================================================
 * Show error message.
 */
void show_error (int err)
{
	if (err == ERR_SYSTEM){ 
		fprintf(stderr, "%s: SYSTEM ERROR %d: %s!\n",
	 		progname, errno, strerror(errno));
	}else{
		fprintf(stderr, "%s: ERROR: %s : %s!\n", progname,
			err_messages[min(err, ERR_LIMIT) - 2], conf_file);
	}
}

/*============================================================================
 * Shut down link.
 */
int shutdown (void)
{
        int err = 0;
        char filename[sizeof(router_dir) + WAN_DRVNAME_SZ + 2];
        int dev;
        char* conf;             /* -> section buffer */
        char* key;              /* -> current configuration record */
        int len;                /* record length */
        FILE* file;
        char devname[WAN_DRVNAME_SZ];

	printf("Shutting down Device %s",conf_file);
        file = fopen(conf_file, "r");
        if (file == NULL) {
                show_error(ERR_SYSTEM);
                return ERR_SYSTEM;
        }

        /* Read [devices] section */
        conf = read_conf_section(file, "devices");
        if (conf == NULL) return ERR_CONFIG;

        /* For each record in [devices] section create a link definition
         * structure and link it to the list of link definitions.
         */
        for (key = conf; !err && *key; key += len) {
                int toknum;
                char* token[MAX_TOKENS];

                len = strlen(key) + 1;
                toknum = tokenize(key, token);
                if (toknum < 2) continue;

                strncpy(devname, token[0], WAN_DRVNAME_SZ);

                if (verbose)
                        printf(" * Shutting down WAN device %s ...\n", devname);                if (strlen(devname) > WAN_DRVNAME_SZ) {
                        show_error(ERR_SYNTAX);
                        return ERR_SYNTAX;
                }

                sprintf(filename, "%s/%s", router_dir, devname);
                dev = open(filename, O_RDONLY);
                if ((dev < 0) || (ioctl(dev, ROUTER_DOWN, NULL) < 0)) {
			fprintf(stderr, "\n\n\tROUTER SHUTDOWN Failed\n");
			fprintf(stderr, "\tIf you router was not running ignore this message\n !!");
			fprintf(stderr, "\tOtherwise, check the /var/log/wanrouter and \n");
			fprintf(stderr, "\t/var/log/messages for errors\n");
                        err = ERR_SYSTEM;
                        show_error(err);
                        return err;
                }

                if (dev >= 0) close(dev);
        }
        free(conf);
        return err;

}

/*============================================================================
 * Configure router.
 * o parse configuration file
 * o configure links
 * o configure logical channels (interfaces)
 */
int configure (void)
{
	int err = 0;
	link_def_t* linkdef;

	/* Parse configuration file */
	if (verbose)
		printf(" * Parsing configuration file %s ...\n", conf_file);

	err = parse_conf_file(conf_file);
	if (err) return err;

	if (link_defs == NULL) {
		if (verbose) printf(" * No link definitions found...\n");
		return 0;
	}

	/* Configure links */
	for (linkdef = link_defs; linkdef; linkdef = linkdef->next) {
		int res;
 		if (verbose) printf(
			" * Configuring device %s (%s)\n",
			linkdef->name,
			linkdef->descr ? linkdef->descr : "no description");
		res = configure_link(linkdef);
	}

	/* Clear definition list */
	while (link_defs != NULL) {
		link_def_t* linkdef = link_defs;

		while (linkdef->chan) {
			chan_def_t* chandef = linkdef->chan;

			if (chandef->conf) free(chandef->conf);
			if (chandef->addr) free(chandef->addr);
			if (chandef->usedby) free(chandef->usedby);
			if (chandef->descr) free(chandef->descr);
			linkdef->chan = chandef->next;
			free(chandef);
		}
		if (linkdef->conf) free(linkdef->conf);
		if (linkdef->descr) free(linkdef->descr);
		link_defs = linkdef->next;
		free(linkdef);
	}
	return err;
}

/*============================================================================
 * Parse configuration file.
 *	Read configuration file and create lists of link and channel
 *	definitions.
 */
int parse_conf_file (char* fname)
{
	int err = 0;
	FILE* file;

	file = fopen(fname, "r");
	if (file == NULL) {
		fprintf(stderr, "\nError: %s not found in /etc directory\n",fname);
		show_error(ERR_SYSTEM);
		return ERR_SYSTEM;
	}

	/* Build a list of link and channel definitions */
	err = build_linkdef_list(file);
	if (!err && link_defs)
		err = build_chandef_list(file);
	fclose(file);
	return err;
}

/*============================================================================
 * Build a list of link definitions.
 */
int build_linkdef_list (FILE* file)
{
	int err = 0;
	char* conf;		/* -> section buffer */
	char* key;		/* -> current configuration record */
	int len;		/* record length */

	/* Read [devices] section */
	conf = read_conf_section(file, "devices");
	if (conf == NULL) return ERR_CONFIG;

	/* For each record in [devices] section create a link definition
	 * structure and link it to the list of link definitions.
	 */
	for (key = conf; !err && *key; key += len) {
		int toknum;
		char* token[MAX_TOKENS];
		link_def_t* linkdef;	/* -> link definition */
		int config_id = 0;

		len = strlen(key) + 1;
		toknum = tokenize(key, token);
		if (toknum < 2) continue;

		strupcase(token[1]);
		config_id = name2val(token[1], config_id_str);
		if (!config_id) {
			if (verbose) printf(
				" * Media ID %s is invalid!\n", token[1]);
			err = ERR_CONFIG;
			show_error(err);
			break;
		}
		linkdef = calloc(1, sizeof(link_def_t));
		if (linkdef == NULL) {
			err = ERR_SYSTEM;
			show_error(err);
			break;
		}

	 	strncpy(linkdef->name, token[0], WAN_DRVNAME_SZ);
		linkdef->config_id = config_id;
		if ((toknum > 2) && token[2])
			linkdef->descr = strdup(token[2]);

		linkdef->conf = read_conf_section(file, linkdef->name);
                if (link_defs) {
			link_def_t* last;
			for (last = link_defs; last->next; last = last->next);
			last->next = linkdef;
		}
		else link_defs = linkdef;
	}
	free(conf);
	return err;
}

/*============================================================================
 * Build a list of channel definitions.
 */
int build_chandef_list (FILE* file)
{
	int err = 0;
	char* conf;		/* -> section buffer */
	char* key;		/* -> current configuration record */
	int len;		/* record length */

	/* Read [interfaces] section */
	conf = read_conf_section(file, "interfaces");
	if (conf == NULL) return 0;

	/* For each record in [interfaces] section create channel definition
	 * structure and link it to the list of channel definitions for
	 * appropriate link.
	 */
	for (key = conf; !err && *key; key += len) {
		int toknum;
		char* token[MAX_TOKENS];
		chan_def_t* chandef;		/* -> channel definition */
		link_def_t* linkdef;		/* -> channel definition */

		len = strlen(key) + 1;
		toknum = tokenize(key, token);
		if (toknum < 2) continue;

		/* find a WAN link definition for this channel */
		for (linkdef = link_defs;
		     linkdef && strcmp(token[1], linkdef->name);
		     linkdef = linkdef->next)
		;
		if (linkdef == NULL) continue;

		/* allocate and initialize channel definition structure */
		chandef = calloc(1, sizeof(chan_def_t));
		if (chandef == NULL) {
			err = ERR_SYSTEM;
			show_error(err);
			break;
		}
	 	strncpy(chandef->name, token[0], WAN_IFNAME_SZ);
		if ((toknum > 2) && token[2])
			chandef->addr = strdup(token[2]);
		if ((toknum > 3) && token[3])
			chandef->usedby = strdup(token[3]);
		if (!(chandef->usedby) || ( 
				( strcmp(chandef->usedby, "WANPIPE") != 0 ) &&
				( strcmp(chandef->usedby, "API") != 0 )) ) {

			if (verbose) printf(" * %s to be used by an Invalid entity: %s\n", chandef->name, chandef->usedby);
			return ERR_CONFIG;
		}

		if ((toknum > 4) && token[4])
			chandef->descr = strdup(token[3]);

		if (verbose)
			printf(" * %s to used by %s\n",
				chandef->name, chandef->usedby);

		chandef->conf = read_conf_section(file, token[0]);

		/* append channel definition structure to the list */
		if (linkdef->chan) {
			chan_def_t* last;

			for (last = linkdef->chan;
			     last->next;
			     last = last->next)
			;
			last->next = chandef;
		}
		else linkdef->chan = chandef;
	}
	free(conf);
	return err;
}

/*============================================================================
 * Configure WAN link.
 */
int configure_link (link_def_t* def)
{
	int err = 0;
	int len = 0;
	int i;
	key_word_t* conf_table = lookup(def->config_id, conf_def_tables);
	char filename[sizeof(router_dir) + WAN_DRVNAME_SZ + 2];
	chan_def_t* chandef;
	char* conf_rec;
	int dev;
	unsigned short hi_DLCI = 15;

	if (def->conf == NULL) {
		show_error(ERR_CONFIG);
		return ERR_CONFIG;
	}

	/* Clear configuration structure */
	memset(&u.linkconf, 0, sizeof(u.linkconf));
	u.linkconf.magic = ROUTER_MAGIC;
	u.linkconf.config_id = def->config_id;

	/* Parse link configuration */
	for (conf_rec = def->conf; *conf_rec; conf_rec += len) {
		int toknum;
		char* tok[MAX_TOKENS];

		len = strlen(conf_rec) + 1;
		toknum = tokenize(conf_rec, tok);
		if (toknum < 2) {
			show_error(ERR_CONFIG);
			return ERR_CONFIG;
		}

		/* Look up a keyword first in common, then in media-specific
		 * configuration definition tables.
		 */
		strupcase(tok[0]);
		err = set_conf_param(
			tok[0], tok[1], common_conftab, &u.linkconf);
		if ((err < 0) && (conf_table != NULL))
			err = set_conf_param(
				tok[0], tok[1], conf_table, &u.linkconf.u);
		if (err < 0) {
			printf(" * Unknown parameter %s\n", tok[0]);
			fprintf(stderr, "\n\n\tERROR in %s !!\n",conf_file);
			fprintf(stderr, "\tPlease check /var/log/wanrouter for errors\n");
			return ERR_CONFIG;
		}
		if (err){ 
			fprintf(stderr,"\n\tError parsing %s configuration file\n",conf_file);
			fprintf(stderr,"\tPlease check /var/log/wanrouter for errors\n\n");
			return err;
		}
	}

	/* Open SDLA device and perform link configuration */
	sprintf(filename, "%s/%s", router_dir, def->name);
	
	/* prepare a list of DLCI(s) and place it in the wandev_conf_t structure
	 * This is done so that we have a list of DLCI(s) available when we 
	 * call the wpf_init() routine in sdla_fr.c (triggered by the ROUTER_
	 * SETUP ioctl call) for running global SET_DLCI_CONFIGURATION cmd. This
	 * is only relevant for a station being a NODE, for CPE it polls the 
	 * NODE (auto config DLCI)
	 */
	
	if ((u.linkconf.config_id == WANCONFIG_FR)){
	   i = 0;
	
	   if(!err) for (chandef = def->chan; chandef; chandef = chandef->next){
 		if (verbose) printf(
			" * Reading DLCI(s) Included : %s\n",
			chandef->addr ? chandef->addr : "not specified");
		u.linkconf.u.fr.dlci[i] = dec_to_uint(chandef->addr,0);
		if(u.linkconf.u.fr.dlci[i] > hi_DLCI) {
			hi_DLCI = u.linkconf.u.fr.dlci[i];
			i++;
		}
		else {
			fprintf(stderr,"\n\tDLCI(s) specified in %s are not in order\n",conf_file); 
			fprintf(stderr,"\n\tor duplicate DLCI(s) assigned!\n");
			err = ERR_SYSTEM;
		}
		
	    }
	}

	dev = open(filename, O_RDONLY);

	if ((dev < 0) || (ioctl(dev, ROUTER_SETUP, &u.linkconf) < 0)) {
		fprintf(stderr, "\n\n\tROUTER_SETUP Failed !!\n");
		fprintf(stderr, "\tWanpipe driver did not load properly\n");
		fprintf(stderr, "\tPlease check /var/log/wanrouter and\n");
		fprintf(stderr, "\t/var/log/messages for errors\n\n"); 
		return ERR_SYSTEM;
	}
	if (u.linkconf.data) free(u.linkconf.data);

	/* Configure logical channels */
	if (!err) for (chandef = def->chan; chandef; chandef = chandef->next) {
 		if (verbose) printf(
			" * Configuring channel %s (%s). Media address: %s\n",
			chandef->name,
			chandef->descr ? chandef->descr : "no description",
			chandef->addr ? chandef->addr : "not specified");
		memset(&u.chanconf, 0, sizeof(u.chanconf));
		u.chanconf.magic = ROUTER_MAGIC;
		u.chanconf.config_id = def->config_id;
		configure_chan(dev, chandef);
	}
	/* clean up */
	if (dev > 0) close(dev);
	return err;
}

/*============================================================================
 * Configure WAN logical channel.
 */
int configure_chan (int dev, chan_def_t* def)
{
	int err = 0;
	int len = 0;
	char* conf_rec;

	/* Prepare configuration data */
	strncpy(u.chanconf.name, def->name, WAN_IFNAME_SZ);
	if (def->addr)
		strncpy(u.chanconf.addr, def->addr, WAN_ADDRESS_SZ);

	if (def->usedby)
		strncpy(u.chanconf.usedby, def->usedby, USED_BY_FIELD);

	if (def->conf) for (conf_rec = def->conf; *conf_rec; conf_rec += len) {
		int toknum;
		char* tok[MAX_TOKENS];

		len = strlen(conf_rec) + 1;
		toknum = tokenize(conf_rec, tok);
		if (toknum < 2) {
			show_error(ERR_CONFIG);
			return ERR_CONFIG;
		}

		/* Look up a keyword first in common, then media-specific
		 * configuration definition tables.
		 */
		strupcase(tok[0]);
		if (set_conf_param(tok[0], tok[1], chan_conftab, &u.chanconf)) {
			printf("Invalid parameter %s\n", tok[0]);
			show_error(ERR_CONFIG);
			return ERR_CONFIG;
		}
	}

	if (ioctl(dev, ROUTER_IFNEW, &u.chanconf) < 0) {
		fprintf(stderr, "\n\n\tINTERFACE Setup Failed\n");
		fprintf(stderr, "\tWanpipe drivers could not setup network interface\n");
		fprintf(stderr, "\tPlease check /var/log/wanrouter and\n");
		fprintf(stderr, "\t/var/log/messages for errors\n\n"); 
		return ERR_SYSTEM;
	}
	return err;
}

/*============================================================================
 * Set configuration parameter.
 *	Look up a keyword in configuration description table to find offset
 *	and data type of the configuration parameter into configuration data
 *	buffer.  Convert parameter to apporriate data type and store it the
 *	configuration data buffer.
 *
 *	Return:	0 - success
 *		1 - error
 *		-1 - not found
 */
int set_conf_param (char* key, char* val, key_word_t* dtab, void* conf)
{
	ulong tmp = 0;

	/* Search a keyword in configuration definition table */
	for (; dtab->keyword && strcmp(dtab->keyword, key); ++dtab);
	if (dtab->keyword == NULL) return -1;	/* keyword not found */

	/* Interprete parameter value according to its data type */

	if (dtab->dtype == DTYPE_FILENAME) {
		return read_data_file(
			val, (void*)((char*)conf + dtab->offset));
	}

	if (verbose) printf(" * Setting %s to %s\n", key, val);
	if (dtab->dtype == DTYPE_STR) {
		strcpy((char*)conf + dtab->offset, val);
		return 0;
	}

	if (!isdigit(*val)) {
		look_up_t* sym;

		strupcase(val);
		for (sym = sym_table;
		     sym->ptr && strcmp(sym->ptr, val);
		     ++sym)
		;
		if (sym->ptr == NULL) {
			if (verbose) printf(" * invalid term %s ...\n", val);
			return 1;
		}
		else tmp = sym->val;
	}
	else tmp = strtoul(val, NULL, 0);

	switch (dtab->dtype) {

	case DTYPE_INT:
		*(int*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_UINT:
		*(uint*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_LONG:
		*(long*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_ULONG:
		*(ulong*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_SHORT:
		*(short*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_USHORT:
		*(ushort*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_CHAR:
		*(char*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_UCHAR:
		*(unsigned char*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_PTR:
		*(void**)((char*)conf + dtab->offset) = (void*)tmp;
		break;
	}
	return 0;
}

/*============================================================================
 * Read configuration file section.
 *	Return a pointer to memory filled with key entries read from given
 *	section or NULL if section is not found. Each key is a zero-terminated
 *	ASCII string with no leading spaces and comments. The last key is
 *	followed by an empty string.
 *	Note:	memory is allocated dynamically and must be released by the
 *		caller.
 */
char* read_conf_section (FILE* file, char* section)
{
	char key[MAX_CFGLINE];		/* key buffer */
	char* buf = NULL;
	int found = 0, offs = 0, len;

	rewind(file);
	while ((len = read_conf_record(file, key)) > 0) {
		char* tmp;

		if (found) {
			if (*key == '[') break;	/* end of section */

			if (buf) tmp = realloc(buf, offs + len + 1);
			else tmp = malloc(len + 1);
			if (tmp) {
				buf = tmp;
				strcpy(&buf[offs], key);
				offs += len;
				buf[offs] = '\0';
			}
			else {	/* allocation failed */
				show_error(ERR_SYSTEM);
 				break;
			}
		}
		else if (*key == '[') {
			tmp = strchr(key, ']');
			if (tmp != NULL) {
				*tmp = '\0';
				if (strcmp(&key[1], section) == 0) {
 					if (verbose) printf(
						" * Reading section [%s]...\n",
						section);
					found = 1;
				}
			}
		}
	}
 	if (verbose && !found) printf(
		" * section [%s] not found!\n", section);
	return buf;
 }

/*============================================================================
 * Read a single record from the configuration file.
 *	Read configuration file stripping comments and leading spaces until
 *	non-empty line is read.  Copy it to the destination buffer.
 *
 * Return string length (incl. terminating zero) or 0 if end of file has been
 * reached.
 */
int read_conf_record (FILE* file, char* key)
{
	char buf[MAX_CFGLINE];		/* line buffer */

	while (fgets(buf, MAX_CFGLINE, file)) {
		char* str;
		int len;

		/* Strip leading spaces and comments */
		for (str = buf; *str && strchr(" \t", *str); ++str);
		len = strcspn(str, "#;\n\r");
		if (len) {
			str[len] = '\0';
			strcpy(key, str);
			return len + 1;
		}
	}
	return 0;
}

/*============================================================================
 * Tokenize string.
 *	Parse a string of the following syntax:
 *		<tag>=<arg1>,<arg2>,...
 *	and fill array of tokens with pointers to string elements.
 *
 *	Return number of tokens.
 */
int tokenize (char* str, char **tokens)
{
	int cnt = 0;

	tokens[0] = strtok(str, "=");
	while (tokens[cnt] && (cnt < MAX_TOKENS - 1)) {
		tokens[cnt] = strstrip(tokens[cnt], " \t");
		tokens[++cnt] = strtok(NULL, ",");
	}
	return cnt;
}

/*============================================================================
 * Strip leading and trailing spaces off the string str.
 */
char* strstrip (char* str, char* s)
{
	char* eos = str + strlen(str);		/* -> end of string */

	while (*str && strchr(s, *str))
		++str;				/* strip leading spaces */
	while ((eos > str) && strchr(s, *(eos - 1)))
		--eos;				/* strip trailing spaces */
	*eos = '\0';
	return str;
}

/*============================================================================
 * Uppercase string.
 */
char* strupcase (char* str)
{
	char* s;

	for(s = str; *s; ++s) *s = toupper(*s);
	return str;
}

/*============================================================================
 * Get a pointer from a look-up table.
 */
void* lookup (int val, look_up_t* table)
{
	while (table->val && (table->val != val)) table++;
	return table->ptr;
}

/*============================================================================
 * Look up a symbolic name in name table.
 *	Return a numeric value associated with that name or zero, if name was
 *	not found.
 */
int name2val (char* name, look_up_t* table)
{
	while (table->ptr && strcmp(name, table->ptr)) table++;
	return table->val;
}

/*============================================================================
 * Create memory image of a file.
 */
int read_data_file (char* name, data_buf_t* databuf)
{
	int err = 0;
	FILE* file;
	void* buf;
	unsigned long fsize;

	databuf->data = NULL;
	databuf->size = 0;
	file = fopen(name, "rb");
	if (file == NULL) {
		if (verbose) printf(" * Can't open data file %s!\n", name);
		show_error(ERR_SYSTEM);
		return ERR_SYSTEM;
	}

	fsize = filesize(file);
	if (!fsize) {
		if (verbose) printf(" * Data file %s is empty!\n", name);
		err = ERR_CONFIG;
		show_error(err);
	}
	else {
		if (verbose) printf(
			" * Reading %lu bytes from %s ...\n",
			fsize, name);
		buf = malloc(fsize);
		if (buf == NULL) {
			err = ERR_SYSTEM;
			show_error(err);
		}
		else if (fread(buf, 1, fsize, file) < fsize) {
			free(buf);
			err = ERR_SYSTEM;
			show_error(err);
		}
		else {
			databuf->data = buf;
			databuf->size = fsize;
		}
	}
	fclose(file);
	return err;
}

/*============================================================================
 * Get file size
 *	Return file length or 0 if error.
 */
unsigned long filesize (FILE* file)
{
	unsigned long size = 0;
	unsigned long cur_pos;

	cur_pos = ftell(file);
	if ((cur_pos != -1L) && !fseek(file, 0, SEEK_END)) {
		size = ftell(file);
		fseek(file, cur_pos, SEEK_SET);
	}
	return size;
}

/*============================================================================
 * Convert decimal string to unsigned integer.
 * If len != 0 then only 'len' characters of the string are converted.
 */
unsigned int dec_to_uint (unsigned char* str, int len)
{
	unsigned val;

	if (!len) len = strlen(str);
	for (val = 0; len && is_digit(*str); ++str, --len)
		val = (val * 10) + (*str - (unsigned)'0')
	;
	return val;
}
/****** End *****************************************************************/

