/*
 ui-history.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

/* command history */
static GList *cmdhist, *histpos;
static gint histlines;

void ui_history_add(WINDOW_REC *window, gchar *text, gboolean prepend)
{
    GList **pcmdhist, *link;
    gint *phistlines;

    g_return_if_fail(text != NULL);

    if (setup_get_bool("toggle_window_history"))
    {
	/* window specific command history */
	pcmdhist = &window->cmdhist;
	phistlines = &window->histlines;
    }
    else
    {
	/* global command history */
	pcmdhist = &cmdhist;
	phistlines = &histlines;
    }

    if (setup_get_int("max_command_history") < 1 || *phistlines < setup_get_int("max_command_history"))
        (*phistlines)++;
    else
    {
        link = g_list_first(*pcmdhist);
        g_free(link->data);
        *pcmdhist = g_list_remove_link(*pcmdhist, link);
        g_list_free_1(link);
    }

    if (prepend)
        *pcmdhist = g_list_prepend(*pcmdhist, g_strdup(text));
    else
        *pcmdhist = g_list_append(*pcmdhist, g_strdup(text));
}

gchar *ui_history_prev(WINDOW_REC *window, gchar *text)
{
    GList *pos, **phistpos;

    phistpos = setup_get_bool("toggle_window_history") ? &window->histpos : &histpos;

    pos = *phistpos;
    if (*phistpos == NULL)
	*phistpos = g_list_last(setup_get_bool("toggle_window_history") ? window->cmdhist : cmdhist);
    else
        *phistpos = (*phistpos)->prev;

    if (*text != '\0' &&
        (pos == NULL || strcmp(pos->data, text) != 0))
    {
        /* save the old entry to history */
        ui_history_add(window, text, FALSE);
    }

    return *phistpos == NULL ? "" : (*phistpos)->data;
}

gchar *ui_history_next(WINDOW_REC *window, gchar *text)
{
    GList *pos, **phistpos;

    phistpos = setup_get_bool("toggle_window_history") ? &window->histpos : &histpos;

    pos = *phistpos;
    if (*phistpos == NULL)
        *phistpos = g_list_first(setup_get_bool("toggle_window_history") ? window->cmdhist : cmdhist);
    else
        *phistpos = (*phistpos)->next;

    if (*text != '\0' &&
        (pos == NULL || strcmp(pos->data, text) != 0))
    {
        /* save the old entry to history */
        ui_history_add(window, text, TRUE);
    }
    return *phistpos == NULL ? "" : (*phistpos)->data;
}

void ui_history_clear_pos(WINDOW_REC *window)
{
    window->histpos = NULL;
    histpos = NULL;
}

static gboolean sig_window_created(WINDOW_REC *window)
{
    window->histlines = 0;
    window->cmdhist = NULL;
    window->histpos = NULL;
    return TRUE;
}

static gboolean sig_window_destroyed(WINDOW_REC *window)
{
    g_list_foreach(window->cmdhist, (GFunc) g_free, NULL);
    g_list_free(window->cmdhist);
    return TRUE;
}

void ui_history_init(void)
{
    histlines = 0;
    cmdhist = NULL; histpos = NULL;
    signal_add("window created", (SIGNAL_FUNC) sig_window_created);
    signal_add("window destroyed", (SIGNAL_FUNC) sig_window_destroyed);
}

void ui_history_deinit(void)
{
    signal_remove("window created", (SIGNAL_FUNC) sig_window_created);
    signal_remove("window destroyed", (SIGNAL_FUNC) sig_window_destroyed);

    g_list_foreach(cmdhist, (GFunc) g_free, NULL);
    g_list_free(cmdhist);
}
