/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998,1999 Bert Vermeulen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <gtk/gtk.h>

#include <lber.h>
#include <ldap.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "common.h"
#include "configfile.h"
#include "query.h"
#include "detail.h"
#include "errorchain.h"
#include "util.h"
#include "debug.h"


struct resultset *cur_resultset = NULL;
struct attrs *cur_attrs = NULL;

extern GtkWidget *main_clist;
extern struct gq_config config;


void free_resultset(struct resultset *set)
{
     struct resultset *nextset;

     while(set) {
	  nextset = set->next;
	  FREE(set, "struct resultset");
	  set = nextset;
     }

}


struct resultset *new_resultset(void)
{
     struct resultset *newset, *set;

     newset = MALLOC(sizeof(struct resultset), "struct resultset");
     newset->objectclass = NULL;
     newset->attributes = NULL;
     newset->num_attributes = 0;
     newset->next = NULL;

     if(!cur_resultset)
	  cur_resultset = newset;
     else {
	  for(set = cur_resultset; set->next; set = set->next)
	       ;
	  set->next = newset;
     }

     return(newset);
}


int new_attr(char *attr)
{
     struct attrs *new_attr, *attrlist;
     int cnt;

     new_attr = MALLOC(sizeof(struct attrs), "struct attrs");
     strncpy(new_attr->name, attr, MAX_ATTR_LEN);
     new_attr->next = NULL;

     cnt = 0;
     if(cur_attrs) {
	  cnt++;
	  attrlist = cur_attrs;
	  while(attrlist->next) {
	       attrlist = attrlist->next;
	       cnt++;
	  }
	  attrlist->next = new_attr;
     }
     else
	  cur_attrs = new_attr;

     new_attr->column = cnt;

     return(cnt);
}

struct attrs *find_attr(char *attr)
{
     struct attrs *attrlist;

     attrlist = cur_attrs;
     while(attrlist) {
	  if(!strcasecmp(attrlist->name, attr))
	       return(attrlist);
	  attrlist = attrlist->next;
     }

     return(NULL);
}

int column_by_attr(char *attribute)
{
     struct attrs *attr;

     attr = find_attr(attribute);
     if(!attr)
	  return(new_attr(attribute));
     else
	  return(attr->column);

}

void free_cur_attrs(void)
{
     struct attrs *next_attrs;

     while(cur_attrs) {
	  next_attrs = cur_attrs->next;;
	  FREE(cur_attrs, "struct attrs");
	  cur_attrs = next_attrs;
     }

}


void query(char *querystring)
{
     LDAP *ld;
     LDAPMessage *res, *e;
     BerElement *ptr;
     GtkWidget *new_main_clist, *scrwin;
     struct ldapserver *server;
     struct resultset *set;
     gchar *cl[MAX_NUM_ATTRIBUTES];
     char tolist[MAX_NUM_ATTRIBUTES][128], message[1152], filter[1024];
     char *attr, *dn, **vals;
     int msg, rc, i, row;
     int cur_col, oc_col, columns_done[MAX_NUM_ATTRIBUTES];

     for(i = 0; i < MAX_NUM_ATTRIBUTES; i++)
	  columns_done[i] = 0;

     server = server_by_name(config.cur_servername);
     if(!server) {
	  statusbar_msg("Oops! No server found!");
	  return;
     }

     if(config.searchtype == ST_FILTER && querystring[0] == 0) {
	  statusbar_msg("Please enter a valid search filter");
	  return;
     }

     if( (ld = open_connection(server)) == NULL)
	  return;

     /* construct filter */
     switch(config.searchtype) {
     case ST_SEARCH:
	  switch(config.search_argument) {
	  case SEARCHARG_BEGINS_WITH:
	       sprintf(filter, "(%s=%s*)", server->searchattr, querystring);
	       break;
	  case SEARCHARG_ENDS_WITH:
	       sprintf(filter, "(%s=*%s)", server->searchattr, querystring);
	       break;
	  case SEARCHARG_CONTAINS:
	       sprintf(filter, "(%s=*%s*)", server->searchattr, querystring);
	       break;
	  case SEARCHARG_EQUALS:
	       sprintf(filter, "(%s=%s)", server->searchattr, querystring);
	       break;

	  };
	  break;
     case ST_FILTER:
	  strncpy(filter, querystring, 1024);
	  break;
     };

     sprintf(message, "searching for %s", filter);
     statusbar_msg(message);

     if(config.sort_search)
	  msg = ldap_search_s(ld, server->basedn, LDAP_SCOPE_SUBTREE,
			      filter, NULL, 0, &res);
     else
	  msg = ldap_search(ld, server->basedn, LDAP_SCOPE_SUBTREE,
			    filter, NULL, 0);

     if(msg == -1) {
	  statusbar_msg(ldap_err2string(msg));
	  return;
     }

     free_resultset(cur_resultset);
     cur_resultset = NULL;

     /* build new clist */
     new_main_clist = gtk_clist_new(MAX_NUM_ATTRIBUTES);
     GTK_CLIST(new_main_clist)->button_actions[2] = GTK_BUTTON_SELECTS;
     GTK_WIDGET_UNSET_FLAGS(GTK_CLIST(new_main_clist), GTK_CAN_FOCUS);
     gtk_widget_show(new_main_clist);
     gtk_clist_column_titles_show(GTK_CLIST(new_main_clist));
     gtk_signal_connect(GTK_OBJECT(new_main_clist), "select_row",
                        GTK_SIGNAL_FUNC(select_entry_callback),
                        NULL);
     gtk_signal_connect(GTK_OBJECT(new_main_clist), "unselect_row",
                        GTK_SIGNAL_FUNC(unselect_entry_callback),
                        NULL);

     scrwin = gtk_object_get_data(GTK_OBJECT(main_clist), "scrwin");
     gtk_object_set_data(GTK_OBJECT(new_main_clist), "scrwin", scrwin);
     gtk_clist_clear(GTK_CLIST(main_clist));
     gtk_widget_destroy(main_clist);
     main_clist = new_main_clist;
     gtk_container_add(GTK_CONTAINER(scrwin), new_main_clist);

     /* reserve columns 0 & 1 for DN and objectClass, respectively */
     if(config.showdn) {
	  column_by_attr("DN");
	  gtk_clist_set_column_title(GTK_CLIST(new_main_clist), 0, "DN");
	  gtk_clist_set_column_width(GTK_CLIST(new_main_clist), 0, 260);
     }
     oc_col = column_by_attr("objectClass");
     gtk_clist_set_column_title(GTK_CLIST(new_main_clist), oc_col, "objectClass");
     gtk_clist_set_column_width(GTK_CLIST(new_main_clist), oc_col, 120);
     columns_done[oc_col] = 1;
     if(config.showoc == 0)
	  gtk_clist_set_column_visibility(GTK_CLIST(new_main_clist), oc_col, 0);


     row = 0;
     if(!config.sort_search)
	  rc = (ldap_result(ld, msg, 0, NULL, &res) == LDAP_RES_SEARCH_ENTRY);
     else
	  rc = (ldap_sort_entries(ld, &res, NULL, strcasecmp) == LDAP_SUCCESS);

     while(rc) {

	  for(e = ldap_first_entry(ld, res); e != NULL; e = ldap_next_entry(ld, e)) {

	       /* not every attribute necessarily comes back for every entry,
		* so clear this every time */
	       for(i = 0; i < MAX_NUM_ATTRIBUTES; i++) {
		    cl[i] = NULL;
		    tolist[i][0] = '\0';
	       }

	       dn = ldap_get_dn(ld, e);
	       /* store for later reference */
	       set = new_resultset();
	       strncpy(set->dn, dn, MAX_DN_LEN);

	       if(config.showdn) {
		    strcpy(tolist[0],dn);
		    cl[0] = tolist[0];
	       }
	       free(dn);

	       for(attr = ldap_first_attribute(ld, e, &ptr); attr != NULL;
		   attr = ldap_next_attribute(ld, e, ptr)) {

		    cur_col = column_by_attr(attr);
		    if(cur_col == MAX_NUM_ATTRIBUTES)
			 break;

		    if(!columns_done[cur_col]) {
			 gtk_clist_set_column_title(GTK_CLIST(new_main_clist), cur_col, attr);
			 gtk_clist_set_column_width(GTK_CLIST(new_main_clist), cur_col, 120);
			 columns_done[cur_col] = 1;
		    }

		    vals = ldap_get_values(ld, e, attr);
		    if(vals) {
			 for(i = 0; vals[i] != NULL; i++) {
			      if(i == 0)
				   strcpy(tolist[cur_col], vals[i]);
			      else {
				   strcat(tolist[cur_col], " ");
				   strcat(tolist[cur_col], vals[i]);
			      }

			 }
			 ldap_value_free(vals);
			 cl[cur_col] = tolist[cur_col];
		    }

	       }

	       /* insert row into result window */
	       gtk_clist_append(GTK_CLIST(new_main_clist), cl);
	       gtk_clist_set_row_data(GTK_CLIST(new_main_clist), row, set);
	       row++;
	  }

	  if(config.sort_search)
	       rc = 0;
	  else
	       if(server->maxentries == 0 || row < server->maxentries)
		    rc = (ldap_result(ld, msg, 0, NULL, &res) == LDAP_RES_SEARCH_ENTRY);
	       else
		    rc = 0;

     }

     if(rc == -1)
	  statusbar_msg(ldap_err2string(msg));
     else {
	  make_message(message, row, "entry", "entries", "found");
	  statusbar_msg(message);
     }

     gtk_clist_column_titles_passive(GTK_CLIST(new_main_clist));
     free_cur_attrs();
     ldap_msgfree(res);

     close_connection(server, FALSE);

}


void results_popup_menu(GdkEventButton *event, struct resultset *set)
{
     GtkWidget *root_menu, *menu, *menu_item;

     root_menu = gtk_menu_item_new_with_label("Root");
     gtk_widget_show(root_menu);

     menu = gtk_menu_new();
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(root_menu), menu);
     menu_item = gtk_tearoff_menu_item_new();
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_widget_set_sensitive(menu_item, FALSE);
     gtk_widget_show(menu_item);

     /* View */
     menu_item = gtk_menu_item_new_with_label("View");
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate",
			       GTK_SIGNAL_FUNC(view_entry),
			       (gpointer) set);
     gtk_widget_show(menu_item);

     /* Edit */
     menu_item = gtk_menu_item_new_with_label("Edit");
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate",
			       GTK_SIGNAL_FUNC(edit_entry_values),
			       (gpointer) set);
     gtk_widget_show(menu_item);

     /* Use as template */
     menu_item = gtk_menu_item_new_with_label("Use as template");
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate",
			       GTK_SIGNAL_FUNC(edit_entry_novalues),
			       (gpointer) set);
     gtk_widget_show(menu_item);

     /* separator */
     menu_item = gtk_menu_item_new();
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_widget_show(menu_item);

     /* Delete */
     menu_item = gtk_menu_item_new_with_label("Delete");
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate",
			       GTK_SIGNAL_FUNC(delete_search_entry),
			       (gpointer) set);
     gtk_widget_show(menu_item);

     gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
		    event->button, event->time);

}


void delete_search_entry(struct resultset *set)
{
     struct ldapserver *server;

     if( (server = server_by_name(config.cur_servername)) == NULL)
	  return;

     if(!set || !set->dn)
	  return;

     delete_entry(server, set->dn);

}


int select_entry_callback(GtkWidget *clist, gint row, gint column,
			   GdkEventButton *event, gpointer data)
{
     struct resultset *set;

     if(event) {
	  set = gtk_clist_get_row_data(GTK_CLIST(clist), row);

	  if(event->button == 1 && event->type == GDK_2BUTTON_PRESS)
	       edit_entry_values(set);
	  else if(event->button == 3) {
	       results_popup_menu(event, set);
	  }
     }

     return(TRUE);
}


int unselect_entry_callback(GtkWidget *clist, gint row, gint column,
			   GdkEventButton *event, gpointer data)
{
     GtkCListRow *clistrow;
     GList *list;
     struct resultset *set;

     if(event && event->button == 3) {

	  list = g_list_nth(GTK_CLIST(clist)->row_list, row);
	  clistrow = list->data;
	  if(clistrow->state != GTK_STATE_SELECTED) {
	       set = gtk_clist_get_row_data(GTK_CLIST(clist), row);
	       results_popup_menu(event, set);

	       /* why TF doesn't this work */
	       gtk_signal_emit_stop_by_name(GTK_OBJECT(clist), "unselect_row");
	  }

     }

     return(TRUE);
}


