/*
 * Copyright (C) 1998  Mark Baysinger (mbaysing@ucsd.edu)
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#ifndef INCLUDED_BNET_PROTOCOL_TYPES
#define INCLUDED_BNET_PROTOCOL_TYPES

#ifdef JUST_NEED_TYPES
#include "bn_type.h"
#else
#define JUST_NEED_TYPES
#include "bn_type.h"
#undef JUST_NEED_TYPES
#endif


/*
 * The "normal" protocol that is used for chatting, ladder info,
 * game listings, etc.
 * FIXME: put the #define's into the PROTO section
 */


/******************************************************/
typedef struct
{
    bn_short type;
    bn_short size;
} t_normal_header;
/******************************************************/


/******************************************************/
typedef struct
{
    t_normal_header h;
} t_normal_generic;
/******************************************************/


/******************************************************/
/*
These two dumps are from the original unpatched Starcraft client:

                          FF 05 28 00 01 00 00 00            ..(.....
D1 43 88 AA DA 9D 1B 00   9A F7 69 AB 4A 41 32 30    .C........i.JA20
35 43 2D 30 34 00 6C 61   62 61 73 73 69 73 74 00    5C-04.labassist.

FF 05 24 00 01 00 00 00   D1 43 88 AA DA 9D 1B 00    ..$......C......
9A F7 69 AB 42 4F 42 20   20 20 20 20 20 20 20 00    ..i.BOB        .
42 6F 62 00                                          Bob.       
*/
#define CLIENT_COMPINFO1 0x05ff
typedef struct
{
    t_normal_header h;
    bn_int          reg_version;  /* 01 00 00 00 */
    bn_int          reg_auth;     /* D1 43 88 AA */
    bn_int          client_id;    /* DA 9D 1B 00 */
    bn_int          client_token; /* 9A F7 69 AB */
    /* host */
    /* user */
} t_client_compinfo1;
#define CLIENT_COMPINFO1_REG_VERSION  0x00000001
#define CLIENT_COMPINFO1_REG_AUTH     0xaa8843d1
#define CLIENT_COMPINFO1_CLIENT_ID    0x001b9dda
#define CLIENT_COMPINFO1_CLIENT_TOKEN 0xab69f79a
/******************************************************/


/******************************************************/
/*
CLIENT_COMPINFO2 was first seen in Starcraft 1.05

                          FF 1E 24 00 01 00 00 00    ..$.....
01 00 00 00 D1 43 88 AA   1C B9 48 00 31 8A F2 89    .....C....H.1...
43 4C 4F 55 44 00 63 6C   6F 75 64 00                CLOUD.cloud.

FF 1E 28 00 01 00 00 00   01 00 00 00 D1 43 88 AA    ..(..........C..
DA 9D 1B 00 9A F7 69 AB   42 4F 42 20 20 20 20 20    ......i.BOB     
20 20 20 00 42 6F 62 00                                 .Bob.
*/
#define CLIENT_COMPINFO2 0x1eff
typedef struct
{
    t_normal_header h;
    bn_int          unknown1;     /* 01 00 00 00 */ /* ??? Version */
    bn_int          unknown2;     /* 01 00 00 00 */ /* ??? Version */
    bn_int          reg_auth;     /* D1 43 88 AA */
    bn_int          client_id;    /* 1C B9 48 00 */ /* DA 9D 1B 00 */
    bn_int          client_token; /* 31 8A f2 89 */ /* 9A F7 69 AB */
    /* host */
    /* user */
} t_client_compinfo2;
#define CLIENT_COMPINFO2_UNKNOWN1     0x00000001
#define CLIENT_COMPINFO2_UNKNOWN2     0x00000001
#define CLIENT_COMPINFO2_REG_AUTH     0xaa8843d1
#define CLIENT_COMPINFO2_CLIENT_ID    0x001b9dda
#define CLIENT_COMPINFO2_CLIENT_TOKEN 0xab69f79a
/******************************************************/


/******************************************************/
/*
Sent in response to CLIENT_COMPINFO[12] along with sessionkey

                          FF 05 14 00 01 00 00 00            ........
D1 43 88 AA 1C B9 48 00   31 8A F2 89                .C....H.1...
*/
#define SERVER_COMPREPLY 0x05ff
typedef struct
{
    t_normal_header h;
    bn_int          reg_version;  /* 01 00 00 00 */
    bn_int          reg_auth;     /* D1 43 88 AA */
    bn_int          client_id;    /* DA 9D 1B 00 */ /* 1C B9 48 00 */
    bn_int          client_token; /* 9A F7 69 AB */ /* 31 8A F2 89 */
} t_server_compreply;
#define SERVER_COMPREPLY_REG_VERSION  0x00000001
#define SERVER_COMPREPLY_REG_AUTH     0xaa8843d1
#define SERVER_COMPREPLY_CLIENT_ID    0x001b9dda
#define SERVER_COMPREPLY_CLIENT_TOKEN 0xab69f79a
/******************************************************/


/******************************************************/
/*
Sent in repsonse to COMPINFO1 along with COMPINFOREPLY.
Used for password hashing by the client.
*/
#define SERVER_SESSIONKEY1 0x28ff
typedef struct
{
    t_normal_header h;
    bn_int          sessionkey;
} t_server_sessionkey1;
/******************************************************/


/******************************************************/
/*
Sent in repsonse to COMPINFO2 along with COMPINFOREPLY.
Used for password hashing by the client.

                          FF 1D 0C 00 40 24 02 00            ....@$..
99 F3 FD 78                                          ...x

FF 1D 0C 00 0C 67 08 00   7A 3C D8 75                .....g..z<.u

FF 1D 0C 00 58 77 00 00   27 45 44 7A                ....Xw..'EDz
*/
#define SERVER_SESSIONKEY2 0x1dff
typedef struct
{
    t_normal_header h;
    bn_int          unknown1; /* F3 4D 00 00 */ /* 40 24 02 00 */ /* 0C 67 08 00 */ /* based on IP somehow? */
    bn_int          sessionkey;
} t_server_sessionkey2;
#define SERVER_SESSIONKEY2_UNKNOWN1 0x00004df3
/******************************************************/


/******************************************************/
/* maybe this is where the timezone is...

FF 12 3C 00 E0 28 02 E4   0A 37 BE 01 E0 50 A3 37    ..<..(...7...P.7
D0 36 BE 01 A4 01 00 00   09 04 00 00 09 04 00 00    .6..............
09 04 00 00 65 6E 75 00   31 00 55 53 41 00 55 6E    ....enu.1.USA.Un
69 74 65 64 20 53 74 61   74 65 73 00                ited States.

still original client, but at a later date
FF 12 3C 00 60 C5 4B 8B   19 DE BE 01 60 55 B1 40    ..<.`.K.....`U.@
E7 DD BE 01 A4 01 00 00   09 04 00 00 09 04 00 00    ................
09 04 00 00 65 6E 75 00   31 00 55 53 41 00 55 6E    ....enu.1.USA.Un
69 74 65 64 20 53 74 61   74 65 73 00                ited States.

FF 12 3C 00 60 EA 02 23   F5 DE BE 01 60 7A 68 D8    ..<.`..#....`zh.
C2 DE BE 01 A4 01 00 00   09 04 00 00 09 04 00 00    ................
09 04 00 00 65 6E 75 00   31 00 55 53 41 00 55 6E    ....enu.1.USA.Un
69 74 65 64 20 53 74 61   74 65 73 00                ited States.

                                      FF 12 35 00                ..5.
20 BA B0 55 F2 7B BE 01   20 62 98 C5 3D 7C BE 01     ..U.{.. b..=|..
E4 FD FF FF 12 04 00 00   12 04 00 00 12 04 00 00    ................
6B 6F 72 00 38 32 00 4B   4F 52 00 4B 6F 72 65 61    kor.82.KOR.Korea
00                                                   .
*/
#define CLIENT_COUNTRYINFO 0x12ff
typedef struct
{
    t_normal_header h;
    bn_long         timestamp1; /* FIXME: current time? */ /* FIXME: is this how timezone is determined? */
    bn_long         timestamp2; /* FIXME: current time? */
    bn_int          unknown5; /* A4 01 00 00 */   /* E4 FD FF FF */  /* A4 01 00 00 */
    bn_int          langid1;  /* 09 04 00 00 */   /* 12 04 00 00 */
    bn_int          langid2;  /* 09 04 00 00 */   /* 12 04 00 00 */
    bn_int          langid3;  /* 09 04 00 00 */   /* 12 04 00 00 */
    /* langstr */
    /* countrycode (long distance phone) */
    /* countryabbrev */
    /* countryname */
} t_client_countryinfo;
#define CLIENT_COUNTRYINFO_TIMESTAMP1_A                         0xe40228e0
#define CLIENT_COUNTRYINFO_TIMESTAMP1_B                         0x01be370a
#define CLIENT_COUNTRYINFO_TIMESTAMP2_A                         0x37a350e0
#define CLIENT_COUNTRYINFO_TIMESTAMP2_B                         0x01be36d0
#define CLIENT_COUNTRYINFO_UNKNOWN5                             0x000001a4
#define CLIENT_COUNTRYINFO_LANGID_USENGLISH                     0x00000409
#define CLIENT_COUNTRYINFO_LANGID_KOREAN                        0x00000412
#define CLIENT_COUNTRYINFO_LANGSTR_USENGLISH                    "enu"
#define CLIENT_COUNTRYINFO_LANGSTR_KOREAN                       "kor"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_USA                        "1"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_CIS                        "7"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_KAZAKHSTAN                 "7"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_KYRGYSTAN                  "7"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_RUSSIA                     "7"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_TAJIKISTAN                 "7"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_UZBEKISTAN                 "7"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_EGYPT                     "20"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SOUTH_AFRICA              "27"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_GREECE                    "30"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NETHERLANDS               "31"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BELGIUM                   "32"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_FRANCE                    "33"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SPAIN                     "34" /* including Baleric Islands */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_HUNGARY                   "36"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_YUGOSLAVIA                "38"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ITALY                     "39"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ROMANIA                   "40"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_LIECHTENSTEIN             "41"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SWITZERLAND               "41"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_CZECHOSLOVAKIA            "42"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_AUSTRIA                   "43"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_UNITED_KINGDOM            "44"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_DENMARK                   "45"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SWEDEN                    "46"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NORWAY                    "47"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_POLAND                    "48"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_GERMANY                   "49"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_PERU                      "51"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MEXICO                    "52"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_CUBA                      "53"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_GUANTANAMO_BAY            "53"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ARGENTINA                 "54"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BRAZIL                    "55"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_CHILE                     "56"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_COLOMBIA                  "57"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_VENEZUELA                 "58"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MALAYSIA                  "60"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_AUSTRALIA                 "61"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_INDONESIA                 "62"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_PHILIPPINES               "63"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NEW_ZEALAND               "64"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SINGAPORE                 "65"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_JAPAN                     "81"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_THAILAND                  "66"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SOUTH_KOREA               "82"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_VIETNAM                   "84"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_CHINA                     "86"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_TURKEY                    "90"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_INDIA                     "91"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BELEM                     "91"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_PAKISTAN                  "92"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_AFGHANISTAN               "93"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SRI_LANKA                 "94"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MYANMAR                   "95" /* Burma */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_IRAN                      "98"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MOROCCO                  "212"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ALGERIA                  "213"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_TUNISIA                  "216"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_LIBYA                    "218"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_GAMBIA                   "220"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SENEGAL                  "221"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MAURITANIA               "222"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MALI                     "223"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_GUINEA                   "224"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_IVORY_COAST              "225"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BURKINA_FASO             "226" /* Upper Volta */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NIGER                    "227"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_TOGO                     "228"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BENIN                    "229"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MAURITIUS                "230"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_LIBERIA                  "231"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SIERRA_LEONE             "232"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_GHANA                    "233"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NIGERIA                  "234"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_CHAD                     "235"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_CENTRAL_AFRICAN_REPUBLIC "236"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_CAMEROON                 "237"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_CAPE_VERDE               "238"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SAO_TOME_AND_PRINCIPE    "239"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_EQUATORIAL_GUINEA        "240"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_GABON                    "241"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_CONGO                    "242"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ZAIRE                    "243" /* Republic of the Congo */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ANGOLA                   "244"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_GUINEA_BISSAU            "245"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_DIEGO_GARCIA             "246"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ASCENSION_ISLAND         "247"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SEYCHELLES               "248"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SUDAN                    "249"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_RWANDA                   "250"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ETHIOPIA                 "251"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SOMALIA                  "252"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_DJIBOUTI                 "253"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_KENYA                    "254"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_TANZANIA                 "255"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_UGANDA                   "256"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BURUNDI                  "257"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MOZAMBIQUE               "258"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ZANZIBAR                 "255" /* should be 259 */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ZAMBIA                   "260"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MADAGASCAR               "261"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_REUNION                  "262"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ZIMBABWE                 "263"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NAMIBIA                  "264" /* South-West Africa */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MALAWI                   "265"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_LESOTHO                  "266"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BOTSWANA                 "267"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SWAZILAND                "268"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_COMOROS                  "269" /* no workie */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MAYOTTE                  "269"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ST_HELENA                "290"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SAN_MARINO               "295" /* should be 295 */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_TRINIDAD                 "296" /* should be 296 */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_TOBAGO                   "296" /* should be 296 */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ARUBA                    "297"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_FAROE_ISLANDS            "298" /* Faeroe */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_GREENLAND                "299"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_GIBRALTAR                "350"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_PORTUGAL                 "351" /* including Azores */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_LUXEMBOURG               "352"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_IRELAND                  "353"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ICELAND                  "354"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ALBANIA                  "355"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MALTA                    "356"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_CYPRUS                   "357"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_FINLAND                  "358"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BULGARIA                 "359"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_LITHUANIA                "370"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_LATVIA                   "371"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MOLDOVA                  "373"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ARMENIA                  "374"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BELARUS                  "375"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MONACO                   "377"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_UKRAINE                  "380"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SERBIA                   "381"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_CROATIA                  "385"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BOSNIA                   "387"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_HERZEGOVINA              "387"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_FYROM                    "389" /* Macedonia */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_FALKLAND_ISLANDS         "500"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BELIZE                   "501"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_GUATEMALA                "502"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_EL_SALVADOR              "503"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_HONDURAS                 "504"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NICARAGUA                "505"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_COSTA_RICA               "506"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_PANAMA                   "507"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ST_PIERRE                "508"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MIQUELON                 "508"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_HAITI                    "509"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_FRENCH_ANTILLES          "590" /* St. Barthelemy, Buadeloupe, and the French side of St. Martin */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BOLIVIA                  "591"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_GUYANA                   "592"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ECUADOR                  "593"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_FRENCH_GUIANA            "594"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_PARAGUAY                 "595"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MARTINIQUE               "596" /* French Antilles */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SURINAME                 "597"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_URUGUAY                  "598"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NETHERLANDS_ANTILLES     "599"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NORTH_MARIANA_ISLANDS    "670" /* Saipan */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_GUAM                     "671"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ANTARCTICA               "672"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_AUSTRALIAN_EXTERNAL_TERRITORIES "672"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BRUNEI_DARUSSALM         "673"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NAURU                    "674"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_PAPUA_NEW_GUINEA         "675"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_TONGA                    "676"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SOLOMON_ISLANDS          "677"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_VANUATU                  "678" /* New Hebrides */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_FIJI                     "679"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_PALAU                    "680"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_WALLIS_AND_FUTUNA        "681"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_COOK_ISLANDS             "682"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NIUE                     "683"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_AMERICAN_SAMOA           "684"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_WESTERN_SAMOA            "685"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_KIRIBATI_REPUBLIC        "686" /* Gilbert Islands */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NEW_CALEDONIA            "687"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_TUVALU                   "688" /* Ellice Islands */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_TAHITI                   "689" /* French Polynesia */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_TOKELAU                  "690"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MICRONESIA               "691"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MARSHALL_ISLANDS         "692"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MIDWAY_ISLANDS           "808"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ANGUILLA                 "809"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ANTIGUA                  "809"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BARBUDA                  "809"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BAHAMAS                  "809"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BARBADOS                 "809"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BERMUDA                  "809"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BRITISH_VIRGIN_ISLANDS   "809"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NORTH_KOREA              "850"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_HONG_KONG                "852"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MACAO                    "853"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_KHMER                    "855" /* Cambodia, Kampuchea */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_LAOS                     "856"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MARISAT_ATLANTIC         "871" /* including Caribbean and Mediterranean */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MARISAT_PACIFIC          "872"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MARISAT_INDIAN           "873"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ATLANTIC_WEST            "874" /* overlaps 871 */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BANGLADESH               "880"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_TAIWAN                   "886"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MALDIVES                 "960"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_LEBANON                  "961"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_JORDAN                   "962"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SYRIA                    "963"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_IRAQ                     "964"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_KUWAIT                   "965"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_SAUDI_ARABIA             "966"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NORTH_YEMEN              "967"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_OMAN                     "968"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_YEMEN                    "969" /* Aden */
#define CLIENT_COUNTRYINFO_COUNTRYCODE_UNITED_ARAB_EMIRATES     "971"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_ISRAEL                   "972"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BAHRAIN                  "973"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_QATAR                    "974"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_BHUTAN                   "975"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_MONGOLIA                 "976"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_NEPAL                    "977"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_TURKMENISTAN             "993"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_AZERBAIJAN               "994"
#define CLIENT_COUNTRYINFO_COUNTRYCODE_GEORGIA                  "995"
#define CLIENT_COUNTRYINFO_COUNTRYABB_USA                       "USA"
#define CLIENT_COUNTRYINFO_COUNTRYABB_KOR                       "KOR"
#define CLIENT_COUNTRYINFO_COUNTRYNAME_USA                      "United States"
#define CLIENT_COUNTRYINFO_COUNTRYNAME_KOR                      "Korea"
/******************************************************/


/******************************************************/
#define CLIENT_CREATEACCTREQ 0x2aff
typedef struct
{
    t_normal_header h;
    bn_int          password_hash1[5]; /* hash of lowercase password w/o null */
    /* player name */
} t_client_createacctreq;
/******************************************************/


/******************************************************/
/*
                          FF 2A 18 00 01 00 00 00            .*......
13 00 00 00 78 52 82 02   00 00 00 00 00 00 00 00    ................
---120 82 130 2---
*/
#define SERVER_CREATEACCTREPLY 0x2aff
typedef struct
{
    t_normal_header h;
    bn_int          result;
    bn_int          unknown1;
    bn_int          unknown2;
    bn_int          unknown3;
    bn_int          unknown4;
} t_server_createacctreply;
#define SERVER_CREATEACCTREPLY_RESULT_OK 0x00000001
#define SERVER_CREATEACCTREPLY_RESULT_NO 0x00000000
#define SERVER_CREATEACCTREPLY_UNKNOWN1  0x00000013
#define SERVER_CREATEACCTREPLY_UNKNOWN2  0x02825278
#define SERVER_CREATEACCTREPLY_UNKNOWN3  0x00000010
#define SERVER_CREATEACCTREPLY_UNKNOWN4  0x00000010
/******************************************************/


/******************************************************/
/*
                          FF 2B 20 00 01 00 00 00            .+ .....
00 00 00 00 4D 00 00 00   0E 01 00 00 20 00 00 00    ....M....... ...
CE 01 00 00 DD 07 00 00                              ........

   FF 2B 20 00 01 00 00   00 00 00 00 00 06 00 00    ..+ ............
00 72 01 00 00 40 00 00   00 A9 07 00 00 FF 07 00    .r...@..........
00                                                   .

from Starcraft 1.05
FF 2B 20 00 01 00 00 00   00 00 00 00 06 00 00 00    .+ .............
7C 01 00 00 20 00 00 00   00 02 00 00 FF 07 00 00    |... ...........
*/
#define CLIENT_UNKNOWN_2B 0x2bff /* FIXME: what is this? */
typedef struct
{
    t_normal_header h;
    bn_int          unknown1; /* 01 00 00 00 */ /* 01 00 00 00 */
    bn_int          unknown2; /* 00 00 00 00 */ /* 00 00 00 00 */
    bn_int          unknown3; /* 4D 00 00 00 */ /* 06 00 00 00 */
    bn_int          unknown4; /* 0E 01 00 00 */ /* 72 01 00 00 */
    bn_int          unknown5; /* 20 00 00 00 */ /* 40 00 00 00 */
    bn_int          unknown6; /* CE 01 00 00 */ /* A9 07 00 00 */
    bn_int          unknown7; /* DD 07 00 00 */ /* FF 07 00 00 */
} t_client_unknown_2b;
#define CLIENT_UNKNOWN_2B_UNKNOWN1 0x00000001
#define CLIENT_UNKNOWN_2B_UNKNOWN2 0x00000000
#define CLIENT_UNKNOWN_2B_UNKNOWN3 0x0000004d
#define CLIENT_UNKNOWN_2B_UNKNOWN4 0x0000010e
#define CLIENT_UNKNOWN_2B_UNKNOWN5 0x00000020
#define CLIENT_UNKNOWN_2B_UNKNOWN6 0x000001ce
#define CLIENT_UNKNOWN_2B_UNKNOWN7 0x000007dd
/******************************************************/


/******************************************************/
/*
FIXME: how does the versionid work?

                          FF 06 14 00 36 38 58 49            ....68XI
50 58 45 53 BB 00 00 00   00 00 00 00                PXES........ 

sent by 1.05 Starcraft
FF 06 14 00 36 38 58 49   52 41 54 53 BD 00 00 00    ....68XIRATS....
00 00 00 00                                          ....
*/
#define CLIENT_PROGIDENT 0x06ff
typedef struct
{
    t_normal_header h;
    bn_int          archtag;
    bn_int          clienttag;
    bn_int          versionid;
    bn_int          unknown1;
} t_client_progident;
#define CLIENT_PROGIDENT_UNKNOWN1 0x00000000
/******************************************************/


/******************************************************/
/*
These formulas seem to be for authenticating the server. 

                          FF 06 5A 00 00 86 BA E3            ..Z.....
09 28 BC 01 49 58 38 36   76 65 72 32 2E 6D 70 71    .(..IX86ver2.mpq
00 41 3D 32 30 31 39 34   39 38 38 39 39 20 42 3D    .A=2019498899 B=
33 34 32 33 32 39 32 33   39 34 20 43 3D 31 37 31    3423292394 C=171
39 30 31 31 32 32 32 20   34 20 41 3D 41 5E 53 20    9011222 4 A=A^S
42 3D 42 2D 43 20 43 3D   43 5E 41 20 41 3D 41 5E    B=B-C C=C^A A=A^
42 00                                                B.

FF 06 59 00 00 C1 12 EC   09 28 BC 01 49 58 38 36    ..Y......(..IX86
76 65 72 35 2E 6D 70 71   00 41 3D 31 38 37 35 35    ver5.mpq.A=18755
39 31 33 34 31 20 42 3D   32 34 39 31 30 39 39 38    91341 B=24910998
30 39 20 43 3D 36 33 34   38 35 36 36 30 34 20 34    09 C=634856604 4
20 41 3D 41 2D 53 20 42   3D 42 5E 43 20 43 3D 43     A=A-S B=B^C C=C
2B 41 20 41 3D 41 5E 42   00                         +A A=A^B.

FF 06 5A 00 00 C1 12 EC   09 28 BC 01 49 58 38 36    ..Z......(..IX86
76 65 72 35 2E 6D 70 71   00 41 3D 31 37 31 32 39    ver5.mpq.A=17129
34 38 34 32 36 20 42 3D   33 36 30 30 30 33 30 36    48426 B=36000306
30 37 20 43 3D 33 33 39   30 34 31 37 39 35 39 20    07 C=3390417959 
34 20 41 3D 41 2D 53 20   42 3D 42 5E 43 20 43 3D    4 A=A-S B=B^C C=
43 2D 41 20 41 3D 41 2D   42 00                      C-A A=A-B.

FF 06 5A 00 00 3A 7F E8   09 28 BC 01 49 58 38 36    ..Z..:...(..IX86
76 65 72 34 2E 6D 70 71   00 41 3D 31 31 38 36 39    ver4.mpq.A=11869
35 38 31 34 31 20 42 3D   31 33 37 37 34 34 31 34    58141 B=13774414
35 37 20 43 3D 31 37 37   32 37 38 37 37 30 35 20    57 C=1772787705
34 20 41 3D 41 5E 53 20   42 3D 42 5E 43 20 43 3D    4 A=A^S B=B^C C=
43 2B 41 20 41 3D 41 5E   42 00                      C+A A=A^B.
*/
#define SERVER_AUTHREQ 0x06ff
typedef struct
{
    t_normal_header h;
    bn_long         timestamp; /* FIXME: file modification time? */
    /* file */
    /* equation */
} t_server_authreq;
/*#define SERVER_AUTHREQ_FILE "IX86ver5.mpq"*/
#define SERVER_AUTHREQ_EQN "A=1712948426 B=3600030607 C=3390417959 4 A=A-S B=B^C C=C-A A=A-B"
/******************************************************/


/******************************************************/
/*
FF 07 40 00 36 38 58 49   52 41 54 53 BD 00 00 00    ..@.68XIRATS....
00 05 00 01 1E 88 D7 08   73 74 61 72 63 72 61 66    ........starcraf
74 2E 65 78 65 20 30 33   2F 30 38 2F 39 39 20 32    t.exe 03/08/99 2
32 3A 34 31 3A 35 30 20   31 30 34 32 34 33 32 00    2:41:50 1042432.

sent by the 1.05 Starcraft
FF 07 40 00 36 38 58 49   52 41 54 53 BD 00 00 00    ..@.68XIRATS....
00 05 00 01 AE AC DE 87   73 74 61 72 63 72 61 66    ........starcraf
74 2E 65 78 65 20 30 33   2F 30 38 2F 39 39 20 32    t.exe 03/08/99 2
32 3A 34 31 3A 35 30 20   31 30 34 32 34 33 32 00    2:41:50 1042432.

sent by the 1.07 Diablo
FF 07 3C 00 36 38 58 49   4C 54 52 44 26 00 00 00    ..<.68XILTRD&...
01 06 05 62 8C 56 E6 21   64 69 61 62 6C 6F 2E 65    ...b.V.!diablo.e
78 65 20 30 39 2F 31 37   2F 39 38 20 31 38 3A 30    xe 09/17/98 18:0
30 3A 34 30 20 37 36 30   33 32 30 00                0:40 760320.
*/
#define CLIENT_AUTHREQ 0x07ff
typedef struct
{
    t_normal_header h;
    bn_int          archtag;
    bn_int          clienttag;
    bn_int          versionid;
    bn_byte         v4;
    bn_byte         v3;
    bn_byte         v2;
    bn_byte         v1;
    bn_int          answer; /* 1E 88 D7 08 */ /* AE AC DE 87 */ /* 5E BE 40 08 */
    /* executable info */
} t_client_authreq;
#define CLIENT_AUTHREQ_ANSWER     0x87deacae
/*                                   executable file    GMT date and time    size in bytes */
#define CLIENT_AUTHREQ_EXEINFO_DRTL "diablo.exe 09/17/98 18:00:40 760320"
#define CLIENT_AUTHREQ_EXEINFO_STAR "starcraft.exe 03/08/99 22:41:50 1042432"
#define CLIENT_AUTHREQ_EXEINFO_SSHR "starcraft.exe 03/08/99 22:41:50 1042432" /* FIXME: wrong */
#define CLIENT_AUTHREQ_EXEINFO_SEXP "starcraft.exe 03/08/99 22:41:50 1042432" /* FIXME: wrong */
/******************************************************/


/******************************************************/
/*
                          FF 07 0A 00 02 00 00 00            ........
00 00                                                ..

FF 07 0A 00 02 00 00 00   00 00                      ..........        
*/
#define SERVER_AUTHREPLY 0x07ff
typedef struct
{
    t_normal_header h;
    bn_int          message;
    /* filename */
    /* unknown */
} t_server_authreply;
#define SERVER_AUTHREPLY_MESSAGE_BADVERSION 0x00000000
#define SERVER_AUTHREPLY_MESSAGE_UPDATE     0x00000001 /* initiate auto-update */
#define SERVER_AUTHREPLY_MESSAGE_OK         0x00000002
/* anything other than these is considered to be ok */
/******************************************************/


/******************************************************/
/*
FF 07 0A 00 02 00 00 00   00 00                      ..........
*/
#define CLIENT_ICONREQ 0x2dff
typedef struct
{
    t_normal_header h;
} t_client_iconreq;
/******************************************************/


/******************************************************/
/*
                          FF 2D 16 00 76 34 1F 8F            .-..v4..
C0 D6 BD 01 69 63 6F 6E   73 2E 62 6E 69 00          ....icons.bni.

FF 2D 16 00 00 77 D0 01   C7 B1 BE 01 69 63 6F 6E    .-...w......icon
73 2E 62 6E 69 00                                    s.bni.
*/
#define SERVER_ICONREPLY 0x2dff
typedef struct
{
    t_normal_header h;
    bn_long         timestamp; /* FIXME: file modification time? */
    /* file */
} t_server_iconreply;
/******************************************************/


/******************************************************/
#define CLIENT_LADDERSEARCHREQ 0x2fff
typedef struct
{
    t_normal_header h;
    bn_int          clienttag;
    bn_int          namecount;
    bn_int          type;
    /* player name */
} t_client_laddersearchreq;
#define CLIENT_LADDERSEARCHREQ_TYPE_HIGHESTRATED 0x00000000
#define CLIENT_LADDERSEARCHREQ_TYPE_MOSTWINS     0x00000002
#define CLIENT_LADDERSEARCHREQ_TYPE_MOSTGAMES    0x00000003
/******************************************************/


/******************************************************/
#define SERVER_LADDERSEARCHREPLY 0x2fff
typedef struct /* FIXME: how does client know how many names?
		  do we send separate replies for each name in the request? */
{
    t_normal_header h;
    bn_int          rank; /* 0 means 1st, etc */
} t_server_laddersearchreply;
#define SERVER_LADDERSEARCHREPLY_RANK_NONE 0xffffffff
/******************************************************/


/******************************************************/
/*
                          FF 30 1C 00 00 00 00 00            .0......
32 37 34 34 37 37 32 39   31 34 38 32 38 00 63 6C    2744772914828.cl
6F 75 64 00                                          oud.
*/
#define CLIENT_CDKEY 0x30ff
typedef struct
{
    t_normal_header h;
    bn_int          unknown1;
    /* cd key */
    /* owner name */
} t_client_cdkey;
#define CLIENT_CDKEY_UNKNOWN1 0x00000000
/******************************************************/


/******************************************************/
/*
                          FF 30 0E 00 01 00 00 00            .0......
63 6C 6F 75 64 00                                    cloud.
*/
#define SERVER_CDKEYREPLY 0x30ff
typedef struct
{
    t_normal_header h;
    bn_int          message;
    /* owner name */
} t_server_cdkeyreply;
#define SERVER_CDKEYREPLY_MESSAGE_OK       0x00000001
#define SERVER_CDKEYREPLY_MESSAGE_BAD      0x00000002
#define SERVER_CDKEYREPLY_MESSAGE_WRONGAPP 0x00000003
#define SERVER_CDKEYREPLY_MESSAGE_ERROR    0x00000004
#define SERVER_CDKEYREPLY_MESSAGE_INUSE    0x00000005
/* (any other value seems to correspond to ok) */
/******************************************************/


/******************************************************/
/*
FF 36 34 00 00 00 00 00   0D 00 00 00 01 00 00 00    .64.............
B5 AE 23 00 50 E5 D5 C0   DB 55 1E 38 0A F5 58 B9    ..#.P....U.8..X.
47 64 C6 C2 9F BB FF B8   81 E7 EB EC 1B 13 C6 38    Gd.............8
52 6F 62 00                                          Rob.

FF 36 34 00 00 00 00 00   0D 00 00 00 01 00 00 00    .64.............
7F D7 00 00 90 64 77 2F   D7 5B 42 38 1F A1 A2 6F    .....dw/.[B8...o
E8 FA BE F8 B6 0B BA 0F   CA 64 3A 17 14 56 83 AB    .........d:..V..
42 6F 62 00                                          Bob.
*/
#define CLIENT_CDKEY2 0x36ff
typedef struct
{
    t_normal_header h;
    bn_int          unknown1;  /* 00 00 00 00 */
    bn_int          unknown2;  /* 0D 00 00 00 */
    bn_int          unknown3;  /* 01 00 00 00 */
    bn_int          unknown4;  /* B5 AE 23 00 */
    bn_int          unknown5;  /* 50 E5 D5 C0 */
    bn_int          unknown6;  /* DB 55 1E 38 */
    bn_int          unknown7;  /* 0A F5 58 B9 */
    bn_int          unknown8;  /* 47 64 C6 C2 */
    bn_int          unknown9;  /* 9F BB FF B8 */
    bn_int          unknown10; /* 81 E7 EB EC */
    bn_int          unknown11; /* 1B 13 C6 38 */
    /* names */
} t_client_cdkey2;
/******************************************************/


/******************************************************/
/*
FF 36 0C 00 01 00 00 00   52 6F 62 00                .6......Rob.

FF 36 0C 00 01 00 00 00   42 6F 62 00                .6......Bob.
*/
#define SERVER_CDKEYREPLY2 0x36ff
typedef struct
{
    t_normal_header h;
    bn_int          message;
    /* names */
} t_server_cdkeyreply2;
#define SERVER_CDKEYREPLY2_MESSAGE_OK       0x00000001
#define SERVER_CDKEYREPLY2_MESSAGE_BAD      0x00000002
#define SERVER_CDKEYREPLY2_MESSAGE_WRONGAPP 0x00000003
#define SERVER_CDKEYREPLY2_MESSAGE_ERROR    0x00000004
#define SERVER_CDKEYREPLY2_MESSAGE_INUSE    0x00000005
/* (any other value seems to correspond to ok) */
/******************************************************/


/******************************************************/
/*
FF 14 08 00 74 65 6E 62                              ....tenb
*/
#define CLIENT_UNKNOWN_14 0x14ff /* FIXME: What is this used for? Another ping/echo?! */
typedef struct
{
    t_normal_header h;
    bn_int          bnettag; /* tenb */
} t_client_unknown_14;
/******************************************************/


/******************************************************/
/*
FF 33 18 00 1A 00 00 00   00 00 00 00 74 6F 73 5F    .3..........tos_
55 53 41 2E 74 78 74 00                              USA.txt.        

                                      FF 33 18 00                .3..
1A 00 00 00 00 00 00 00   74 6F 73 5F 55 53 41 2E    ........tos_USA.
74 78 74 00                                          txt.
*/
#define CLIENT_TOSREQ 0x33ff
typedef struct
{
    t_normal_header h;
    bn_int          unknown1; /* 1a 00 00 00 */
    bn_int          unknown2; /* 00 00 00 00 */
    /* file */
} t_client_tosreq;
#define CLIENT_TOSREQ_UNKNOWN1 0x0000001a
#define CLIENT_TOSREQ_UNKNOWN2 0x00000000
#define CLIENT_TOSREQ_FILEUSA  "tos_USA.txt"
/******************************************************/


/******************************************************/
/*
                          FF 33 1C 00 1A 00 00 00            .3......
00 00 00 00 30 C3 89 86   09 4F BD 01 74 6F 73 2E    ....0....O..tos.
74 78 74 00                                          txt.

                          FF 33 20 00 1A 00 00 00            .3 .....
00 00 00 00 00 38 51 E2   30 A1 BD 01 74 6F 73 5F    .....8Q.0...tos_
55 53 41 2E 74 78 74 00                              USA.txt.
*/
#define SERVER_TOSREPLY 0x33ff
typedef struct
{
    t_normal_header h;
    bn_int          unknown1; /* 1a 00 00 00 */ /* same as in TOSREQ */
    bn_int          unknown2; /* 00 00 00 00 */
    bn_long         timestamp; /* FIXME: file modification time? */
    /* file */
} t_server_tosreply;
#define SERVER_TOSREPLY_UNKNOWN1 0x0000001a
#define SERVER_TOSREPLY_UNKNOWN2 0x00000000
/******************************************************/


/******************************************************/
/*
            FF 26 9E 01   01 00 00 00 13 00 00 00        .&..........
78 52 82 02 42 6F 62 00   70 72 6F 66 69 6C 65 5C    xR..Bob.profile\
73 65 78 00 70 72 6F 66   69 6C 65 5C 61 67 65 00    sex.profile\age.
70 72 6F 66 69 6C 65 5C   6C 6F 63 61 74 69 6F 6E    profile\location
00 70 72 6F 66 69 6C 65   5C 64 65 73 63 72 69 70    .profile\descrip
74 69 6F 6E 00 52 65 63   6F 72 64 5C 53 45 58 50    tion.Record\SEXP
5C 30 5C 77 69 6E 73 00   52 65 63 6F 72 64 5C 53    \0\wins.Record\S
45 58 50 5C 30 5C 6C 6F   73 73 65 73 00 52 65 63    EXP\0\losses.Rec
6F 72 64 5C 53 45 58 50   5C 30 5C 64 69 73 63 6F    ord\SEXP\0\disco
6E 6E 65 63 74 73 00 52   65 63 6F 72 64 5C 53 45    nnects.Record\SE
58 50 5C 30 5C 6C 61 73   74 20 67 61 6D 65 00 52    XP\0\last game.R
65 63 6F 72 64 5C 53 45   58 50 5C 30 5C 6C 61 73    ecord\SEXP\0\las
74 20 67 61 6D 65 20 72   65 73 75 6C 74 00 52 65    t game result.Re
63 6F 72 64 5C 53 45 58   50 5C 31 5C 77 69 6E 73    cord\SEXP\1\wins
00 52 65 63 6F 72 64 5C   53 45 58 50 5C 31 5C 6C    .Record\SEXP\1\l
6F 73 73 65 73 00 52 65   63 6F 72 64 5C 53 45 58    osses.Record\SEX
50 5C 31 5C 64 69 73 63   6F 6E 6E 65 63 74 73 00    P\1\disconnects.
52 65 63 6F 72 64 5C 53   45 58 50 5C 31 5C 72 61    Record\SEXP\1\ra
74 69 6E 67 00 52 65 63   6F 72 64 5C 53 45 58 50    ting.Record\SEXP
5C 31 5C 68 69 67 68 20   72 61 74 69 6E 67 00 52    \1\high rating.R
65 63 6F 72 64 5C 53 45   58 50 5C 31 5C 72 61 6E    ecord\SEXP\1\ran
6B 00 52 65 63 6F 72 64   5C 53 45 58 50 5C 31 5C    k.Record\SEXP\1\
68 69 67 68 20 72 61 6E   6B 00 52 65 63 6F 72 64    high rank.Record
5C 53 45 58 50 5C 31 5C   6C 61 73 74 20 67 61 6D    \SEXP\1\last gam
65 00 52 65 63 6F 72 64   5C 53 45 58 50 5C 31 5C    e.Record\SEXP\1\
6C 61 73 74 20 67 61 6D   65 20 72 65 73 75 6C 74    last game result
00 00                                                ..

            FF 26 C2 01   05 00 00 00 13 00 00 00        .&..........
EE E4 84 03 6E 73 6C 40   63 6C 6F 75 64 00 63 6C    ....nsl@cloud.cl
6F 75 64 00 67 75 65 73   74 00 48 65 72 6E 40 73    oud.guest.Hern@s
65 65 6D 65 00 6F 72 69   6F 6E 40 00 70 72 6F 66    eeme.orion@.prof
69 6C 65 5C 73 65 78 00   70 72 6F 66 69 6C 65 5C    ile\sex.profile\
61 67 65 00 70 72 6F 66   69 6C 65 5C 6C 6F 63 61    age.profile\loca
74 69 6F 6E 00 70 72 6F   66 69 6C 65 5C 64 65 73    tion.profile\des
63 72 69 70 74 69 6F 6E   00 52 65 63 6F 72 64 5C    cription.Record\
53 74 61 72 5C 30 5C 77   69 6E 73 00 52 65 63 6F    Star\0\wins.Reco
72 64 5C 53 74 61 72 5C   30 5C 6C 6F 73 73 65 73    rd\Star\0\losses
00 52 65 63 6F 72 64 5C   53 74 61 72 5C 30 5C 64    .Record\Star\0\d
69 73 63 6F 6E 6E 65 63   74 73 00 52 65 63 6F 72    isconnects.Recor
64 5C 53 74 61 72 5C 30   5C 6C 61 73 74 20 67 61    d\Star\0\last ga
6D 65 00 52 65 63 6F 72   64 5C 53 74 61 72 5C 30    me.Record\Star\0
5C 6C 61 73 74 20 67 61   6D 65 20 72 65 73 75 6C    \last game resul
74 00 52 65 63 6F 72 64   5C 53 74 61 72 5C 31 5C    t.Record\Star\1\
77 69 6E 73 00 52 65 63   6F 72 64 5C 53 74 61 72    wins.Record\Star
5C 31 5C 6C 6F 73 73 65   73 00 52 65 63 6F 72 64    \1\losses.Record
5C 53 74 61 72 5C 31 5C   64 69 73 63 6F 6E 6E 65    \Star\1\disconne
63 74 73 00 52 65 63 6F   72 64 5C 53 74 61 72 5C    cts.Record\Star\
31 5C 72 61 74 69 6E 67   00 52 65 63 6F 72 64 5C    1\rating.Record\
53 74 61 72 5C 31 5C 68   69 67 68 20 72 61 74 69    Star\1\high rati
6E 67 00 52 65 63 6F 72   64 5C 53 74 61 72 5C 31    ng.Record\Star\1
5C 72 61 6E 6B 00 52 65   63 6F 72 64 5C 53 74 61    \rank.Record\Sta
72 5C 31 5C 68 69 67 68   20 72 61 6E 6B 00 52 65    r\1\high rank.Re
63 6F 72 64 5C 53 74 61   72 5C 31 5C 6C 61 73 74    cord\Star\1\last
20 67 61 6D 65 00 52 65   63 6F 72 64 5C 53 74 61     game.Record\Sta
72 5C 31 5C 6C 61 73 74   20 67 61 6D 65 20 72 65    r\1\last game re
73 75 6C 74 00 00                                    sult..
*/
#define CLIENT_STATSREQ 0x26ff
typedef struct
{
    t_normal_header h;
    bn_int          name_count;
    bn_int          key_count;
    bn_int          unknown1; /* 78 52 82 02 */
    /* player name */
    /* field key ... */
} t_client_statsreq;
#define CLIENT_STATSREQ_UNKNOWN1 0x02825278
/******************************************************/


/******************************************************/
/*
                          FF 26 23 00 01 00 00 00            .&#.....
13 00 00 00 78 52 82 02   00 00 00 00 00 00 00 00    ....xR..........
00 00 00 00 00 00 00 00   00 00 00 00 00 00          ..............

            FF 26 13 02   05 00 00 00 13 00 00 00        .&..........
EE E4 84 03 20 20 A1 F0   00 20 20 A1 F0 00 68 74    ....  ...  ...ht
74 70 3A 2F 2F 6E 73 6C   2E 6B 6B 69 72 69 2E 6F    tp://nsl.kkiri.o
72 67 00 20 20 20 20 20   20 20 20 20 20 20 A2 CB    rg.           ..
20 50 72 6F 74 6F 73 73   20 69 73 20 54 68 65 20     Protoss is The 
42 65 73 74 20 A2 CB 20   0D 0A 0D 0A 20 20 20 49    Best .. ....   I
66 20 59 6F 75 20 57 61   6E 74 20 54 6F 20 4B 6E    f You Want To Kn
6F 77 20 41 62 6F 75 74   20 55 73 2C 20 47 6F 0D    ow About Us, Go.
0A 0D 0A 20 20 20 20 20   20 20 20 20 20 20 20 20    ...             
20 68 74 74 70 3A 2F 2F   6E 73 6C 2E 6B 6B 69 72     http://nsl.kkir
69 2E 6F 72 67 00 38 38   00 37 30 00 33 00 32 39    i.org.88.70.3.29
32 36 30 33 31 30 20 33   39 36 31 36 37 38 32 34    260310 396167824
30 00 4C 4F 53 53 00 30   00 30 00 30 00 30 00 00    0.LOSS.0.0.0.0..
00 00 00 00 00 6D 00 31   35 00 53 69 6E 67 61 70    .....m.15.Singap
6F 72 65 20 00 00 32 37   00 31 00 36 00 32 39 32    ore ..27.1.6.292
35 34 32 33 37 20 32 34   35 37 32 33 30 39 38 00    54237 245723098.
44 52 41 57 00 30 00 30   00 30 00 30 00 00 00 00    DRAW.0.0.0.0....
00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00    ................
00 00 00 00 00 00 00 00   00 00 31 00 00 00 32 39    ..........1...29
32 35 39 38 31 32 20 31   31 35 33 38 33 37 39 34    259812 115383794
36 00 57 49 4E 00 30 00   30 00 30 00 30 00 00 00    6.WIN.0.0.0.0...
00 00 00 00 B0 C5 BD C3   B1 E2 00 3F 3F 00 B1 D9    ...........??...
B0 C5 C1 F6 20 BE F8 C0   BD 2E 00 BA B0 C0 DA B8    .... ...........
AE 20 6F 72 69 6F 6E 20   2C 2C 20 C3 CA C4 DA C6    . orion ,, .....
C4 C0 CC B0 A1 20 BE C6   B3 E0 BF EB 2E 2E 0D 0A    ..... ..........
C4 ED C4 ED 2E 2E 0D 0A   0D 0A C1 B9 B6 F3 20 C0    .............. .
DF C7 CF B4 C2 20 B3 D1   20 3A 20 6E 73 6C B3 D1    ..... .. : nsl..
B5 E9 0D 0A C0 DF C7 CF   B4 C2 20 C7 C1 C5 E4 20    .......... .... 
3A 20 6E 73 6C 40 74 6F   74 6F 72 6F 00 35 31 00    : nsl@totoro.51.
34 38 00 36 00 32 39 32   35 39 32 36 33 20 39 35    48.6.29259263 95
36 35 30 35 39 30 32 00   4C 4F 53 53 00 36 00 36    6505902.LOSS.6.6
00 32 00 39 39 30 00 31   30 32 37 00 00 00 32 39    .2.990.1027...29
32 35 39 32 35 38 20 33   31 32 32 37 39 38 38 32    259258 312279882
00 4C 4F 53 53 00 00                                 .LOSS..
*/
#define SERVER_STATSREPLY 0x26ff
typedef struct
{
    t_normal_header h;
    bn_int          name_count;
    bn_int          key_count;
    bn_int          unknown1; /* 78 52 82 02 */ /* EE E4 84 03 */ /* same as request */
    /* field values ... */
} t_server_statsreply;
/******************************************************/


/******************************************************/
/*
FF 29 25 00 CF 17 28 00   A3 D3 2C 5C F4 18 02 40    .)%...(...,\...@
F9 B8 EA F4 A5 B1 3F 39   85 89 2D DB 18 2D B9 D4    ......?9..-..-..
52 6F 73 73 00                                       Ross.
*/
#define CLIENT_LOGINREQ 0x29ff
typedef struct
{
    t_normal_header h;
    bn_int          ticks;
    bn_int          sessionkey;
    bn_int          password_hash2[5]; /* hash of ticks, key, and hash1 */
    /* player name */
} t_client_loginreq;
/******************************************************/


/******************************************************/
#define SERVER_LOGINREPLY 0x29ff
typedef struct
{
    t_normal_header h;
    bn_int          message;
} t_server_loginreply;
#define SERVER_LOGINREPLY_MESSAGE_FAIL    0x00000000
#define SERVER_LOGINREPLY_MESSAGE_SUCCESS 0x00000001
/******************************************************/


/******************************************************/
/*
FF 31 3B 00 22 1A 9A 00   64 B7 C5 21 2C 82 57 F4    .1;."...d..!,.W.
0A 36 73 25 1E A5 42 5F   FA 36 54 97 BC 65 3F E1    .6s%..B_.6T..e?.
7D 5A 54 17 4C 33 B9 1A   09 25 49 45 99 52 69 45    }ZT.L3...%IE.RiE
B1 E6 5C 9C 77 72 73 70   6F 69 00                   ..\.wrspoi.

FF 31 3B 00 3A 5B 9B 00   64 B7 C5 21 99 32 14 B7    .1;.:[..d..!.2..
89 02 3C 28 4A 75 84 05   70 EF B5 A7 99 CA 7E 12    ..<(Ju..p.....~.
7D 5A 54 17 4C 33 B9 1A   09 25 49 45 99 52 69 45    }ZT.L3...%IE.RiE
B1 E6 5C 9C 77 72 73 70   6F 69 00                   ..\.wrspoi.
*/
#define CLIENT_CHANGEPASSREQ 0x31ff
typedef struct
{
    t_normal_header h;
    bn_int          ticks;
    bn_int          sessionkey;
    bn_int          oldpassword_hash2[5]; /* hash of ticks, key, hash1 */
    bn_int          newpassword_hash1[5]; /* hash of lowercase password w/o null */
    /* player name */
} t_client_changepassreq;
/******************************************************/


/******************************************************/
/*
FF 31 08 00 01 00 00 00                              .1......

FF 31 08 00 00 00 00 00                              .1......
*/
#define SERVER_CHANGEPASSACK 0x31ff
typedef struct
{
    t_normal_header h;
    bn_int          message;
} t_server_changepassack;
#define SERVER_CHANGEPASSACK_MESSAGE_FAIL    0x00000000
#define SERVER_CHANGEPASSACK_MESSAGE_SUCCESS 0x00000001
/******************************************************/


/******************************************************/
/*
                         FF 0A 0F 00 0F                    .....
                         4D 79 41 63 63 6F 75 6E           MyAccoun
74 00 00                                                   t..
*/
#define CLIENT_PLAYERINFOREQ 0x0aff
typedef struct
{
    t_normal_header h;
    /* player name */
    /* player info */ /* used by Diablo */
} t_client_playerinforeq;
/******************************************************/


/******************************************************/
/*
                          FF 0A 29 00 4D 79 41 63            ..).MyAc
63 6F 75 6E 74 00 50 58   45 53 20 30 20 30 20 30    count.PXES 0 0 0
20 30 20 30 20 30 00 4D   79 41 63 63 6F 75 6E 74     0 0 0.MyAccount
00                                                   .

FF 0A 2F 00 6C 61 77 75   65 66 00 4C 54 52 44 20    ../.lawuef.LTRD
31 20 30 20 30 20 33 30   20 31 30 20 32 30 20 32    1 0 0 30 10 20 2
35 20 31 30 30 20 30 00   6C 61 77 75 65 66 00       5 100 0.lawuef.
*/
#define SERVER_PLAYERINFOREPLY 0x0aff
typedef struct
{
    t_normal_header h;
    /* player name */
    /* status */
    /* player name?! (maybe character name?) */
} t_server_playerinforeply;
/*
 * status string:
 *
 * for STAR, SEXP, SSHR:
 * "%s %u %u %u %u %u"
 *  client tag (RATS, PXES, RHSS)
 *  rating
 *  number (ladder rank)
 *  stars  (normal wins)
 *  unknown3 (always zero?)
 *  unknown4 (always zero?) FIXME: I don't see this last one in any dumps...
                                   is this only a SEXP thing?
 *
 * for DRTL:
 * "%s %u %u %u %u %u %u %u %u %u"
 *  client tag (LTRD)
 *  level
 *  class (0==warrior, 1==rogue, 2==sorcerer)
 *  dots (times killed diablo)
 *  strength
 *  magic
 *  dexterity
 *  vitality
 *  gold
 *  unknown2 (always zero?)
 */
#define PLAYERINFO_DRTL_CLASS_WARRIOR  0
#define PLAYERINFO_DRTL_CLASS_ROGUE    1
#define PLAYERINFO_DRTL_CLASS_SORCERER 2
/******************************************************/


/******************************************************/
#define CLIENT_PROGIDENT2 0x0bff
typedef struct
{
    t_normal_header h;
    bn_int          clienttag;
} t_client_progident2;
/******************************************************/


/******************************************************/
#define CLIENT_JOINCHANNEL 0x0cff
typedef struct
{
    t_normal_header h;
    bn_int          channelflag;
} t_client_joinchannel;
#define CLIENT_JOINCHANNEL_NORMAL  0x00000000
#define CLIENT_JOINCHANNEL_GENERIC 0x00000001
#define CLIENT_JOINCHANNEL_CREATE  0x00000002
/******************************************************/


/******************************************************/
#define SERVER_CHANNELLIST 0x0bff
typedef struct
{
    t_normal_header h;
    /* channel names */
} t_server_channellist;
/******************************************************/


/******************************************************/
/*
We don't use this for now. It makes the client put
the list of IPs/hostnames into the registry.

FF 04 8F 00 00 00 00 00   32 30 39 2E 36 37 2E 31    ........209.67.1
33 36 2E 31 37 34 3B 32   30 37 2E 36 39 2E 31 39    36.174;207.69.19
34 2E 32 31 30 3B 32 30   37 2E 36 39 2E 31 39 34    4.210;207.69.194
2E 31 38 39 3B 32 31 36   2E 33 32 2E 37 33 2E 31    .189;216.32.73.1
37 34 3B 32 30 39 2E 36   37 2E 31 33 36 2E 31 37    74;209.67.136.17
31 3B 32 30 36 2E 37 39   2E 32 35 34 2E 31 39 32    1;206.79.254.192
3B 32 30 37 2E 31 33 38   2E 33 34 2E 33 3B 32 30    ;207.138.34.3;20
39 2E 36 37 2E 31 33 36   2E 31 37 32 3B 65 78 6F    9.67.136.172;exo
64 75 73 2E 62 61 74 74   6C 65 2E 6E 65 74 00       dus.battle.net.
*/
#define SERVER_SERVERLIST 0x04ff
typedef struct
{
    t_normal_header h;
    bn_int          unknown1; /* 00 00 00 00 */
} t_server_serverlist;
/******************************************************/


/******************************************************/
/*
FF 0F 30 00 01 00 00 00   00 00 00 00 00 00 00 00    ..0.............
00 00 00 00 00 00 00 00   00 00 00 00 52 6F 73 73    ............Ross
00 52 41 54 53 20 30 20   30 20 30 20 30 20 30 00    .RATS 0 0 0 0 0.
*/
#define SERVER_MESSAGE 0x0fff
typedef struct
{
    t_normal_header h;
    bn_int          type;
    bn_int          flags;
    bn_int          latency;
    bn_int          unknown1;
    bn_int          unknown2;
    bn_int          unknown3;
    /* player name */
    /* text */
} t_server_message;
/* For MT_ADD, the text portion looks like:
 *
 * for STAR, SEXP, SSHR:
 * "%s %u %u %u %u %u"
 *  client tag (RATS, PXES, RHSS)
 *  rating
 *  number (ladder rank)
 *  stars  (normal wins)
 *  unknown3 (always zero?)
 *  unknown4 (always zero?)
 *
 * for DRTL:
 * "%s %u %u %u %u %u %u %u %u %u"
 *  client tag (LTRD)
 *  level
 *  class (0==warrior, 1==rogue, 2==sorcerer)
 *  dots (times killed diablo)
 *  strength
 *  magic
 *  dexterity
 *  vitality
 *  gold
 *  unknown2 (always zero?)
 */
#define MT_ADD                 0x00000001 /* USER,SHOWUSER */
#define MT_JOIN                0x00000002
#define MT_PART                0x00000003 /* LEAVE */
#define MT_WHISPER             0x00000004
#define MT_MESSAGE             0x00000005 /* TALK */
#define MT_BROADCAST           0x00000006
#define MT_JOINING             0x00000007 /* CHANNEL */
/* unused                      0x00000008 */
#define MT_USERFLAGS           0x00000009
#define MT_WHISPERACK          0x0000000a /* WHISPERSENT */
/* unused                      0x0000000b */
/* unused                      0x0000000c */
#define MT_CHANNELFULL         0x0000000d
#define MT_CHANNELDOESNOTEXIST 0x0000000e
#define MT_CHANNELRESTRICTED   0x0000000f
/* unused                      0x00000010 */
/* unused                      0x00000011 */
#define MT_WARN                0x00000012 /* INFO */
#define MT_ERROR               0x00000013
/* unused                      0x00000014 */
/* unused                      0x00000015 */
/* unused                      0x00000016 */
#define MT_EMOTE               0x00000017

/* flag bits for above struct */
/*      Blizzard Entertainment employee */
#define MF_BLIZZARD 0x00000001 /* blue Blizzard logo */
/*      Channel operator */
#define MF_GAVEL    0x00000002 /* gavel */
/*      Speaker in moderated channel */
#define MF_VOICE    0x00000004 /* megaphone */
/*      System operator */
#define MF_BNET     0x00000008 /* blue Blizzard or red BNETD logo */
/*      Chat bot or other user without UDP support */
#define MF_PLUG     0x00000010 /* tiny plug to right of icon */
/*      Squelched user */
#define MF_X        0x00000020 /* big red X */
/*      Special guest of Blizzard Entertainment */
#define MF_SHADES   0x00000040 /* sunglasses */
/* unused           0x00000080 */
/* unused           0x00000100 */
/*      Registered Professional Gamers Leauge player */
#define MF_PGLPLAY  0x00000200 /* PGL player logo */
/*      Registered Professional Gamers Leauge official */
#define MF_PGLOFFL  0x00000400 /* PGL official logo */
/******************************************************/


/******************************************************/
#define CLIENT_MESSAGE 0x0eff
typedef struct
{
    t_normal_header h;
    /* text */
} t_client_message;
/******************************************************/


/******************************************************/
/*
FF 09 17 00 03 00 00 00   FF FF 00 00 00 00 00 00    ................
19 00 00 00 00 00 00                                 .......
*/
#define CLIENT_GAMELISTREQ 0x09ff
typedef struct
{
    t_normal_header h;
    bn_short        gametype;
    bn_short        unknown1;
    bn_int          unknown2;
    bn_int          unknown3;
    bn_int          maxgames;
    /* game name */
} t_client_gamelistreq;
#define CLIENT_GAMELISTREQ_ALL       0x0000
#define CLIENT_GAMELISTREQ_MELEE     0x0002
#define CLIENT_GAMELISTREQ_FFA       0x0003
#define CLIENT_GAMELISTREQ_ONEONONE  0x0004
#define CLIENT_GAMELISTREQ_CTF       0x0005
#define CLIENT_GAMELISTREQ_GREED     0x0006
#define CLIENT_GAMELISTREQ_SLAUGHTER 0x0007
#define CLIENT_GAMELISTREQ_SDEATH    0x0008
#define CLIENT_GAMELISTREQ_LADDER    0x0009
#define CLIENT_GAMELISTREQ_MAPSET    0x000a
#define CLIENT_GAMELISTREQ_TEAMMELEE 0x000b
#define CLIENT_GAMELISTREQ_TEAMFFA   0x000c
#define CLIENT_GAMELISTREQ_TEAMCTF   0x000d
#define CLIENT_GAMELISTREQ_PGL       0x000e
#define CLIENT_GAMELISTREQ_DIABLO    0x0409 /* FIXME: this should be the langid */
/* Diablo reports differently than it is listed in GAMELIST */
#define CLIENT_GAMETYPE_DIABLO1      0x00000005
#define CLIENT_GAMETYPE_DIABLO2      0x00000009
#define CLIENT_GAMETYPE_DIABLO3      0x0000000c
/******************************************************/


/******************************************************/
/*
FF 09 35 00 01 00 00 00   00 00 00 00 03 00 01 00    ..5.............
00 00 00 00 02 00 17 E0   80 7B 4F 0D 00 00 00 00    .........{O.....
00 00 00 00 04 00 00 00   64 00 00 00 4D 79 47 61    ........d...MyGa
6D 65 00 00 00 

FF 09 5B 00 01 00 00 00   00 00 00 00 03 00 01 00    ..[.............
02 00 17 E0 80 7B 4F 0D   00 00 00 00 00 00 00 00    .....{O.........
04 00 00 00 2B 00 00 00   47 61 6D 65 00 50 61 73    ....+...Game.Pas
73 00 2C 33 34 2C 31 32   2C 35 2C 31 2C 33 2C 31    s.,34,12,5,1,3,1
2C 63 63 63 33 36 34 30   36 2C 2C 42 6F 62 0D 43    ,ccc36406,,Bob.C
68 61 6C 6C 65 6E 67 65   72 0D 00                   hallenger..  

FF 09 D4 03 0A 00 00 00   0C 00 00 00 09 04 00 00    ................
02 00 17 E0 CD E8 B5 E1   00 00 00 00 00 00 00 00    ................
00 00 00 00 3C 00 00 00   4A 65 73 73 65 27 73 20    ....<...Jesse's
57 6F 72 6C 64 00 00 32   0D 4C 69 7A 7A 69 65 2E    World..2.Lizzie.
42 6F 72 64 65 6E 0D 4C   54 52 44 20 34 30 20 31    Borden.LTRD 40 1
20 33 20 31 32 31 20 31   32 36 20 33 30 36 20 31     3 121 126 306 1
33 36 20 35 34 37 30 32   20 30 00 ...
*/
#define SERVER_GAMELISTREPLY 0x09ff
typedef struct
{
    t_normal_header h;
    bn_int          gamecount;
    /* games */
} t_server_gamelistreply;

typedef struct
{
    bn_int   unknown7;
    bn_short gametype;
    bn_short unknown1; /* langid under Diablo... */
    bn_short unknown3;
    bn_short port;     /* big endian byte order... at least they are consistant! */
    bn_int   game_ip;  /* big endian byte order */
    bn_int   unknown4;
    bn_int   unknown5; /* FIXME: got to figure out where latency is */
    bn_int   status;
    bn_int   unknown6;
    /* game name */
    /* clear password */
    /* info */
} t_server_gamelistreply_game;
#define SERVER_GAMELISTREPLY_GAME_UNKNOWN7       0x00000000 /* 0x0000000c */
#define SERVER_GAMELISTREPLY_GAME_UNKNOWN1           0x0001 /* 0x0000 */
#define SERVER_GAMELISTREPLY_GAME_UNKNOWN3           0x0002
#define SERVER_GAMELISTREPLY_GAME_UNKNOWN4       0x00000000
#define SERVER_GAMELISTREPLY_GAME_UNKNOWN5       0x00000000
#define SERVER_GAMELISTREPLY_GAME_STATUS_OPEN    0x00000004
#define SERVER_GAMELISTREPLY_GAME_STATUS_FULL    0x00000006
#define SERVER_GAMELISTREPLY_GAME_STATUS_STARTED 0x0000000e
#define SERVER_GAMELISTREPLY_GAME_STATUS_DONE    0x0000000c
#define SERVER_GAMELISTREPLY_GAME_UNKNOWN6       0x0000002b /* latency? */
/******************************************************/


/******************************************************/
#define CLIENT_STARTGAME1 0x08ff /* original starcraft or shareware (1.01) */
typedef struct
{
    t_normal_header h;
    bn_int          status;
    bn_int          unknown3;
    bn_short        gametype;
    bn_short        unknown1;
    bn_int          unknown4;
    bn_int          unknown5;
    /* game name */
    /* game password */
    /* game info */
} t_client_startgame1;
/* I have also seen 1,5,7,f */
#define CLIENT_STARTGAME1_STATUSMASK     0x0000000f
#define CLIENT_STARTGAME1_STATUS_OPEN    0x00000004
#define CLIENT_STARTGAME1_STATUS_FULL    0x00000006
#define CLIENT_STARTGAME1_STATUS_STARTED 0x0000000e
#define CLIENT_STARTGAME1_STATUS_DONE    0x0000000c
/******************************************************/


/******************************************************/
/*
FF 1B 14 00 02 00 17 E0   80 7B 3F 54 00 00 00 00    .........{?T....
00 00 00 00                                          ....
*/
#define CLIENT_UNKNOWN_1B 0x1bff
typedef struct
{
    t_normal_header h;
    bn_short        unknown1; /* FIXME: This "2" is the same as in the game
				 listings. What do they mean? */
    bn_short        port;     /* big endian byte order */
    bn_int          ip;       /* big endian byte order */
    bn_int          unknown2;
    bn_int          unknown3;
} t_client_unknown_1b;
#define CLIENT_UNKNOWN_1B_UNKNOWN1 0x0002
#define CLIENT_UNKNOWN_1B_UNKNOWN2 0x00000000
#define CLIENT_UNKNOWN_1B_UNKNOWN3 0x00000000
/******************************************************/


/******************************************************/
/*
FF 1A 4C 00 01 00 00 00   00 00 00 00 00 00 00 00    ..L.............
0F 00 00 00 00 00 00 00   E0 17 00 00 61 6E 73 00    ............ans.
65 6C 6D 6F 00 30 0D 77   61 72 72 69 6F 72 0D 4C    elmo.0.warrior.L
54 52 44 20 31 20 30 20   30 20 33 30 20 31 30 20    TRD 1 0 0 30 10
32 30 20 32 35 20 31 30   30 20 30 00                20 25 100 0.
*/
#define CLIENT_STARTGAME3 0x1aff /* Starcraft 1.03, Diablo 1.07 */
typedef struct
{
    t_normal_header h;
    bn_int          status;
    bn_int          unknown3;
    bn_short        gametype;
    bn_short        unknown1;
    bn_int          unknown6;
    bn_int          unknown4; /* port # under diablo */
    bn_int          unknown5;
    /* game name */
    /* game password */
    /* game info */
} t_client_startgame3;
#define CLIENT_STARTGAME3_STATUSMASK      0x0000000f
#define CLIENT_STARTGAME3_STATUS_OPEN1    0x00000001 /* used by Diablo */
#define CLIENT_STARTGAME3_STATUS_OPEN     0x00000004
#define CLIENT_STARTGAME3_STATUS_FULL     0x00000006
#define CLIENT_STARTGAME3_STATUS_STARTED  0x0000000e
#define CLIENT_STARTGAME3_STATUS_DONE     0x0000000c
/******************************************************/


/******************************************************/
/*
FF 1C 49 00 00 00 00 00   00 00 00 00 03 00 01 00    ..I.............
00 00 00 00 00 00 00 00   74 65 61 6D 6D 65 6C 65    ........teammele
65 00 00 2C 2C 2C 2C 31   2C 33 2C 31 2C 33 65 33    e..,,,,1,3,1,3e3
37 61 38 34 63 2C 37 2C   61 6E 73 65 6C 6D 6F 0D    7a84c,7,anselmo.
4F 63 74 6F 70 75 73 0D   00                         Octopus.. 
*/
#define CLIENT_STARTGAME4 0x1cff /* Brood Wars or newer Starcraft (1.04, 1.05) */
typedef struct
{
    t_normal_header h;
    bn_int          status;
    bn_int          unknown2; /* 00 00 00 00 */
    bn_short        gametype;
    bn_short        unknown3; /* 01 00 */
    bn_int          unknown4; /* 00 00 00 00 */
    bn_int          unknown5; /* 00 00 00 00 */
    /* game name */
    /* game password */
    /* game info */
} t_client_startgame4;
#define CLIENT_STARTGAME4_STATUSMASK     0x0000000f
#define CLIENT_STARTGAME4_STATUS_INIT    0x00000000
#define CLIENT_STARTGAME4_STATUS_OPEN1   0x00000001
#define CLIENT_STARTGAME4_STATUS_OPEN2   0x00000005
#define CLIENT_STARTGAME4_STATUS_OPEN3   0x00000006
#define CLIENT_STARTGAME4_STATUS_FULL1   0x00000004
#define CLIENT_STARTGAME4_STATUS_FULL2   0x00000007
#define CLIENT_STARTGAME4_STATUS_STARTED 0x0000000e
#define CLIENT_STARTGAME4_STATUS_DONE1   0x0000000c
#define CLIENT_STARTGAME4_STATUS_DONE2   0x0000000d
/******************************************************/


/******************************************************/
#define SERVER_STARTGAME1_ACK 0x08ff
typedef struct
{
    t_normal_header h;
    bn_int          reply;
} t_server_startgame1_ack;    
#define SERVER_STARTGAME1_ACK_NO 0x00000000
#define SERVER_STARTGAME1_ACK_OK 0x00000001
/******************************************************/


/******************************************************/
#define SERVER_STARTGAME3_ACK 0x1aff
typedef struct
{
    t_normal_header h;
    bn_int          reply;
} t_server_startgame3_ack;    
#define SERVER_STARTGAME3_ACK_NO 0x00000000
#define SERVER_STARTGAME3_ACK_OK 0x00000001
/******************************************************/


/******************************************************/
#define SERVER_STARTGAME4_ACK 0x1cff
typedef struct
{
    t_normal_header h;
    bn_int          reply;
} t_server_startgame4_ack;    
#define SERVER_STARTGAME4_ACK_NO 0x00000001
#define SERVER_STARTGAME4_ACK_OK 0x00000000
/******************************************************/


/******************************************************/
#define CLIENT_CLOSEGAME 0x02ff
typedef struct
{
    t_normal_header h;
} t_client_closegame;
/******************************************************/


/******************************************************/
#define CLIENT_LEAVECHANNEL 0x10ff
typedef struct
{
    t_normal_header h;
} t_client_leavechannel;
/******************************************************/


/******************************************************/
/*
FF 32 2A 00 1A 29 25 72   77 C3 3C 25 6B 4D 7A A4    .2*..)%rw.<%kMz.
3B 92 38 D5 01 F4 A5 6B   28 32 29 43 68 61 6C 6C    ;.8....k(2)Chall
65 6E 67 65 72 2E 73 63   6D 00                      enger.scm.    

FF 32 2C 00 21 F8 16 2D   99 D9 BC A4 A6 5C BA 60    .2,.!..-.....\.`
71 DE 6D 64 6F BC A5 03   28 34 29 44 69 72 65 20    q.mdo...(4)Dire 
53 74 72 61 69 74 73 2E   73 63 6D 00                Straits.scm.    
*/
#define CLIENT_MAPAUTHREQ 0x32ff
typedef struct
{
    t_normal_header h;
    bn_int          unknown1; /* FIXME: maybe these are a hash of some game info? */
    bn_int          unknown2;
    bn_int          unknown3;
    bn_int          unknown4;
    bn_int          unknown5;
    /* mapfile */
} t_client_mapauthreq;
/******************************************************/


/******************************************************/
/*
FF 32 08 00 01 00 00 00                              .2......
*/
#define SERVER_MAPAUTHREPLY 0x32ff
typedef struct
{
    t_normal_header h;
    bn_int          response;
} t_server_mapauthreply;
#define SERVER_MAPAUTHREPLY_NO        0x00000000
#define SERVER_MAPAUTHREPLY_OK        0x00000001
#define SERVER_MAPAUTHREPLY_LADDER_OK 0x00000002
/******************************************************/


/******************************************************/
/*
FF 15 14 00 36 38 58 49 52 41 54 53 00 00 00 00  ....68XIRATS....
AF 14 55 36                                      ..U6            
*/
#define CLIENT_ADREQ 0x15ff
typedef struct
{
    t_normal_header h;
    bn_int          archtag;
    bn_int          clienttag;
    bn_int          prev_adid; /* zero if first request */
    bn_int          ticks; /* Unix-style time */
} t_client_adreq;
/******************************************************/


/******************************************************/
/*
Sent in response to a CLIENT_ADREQ to tell the client which
banner to display next

FF 15 3A 00 72 00 00 00   2E 70 63 78 50 15 7A 1C    ..:.r....pcxP.z.
CE 0F BD 01 61 64 30 30   30 30 37 32 2E 70 63 78    ....ad000072.pcx
00 68 74 74 70 3A 2F 2F   77 77 77 2E 62 6C 69 7A    .http://www.bliz
7A 61 72 64 2E 63 6F 6D   2F 00                      zard.com/.

FF 15 3C 00 C3 00 00 00   2E 73 6D 6B 00 9B 36 A6    ..<......smk..6.
8F 5B BE 01 61 64 30 30   30 30 63 33 2E 73 6D 6B    .[..ad0000c3.smk
00 68 74 74 70 3A 2F 2F   77 77 77 2E 66 61 74 68    .http://www.fath
65 72 68 6F 6F 64 2E 6F   72 67 2F 00                erhood.org/.

                          FF 15 36 00 2B 51 02 00            ..6.+Q..
2E 70 63 78 00 00 00 00   58 01 B2 00 61 64 30 32    .pcx....X...ad02
35 31 32 62 2E 70 63 78   00 68 74 74 70 3A 2F 2F    512b.pcx.http://
77 77 77 2E 66 73 67 73   2E 63 6F 6D 2F 00          www.fsgs.com/.
*/
#define SERVER_ADREPLY 0x15ff
typedef struct
{
    t_normal_header h;
    bn_int          adid;
    bn_int          extensiontag; /* unlike other tags, this one is "forward" */
    bn_long         timestamp; /* FIXME: file modification time? */
    /* filename */
    /* link URL */
} t_server_adreply;
/******************************************************/


/******************************************************/
#define CLIENT_ADACK 0x21ff
/*
Sent after client has displayed the banner

0000:   FF 21 36 00 36 38 58 49   52 41 54 53 72 00 00 00    .!6.68XIRATSr...
0010:   61 64 30 30 30 30 37 32   2E 70 63 78 00 68 74 74    ad000072.pcx.htt
0020:   70 3A 2F 2F 77 77 77 2E   62 6C 69 7A 7A 61 72 64    p://www.blizzard
0030:   2E 63 6F 6D 2F 00                                    .com/.

0000:   FF 21 38 00 36 38 58 49   4C 54 52 44 C3 00 00 00    .!8.68XILTRD....
0010:   61 64 30 30 30 30 63 33   2E 73 6D 6B 00 68 74 74    ad0000c3.smk.htt
0020:   70 3A 2F 2F 77 77 77 2E   66 61 74 68 65 72 68 6F    p://www.fatherho
0030:   6F 64 2E 6F 72 67 2F 00                              od.org/.
*/
typedef struct
{
    t_normal_header h;
    bn_int          archtag;
    bn_int          clienttag;
    bn_int          adid;
    /* adfile */
    /* adlink */
} t_client_adack;
/******************************************************/


/******************************************************/
/*
Sent if the user clicks on the adbanner
*/
#define CLIENT_ADCLICK 0x16ff
typedef struct
{
    t_normal_header h;
    bn_int          adid;
    bn_int          unknown1;
} t_client_adclick;
/******************************************************/


/******************************************************/
#define CLIENT_LADDERREQ 0x2eff
typedef struct
{
    t_normal_header h;
    bn_int          clienttag;
    bn_int          unknown1; /* 01 00 00 00 */
    bn_int          type;
    bn_int          startplace; /* start listing on this entries */
    bn_int          count; /* how many entries to list */
} t_client_ladderreq;
#define CLIENT_LADDERREQ_TYPE_HIGHESTRATED 0x00000000
#define CLIENT_LADDERREQ_TYPE_MOSTWINS     0x00000002
#define CLIENT_LADDERREQ_TYPE_MOSTGAMES    0x00000003
/******************************************************/


/******************************************************/
/*
Sent in repsonse to CLIENT_LADDERREQ

FF 2E AB 03 52 41 54 53   01 00 00 00 00 00 00 00    ....RATS........
00 00 00 00 0A 00 00 00   27 00 00 00 01 00 00 00    ........'.......
01 00 00 00 74 06 00 00   00 00 00 00 27 00 00 00    ....t.......'...
01 00 00 00 01 00 00 00   74 06 00 00 00 00 00 00    ........t.......
00 00 00 00 00 00 00 00   FF FF FF FF 74 06 00 00    ............t...
00 00 00 00 00 00 00 00   12 27 4E FF 2D DD BE 01    .........'N.-...
12 27 4E FF 2D DD BE 01   6E 65 6D 62 69 3A 29 6B    .'N.-...nembi:)k
69 6C 6C 65 72 00 1D 00   00 00 03 00 00 00 00 00    iller...........
00 00 67 06 00 00 00 00   00 00 1D 00 00 00 03 00    ..g.............
00 00 00 00 00 00 67 06   00 00 00 00 00 00 01 00    ......g.........
00 00 00 00 00 00 FF FF   FF FF 67 06 00 00 01 00    ..........g.....
00 00 00 00 00 00 D8 5B   9E D0 32 DD BE 01 D8 5B    .......[..2....[
9E D0 32 DD BE 01 53 4B   45 4C 54 4F 4E 00 1F 00    ..2...SKELTON...
00 00 03 00 00 00 00 00   00 00 EF 05 00 00 00 00    ................
00 00 1F 00 00 00 03 00   00 00 00 00 00 00 EF 05    ................
00 00 00 00 00 00 02 00   00 00 00 00 00 00 FF FF    ................
FF FF EF 05 00 00 02 00   00 00 00 00 00 00 62 26    ..............b&
55 0C 51 DC BE 01 62 26   55 0C 51 DC BE 01 7A 69    U.Q...b&U.Q...zi
7A 69 62 65 5E 2E 7E 00   19 00 00 00 02 00 00 00    zibe^.~.........
00 00 00 00 FC 05 00 00   00 00 00 00 18 00 00 00    ................
02 00 00 00 00 00 00 00   EE 05 00 00 00 00 00 00    ................
03 00 00 00 00 00 00 00   FF FF FF FF FC 05 00 00    ................
03 00 00 00 00 00 00 00   A0 25 4F 31 90 DE BE 01    .........%O1....
8C F1 7F 9F 66 DD BE 01   59 4F 4F 4A 49 4E 27 53    ....f...YOOJIN'S
00 1D 00 00 00 02 00 00   00 00 00 00 00 EC 05 00    ................
00 00 00 00 00 1D 00 00   00 02 00 00 00 00 00 00    ................
00 EC 05 00 00 00 00 00   00 04 00 00 00 00 00 00    ................
00 FF FF FF FF EC 05 00   00 03 00 00 00 00 00 00    ................
00 F4 58 78 82 2F D7 BE   01 F4 58 78 82 2F D7 BE    ..Xx./....Xx./..
01 3D 7B 5F 7C 5F 7D 3D   00 1A 00 00 00 00 00 00    .={_|_}=........
00 00 00 00 00 E2 05 00   00 00 00 00 00 1A 00 00    ................
00 00 00 00 00 00 00 00   00 E2 05 00 00 00 00 00    ................
00 05 00 00 00 00 00 00   00 FF FF FF FF E2 05 00    ................
00 05 00 00 00 00 00 00   00 F2 DC 0D 1F 6C DD BE    .............l..
01 F2 DC 0D 1F 6C DD BE   01 5B 53 50 41 43 45 5D    .....l...[SPACE]
2D 31 2D 54 2E 53 2E 4A   00 15 00 00 00 02 00 00    -1-T.S.J........
00 01 00 00 00 E0 05 00   00 00 00 00 00 15 00 00    ................
00 02 00 00 00 01 00 00   00 E0 05 00 00 00 00 00    ................
00 06 00 00 00 00 00 00   00 FF FF FF FF E0 05 00    ................
00 06 00 00 00 00 00 00   00 7A C9 F6 6E 0B DE BE    .........z..n...
01 7A C9 F6 6E 0B DE BE   01 5B 46 65 77 5D 2D 44    .z..n....[Few]-D
2E 73 00 23 00 00 00 00   00 00 00 00 00 00 00 DF    .s.#............
05 00 00 00 00 00 00 23   00 00 00 00 00 00 00 00    .......#........
00 00 00 DF 05 00 00 00   00 00 00 07 00 00 00 00    ................
00 00 00 FF FF FF FF E2   05 00 00 04 00 00 00 00    ................
00 00 00 6E D9 DC 3A E2   DA BE 01 6E D9 DC 3A E2    ...n..:....n..:.
DA BE 01 5B 4C 2E 73 5D   2D 43 6F 6F 6C 00 1F 00    ...[L.s]-Cool...
00 00 07 00 00 00 02 00   00 00 DD 05 00 00 00 00    ................
00 00 1F 00 00 00 07 00   00 00 02 00 00 00 DD 05    ................
00 00 00 00 00 00 08 00   00 00 00 00 00 00 FF FF    ................
FF FF 07 06 00 00 02 00   00 00 00 00 00 00 B6 CE    ................
B1 D8 7A D8 BE 01 B6 CE   B1 D8 7A D8 BE 01 52 6F    ..z.......z...Ro
60 4C 65 58 7E 50 72 4F   27 5A 65 4E 00 18 00 00    `LeX~PrO'ZeN....
00 04 00 00 00 00 00 00   00 DD 05 00 00 00 00 00    ................
00 18 00 00 00 04 00 00   00 00 00 00 00 DD 05 00    ................
00 00 00 00 00 09 00 00   00 00 00 00 00 FF FF FF    ................
FF DD 05 00 00 04 00 00   00 00 00 00 00 50 76 4C    .............PvL
F7 AD D7 BE 01 50 76 4C   F7 AD D7 BE 01 48 61 6E    .....PvL.....Han
5F 65 53 54 68 65 72 2E   27 27 00                   _eSTher.''.
*/
#define SERVER_LADDERREPLY 0x2eff
typedef struct
{
    t_normal_header h;
    bn_int          clienttag;
    bn_int          unknown1; /* 01 00 00 00 */
    bn_int          type;
    bn_int          startplace; /* start listing on this entries */
    bn_int          count; /* how many entries to list */
    /* ladder entry */
    /* player name */
} t_server_ladderreply;
#define SERVER_LADDERREPLY_UNKNOWN1 0x00000001

typedef struct
{
    bn_int wins;
    bn_int loss;
    bn_int disconnect;
    bn_int rating;
    bn_int unknown;
} t_ladder_data;

typedef struct
{
    t_ladder_data current;
    t_ladder_data active;
    bn_int        ttest[6]; /* 00 00 00 00  00 00 00 00  FF FF FF FF  74 06 00 00  00 00 00 00 00  00 00 00 */
    bn_long       lastgame_current; /* timestamp */
    bn_long       lastgame_active;  /* timestamp */
} t_ladder_entry;
/******************************************************/


/******************************************************/
/*
                                    FF 25 08 00 EA 7F DB 02   .%......
*/
#define CLIENT_ECHOREPLY 0x25ff
typedef struct
{
    t_normal_header h;
    bn_int          ticks;
} t_client_echoreply;
/******************************************************/


/******************************************************/
#define SERVER_ECHOREQ 0x25ff
typedef struct
{
    t_normal_header h;
    bn_int          ticks;
} t_server_echoreq;
/******************************************************/



/******************************************************/
/* What I'm calling the ping happens every 90 seconds during gameplay.  I'm
 * not exactly sure what it is, but it didn't hurt that we didn't respond up
 * to now...  I went ahead and coded a response in.
 * This seems to be associated with a UDP packet 7 from the client:
 *  7: cli class=normal[0x0001] type=CLIENT_PINGREQ[0x00ff] length=4
 *  0000:   FF 00 04 00                                          ....
 *  5: clt prot=UDP from=128.123.62.23:6112 to=128.123.62.23:6112 length=8
 *  0000:   07 00 00 00 2E 95 9D 00                              ........
 *  7: srv class=normal[0x0001] type=SERVER_PINGREPLY[0x00ff] length=4
 *  0000:   FF 00 04 00                                          
 * It looks like:
 * 
 * 6: recv normal packet type 0x00ff length 4
 * 0000:   FF 00 04 00                                          ....
 * 6: send normal packet type 0x00ff length 4
 * 0000:   FF 00 04 00                                          ....
 */
#define CLIENT_PINGREQ 0x00ff
typedef struct
{
    t_normal_header h;
} t_client_pingreq;
/******************************************************/


/******************************************************/
#define SERVER_PINGREPLY 0x00ff
typedef struct
{
    t_normal_header h;
} t_server_pingreply;
/******************************************************/


/******************************************************/
/*
FF 2C 3D 02 00 00 00 00   08 00 00 00 03 00 00 00    .,=.............
00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00    ................
00 00 00 00 00 00 00 00   00 00 00 00 52 6F 73 73    ............Ross
5F 43 4D 00 00 00 00 00   00 00 00 4F 6E 20 6D 61    _CM........On ma
70 20 22 43 68 61 6C 6C   65 6E 67 65 72 22 3A 0A    p "Challenger":.
00 52 6F 73 73 5F 43 4D   20 77 61 73 20 50 72 6F    .Ross_CM was Pro
74 6F 73 73 20 61 6E 64   20 70 6C 61 79 65 64 20    toss and played
66 6F 72 20 33 31 20 6D   69 6E 75 74 65 73 0A 0A    for 31 minutes..
20 20 4F 76 65 72 61 6C   6C 20 53 63 6F 72 65 20      Overall Score
32 38 31 30 35 0A 20 20   20 20 20 20 20 20 20 31    28105.         1
31 37 30 30 20 66 6F 72   20 55 6E 69 74 73 0A 20    1700 for Units.
20 20 20 20 20 20 20 20   20 31 34 37 35 20 66 6F             1475 fo
72 20 53 74 72 75 63 74   75 72 65 73 0A 20 20 20    r Structures.
20 20 20 20 20 20 31 34   39 33 30 20 66 6F 72 20          14930 for
52 65 73 6F 75 72 63 65   73 0A 0A 20 20 55 6E 69    Resources..  Uni
74 73 20 53 63 6F 72 65   20 31 31 37 30 30 0A 20    ts Score 11700.
20 20 20 20 20 20 20 20   20 20 20 37 30 20 55 6E               70 
*/
#define CLIENT_GAME_REPORT 0x2cff
typedef struct
{
    t_normal_header h;
    bn_int          unknown1; /* 0x00000000 */
    bn_int          count;    /* (number of player slots (will be 8 for now) */
    /* results... */
    /* names... */
    /* report header */
    /* report body */
} t_client_game_report;

typedef struct
{
    bn_int result;
} t_client_game_report_result;
#define CLIENT_GAME_REPORT_RESULT_PLAYING    0x00000000
#define CLIENT_GAME_REPORT_RESULT_WIN        0x00000001
#define CLIENT_GAME_REPORT_RESULT_LOSS       0x00000002
#define CLIENT_GAME_REPORT_RESULT_DRAW       0x00000003
#define CLIENT_GAME_REPORT_RESULT_DISCONNECT 0x00000004
/******************************************************/


/******************************************************/
#define CLIENT_JOIN_GAME 0x22ff
typedef struct
{
    t_normal_header h;
    bn_int          clienttag;
    bn_int          versiontag;
    /* game name */
    /* game password */
} t_client_join_game;
/******************************************************/


/******************************************************/
/*
FF 27 84 00 01 00 00 00  04 00 00 00 52 6F 73 73    .'..........Ross
5F 43 4D 00 70 72 6F 66  69 6C 65 5C 73 65 78 00    _CM.profile\sex.
70 72 6F 66 69 6C 65 5C  61 67 65 00 70 72 6F 66    profile\age.prof
69 6C 65 5C 6C 6F 63 61  74 69 6F 6E 00 70 72 6F    ile\location.pro
66 69 6C 65 5C 64 65 73  63 72 69 70 74 69 6F 6E    file\description
00 61 73 64 66 00 61 73  66 00 61 73 64 66 00 61    .asdf.asf.asdf.a
73 64 66 61 73 64 66 61  73 64 66 61 73 64 66 0D    sdfasdfasdfasdf.
0A 61 73 64 0D 0A 66 61  73 64 0D 0A 66 61 73 64    .asd..fasd..fasd
66 0D 0A 00                                         f...
*/
#define CLIENT_STATSUPDATE 0x27ff
typedef struct
{
    t_normal_header h;
    bn_int          name_count;
    bn_int          key_count;
    /* names... */
    /* values... */
} t_client_statsupdate;
/******************************************************/


#endif
