/* $Id: tree.c,v 1.15 1997/01/31 00:17:04 tdyas Exp $
 * tree.c: Basic device tree traversal/scanning for the Linux
 *         prom library.
 *
 * Copyright (C) 1995 David S. Miller (davem@caip.rutgers.edu)
 * Copyright (C) 1997 Jakub Jelinek (jj@sunsite.mff.cuni.cz)
 */

#include "tilo.h"

static char promlib_buf[128];

/* Return the child of node 'node' or zero if no this node has no
 * direct descendent.
 */
int prom_getchild(int node)
{
	int cnode;

	if (node == -1)
		return 0;

	if (prom_vers != PROM_P1275)
		cnode = prom_nodeops->no_child(node);
	else
		cnode = p1275_cmd ("child", 1, node);
		
	if (cnode == 0 || cnode == -1)
		return 0;

	return cnode;
}

/* Return the next sibling of node 'node' or zero if no more siblings
 * at this level of depth in the tree.
 */
int prom_getsibling(int node)
{
	int sibnode;

	if (node == -1)
		return 0;

	if (prom_vers != PROM_P1275)
		sibnode = prom_nodeops->no_nextnode(node);
	else
		sibnode = p1275_cmd ("peer", 1, node);
		
	if (sibnode == 0 || sibnode == -1)
		return 0;

	return sibnode;
}

/* Return the length in bytes of property 'prop' at node 'node'.
 * Return -1 on error.
 */
int prom_getproplen(int node, char *prop)
{
	int ret;

	if((!node) || (!prop))
		ret = -1;
	else {
		if (prom_vers != PROM_P1275)
			ret = prom_nodeops->no_proplen(node, prop);
		else
			ret = p1275_cmd ("getproplen", 2, node, prop);
	}
	return ret;
}

/* Acquire a property 'prop' at node 'node' and place it in
 * 'buffer' which has a size of 'bufsize'.  If the acquisition
 * was successful the length will be returned, else -1 is returned.
 */
int prom_getproperty(int node, char *prop, char *buffer, int bufsize)
{
	int plen, ret;

	plen = prom_getproplen(node, prop);
	if((plen > bufsize) || (plen == 0) || (plen == -1))
		ret = -1;
	else {
		/* Ok, things seem all right. */
		if (prom_vers != PROM_P1275)
			ret = prom_nodeops->no_getprop(node, prop, buffer);
		else
			ret = p1275_cmd ("getprop", 4, node, prop, buffer, bufsize);
	}
	return ret;
}

/* Acquire an integer property and return its value.  Returns -1
 * on failure.
 */
int prom_getint(int node, char *prop)
{
	static int intprop;

	if(prom_getproperty(node, prop, (char *) &intprop, sizeof(int)) != -1)
		return intprop;

	return -1;
}

/* Acquire an integer property, upon error return the passed default
 * integer.
 */
int prom_getintdefault(int node, char *property, int deflt)
{
	int retval;

	retval = prom_getint(node, property);
	if(retval == -1) return deflt;

	return retval;
}

/* Acquire a boolean property, 1=TRUE 0=FALSE. */
int prom_getbool(int node, char *prop)
{
	int retval;

	retval = prom_getproplen(node, prop);
	if(retval == -1) return 0;
	return 1;
}

/* Acquire a property whose value is a string, returns a null
 * string on error.  The char pointer is the user supplied string
 * buffer.
 */
void prom_getstring(int node, char *prop, char *user_buf, int ubuf_size)
{
	int len;

	len = prom_getproperty(node, prop, user_buf, ubuf_size);
	if(len != -1) return;
	user_buf[0] = 0;
	return;
}


/* Does the device at node 'node' have name 'name'?
 * YES = 1   NO = 0
 */
int prom_nodematch(int node, char *name)
{
	static char namebuf[128];
	prom_getproperty(node, "name", namebuf, sizeof(namebuf));
	if(strcmp(namebuf, name) == 0) return 1;
	return 0;
}

/* Search siblings at 'node_start' for a node with name
 * 'nodename'.  Return node if successful, zero if not.
 */
int prom_searchsiblings(int node_start, char *nodename)
{

	int thisnode, error;

	for(thisnode = node_start; thisnode;
	    thisnode=prom_getsibling(thisnode)) {
		error = prom_getproperty(thisnode, "name", promlib_buf,
					 sizeof(promlib_buf));
		/* Should this ever happen? */
		if(error == -1) continue;
		if(strcmp(nodename, promlib_buf)==0) return thisnode;
	}

	return 0;
}

/* Return the first property type for node 'node'.
 */
char * prom_firstprop(int node, char *buffer)
{
	char *ret;

	*buffer = 0;
	if (node == -1) return buffer;
	if (prom_vers != PROM_P1275) {
		ret = prom_nodeops->no_nextprop(node, (char *) 0x0);
		strcpy (buffer, ret);
	} else {
		p1275_cmd ("nextprop", 3, node, (char *)0, buffer);
	}
	return buffer;
}

/* Return the property type string after property type 'oprop'
 * at node 'node' .  Returns NULL string if no more
 * property types for this node.
 */
char * prom_nextprop(int node, char *oprop, char *buffer)
{
	char *ret;
	char buf[32];

	if (node == -1) {
		*buffer = 0;
		return buffer;
	}
	if (oprop == buffer) {
		strcpy (buf, oprop);
		oprop = buf;
	}
	*buffer = 0;
	if(node == -1) return buffer;
	if (prom_vers != PROM_P1275) {
		ret = prom_nodeops->no_nextprop(node, oprop);
		strcpy (buffer, ret);
	} else {
		p1275_cmd ("nextprop", 3, node, oprop, buffer);
	}
	return buffer;
}

int prom_node_has_property(int node, char *prop)
{
	char buf [32];

	*buf = 0;
	do {
		prom_nextprop(node, buf, buf);
		if(!strcmp(buf, prop))
		   return 1;
	} while (*buf);
	return 0;
}
