;;; elmo-maildir.el -- Maildir interface for ELMO.

;; Copyright 1998,1999,2000 Yuuichi Teranishi <teranisi@gohome.org>

;; Author: Yuuichi Teranishi <teranisi@gohome.org>
;; Keywords: mail, net news
;; Time-stamp: <2000-01-07 10:27:39 teranisi>

;; This file is part of ELMO (Elisp Library for Message Orchestration).

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.
;;

;;; Commentary:
;; 

;;; Code:
;; 

(require 'elmo-localdir)

(defun elmo-maildir-number-to-filename (dir number loc-alist 
					    &optional candidates)
  (let* ((candidates (or candidates
			 (directory-files 
			  (expand-file-name 
			   "cur"
			   dir) t)))
	 (location (cdr (assq number loc-alist))))
    (if location 
	(elmo-maildir-get-filename 
	 location candidates))))

(defun elmo-maildir-get-filename (location candidates)
  (catch 'done
    (while candidates
      (if (string-match (symbol-name location) (car candidates))
	  (throw 'done (car candidates)))
      (setq candidates (cdr candidates)))))

(defsubst elmo-maildir-list-location (dir)
  (let* ((cur-dir (expand-file-name "cur" dir))
	 (cur (directory-files cur-dir
			       nil "^[^.].*$" t))
	 seen-list seen sym list
	 )
;    (setq cur (sort cur
;		    (lambda (x y)
;		      (< (elmo-get-last-modification-time x cur-dir)
;			 (elmo-get-last-modification-time y cur-dir)))))
    (setq list
	  (mapcar 
	   (lambda (x)
	     (if (string-match "^\\([^:]+\\):\\([^:]+\\)$" x)
		 (progn
		   (setq seen nil)
		   (save-match-data
		     (if (string-match
			  "S"
			  (elmo-match-string 2 x))
			 (setq seen t)))
		   (setq sym (intern (elmo-match-string 1 x)))
		   (if seen
		       (setq seen-list (cons sym seen-list)))
		   sym)
	       (intern x)))
	   cur))
    (cons list seen-list)))

(defun elmo-maildir-msgdb-create-entity (dir number loc-alist files)
  (elmo-localdir-msgdb-create-overview-entity-from-file
   number
   (elmo-maildir-number-to-filename dir
				    number loc-alist
				    files)))

(defun elmo-maildir-update-current (spec)
  "Move all new msgs to cur in the maildir"
  (let* ((maildir (elmo-localdir-get-folder-directory spec))
	 (news (directory-files (expand-file-name "new"
						  maildir)
				nil
				"^[^.].*$" t)))
    ;; move new msgs to cur directory.
    (mapcar (lambda (x)
	      (rename-file 
	       (expand-file-name x (expand-file-name "new" maildir))
	       (expand-file-name (concat x ":2,")
				 (expand-file-name "cur" maildir))))
	    news)))

(defun elmo-maildir-set-mark (filename mark)
  "Mark the file in the maildir. MARK is a character."
  (if (string-match "^\\([^:]+:2,\\)\\(.*\\)$" filename)
      (let ((flaglist (string-to-char-list (elmo-match-string 
					    2 filename))))
	(unless (memq mark flaglist)
	  (setq flaglist (sort (cons mark flaglist) '<))
	  (rename-file filename
		       (concat (elmo-match-string 1 filename)
			       (char-list-to-string flaglist)))))))

(defun elmo-maildir-delete-mark (filename mark)
  "Mark the file in the maildir. MARK is a character."
  (if (string-match "^\\([^:]+:2,\\)\\(.*\\)$" filename)
      (let ((flaglist (string-to-char-list (elmo-match-string 
					    2 filename))))
	(when (memq mark flaglist)
	  (setq flaglist (delq mark flaglist))
	  (rename-file filename
		       (concat (elmo-match-string 1 filename)
			       (if flaglist
				   (char-list-to-string flaglist))))))))

(defun elmo-maildir-mark-as-important (spec msgs &optional msgdb)
  (let* ((dir (elmo-localdir-get-folder-directory spec))
	 (loc-alist (if msgdb (elmo-msgdb-get-location msgdb)
		      (elmo-msgdb-location-load (elmo-msgdb-expand-path 
						 nil spec))))
	 (candidates (directory-files (expand-file-name "cur" dir) t))
	 loc markings)
    (while msgs
      (if (setq loc (cdr (assq (car msgs) loc-alist)))
	  (setq markings (cons loc markings))) ; append to markings.
      (setq msgs (cdr msgs)))
    (while markings
      (elmo-maildir-set-mark
       (elmo-maildir-get-filename (car markings)
				  candidates)
       ?F) ; set flagged.
      (setq markings (cdr markings)))))
  
(defun elmo-maildir-unmark-important (spec msgs &optional msgdb)
  (let* ((dir (elmo-localdir-get-folder-directory spec))
	 (loc-alist (if msgdb (elmo-msgdb-get-location msgdb)
		      (elmo-msgdb-location-load (elmo-msgdb-expand-path 
						 nil spec))))
	 (candidates (directory-files (expand-file-name "cur" dir) t))
	 loc markings)
    (while msgs
      (if (setq loc (cdr (assq (car msgs) loc-alist)))
	  (setq markings (cons loc markings))) ; append to markings.
      (setq msgs (cdr msgs)))
    (while markings
      (elmo-maildir-delete-mark
       (elmo-maildir-get-filename (car markings)
				  candidates)
       ?F) ; set flagged.
      (setq markings (cdr markings)))))

(defun elmo-maildir-mark-as-unread (spec msgs &optional msgdb)
  (let* ((dir (elmo-localdir-get-folder-directory spec))
	 (loc-alist (if msgdb (elmo-msgdb-get-location msgdb)
		      (elmo-msgdb-location-load (elmo-msgdb-expand-path 
						 nil spec))))
	 (candidates (directory-files (expand-file-name "cur" dir) t))
	 loc markings)
    (while msgs
      (if (setq loc (cdr (assq (car msgs) loc-alist)))
	  (setq markings (cons loc markings))) ; append to markings.
      (setq msgs (cdr msgs)))
    (while markings
      (elmo-maildir-delete-mark
       (elmo-maildir-get-filename (car markings)
				  candidates)
       ?S) ; set seen mark.
      (setq markings (cdr markings)))))

(defun elmo-maildir-mark-as-read (spec msgs &optional msgdb)
  (let* ((dir (elmo-localdir-get-folder-directory spec))
	 (loc-alist (if msgdb (elmo-msgdb-get-location msgdb)
		      (elmo-msgdb-location-load (elmo-msgdb-expand-path 
						 nil spec))))
	 (candidates (directory-files (expand-file-name "cur" dir) t))
	 loc markings)
    (while msgs
      (if (setq loc (cdr (assq (car msgs) loc-alist)))
	  (setq markings (cons loc markings))) ; append to markings.
      (setq msgs (cdr msgs)))
    (while markings
      (elmo-maildir-set-mark
       (elmo-maildir-get-filename (car markings)
				  candidates)
       ?S) ; set seen mark.
      (setq markings (cdr markings)))))

(defun elmo-maildir-msgdb-create (spec numlist new-mark 
				       already-mark seen-mark 
				       important-mark 
				       seen-list
				       &optional msgdb)
  (when numlist
    (let* ((dir (elmo-localdir-get-folder-directory spec))
	   (loc-alist (if msgdb (elmo-msgdb-get-location msgdb)
			(elmo-msgdb-location-load (elmo-msgdb-expand-path 
						   nil spec))))
	   (loc-seen (elmo-maildir-list-location dir))
	   (files (directory-files (expand-file-name "cur" dir) t))
	   (loc-list  (car loc-seen))
	   (seen-list (cdr loc-seen))
	   overview number-alist mark-alist entity
	   i percent num location pair)
      (setq num (length numlist))
      (setq i 0)
      (message "creating msgdb...")
      (while numlist
	(setq entity
	      (elmo-maildir-msgdb-create-entity
	       dir (car numlist) loc-alist files))
	(if (null entity)
	    ()
	  (setq overview 
		(elmo-msgdb-append-element
		 overview entity))
	  (setq number-alist
		(elmo-msgdb-number-add number-alist
				       (elmo-msgdb-overview-entity-get-number
					entity)
				       (elmo-msgdb-overview-entity-get-id
					entity)))
	  (setq location (cdr (assq (car numlist) loc-alist)))
	  (unless (memq location seen-list)
	    (setq mark-alist
		  (elmo-msgdb-mark-append mark-alist 
					  (elmo-msgdb-overview-entity-get-number
					   entity)
					;(nth 0 entity)
					  (or (elmo-msgdb-global-mark-get 
					       (elmo-msgdb-overview-entity-get-id
						entity))
					      (if (elmo-cache-exists-p 
						   (elmo-msgdb-overview-entity-get-id
						    entity))
						  already-mark
						new-mark))))))
	(setq i (1+ i))
	(setq percent (/ (* i 100) num))
	(message "creating msgdb...%d%%" percent)
	(setq numlist (cdr numlist)))
      (message "creating msgdb...done.")
      (list overview number-alist mark-alist loc-alist))))

(defalias 'elmo-maildir-msgdb-create-as-numlist 'elmo-maildir-msgdb-create)

(defalias 'elmo-maildir-list-folders 'elmo-localdir-list-folders)

(defun elmo-maildir-append-msg (spec string)
  nil)

(defun elmo-maildir-delete-msg (spec number loc-alist)
  (let (file
	(dir (elmo-localdir-get-folder-directory spec)))
    (setq file (elmo-maildir-number-to-filename
		dir number loc-alist))
    (if (and (file-writable-p file) 
	     (not (file-directory-p file)))
	(progn (delete-file file)
	       t))))

(defun elmo-maildir-read-msg (spec number outbuf &optional msgdb)
  (save-excursion
    (let* ((loc-alist (if msgdb (elmo-msgdb-get-location msgdb)
			(elmo-msgdb-location-load (elmo-msgdb-expand-path 
						   nil spec))))
	   (dir (elmo-localdir-get-folder-directory spec))
	   (file (elmo-maildir-number-to-filename
		  dir number loc-alist)))
      (set-buffer outbuf)
      (erase-buffer)
      (when (file-exists-p file)
	(as-binary-input-file (insert-file-contents file))
	(elmo-delete-cr-get-content-type)))))

(defun elmo-maildir-delete-msgs (spec msgs &optional msgdb)
  (let ((loc-alist (if msgdb (elmo-msgdb-get-location msgdb)
		     (elmo-msgdb-location-load (elmo-msgdb-expand-path 
						nil spec)))))
    (mapcar '(lambda (msg) (elmo-maildir-delete-msg spec msg
						    loc-alist))
	    msgs)))

(defsubst elmo-maildir-list-folder-subr (spec &optional nonsort)
  (let* ((dir (elmo-localdir-get-folder-directory spec))
	 (flist (elmo-list-folder-by-location 
		 spec
		 (car (elmo-maildir-list-location dir)))))
    (if nonsort
	(cons (or (elmo-max-of-list flist) 0) (length flist))
      (sort flist '<))))

(defun elmo-maildir-list-folder (spec); called by elmo-maildir-search()
  (elmo-maildir-update-current spec)
  (elmo-maildir-list-folder-subr spec))

(defun elmo-maildir-max-of-folder (spec)
  (elmo-maildir-list-folder-subr spec t))

(defalias 'elmo-maildir-check-validity 'elmo-localdir-check-validity)

(defalias 'elmo-maildir-sync-validity  'elmo-localdir-sync-validity)

(defun elmo-maildir-folder-exists-p (spec)
  (file-directory-p (elmo-localdir-get-folder-directory spec)))

(defun elmo-maildir-folder-creatable-p (spec)
  t)

(defun elmo-maildir-create-folder (spec)
  (save-excursion
    (let ((dir (elmo-localdir-get-folder-directory spec)))
      (if (file-directory-p dir)
	  ()
	(if (or (file-exists-p dir)
		(file-exists-p (concat dir "/new"))
		(file-exists-p (concat dir "/cur"))
		(file-exists-p (concat dir "/tmp")))
	    (error "Create folder failed.")
	  (if (string= (substring dir -1) "/")
	      (setq dir (substring dir 0 -1))
	    t)
	  (elmo-make-directory dir) (set-file-modes dir 448)
	  (elmo-make-directory (concat dir "/new"))
	  (set-file-modes (concat dir "/new") 448)
	  (elmo-make-directory (concat dir "/cur"))
	  (set-file-modes (concat dir "/cur") 448)
	  (elmo-make-directory (concat dir "/tmp"))
	  (set-file-modes (concat dir "/tmp") 448)
	  t
	  )))))

(defun elmo-maildir-delete-folder (spec)
  (let* ((dir (elmo-localdir-get-folder-directory spec)))
    (if (not (file-directory-p dir))
	(error "no such directory: %s" dir)
	  (elmo-delete-directory (concat dir "/new") t)
	  (elmo-delete-directory (concat dir "/cur") t)
	  (elmo-delete-directory (concat dir "/tmp") t)
      (elmo-delete-directory dir t)
      t)))

(defvar elmo-maildir-header-chop-length 2048)

(defun elmo-maildir-search (spec condition &optional from-msgs msgdb)
  (save-excursion
    (let* ((msgs (or from-msgs (elmo-maildir-list-folder spec)))
	   (loc-alist (if msgdb (elmo-msgdb-get-location msgdb)
			(elmo-msgdb-location-load (elmo-msgdb-expand-path 
						   nil spec))))
	   (dir (elmo-localdir-get-folder-directory spec))
	   (candidates (directory-files (expand-file-name "cur"
							  dir) t))
	   case-fold-search ret-val
	   percent i num
	   (num (length msgs))
	   msg-num)
      (while msgs
	(setq msg-num (car msgs))
	(if (elmo-file-field-condition-match 
	     (elmo-maildir-number-to-filename
	      dir (car msgs) loc-alist candidates)
	     condition)
	    (setq ret-val (append ret-val (list msg-num))))
	(setq i (1+ i))
	(setq percent (/ (* i 100) num))
	(message "searching...%d%%" percent)
	(setq msgs (cdr msgs)))
      ret-val)))

;;; (maildir) -> maildir
(defun elmo-maildir-copy-msgs (dst-spec msgs src-spec 
					&optional loc-alist same-number)
  (let (srcfile)
    (while msgs
      (setq srcfile 
	    (elmo-maildir-get-msg-filename src-spec (car msgs) loc-alist))
      (elmo-copy-file
       ;; src file
       srcfile
       ;; dst file
       (expand-file-name 
	(file-name-nondirectory srcfile)
	(concat (elmo-localdir-get-folder-directory dst-spec) "/cur")))
      (setq msgs (cdr msgs))))
  t)

(defun elmo-maildir-use-cache-p (spec number)
  nil)

(defun elmo-maildir-local-file-p (spec number)
  t)

(defun elmo-maildir-get-msg-filename (spec number &optional loc-alist)
  (let* ((dir (elmo-localdir-get-folder-directory spec))
	 (loc-alist (or loc-alist
			(elmo-msgdb-location-load (elmo-msgdb-expand-path 
						   nil spec))))
	 (candidates (directory-files (expand-file-name "cur" dir) t))
	 loc)
    (setq loc (cdr (assq number loc-alist)))
    (elmo-maildir-get-filename loc candidates)))

(defalias 'elmo-maildir-sync-number-alist 
  'elmo-generic-sync-number-alist)
(defalias 'elmo-maildir-list-folder-unread 
  'elmo-generic-list-folder-unread)
(defalias 'elmo-maildir-list-folder-important
  'elmo-generic-list-folder-important)

(provide 'elmo-maildir)

;;; elmo-maildir.el ends here
