/*  Spruce
 *  Copyright (C) 1999 Susixware
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "sockio.h"

guint timeout = 30;  /* timeout in seconds */

Sigfunc *signal(gint signo, Sigfunc *func)
{
   struct sigaction act, oact;

   act.sa_handler = func;
   sigemptyset(&act.sa_mask);
   act.sa_flags = 0;
   if (signo == SIGALRM)
   {
#ifdef SA_INTERRUPT
      act.sa_flags |= SA_INTERRUPT;        /* SunOS 4.x */
#endif
   }
   else
   {
#ifdef SA_RESTART
      act.sa_flags |= SA_RESTART;          /* SVR4 and 4.4BSD */
#endif
   }
   if (sigaction(signo, &act, &oact) < 0)
      return (SIG_ERR);
   return (oact.sa_handler);
}

static void sockio_alarm(gint signo)
{
   return;
}

gint connect_timeo(gint sockfd, const struct sockaddr *saptr, guint salen, gint nsec)
{
   Sigfunc *sigfunc;
   gint n;

   sigfunc = signal(SIGALRM, sockio_alarm);
   if (alarm(nsec) != 0)
      fprintf(stderr, "connect_timeo: alarm was already set\n");

   if ( (n = connect(sockfd, (struct sockaddr *) saptr, salen)) < 0)
   {
      close(sockfd);
      if (errno == EINTR)
         errno = ETIMEDOUT;
   }
   alarm(0);
   signal(SIGALRM, sigfunc);       /* restore previous signal handler */

   return n;
}

ssize_t recvch(gint sock, gchar *ptr)
{
   static gint read_cnt = 0;
   static gchar *read_ptr;
   static gchar read_buf[512];

   if (read_cnt <= 0)
   {
      again:
         if ( (read_cnt = read(sock, read_buf, sizeof(read_buf))) < 0)
         {
            if (errno == EINTR)
               goto again;
            return -1;
         }
         else if (read_cnt == 0)
            return 0;
         read_ptr = read_buf;
   }
   read_cnt--;
   *ptr = *read_ptr++;
   return 1;
}

ssize_t recvline(gint sock, void *vptr, size_t maxlen)
{
   gint n, rc;
   gchar c, *ptr;

   ptr = vptr;
   for (n = 1; n < maxlen; n++)
   {
      if ( (rc = recvch(sock, &c)) == 1)
      {
         *ptr++ = c;
         if (c == '\n')
         break;                      /* newline is stored, like fgets() */
      }
      else if (rc == 0)
      {
         if (n == 1)
            return 0;                   /* EOF, no data read */
         else
            break;                      /* EOF, some data read */
      }
      else
         return -1;                     /* error, errno set by read() */
   }
	
   *ptr = 0;                            /* null terminate like fgets() */
   return n;
}

gint sockfd_readable_timeo(gint sockfd, guint nsec)
{
   fd_set rset;
   struct timeval tv;

   FD_ZERO(&rset);
   FD_SET(sockfd, &rset);

   tv.tv_sec = nsec;
   tv.tv_usec = 0;

   return (select(sockfd + 1, &rset, NULL, NULL, &tv));
}

ssize_t recvline_timeo(gint sockfd, void *vptr, size_t maxlen, guint nsec)
{
   gint n = 0;
   struct timeval tv;
   gchar *ptr;

   tv.tv_sec = nsec;
   tv.tv_usec = 0;
   setsockopt(sockfd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));

   ptr = vptr;

   n = recvfrom(sockfd, ptr, maxlen, 0, NULL, NULL);
   if (n < 0)
   {
      if (errno == EWOULDBLOCK)
         errno = 0;
      return -1;
   }

   *(ptr + n) = 0;

   return n;
}
