/*  Spruce
 *  Copyright (C) 1999 Susixware
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "smtpio.h"

static gchar *smtp_errlist[] = {"",
                               ": Hostname failed to resolve", 
                               ": Socket failed to open",
                               ": No route to host?",
                               ": Server failed to acknowledge",
                               ": HELO send()",
                               ": HELO recv()",
                               ": MAIL FROM send()",
                               ": MAIL FROM recv()",
                               ": RCPT TO send()",
                               ": RCPT TO recv",
                               ": DATA send()",
                               ": DATA recv()",
                               ": BODY recv()",
                               ": RSET send()",
                               ": RSET recv()",
                               ": QUIT send()",
                               ": QUIT recv()",
                               ": Sender invalid",
                               ": Recipient invalid"};

extern gint errno;
extern gint timeout;
SMTPError smtperrno;

void smtperror (gchar *reason, gint len)
{
   strncpy(reason, smtp_errlist[smtperrno], len);
}

void email_destroy (Email *msg)
{
   g_free(msg->to);
   g_free(msg->to_addr);
   /* if msg->cc is not NULL, destroy it */
   if (msg->cc != (gchar*) NULL && strcmp(msg->cc, "") != 0)
      g_free(msg->cc);
   /* if msg->cc_addr is not NULL, destroy it */
   if (msg->cc_addr != (gchar*) NULL && strcmp(msg->cc_addr, "") != 0)
      g_free(msg->cc_addr);
   g_free(msg->from);
   g_free(msg->from_addr);
   g_free(msg->replyto);
   g_free(msg->header);
   g_free(msg->subject);
   g_free(msg->xmailer);
   if (msg->mime_version != (gchar*) NULL)
      	g_free(msg->mime_version);
   if (msg->content_type != (gchar*) NULL)
   	g_free(msg->content_type);
   if (msg->content_transfer_encoding != (gchar*) NULL)
   	g_free(msg->content_transfer_encoding);
   g_free(msg->body);
}

gint format_date (gchar *date, size_t date_size)
{
   time_t t;
   time(&t);

   if (strftime(date, date_size, "%a, %d %b %Y %H:%M:%S GMT", gmtime(&t)) == 0)
      /* No characters written to date string. */
      return ERROR;
   else
      return SUCCESS;
}

gint format_header (Email *msg)
{
   /* format the header information */
   gchar date[60], *hdrend;
   gint hdrsize;
   static Version smtpio_ver;
   
   smtpio_ver.part.major = VER_MAJOR;
   smtpio_ver.part.delim1 = '.';
   smtpio_ver.part.minor = VER_MINOR;
   smtpio_ver.part.delim2 = '.';
   smtpio_ver.part.release = VER_REL;
   smtpio_ver.part.null = '\0';

   /* Get the date/time. */
   format_date(date, 60);

   /* Calculate header size. */
   hdrsize = 200 + strlen(date) + strlen(msg->from) + strlen(msg->to);
   if (msg->cc != (gchar *)NULL && strcmp(msg->cc, "") != 0)
      hdrsize += strlen(msg->cc);
   if (msg->subject != (gchar *)NULL)
      hdrsize += strlen(msg->subject);
   if (msg->replyto != (gchar *)NULL)
      hdrsize += strlen(msg->replyto);
   if (msg->mime_version != (gchar *)NULL)
      hdrsize += strlen(msg->mime_version);
   if (msg->content_type != (gchar *)NULL)
      hdrsize += strlen(msg->content_type);
   if (msg->content_transfer_encoding != (gchar *)NULL)
      hdrsize += strlen(msg->content_transfer_encoding);
   hdrsize += strlen(msg->xmailer) + strlen(smtpio_ver.string) + 1;
   
   /* Create the header. */
   msg->header = g_malloc(hdrsize);
   hdrend = msg->header + sprintf(msg->header, "Date: %s\nFrom: %s\nTo: %s\n",
				  date, msg->from, msg->to);
   if (msg->cc != (gchar *)NULL && strcmp(msg->cc, "") != 0)
      hdrend += sprintf(hdrend, "CC: %s\n", msg->cc);
   if (msg->subject != (gchar *)NULL)
      hdrend += sprintf(hdrend, "Subject: %s\n", msg->subject);
   if (msg->replyto != (gchar *)NULL)
      hdrend += sprintf(hdrend, "Reply-To: %s\n", msg->replyto);
   hdrend += sprintf(hdrend, "X-Mailer: %s w/smtpio %s\n", msg->xmailer, 
	   smtpio_ver.string);
   if (msg->mime_version != (gchar *)NULL)
      hdrend += sprintf(hdrend, "MIME-Version: %s\n", msg->mime_version);
   if (msg->content_type != (gchar *)NULL)
      hdrend += sprintf(hdrend, "Content-Type: %s\n", msg->content_type);
   if (msg->content_transfer_encoding != (gchar *)NULL)
      hdrend += sprintf(hdrend, "Content-Transfer-Encoding: %s\n", msg->content_transfer_encoding);
	
   return SUCCESS;
}

gint get_email_addr_from_text (gchar *addr, gchar *email)
{
   /* get the actual email address from the string passed and place it in *addr
    * we can assume the address will be in one of the following forms:
    * 1) The Name <person@host.com>
    * 2) person@host.com
    * 3) <person@host.com>
   */
   
   gchar *addr_strt;         /* points to start of addr */
   gchar *addr_end;          /* points to end of addr */
   gchar *ptr1, *ptr2;
   
   
   /* check the incoming args */
   if(!addr || !email || *email=='\0')
      return ERROR;

   /* scan the string for an open brace */
   for(addr_strt = email; *addr_strt; addr_strt++) 
      if(*addr_strt == '<')
         break;

   if(*addr_strt)  /* we found an open brace */
   {
      /* let's look for it's counterpart */
      for(addr_end = addr_strt; *addr_end; addr_end++)
         if(*addr_end == '>') 
            break;

      /* if we didn't find it, or braces are empty... */
      if(!(*addr_end) || (addr_strt == addr_end - 1))
         return ERROR;
			
      /* addr_strt points to '<' and addr_end points to '>'.
       * Now let's adjust 'em slightly to point to the beginning
       * and ending of the email addy
		 */
      addr_strt++;
      addr_end--;
   }
   else    /* no open brace...assume type 2? */
   {
      addr_strt = email;
			
      /* find the end of the string */
      for(addr_end = addr_strt; *addr_end; addr_end++);
 
      addr_end--;       /* points to NULL, move it back one char */
   }

   /* now addr_strt & addr_end point to the beginning & ending of the email addy */

   /* copy the string into addr */
   for(ptr1 = addr_strt, ptr2 = addr; ptr1 <= addr_end; ptr1++, ptr2++)
      *ptr2 = *ptr1;

   for(ptr1 = addr_strt; ptr1 <= addr_end; ptr1++)    /* look for an '@' sign */
      if(*ptr1 == '@')
         break;

   if(*ptr1 != '@')
   {
      /* here we found out the name doesn't have an '@' part
       * let's figure out what machine we're on & stick it on the end
       * woops.. forgot the '@' sign, gotta stick that in the reply
       * ptr2 should still point to the next place to copy to in addr
       */
      
      *ptr2++ = '@';
      if(gethostname(ptr2, MAXHOSTNAMELEN))
         return ERROR;
   }
   else               /* there's an @ sign already in the email addy */
      *ptr2 = '\0';   /* add a null to the end of the string */

   return SUCCESS;
}

gint helo (gint sock)
{
   /* say hello to the server */
   gchar buffer[256];
   gchar command[256];
   gchar localhost[MAXHOSTNAMELEN];
   gint len, ok = 0;

   /* get the localhost name */
   gethostname (localhost, MAXHOSTNAMELEN);

   /* hiya server! how are you today? */
   g_snprintf (command, 255, "HELO %s\r\n", localhost);
   if (send (sock, command, strlen(command), 0) < 0)
   {
      smtperrno = S_ERR_HELO_SEND;
      return ERROR;
   }

   if ( ((len = recvline(sock, buffer, 255)) < 0) 
        || ((ok = find_string(buffer, ReqMlActOK)) == -1) )
   {
      /* the smtp server is not being friendly today,
       * because it refused to say hello back to us :(
       */
      smtperrno = S_ERR_HELO_RECV;

      if (ok < 0 && len > 0)  /* if we got a response, even tho it didn't    */
      {                       /* like it, lets ignore it and continue anyway */
         fprintf(stderr, "Warning: smtp server did not like our HELO.\n");
         return SUCCESS;
      }

      return ERROR;  /* it failed to respond at all, so fuck it */
   }

   return SUCCESS;
}

gint mail (gint sock, gchar *sender)
{
   /* we gotta tell the smtp server who we are. (our email addy) */
   gchar buffer[256];
   gchar command[256];

   g_snprintf(command, 255, "MAIL FROM: %s\r\n", sender);
   if (send (sock, command, strlen(command), 0) < 0)
   {
      smtperrno = S_ERR_MAIL_SEND;
      return ERROR;
   }

   if ( (recvline(sock, buffer, 255) < 0)
       || (find_string(buffer, ReqMlActOK) == -1) )
   {
      /* we should have gotten a message from the smtp server saying:
       * 250 user@host.com... Sender ok
       */
      smtperrno = S_ERR_MAIL_RECV;
      return ERROR;
   }

   return SUCCESS;
}

gint rcpt (gint sock, gchar *recipient, gchar *orig_rcpt)
{
  /* we gotta tell the smtp server who we are going to be sending
   * our email to */
  static gchar buffer[256];
  static gchar command[512];

#undef I_KNOW_WHAT_ORCPT_IS_FOR
#ifdef I_KNOW_WHAT_ORCPT_IS_FOR
   if (orig_rcpt == (gchar *)NULL || strcmp(recipient, orig_rcpt) == 0)
      /* No original recipient specified or same as recipient. */ 
      g_snprintf(command, 511, "RCPT TO: %s\r\n", recipient);
   else
      g_snprintf(command, 511, "RCPT TO: %s ORCPT=%s\r\n", recipient, orig_rcpt);
#else
   g_snprintf(command, 511, "RCPT TO: %s\r\n", recipient);
#endif
  
   if (send (sock, command, strlen(command), 0) < 0)
   {
       smtperrno = S_ERR_RCPT_SEND;
       return ERROR;
   }

   if ( (recvline(sock, buffer, 255) < 0)
        || (find_string(buffer, ReqMlActOK) == -1) )
   {
      /* we should have gotten a message from the smtp server saying:
       * 250 user@host.com... Recipient ok
       */
      smtperrno = S_ERR_RCPT_RECV;
      return ERROR;
   }
  
   return SUCCESS;
}

gint data (gint sock, gchar *header, gchar *body)
{
   /* now we can actually send what's important :p */
   gchar buffer[256];
   gchar command[256];
   gchar data[6];
   gint i;

   strncpy(command, "DATA\r\n", 255);
   if (send (sock, command, strlen(command), 0) < 0)
   {
      smtperrno = S_ERR_DATA_SEND;
      return ERROR;
   }
	
   if ( (recvline(sock, buffer, 255) < 0)
       || (find_string(buffer, StrtMailInpt) == -1) )
   {
      /* we should have gotten instructions on how to use the DATA command:
       * 354 Enter mail, end with "." on a line by itself
       */
      smtperrno = S_ERR_DATA_RECV;
      return ERROR;
   }
	
   /* now to send the actual data */
   /* send the header information */
   for (i = 0; i < strlen(header); i++)
   {
      if (header[i] == '\n')
         strncpy(data, "\r\n", 3);
      else
         g_snprintf(data, 3, "%c", header[i]);
      send (sock, data, strlen(data), 0);
   }
   send (sock, "\r\n\r\n", 4, 0);
	
   /* send the body of the message */
   for (i = 0; i < strlen(body); i++)
   {
      if (body[i] == '.' && body[i-1] == '\n' && body[i+1] == '\n')
      {
         /* if we have a lone period on a line... */
         strcpy(data, "..");
      }
      else
         if (body[i] == '\n' && body[i-1] != '\r')
         {
            strcpy(data, "\r\n");
         }
         else
         {
            data[0] = body[i];
            data[1] = '\0';
         }
      send (sock, data, strlen(data), 0);
   }
   /* terminate the message body */
   send (sock, "\r\n.\r\n", 5, 0);	

   if ( (recvline(sock, buffer, 255) < 0)
       || (find_string(buffer, ReqMlActOK) == -1) )
   {
      /* Should've gotten a respose saying our message is ready for delivery */
      smtperrno = S_ERR_BODY_RECV;
      return ERROR;
   }

   return SUCCESS;
}

gint rset (gint sock)
{
   /* we are going to reset the smtp server (just to be nice) */
   gchar buffer[256];
   gchar command[256];

   strncpy(command, "RSET\r\n", 255);
   if (send (sock, command, strlen(command), 0) < 0)
   {
      smtperrno = S_ERR_RSET_SEND;
      return ERROR;
   }
	
   if ( (recvline(sock, buffer, 255) < 0)
       || (find_string(buffer, ReqMlActOK) == -1) )
   {
      /* the server should have told us that it's been reset */
      smtperrno = S_ERR_RSET_RECV;
      return ERROR;
   }

   return SUCCESS;
}

gint quit (gint sock)
{
   /* lets tell the server we want to quit now */
   gchar buffer[256];
   gchar command[256];

   strncpy(command, "QUIT\r\n", 255);
   if (send (sock, command, strlen(command), 0) < 0)
   {
      smtperrno = S_ERR_QUIT_SEND;
      return ERROR;
   }
	
   if ( (recvline(sock, buffer, 255) < 0)
       || (find_string(buffer, ServClose) == -1) )
   {
      /* mr server didn't say bye-bye :( */
      smtperrno = S_ERR_QUIT_RECV;
      return ERROR;
   }

   return SUCCESS;
}

gint send_message (Server *server, Email *msg)
{
   /* this will interface with a smtp daemon on a foreign host
    * and send an email message. You should call format_header()
    * previous to calling this function. You may also want to call
    * email_addr_from_text() to get the actual address. The variables sender[]
    * and recipient[] must be in the format: user@host.com
    * which is the format that email_addr_from_text() will return.
    * "server" is of type SmtpHost and contains a hostname, ip, and port
    * number.
    */

   gint sock;
   gint fields, i;
   gchar buffer[513];
   gchar *buff = NULL;
	
   /* make sure we aren't claiming an error */
   smtperrno = S_NO_ERROR;

   /* set the bits to zero (yeah, I know...kinda complex 
    * to do it this way but it's fun??) then set it to DELIVERING
    * Tell me again why we don't just go: msg->status = DELIVERING; ?
    */
   msg->status = (msg->status & 0) | DELIVERING;

   /* resolve the hostname */
   if ( !Resolve(server) )
   {
      smtperrno = S_ERR_RESOLVE;
      return ERROR;
   }

   /* assign and open a socket */
   sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
   if ( sock < 0 )    /* if socket fails to open */
   {
      smtperrno = S_ERR_SOCKET;
      return ERROR;
   }

   /* lets set up information about the server we will be connecting to... */
   server->sin.sin_family = AF_INET;
   server->sin.sin_port = htons(server->port);

   /* call the server */
   if (connect_timeo(sock, (struct sockaddr*)&server->sin, sizeof(server->sin), timeout) < 0)
   {
      /* we failed to connect */
      smtperrno = S_ERR_CONNECT;
      close(sock);
      return ERROR;
   }
	
   if ( (recvline(sock, buffer, 512) < 0)
       || (find_string(buffer, ServiceRdy) == -1) )
   {
      /* the smtp server failed to send a response (usually version info) */
      smtperrno = S_ERR_SERV_ACK;
      close(sock);
      return ERROR;
   }

   /* Because some servers send more than just their version info...
    * (like NT SMTP servers...), lets keep checking the buffer until
    * there is nothing left. */
	/* TODO: Fix this so that we don't have to do this horrid hack */
   if (find_string(buffer, "Microsoft") > -1)
      recvline(sock, buffer, 512);
   else
      if (find_string(buffer, "earthlink.net") > -1)
      {
         /* Earthlink likes to send 2 extra lines of data... */
         recvline(sock, buffer, 512);
         recvline(sock, buffer, 512);
      }

   /* send the email */
   if ( !helo(sock) )
   {
      close(sock);
      return ERROR;
   }
   if ( !mail(sock, msg->from_addr) )
   {
      close(sock);
      return ERROR;
   }

   /* Determine number of direct recipients.  Do not CC if no direct 
      recipients are specified. */
   fields = get_num_fields(msg->to_addr, ';');
   
   if (fields > 0)
   {
      /* We will consider the first (or only) recipient to be the original recipient. */
      gchar *orig_rcpt = get_field(msg->to_addr, 1, ';');

      /* Do a RCPT TO for each of the direct recipients. */
      for (i = 1; i <= fields; i++)
      {	   
         if (i != 1)
            buff = get_field(msg->to_addr, i, ';');
         else
            buff = orig_rcpt;

         if ( !rcpt(sock, buff, (gchar *)NULL) )
         {
            g_free(buff);
            close(sock);
            return ERROR;
         }
	  
         if (i != 1)
            g_free(buff);
      }

      /* Do a RCPT TO for each of the carbon copies. */
      fields = get_num_fields(msg->cc_addr, ';');
      if (fields > 0)
      {
         for (i = 1; i <= fields; i++)
         {
            buff = get_field(msg->cc_addr, i, ';');

            if ( !rcpt(sock, buff, orig_rcpt) )
            {
               g_free(buff);
               close(sock);
               return ERROR;
            }

            g_free(buff);
         }
      }

      g_free (orig_rcpt);
   }

   if ( !data(sock, msg->header, msg->body) )
   {
      close(sock);
      return ERROR;
   }
   if ( !rset(sock) )
   {
      close(sock);
      return ERROR;
   }
   if ( !quit(sock) )
   {
      close(sock);
      return ERROR;
   }

   close(sock);                             /* close the socket */
	
   msg->status = msg->status | DELIVERED;   /* update status to delivered */
	
   return SUCCESS;                          /* everything went smoothly :) */
}
