/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1999 */
/* See the file NOTICE for conditions of use and distribution. */

/* A number of functions for driving outgoing SMTP calls. */


#include "exim.h"



/*************************************************
*           Connect to remote host               *
*************************************************/

/* Create a socket, and connect it to a remote host. IPv6 addresses are
detected by checking for a colon in the address.

Arguments:
  host        host item containing name and address
  port        remote port to connect to, in network byte order
  interface   outgoing interface address or NULL
  timeout     timeout value or 0
  keepalive   TRUE to use keepalive

Returns:      connected socket number, or -1 with errno set
*/

int
smtp_connect(host_item *host, int port, char *interface, int timeout,
  BOOL keepalive)
{
#if HAVE_IPV6
struct sockaddr_in6 s_in;
#else
struct sockaddr_in s_in;
#endif

int rc, save_errno;
int host_af = (strchr(host->address, ':') != NULL)? AF_INET6 : AF_INET;
int sock = socket(host_af, SOCK_STREAM, 0);

if (sock < 0)
  log_write(0, LOG_PANIC_DIE, "socket creation failed: %s", strerror(errno));

HDEBUG(1) debug_printf("Connecting to %s [%s] ... ", host->name, host->address);

/* Clear the socket block */

memset(&s_in, 0, sizeof(s_in));

/* Bind to a specific interface if requested. On an IPv6 system, this has
to be of the same family as the address we are calling. */

if (interface != NULL)
  {
  #if HAVE_IPV6
  if (strchr(interface, ':') != NULL)
    {
    rc = inet_pton(AF_INET6, interface, &s_in.sin6_addr);
    s_in.sin6_family = AF_INET6;
    }
  else
    {
    rc = inet_pton(AF_INET, interface, &s_in.sin6_flowinfo);
    s_in.sin6_family = AF_INET;
    }

  if (rc != 1)
    log_write(0, LOG_PANIC_DIE, "unable to parse \"%s\"", interface);

  if (s_in.sin6_family == host_af)
    {
    s_in.sin6_port = 0;
    rc = bind(sock, (struct sockaddr *)&s_in, sizeof(s_in));
    }

  /* Handle IPv4 binding */

  #else
  s_in.sin_family = AF_INET;
  s_in.sin_port = 0;
  s_in.sin_addr.s_addr = (S_ADDR_TYPE)inet_addr(interface);
  rc = bind(sock, (struct sockaddr *)&s_in, sizeof(s_in));
  #endif

  /* Check that it bound successfully. */

  if (rc < 0)
    log_write(0, LOG_PANIC_DIE, "unable to bind outgoing SMTP call to %s: %s",
      interface, strerror(errno));
  }

/* Set up the remote address and port on an IPv6 system. You wouldn't have
thought it would take much to put the IPv6 address in the same place as the
IPv4 one, but no... */

#if HAVE_IPV6
s_in.sin6_family = host_af;
s_in.sin6_port = port;

if (host_af == AF_INET6)
  {
  if (inet_pton(host_af, host->address, &s_in.sin6_addr) != 1)
    log_write(0, LOG_PANIC_DIE, "unable to parse \"%s\"", host->address);
  }
else
  {
  if (inet_pton(host_af, host->address, &s_in.sin6_flowinfo) != 1)
    log_write(0, LOG_PANIC_DIE, "unable to parse \"%s\"", host->address);
  }

/* Set up the address and port on an IPv4 system. */

#else
s_in.sin_family = AF_INET;
s_in.sin_port = port;
s_in.sin_addr.s_addr = (S_ADDR_TYPE)inet_addr(host->address);
#endif

/* If no connection timeout is set, just call connect() without setting
a timer, thereby allowing the inbuilt timeout to operate. */

sigalrm_seen = FALSE;

if (timeout > 0)
  {
  os_non_restarting_signal(SIGALRM, sigalrm_handler);
  alarm(timeout);
  }

rc = connect(sock, (struct sockaddr *)&s_in, sizeof(s_in));
save_errno = errno;

if (timeout > 0)
  {
  alarm(0);
  signal(SIGALRM, SIG_IGN);
  }

/* A failure whose error code is "Interrupted system call" is in fact
an externally applied timeout if the signal handler has been run. */

if (rc < 0)
  {
  if (save_errno == EINTR && sigalrm_seen) save_errno = ETIMEDOUT;
  close(sock);
  HDEBUG(1) debug_printf("failed\n");
  errno = save_errno;
  return -1;
  }

HDEBUG(1) debug_printf("connected\n");

/* Add keepalive to the socket if requested, before returning it. */

if (keepalive)
  {
  int fodder = 1;
  if (setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE,
      (char *)(&fodder), sizeof(fodder)) != 0)
    log_write(4, LOG_MAIN, "send setsockopt(SO_KEEPALIVE) failed on "
      "connection to %s: %s", host->address, strerror(errno));
  }

return sock;
}



/*************************************************
*             Write SMTP command                 *
*************************************************/

/* The formatted command is left in big_buffer so that it can be reflected in
any error message.

Arguments:
  sock       the socket to write to
  format     a format, starting with one of
             of HELO, MAIL FROM, RCPT TO, DATA, ".", or QUIT.
  ...        data for the format

Returns:     TRUE if successful, FALSE if not, with errno set
*/

BOOL
smtp_write_command(int sock, char *format, ...)
{
int count, rc;
va_list ap;
va_start(ap, format);
if (!string_vformat(big_buffer, big_buffer_size, format, ap))
  log_write(0, LOG_PANIC_DIE, "overlong write_command in outgoing SMTP");
va_end(ap);
count = (int)strlen(big_buffer);
HDEBUG(1) debug_printf("  SMTP>> %s", big_buffer);
rc = send(sock, big_buffer, count, 0);
big_buffer[count-2] = 0;     /* remove \r\n for debug and error message */
if (rc > 0) return TRUE;
HDEBUG(1) debug_printf("send failed: %s\n", strerror(errno));
return FALSE;
}



/*************************************************
*              Read SMTP response                *
*************************************************/

/* This function reads an SMTP response with a timeout, and returns the
response in the given buffer. It also analyzes the first digit of the reply
code and returns FALSE if it is not acceptable. The following is an abstract of
the theory of reply codes from RFC 821:

  1yz   Positive Preliminary reply (not used in SMTP)
  2yz   Positive Completion reply
  3yz   Positive Intermediate reply
  4yz   Transient Negative Completion reply
  5yz   Permanent Negative Completion reply

For all SMTP commands:
  4yz => fail, but try again
  5yz => fail, but don't try again

For all except DATA (including after final '.'):
  2yz => success
  3yz => error

For DATA:
  2yz => error
  3yz => send data

A zero is added after any data that is read, to make it a valid C string. FALSE
is also returned after a reading error. In this case buffer[0] will be zero,
and the error code will be in errno.

Arguments:
  sock      the socket to read from
  buffer    where to put the response
  size      the size of the buffer
  okdigit   the expected first digit of the response
  timeout   the timeout to use

Returns:    TRUE if a valid, non-error response was received; else FALSE
*/

BOOL
smtp_read_response(int sock, char *buffer, int size, int okdigit, int timeout)
{
int count, rc;
fd_set select_inset;
struct timeval tv;
char *ptr = buffer;
char *readptr = buffer;

/* Ensure errno starts out zero */

errno = 0;

/* Loop for handling SMTP responses that do not all come in one packet
(multiline or otherwise). Each call to recv is timed by means of the
timeout in the select() function. This works on all OS - and is more
efficient that the use of signal() and alarm(). */

for (;;)
  {
  /* If buffer is too full, something has gone wrong. */

  if (size < 10)
    {
    *readptr = 0;
    errno = ERRNO_SMTPFORMAT;
    return FALSE;
    }

  /* Loop to cover select() getting interrupted, and the possibility of
  select() returning with a positive result but no ready descriptor. Is
  this in fact possible? */

  for (;;)
    {
    FD_ZERO (&select_inset);
    FD_SET (sock, &select_inset);
    tv.tv_sec = timeout;
    tv.tv_usec = 0;
    rc = select(sock + 1, (SELECT_ARG2_TYPE *)&select_inset, NULL, NULL, &tv);

    /* If some interrupt arrived, just retry. We presume this to be rare,
    but it can happen (e.g. the SIGUSR1 signal sent by exiwhat causes
    select() to exit). */

    if (rc < 0 && errno == EINTR)
      {
      HDEBUG(9) debug_printf("EINTR while selecting for SMTP response\n");
      continue;
      }

    /* Handle a timeout, and treat any other error as a timeout */

    if (rc <= 0)
      {
      errno = ETIMEDOUT;
      buffer[0] = 0;
      return FALSE;
      }

    /* If the socket is ready, initialize empty buffer in case nothing gets
    read, then read the response and break out of this select retry loop. */

    if (FD_ISSET(sock, &select_inset))
      {
      *readptr = 0;
      count = recv(sock, readptr, size-1, 0);
      break;
      }
    }

  /* Handle an EOF (i.e. close down of the connection). */

  if (count == 0)
    {
    buffer[0] = 0;
    errno = 0;
    return FALSE;
    }

  /* Any other error in reading returns FALSE, leaving errno unchanged. */

  else if (count < 0)
    {
    buffer[0] = 0;
    return FALSE;
    }

  /* Adjust size in case we have to read another packet, and adjust the
  count to be the length of the line we are about to inspect. */

  size -= count;
  count += readptr - ptr;

  /* See if the final two characters in the buffer are \r\n. If not, we
  have to read another packet. At least, that is what we should do on a strict
  interpretation of the RFC. In practice, it seems that there are sites sending
  only LF at the ends of responses and other MTAs cope with this. So we have to
  follow the crowd. */

  /*** if (count < 2 || ptr[count-1] != '\n' || ptr[count-2] != '\r') ***/

  if (ptr[count-1] != '\n')
    {
    DEBUG(9)
      {
      int i;
      debug_printf("SMTP input line incomplete in one buffer:\n  ");
      for (i = 0; i < count; i++)
        {
        int c = (uschar)(ptr[i]);
        if (mac_isprint(c)) debug_printf("%c", c); else debug_printf("<%d>", c);
        }
      debug_printf("\n");
      }
    readptr = ptr + count;
    continue;
    }

  /* Ensure the buffer contains a C string, and remove any whitespace at the
  end of it. This gets rid of CR, LF etc. at the end. Show it, if debugging,
  formatting multi-line responses. */

  while (count > 0 && isspace(ptr[count-1])) count--;
  ptr[count] = 0;

  HDEBUG(1)
    {
    char *s = ptr;
    char *t = ptr;

    while (*t != 0)
      {
      while (*t != 0 && *t != '\n') t++;
      debug_printf("  %s %*s\n", (s == ptr)? "SMTP<<" : "      ", t-s, s);
      if (*t == 0) break;
      s = t = t + 1;
      }
    }

  /* Check the format of the response: it must start with three digits; if
  these are followed by a space or end of line, the response is complete. If
  they are followed by '-' this is a multi-line response and we must look for
  another line until the final line is reached. The only use made of multi-line
  responses is to pass them back as error messages. We therefore just
  concatenate them all within the buffer, which should be large enough to
  accept any reasonable number of lines. A multiline response may already
  have been read in one packet - hence the loop here. */

  for(;;)
    {
    char *p;
    if (count < 3 ||
       !isdigit(ptr[0]) || !isdigit(ptr[1]) || !isdigit(ptr[2]) ||
       (ptr[3] != '-' && ptr[3] != ' ' && ptr[3] != 0))
      {
      errno = ERRNO_SMTPFORMAT;    /* format error */
      return FALSE;
      }

    /* If a single-line response, exit the loop */

    if (ptr[3] != '-') break;

    /* For a multi-line response see if the next line is already read, and if
    so, stay in this loop to check it. */

    p = ptr + 3;
    while (*(++p) != 0)
      {
      if (*p == '\n')
        {
        ptr = ++p;
        break;
        }
      }
    if (*p == 0) break;   /* No more lines to check */
    }

  /* End of response. If the last of the lines we are looking at is the final
  line, we are done. Otherwise more data has to be read. */

  if (ptr[3] != '-') break;

  /* Move the reading pointer upwards in the buffer and insert \n in case this
  is an error message that subsequently gets printed. Set the scanning pointer
  to the reading pointer position. */

  ptr += count;
  *ptr++ = '\n';
  size--;
  readptr = ptr;
  }

/* Return a value that depends on the SMTP return code. On some systems a
non-zero value of errno has been seen at this point, so ensure it is zero,
because the caller of this function looks at errno when FALSE is returned, to
distinguish between an unexpected return code and other errors such as
timeouts, lost connections, etc. */

errno = 0;
return buffer[0] == okdigit;
}

/* End of smtp_out.c */
