/********************************************************************

void conj_gradient_method(pcode,pval_min,vect,
			  len,pfunction,pgradient,max_iterations)

********************************************************************/
#include <stdio.h>
#include <math.h>

#include "wnlib.h"
#include "wnmem.h"
#include "wnasrt.h"
#include "wnswap.h"
#include "wnabs.h"
#include "wnrndd.h"

#include "wnvect.h"

#include "wnconj.h"



#define GOLDEN_RATIO          1.618034
#define GOLDEN_SECTION        0.3819660



local bool force_optimize_stop_flag;

local wn_memgp old_group;

local double *buffer_vect;



void wn_force_conj_gradient_stop(void)
{
  force_optimize_stop_flag = TRUE;
}


local bool vects_equal(double v1[],double v2[],int len)
{
  int i;

  for(i=0;i<len;++i)
  {
    if(v1[i] != v2[i])
    {
      return(FALSE);
    }
  }

  return(TRUE);
}


local bool parabola_fit_improvement_wrong(double y1,double y0,double b)
{
  double expected_improvement;

  expected_improvement = y1-b;
  if(!(expected_improvement > 0.0))  /* weird form to handle Nan problems */
  {
    return(TRUE);
  }

  return(
	  (y0 > y1-0.75*expected_improvement)  /* not enough improvement */
	    ||
	  (y0 < y1-1.25*expected_improvement)  /* too much improvement */
        );
}


local void eval_function
(
  double *pval,
  double vect[],
  double direction[],
  double x,
  int len,
  double (*pfunction)(double vect[])
)
{
  wn_add_vect_and_scaled_vect(buffer_vect,vect,direction,x,len);

  *pval = (*pfunction)(buffer_vect);
}


local double *save_vect,*save_direction;
local int save_len;
local double (*save_pfunction)(double vect[]);

local double simple_eval_function(double x)
{
  double ret;

  eval_function(&ret,save_vect,save_direction,x,save_len,save_pfunction);

  return(ret);
}


local bool x0_too_far_out(double x0,double x2,double threshold)
{
  wn_assert(threshold > 1.0);

  if(x2 > 0.0)
  {
    return(x0 > x2*threshold);
  }
  else
  {
    wn_assert(x2 < 0.0);  /* x2 == 0 not allowed */

    return(x0_too_far_out(-x0,-x2,threshold));
  }
}


local bool x0_too_far_in(double x0,double x2,double threshold)
{
  wn_assert(threshold < 1.0);

  if(x2 > 0.0)
  {
    return(x0 < x2*threshold);
  }
  else
  {
    wn_assert(x2 < 0.0);  /* x0 == 0 not allowed */

    return(x0_too_far_in(-x0,-x2,threshold));
  }
}


local void line_minimize
(
  int *pcode,
  double *pval,
  double *pjump_len,
  double vect[],
  double direction[],
  double last_val,
  double last_g[],
  double last_jump_len,
  int len,
  double (*pfunction)(double vect[])
)
{
  double a,x0,y0,b,x1,y1,dy1,x2,y2;
  int code;

  *pcode = WN_SUCCESS;

  x1 = 0.0;
  y1 = last_val;
  dy1 = wn_dot_vects(direction,last_g,len);

  last_jump_len = -wn_sign(dy1)*wn_abs(last_jump_len);
  x2 = last_jump_len;

  eval_function(&y2,vect,direction,x2,len,pfunction);

  wn_fit_parabola_2pd(&code,&a,&x0,&b,x1,y1,dy1,x2,y2);
  wn_assert(code == WN_SUCCESS);

  /* look for excuses to say that parabolic fit is no good */
  if(!(a > 0))
  {
    /*
    printf("downward facing parabola.\n");
    */
    x0 = x2+GOLDEN_RATIO*x2;  /* project outward */
    eval_function(&y0,vect,direction,x0,len,pfunction);
    goto parabolic_fit_failed;
  }
  if(x0_too_far_out(x0,x2,100.0))
  {
    /*
    printf("x0 too far out.\n");
    */
    x0 = 100.0*x2;  /* project outward */
    eval_function(&y0,vect,direction,x0,len,pfunction);
    goto parabolic_fit_failed;
  }
  if(x0_too_far_in(x0,x2,0.01))
  {
    /*
    printf("x0 too far in.\n");
    */
    x0 = 0.01*x2;  /* project outward */
    eval_function(&y0,vect,direction,x0,len,pfunction);
    goto parabolic_fit_failed;
  }
  if(!(b < y1))  /* no improvement expected,weird form for Nan problems */
  {
    /*
    printf("no improvement expected.\n");
    */
    x0 = GOLDEN_SECTION*x2;
    eval_function(&y0,vect,direction,x0,len,pfunction);
    goto parabolic_fit_failed;
  }

  eval_function(&y0,vect,direction,x0,len,pfunction);

  if(parabola_fit_improvement_wrong(y1,y0,b))
  {
    /*
    printf("poor parabola fit detected.\n");
    */
    goto parabolic_fit_failed;
  }

  /* parabolic fit succeeded */

  if(y0 > y1)
  {
    x0 = x1;
    y0 = y1;
  }
  *pval = y0;
  *pjump_len = x0;
  wn_copy_vect(vect,buffer_vect,len);

  return;

parabolic_fit_failed:
  save_vect = vect;
  save_direction = direction;
  save_len = len;
  save_pfunction = pfunction;

  wn_minimize_1d_raw(pcode,&y1,&y0,&y2,&x1,&x0,&x2,y1,(simple_eval_function),3);
  if(*pcode != WN_SUCCESS)
  {
    return;
  }

  if(y0 <= last_val)
  {
    *pval = y0;
    *pjump_len = x0;
    wn_add_scaled_vect(vect,direction,x0,len);
  }
  else
  {
    *pval = last_val;
    *pjump_len = 0.0;
  }

  return;
}


void wn_conj_gradient_method
(
  int *pcode,
  double *pval_min,
  double vect[],
  int len,
  double (*pfunction)(double vect[]),
  void (*pgradient)(double grad[],double vect[]),
  int max_iterations
)
{
  int j,iteration,no_move_count;
  double norm2_g,norm2_last_g,val,last_val,numerator,beta,
	 jump_len,last_jump_len;
  double *g,*last_g,*direction;

  old_group = wn_curgp();
  wn_gpmake("no_free");

  force_optimize_stop_flag = FALSE;

  wn_make_vect(&buffer_vect,len);
  /*
  wn_make_vect(&g,len);
  */
  g = buffer_vect;  /* share memory with buffer_vect */
  wn_make_vect(&last_g,len);
  wn_make_vect(&direction,len);

  jump_len = 1.0;
  no_move_count = 0;

  wn_gppush(old_group);

  val = (*pfunction)(vect);
  (*pgradient)(g,vect);

  wn_copy_vect(direction,g,len);

  for(iteration=0;;++iteration)
  {
    norm2_g = wn_norm2_vect(g,len);

    if(norm2_g == 0.0)   /* unlikely */
    {
      wn_gppop();
      wn_gpfree();
      *pval_min = val;
      *pcode = WN_SUCCESS;
      return;
    }

    last_jump_len = jump_len;
    last_val = val;
    wn_copy_vect(last_g,g,len);
    norm2_last_g = norm2_g;

    line_minimize(pcode,&val,&jump_len,
		  vect,
		  direction,last_val,last_g,last_jump_len,len,pfunction);
    if(*pcode != WN_SUCCESS)
    {
      return;
    }

    if(force_optimize_stop_flag||(iteration >= max_iterations))
    {
      force_optimize_stop_flag = FALSE;
      wn_gppop();
      wn_gpfree();
      *pval_min = val;
      *pcode = WN_SUBOPTIMAL;
      return;
    }
    wn_assert(val <= last_val);
    if(val == last_val)
    {
      if(no_move_count >= 2)
      {
        wn_gppop();
        wn_gpfree();
        *pval_min = val;
        *pcode = WN_SUCCESS;
        return;
      }
      else
      {
	++no_move_count;
	jump_len = last_jump_len;
      }
    }
    else
    {
      no_move_count = 0;
    }


    (*pgradient)(g,vect);

    numerator = 0.0;
    for(j=0;j<len;++j)
    {
      /*
      numerator += g[j]*g[j];
      */
      numerator += g[j]*(g[j]-last_g[j]);
    }

    beta = numerator/norm2_last_g;

    /*
    wn_copy_vect(direction,g,len);
    */
    wn_add_vect_and_scaled_vect(direction,g,direction,beta,len);

    /*
    printf("norm(g) = %lg\n",wn_norm2_vect(g,len));
    printf("ob = %lg,beta = %lg,numerator=%lg,denom=%lg,norm2(direction)=%lg\n",
	   val,beta,numerator,norm2_last_g,wn_norm2(direction));
    printf("iteration = %d,ob = %lg\n",iteration,val);
    */
  }
}

