/*
 * smidump.c --
 *
 *      Dump a MIB module conforming to a given format.
 *
 * Copyright (c) 1999 Frank Strauss, Technical University of Braunschweig.
 * Copyright (c) 1999 J. Schoenwaelder, Technical University of Braunschweig.
 *
 * See the file "COPYING" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * @(#) $Id: smidump.c,v 1.19 1999/10/05 06:31:01 strauss Exp $
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "smi.h"
#include "smidump.h"



typedef struct {
    char *name;				/* Name of the output driver. */
    int (*func) (char *moduleName);	/* Output generating function. */
    char *descr;			/* Short description. */
} Driver;
 


static Driver driverTable[] = {
    { "SMIng",	   dumpSming,    "SMIng" },
    { "SMIv2",	   dumpSmiV2,    "SMIv2 (RFC 2578, RFC 2579, RFC 2580)"},
    { "SMIv1",	   dumpSmiV1,    "SMIv1 (RFC 1155, RFC 1212, RFC 1215)"},
    { "imports",   dumpImports,  "recursive list of all imports" },
    { "types",	   dumpTypes,    "recursive list of all derived types" },
    { "tree",	   dumpTree,     "structure of the OID tree" },
    { "mosy",	   dumpMosy,     "output format generated by the mosy compiler" },
    { "corba-idl", dumpCorbaIdl, "corba IDL interface definitions (JIDM)" },
    { "corba-oid", dumpCorbaOid, "corba OID definitions (JIDM)" },
    { "ucd-h",     dumpUcdH,     "UCD SNMP mib module C header" },
    { "ucd-c",	   dumpUcdC,     "UCD SNMP mib module C code" },
#if 0
    { "fig",       dumpFig,	 "graphics in xfig fig format" },
    { "java",      dumpJava,     "java manager stub code (JMGMT)" },
    { "jdmk",      dumpJdmk,     "java manager stub code (JDMK)" },
#endif
    { NULL, NULL }
};



void *xmalloc(size_t size)
{
    char *m = malloc(size);
    if (! m) {
	fprintf(stderr, "smidump: malloc failed - running out of memory\n");
	exit(1);
    }
    return m;
}



char *xstrdup(const char *s)
{
    char *m = strdup(s);
    if (! m) {
	fprintf(stderr, "smidump: strdup failed - running out of memory\n");
	exit(1);
    }
    return m;
}



void xfree(void *ptr)
{
    free(ptr);
}



static void formats()
{
    Driver *driver = driverTable;
    
    for (driver = driverTable; driver->name; driver++) {
	fprintf(stderr, "  %-12s: %s\n", driver->name,
 		driver->descr ? driver->descr : "...");
    }
}



static void usage()
{
    fprintf(stderr,
	    "Usage: smidump [-Vhls] [-f <format>] [-p <module>] <module_or_path>\n"
	    "-V                    show version and license information\n"
	    "-h                    show usage information\n"
	    "-l <level>            set maximum level of errors and warnings\n"
	    "-s                    do not generate any comments\n"
	    "-f <format>           use <format> when dumping (default %s)\n"
	    "-p <module>           preload <module>\n"
	    "<module_or_path>      plain name of MIB module or file path\n\n",
	    driverTable->name);
    fprintf(stderr, "Supported formats are:\n");
    formats();
}



static void version()
{
    printf("smidump " VERSION "\n");
}



int
main(argc, argv)
    int argc;
    char *argv[];
{
    char c;
    char *modulename;
    int flags;
    int errors = 0;
    int silent = 0;
    Driver *driver = driverTable;
    
    smiInit();

    flags = smiGetFlags();
    
    flags |= SMI_FLAG_ERRORS;
    smiSetFlags(flags);

    while ((c = getopt(argc, argv, "Vhl:sf:p:")) != -1) {
	switch (c) {
	case 'V':
	    version();
	    exit(0);
	case 'h':
	    usage();
	    exit(0);
	case 'l':
	    smiSetErrorLevel(atoi(optarg));
	    break;
	case 's':
	    silent++;
	    break;
	case 'p':
	    smiLoadModule(optarg);
	    break;
	case 'f':
	    for (driver = driverTable; driver->name; driver++) {
		if (strcasecmp(driver->name, optarg) == 0) {
		    break;
		}
	    }
	    if (!driver->name) {
		fprintf(stderr, "smidump: invalid dump format `%s'"
			" - supported formats are:\n", optarg);
		formats();
		exit(1);
	    }
	    break;
	default:
	    usage();
	    exit(1);
	}
    }

    while (optind < argc) {
	modulename = smiLoadModule(argv[optind]);
	if (modulename) {
	    errors += (driver->func)(modulename);
	} else {
	    fprintf(stderr, "smidump: cannot locate module `%s'\n",
		    argv[optind]);
	}
	optind++;
    }

    smiExit();
    
    exit(errors);
}
