/* $Id: visual.c,v 1.33 1999/01/29 22:29:25 marcus Exp $
******************************************************************************

   Display-trueemu : visual management

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <ggi/internal/ggi-dl.h>
#include <ggi/display/trueemu.h>


static gg_option trueemu_options[3] =
{
	{ "parent", ""    },
	{ "dither", "4"   },
	{ "model",  "rgb" }
};

int GGIdlinit(ggi_visual *vis, const char *args,void *argptr)
{
	TrueemuHook *th;

	char target[1024];


	/* initialize */

	GGIDPRINT("display-trueemu: GGIdlinit start.\n");

	th = LIBGGI_PRIVATE(vis) = _ggi_malloc(sizeof(TrueemuHook));

	if ((th->flush_lock = ggLockCreate()) == NULL) {
		free(th);
		return GGI_DL_ERROR;
	}

	vis->gc = _ggi_malloc(sizeof(ggi_gc));
	th->opmansync = _ggi_malloc(sizeof(_ggi_opmansync));


	th->flags  = 0;
	th->fb_ptr = NULL;

	
	/* handle arguments */

	if (getenv("GGI_TRUEEMU_OPTIONS") != NULL) {
		if (ggParseOptions(getenv("GGI_TRUEEMU_OPTIONS"),
			trueemu_options, 3) == NULL) {
			
			fprintf(stderr, "display-trueemu: error in "
				"$GGI_TRUEEMU_OPTIONS.\n");
			free(th);
			return GGI_DL_ERROR;
		}
	}
	
	if (args) {
		args = ggParseOptions((char *) args, trueemu_options, 3);

		if (args == NULL) {
			fprintf(stderr, "display-trueemu: error in "
				"arguments.\n");
			free(th);
			return GGI_DL_ERROR;
		}
	}


	/* parse the results */

	GGIDPRINT("trueemu: parent is '%s'.\n", trueemu_options[0].result);
	GGIDPRINT("trueemu: dither is '%s'.\n", trueemu_options[1].result);
	GGIDPRINT("trueemu: model  is '%s'.\n", trueemu_options[2].result);
	
	ggiParseMode(trueemu_options[0].result, &th->mode);

	switch(trueemu_options[1].result[0]) {

		case '0': th->flags |= TRUEEMU_F_DITHER_0; break;
		case '2': th->flags |= TRUEEMU_F_DITHER_2; break;
		case '4': th->flags |= TRUEEMU_F_DITHER_4; break;

		default:
			fprintf(stderr, "display-trueemu: Unknown dither "
				"'%s'.\n", trueemu_options[1].result);
	}

	switch (trueemu_options[2].result[0]) {

		case 'r': th->flags |= TRUEEMU_F_RGB;    break;
		case 'c': th->flags |= TRUEEMU_F_CUBE;   break;
		case 'p': th->flags |= TRUEEMU_F_PASTEL; break;

		default:
			fprintf(stderr, "display-trueemu: Unknown model "
				"'%s'.\n", trueemu_options[2].result);
	}


	/* setup mansync */
	if (! _ggiAddDL(vis, "helper-mansync", NULL, th->opmansync, 0)) {
		fprintf(stderr, "display-trueemu: Cannot load "
			"helper-mansync!\n");
		return GGI_DL_ERROR;
	}

	MANSYNC_init(vis);

	/* open the parent visual */
	while (args && *args && isspace((unsigned char) *args)) {
		args++;
	}

	GGIDPRINT("display-trueemu: opening target: %s\n", args);
        
	strcpy(target, "");
	
	if (args) {
		if (ggParseTarget((char *) args, target, 1024) == NULL) {
			/* error occurred */
			free(th);
			return GGI_DL_ERROR;
		}
	}

	if (*target == 0) {
		strcpy(target, "auto");
	}

	th->parent = ggiOpen(target, NULL);

	if (th->parent == NULL) {
		fprintf(stderr, "display-trueemu: Failed to open "
			"target: '%s'\n", target);
		free(th);
		return GGI_DL_ERROR;
	}
	
	ggiSetFlags(th->parent, GGIFLAG_ASYNC);

	/* add giiInputs, if we have them */

	if (th->parent->input) {
		vis->input=giiJoinInputs(vis->input, th->parent->input);
		th->parent->input=NULL; /* destroy old reference */
	}

	/* Mode management */
	vis->opdisplay->getmode = GGI_trueemu_getmode;
	vis->opdisplay->setmode = GGI_trueemu_setmode;
	vis->opdisplay->checkmode = GGI_trueemu_checkmode;
	vis->opdisplay->getapi = GGI_trueemu_getapi;
	vis->opdisplay->flush = GGI_trueemu_flush;
	vis->opdisplay->setflags = GGI_trueemu_setflags;
	
	GGIDPRINT("display-trueemu: GGIdlinit succeeded.\n");

	return GGI_DL_OPDISPLAY;
}

int GGIdlcleanup(ggi_visual *vis)
{
	TrueemuHook *th = TRUEEMU_PRIV(vis);
	
	GGIDPRINT("display-trueemu: GGIdlcleanup start.\n");

	MANSYNC_deinit(vis);

	if (th->fb_ptr != NULL) {
		GGI_trueemu_resetmode(vis);
	}

	if (th->parent != NULL) {
		ggiClose(th->parent);
		th->parent = NULL;
	}

	LIBGGI_PRIVATE(vis) = NULL;

	ggLockDestroy(th->flush_lock);

	free(th->opmansync);
	free(vis->gc);
	free(th);

	GGIDPRINT("display-trueemu: GGIdlcleanup done.\n");

	return 0;
}

#include <ggi/internal/ggidlinit.h>
