
/*
 * bltGrPs.c --
 *
 *      This module implements the "postscript" operation for BLT graph widget.
 *
 * Copyright 1991-1998 Lucent Technologies, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the names
 * of Lucent Technologies any of their entities not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and
 * fitness.  In no event shall Lucent Technologies be liable for any
 * special, indirect or consequential damages or any damages
 * whatsoever resulting from loss of use, data or profits, whether in
 * an action of contract, negligence or other tortuous action, arising
 * out of or in connection with the use or performance of this
 * software.
 */

/*
 * -----------------------------------------------------------------
 *
 * PostScript routines to print a graph
 *
 * -----------------------------------------------------------------
 */
#include "bltGraph.h"
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#if defined(__STDC__)
#include <stdarg.h>
#else
#include <varargs.h>
#endif

static int StringToColorMode _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, Tk_Window tkwin, char *string, char *widgRec,
	int offset));
static char *ColorModeToString _ANSI_ARGS_((ClientData clientData,
	Tk_Window tkwin, char *widgRec, int offset,
	Tcl_FreeProc **freeProcPtr));

static Tk_CustomOption colorModeOption =
{
    StringToColorMode, ColorModeToString, (ClientData)0,
};

extern Tk_CustomOption bltDistanceOption;
extern Tk_CustomOption bltPositiveDistanceOption;
extern Tk_CustomOption bltPadOption;

#define DEF_PS_CENTER		"yes"
#define DEF_PS_COLOR_MAP	(char *)NULL
#define DEF_PS_COLOR_MODE	"color"
#define DEF_PS_DECORATIONS	"yes"
#define DEF_PS_FONT_MAP		(char *)NULL
#define DEF_PS_FOOTER		"no"
#define DEF_PS_HEIGHT		"0"
#define DEF_PS_LANDSCAPE	"no"
#define DEF_PS_MAXPECT		"no"
#define DEF_PS_PADX		"1.0i"
#define DEF_PS_PADY		"1.0i"
#define DEF_PS_PAPERHEIGHT	"11.0i"
#define DEF_PS_PAPERWIDTH	"8.5i"
#define DEF_PS_PREVIEW		"no"
#define DEF_PS_WIDTH		"0"

static Tk_ConfigSpec configSpecs[] =
{
    {TK_CONFIG_BOOLEAN, "-center", "center", "Center",
	DEF_PS_CENTER, Tk_Offset(PostScript, center),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_STRING, "-colormap", "colorMap", "ColorMap",
	DEF_PS_COLOR_MAP, Tk_Offset(PostScript, colorVarName),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-colormode", "colorMode", "ColorMode",
	DEF_PS_COLOR_MODE, Tk_Offset(PostScript, colorMode),
	TK_CONFIG_DONT_SET_DEFAULT, &colorModeOption},
    {TK_CONFIG_BOOLEAN, "-decorations", "decorations", "Decorations",
	DEF_PS_DECORATIONS, Tk_Offset(PostScript, decorations),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_STRING, "-fontmap", "fontMap", "FontMap",
	DEF_PS_FONT_MAP, Tk_Offset(PostScript, fontVarName),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_BOOLEAN, "-footer", "footer", "Footer",
	DEF_PS_FOOTER, Tk_Offset(PostScript, footer),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-height", "height", "Height",
	DEF_PS_HEIGHT, Tk_Offset(PostScript, reqHeight),
	TK_CONFIG_DONT_SET_DEFAULT, &bltDistanceOption},
    {TK_CONFIG_BOOLEAN, "-landscape", "landscape", "Landscape",
	DEF_PS_LANDSCAPE, Tk_Offset(PostScript, landscape),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_BOOLEAN, "-maxpect", "maxpect", "Maxpect",
	DEF_PS_MAXPECT, Tk_Offset(PostScript, maxpect),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-padx", "padX", "PadX",
	DEF_PS_PADX, Tk_Offset(PostScript, padX), 0, &bltPadOption},
    {TK_CONFIG_CUSTOM, "-pady", "padY", "PadY",
	DEF_PS_PADY, Tk_Offset(PostScript, padY), 0, &bltPadOption},
    {TK_CONFIG_CUSTOM, "-paperheight", "paperHeight", "PaperHeight",
	DEF_PS_PAPERHEIGHT, Tk_Offset(PostScript, reqPaperHeight),
	0, &bltPositiveDistanceOption},
    {TK_CONFIG_CUSTOM, "-paperwidth", "paperWidth", "PaperWidth",
	DEF_PS_PAPERWIDTH, Tk_Offset(PostScript, reqPaperWidth),
	0, &bltPositiveDistanceOption},
    {TK_CONFIG_BOOLEAN, "-preview", "preview", "Preview",
	DEF_PS_PREVIEW, Tk_Offset(PostScript, addPreview),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-width", "width", "Width",
	DEF_PS_WIDTH, Tk_Offset(PostScript, reqWidth),
	TK_CONFIG_DONT_SET_DEFAULT, &bltDistanceOption},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

extern void Blt_PrintMarkers _ANSI_ARGS_((Graph *graphPtr, Printable printable,
	int under));
extern void Blt_PrintElements _ANSI_ARGS_((Graph *graphPtr, Printable printable));
extern void Blt_PrintActiveElements _ANSI_ARGS_((Graph *graphPtr,
	Printable printable));
extern void Blt_PrintLegend _ANSI_ARGS_((Graph *graphPtr, Printable printable));
extern void Blt_PrintGrid _ANSI_ARGS_((Graph *graphPtr, Printable printable));
extern void Blt_PrintAxes _ANSI_ARGS_((Graph *graphPtr, Printable printable));
extern void Blt_PrintAxisLimits _ANSI_ARGS_((Graph *graphPtr,
	Printable printable));

/*
 *----------------------------------------------------------------------
 *
 * StringToColorMode --
 *
 *	Convert the string representation of a PostScript color mode
 *	into the enumerated type representing the color level:
 *
 *	    PS_MODE_COLOR 	- Full color
 *	    PS_MODE_GREYSCALE  	- Color converted to greyscale
 *	    PS_MODE_MONOCHROME 	- Only black and white
 *
 * Results:
 *	A standard Tcl result.  The color level is written into the
 *	page layout information structure.
 *
 * Side Effects:
 *	Future invocations of the "postscript" option will use this
 *	variable to determine how color information will be displayed
 *	in the PostScript output it produces.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
StringToColorMode(clientData, interp, tkwin, string, widgRec, offset)
    ClientData clientData;	/* Not used. */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* Not used. */
    char *string;		/* New legend position string */
    char *widgRec;		/* Graph widget record */
    int offset;			/* Offset of colorMode field in record */
{
    PsColorMode *modePtr = (PsColorMode *) (widgRec + offset);
    unsigned int length;
    char c;

    c = string[0];
    length = strlen(string);
    if ((c == 'c') && (strncmp(string, "color", length) == 0)) {
	*modePtr = PS_MODE_COLOR;
    } else if ((c == 'g') && (strncmp(string, "grayscale", length) == 0)) {
	*modePtr = PS_MODE_GREYSCALE;
    } else if ((c == 'g') && (strncmp(string, "greyscale", length) == 0)) {
	*modePtr = PS_MODE_GREYSCALE;
    } else if ((c == 'm') && (strncmp(string, "monochrome", length) == 0)) {
	*modePtr = PS_MODE_MONOCHROME;
    } else {
	Tcl_AppendResult(interp, "bad color mode \"", string, "\": should be \
\"color\", \"greyscale\", or \"monochrome\"", (char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NameOfColorMode --
 *
 *	Convert the PostScript mode value into the string representing
 *	a valid color mode.
 *
 * Results:
 *	The static string representing the color mode is returned.
 *
 *----------------------------------------------------------------------
 */
static char *
NameOfColorMode(colorMode)
    PsColorMode colorMode;
{
    switch (colorMode) {
    case PS_MODE_COLOR:
	return "color";
    case PS_MODE_GREYSCALE:
	return "greyscale";
    case PS_MODE_MONOCHROME:
	return "monochrome";
    default:
	return "unknown color mode";
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ColorModeToString --
 *
 *	Convert the current color mode into the string representing a
 *	valid color mode.
 *
 * Results:
 *	The string representing the color mode is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static char *
ColorModeToString(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* Not used. */
    Tk_Window tkwin;		/* Not used. */
    char *widgRec;		/* PostScript structure record */
    int offset;			/* field of colorMode in record */
    Tcl_FreeProc **freeProcPtr;	/* Not used. */
{
    PsColorMode mode = *(PsColorMode *) (widgRec + offset);

    return NameOfColorMode(mode);
}

void
Blt_DestroyPostScript(graphPtr)
    Graph *graphPtr;
{
    Tk_FreeOptions(configSpecs, (char *)graphPtr->postscript,
	graphPtr->display, 0);
    free((char *)graphPtr->postscript);
}

/*
 *----------------------------------------------------------------------
 *
 * CgetOp --
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
CgetOp(graphPtr, interp, argc, argv)
    Graph *graphPtr;
    Tcl_Interp *interp;
    int argc;
    char *argv[];
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;

    if (Tk_ConfigureValue(interp, graphPtr->tkwin, configSpecs, (char *)psPtr,
	    argv[3], 0) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureOp --
 *
 *      This procedure is invoked to print the graph in a file.
 *
 * Results:
 *      A standard TCL result.
 *
 * Side effects:
 *      A new PostScript file is created.
 *
 * ----------------------------------------------------------------------
 */
static int
ConfigureOp(graphPtr, interp, argc, argv)
    Graph *graphPtr;
    Tcl_Interp *interp;
    int argc;			/* Number of options in argv vector */
    char **argv;		/* Option vector */
{
    int flags = TK_CONFIG_ARGV_ONLY;
    PostScript *psPtr = (PostScript *)graphPtr->postscript;

    if (argc == 3) {
	return Tk_ConfigureInfo(interp, graphPtr->tkwin, configSpecs,
		(char *)psPtr, (char *)NULL, flags);
    } else if (argc == 4) {
	return Tk_ConfigureInfo(interp, graphPtr->tkwin, configSpecs,
		(char *)psPtr, argv[3], flags);
    }
    if (Tk_ConfigureWidget(interp, graphPtr->tkwin, configSpecs, argc - 3,
	    argv + 3, (char *)psPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * --------------------------------------------------------------------------
 *
 * ComputeBoundingBox --
 *
 * 	Computes the bounding box for the PostScript plot.  First get
 * 	the size of the plot (by default, it's the size of graph's X
 * 	window).  If the plot plus the page border is bigger than the
 * 	designated paper size, or if the "-maxpect" option is turned
 * 	on, scale the plot to the page.
 *
 *	Note: All coordinates/sizes are in screen coordinates, not
 *	      PostScript coordinates.  This includes the computed
 *	      bounding box and paper size.  They will be scaled to
 *	      printer points later.
 *
 * Results:
 *	Returns the height of the paper in screen coordinates.
 *
 * Side Effects:
 *	The graph dimensions (width and height) are changed to the
 *	requested PostScript plot size.
 *
 * --------------------------------------------------------------------------
 */
static int
ComputeBoundingBox(graphPtr, psPtr)
    Graph *graphPtr;
    PostScript *psPtr;
{
    int paperWidth, paperHeight;
    int x, y, hSize, vSize, hBorder, vBorder;
    float hScale, vScale, scale;

    x = psPtr->padLeft;
    y = psPtr->padTop;
    hBorder = PADDING(psPtr->padX);
    vBorder = PADDING(psPtr->padY);

    if (psPtr->reqWidth > 0) {
	graphPtr->width = psPtr->reqWidth;
    }
    if (psPtr->reqHeight > 0) {
	graphPtr->height = psPtr->reqHeight;
    }
    if (psPtr->landscape) {
	hSize = graphPtr->height;
	vSize = graphPtr->width;
    } else {
	hSize = graphPtr->width;
	vSize = graphPtr->height;
    }
    /*
     * If the paper size wasn't specified, set it to the graph size plus
     * the paper border.
     */
    paperWidth = psPtr->reqPaperWidth;
    paperHeight = psPtr->reqPaperHeight;
    if (paperWidth < 1) {
	paperWidth = hSize + hBorder;
    }
    if (paperHeight < 1) {
	paperHeight = vSize + vBorder;
    }
    hScale = vScale = 1.0;
    /*
     * Scale the plot size (the graph itself doesn't change size) if
     * it's bigger than the paper or if -maxpect was set.
     */
    if ((psPtr->maxpect) || ((hSize + hBorder) > paperWidth)) {
	hScale = (float)(paperWidth - hBorder) / (float)hSize;
    }
    if ((psPtr->maxpect) || ((vSize + vBorder) > paperHeight)) {
	vScale = (float)(paperHeight - vBorder) / (float)vSize;
    }
    scale = MIN(hScale, vScale);
    if (scale != 1.0) {
	hSize = (int)((hSize * scale) + 0.5);
	vSize = (int)((vSize * scale) + 0.5);
    }
    psPtr->pageScale = scale;
    if (psPtr->center) {
	if (paperWidth > hSize) {
	    x = (paperWidth - hSize) / 2;
	}
	if (paperHeight > vSize) {
	    y = (paperHeight - vSize) / 2;
	}
    }
    psPtr->left = x;
    psPtr->bottom = y;
    psPtr->right = x + hSize - 1;
    psPtr->top = y + vSize - 1;

    graphPtr->flags |= LAYOUT_NEEDED | COORDS_WORLD;
    Blt_LayoutGraph(graphPtr);
    return paperHeight;
}

/*
 * --------------------------------------------------------------------------
 *
 * PreviewImage --
 *
 * 	Generates a EPSI thumbnail of the graph.  The thumbnail is
 *	restricted to a certain size.  This is to keep the size of the
 *	PostScript file small and the processing time low.
 *
 *	The graph is drawn into a pixmap.  We then take a snapshot
 *	of that pixmap, and rescale it to a smaller image.  Finally,
 * 	the image is dumped to PostScript.
 *
 * Results:
 *	None.
 *
 * --------------------------------------------------------------------------
 */
static void
PreviewImage(graphPtr, printable)
    Graph *graphPtr;
    Printable printable;
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;
    int noBackingStore = 0;
    Pixmap drawable;
    Colorimage image;
    int nLines;
    ImageRegion region;
    Tcl_DString dString;

    /* Create a pixmap and draw the graph into it. */

    drawable = Tk_GetPixmap(graphPtr->display, Tk_WindowId(graphPtr->tkwin),
	graphPtr->width, graphPtr->height, Tk_Depth(graphPtr->tkwin));
    Blt_DrawGraph(graphPtr, drawable, noBackingStore);

    /* Get a color image from the pixmap */

    region.x = region.y = region.width = region.height = 0;
    image = Blt_DrawableToColorimage(graphPtr->tkwin, drawable,
	graphPtr->width, graphPtr->height, &region);
    Tk_FreePixmap(graphPtr->display, drawable);
    if (image == NULL) {
	return;			/* Can't grab pixmap? */
    }
#ifdef THUMBNAIL_PREVIEW
    {
	float scale, xScale, yScale;
	Colorimage destImage;
	ImageRegion destRegion;

	/* Scale the source image into a size appropriate for a thumbnail. */
#define PS_MAX_PREVIEW_WIDTH	300.0
#define PS_MAX_PREVIEW_HEIGHT	300.0
	xScale = PS_MAX_PREVIEW_WIDTH / (float)graphPtr->width;
	yScale = PS_MAX_PREVIEW_HEIGHT / (float)graphPtr->height;
	scale = MIN(xScale, yScale);

	destRegion.width = (int)(scale * region.width + 0.5);
	destRegion.height = (int)(scale * region.height + 0.5);
	destRegion.x = destRegion.y = 0;

	destImage = Blt_ResampleColorimage(image, &region, &destRegion,
	    bltBoxFilter, bltBoxFilter);
	Blt_FreeColorimage(image);
	image = destImage;
    }
#endif /* THUMBNAIL_PREVIEW */
    Blt_ColorimageToGreyscale(image, image);
    if (psPtr->landscape) {
	Colorimage oldImage;

	oldImage = image;
	image = Blt_RotateColorimage(image, 90.0);
	Blt_FreeColorimage(oldImage);
    }
    Tcl_DStringInit(&dString);
    /* Finally, we can generate PostScript for the image */
    nLines = Blt_ColorimageToPsData(image, 1, &dString, "%");

    Tcl_DStringAppend(printable->dStrPtr, "%%BeginPreview: ", -1);
    Blt_PrintFormat(printable, "%d %d 8 %d\n", ColorimageWidth(image),
	ColorimageHeight(image), nLines);
    Tcl_DStringAppend(printable->dStrPtr, Tcl_DStringValue(&dString), -1);
    Tcl_DStringAppend(printable->dStrPtr, "%%EndPreview\n\n", -1);
    Tcl_DStringFree(&dString);
    Blt_FreeColorimage(image);
}

/*
 * --------------------------------------------------------------------------
 *
 * PrintPreamble
 *
 *    	The PostScript preamble calculates the needed translation and scaling
 *    	to make X11 coordinates compatible with PostScript.
 *
 * ---------------------------------------------------------------------
 */

#ifdef TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif /* HAVE_SYS_TIME_H */
#endif /* TIME_WITH_SYS_TIME */

static int
PostScriptPreamble(graphPtr, fileName, printable)
    Graph *graphPtr;
    char *fileName;
    Printable printable;
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;
    time_t ticks;
    char date[200];		/* Hold the date string from ctime() */
    char *version;
    float dpiX, dpiY;
    float xPixelsToPica, yPixelsToPica;	/* Scales to convert pixels to pica */
    Screen *screenPtr;
    char *nl;
    int paperHeightPixels;

    paperHeightPixels = ComputeBoundingBox(graphPtr, psPtr);
    if (fileName == NULL) {
	fileName = Tk_PathName(graphPtr->tkwin);
    }
    Blt_PrintAppend(printable, "%!PS-Adobe-3.0 EPSF-3.0\n", (char *)NULL);

    /*
     * Compute the scale factors to convert PostScript to X11 coordinates.
     * Round the pixels per inch (dpi) to an integral value before computing
     * the scale.
     */
#define MM_INCH (float)25.4
#define PICA_INCH (float)72.0

    screenPtr = Tk_Screen(graphPtr->tkwin);
    dpiX = (WidthOfScreen(screenPtr) * MM_INCH) / WidthMMOfScreen(screenPtr);
    xPixelsToPica = PICA_INCH / dpiX;
    dpiY = (HeightOfScreen(screenPtr) * MM_INCH) / HeightMMOfScreen(screenPtr);
    yPixelsToPica = PICA_INCH / dpiY;

    /*
     * The "BoundingBox" comment is required for EPS files. The box
     * coordinates are integers, so we need round away from the
     * center of the box.
     */
    Blt_PrintFormat(printable, "%%%%BoundingBox: %d %d %d %d\n",
	(int)floor(psPtr->left * xPixelsToPica),
	(int)floor((paperHeightPixels - psPtr->top) * yPixelsToPica),
	(int)ceil(psPtr->right * xPixelsToPica),
	(int)ceil((paperHeightPixels - psPtr->bottom) * yPixelsToPica));

    Blt_PrintAppend(printable, "%%Pages: 0\n", (char *)NULL);

    version = Tcl_GetVar(graphPtr->interp, "blt_version", TCL_GLOBAL_ONLY);
    if (version == NULL) {
	version = "???";
    }
    Blt_PrintFormat(printable, "%%%%Creator: (BLT %s %s)\n", version,
	Tk_Class(graphPtr->tkwin));

    ticks = time((time_t *) NULL);
    strcpy(date, ctime(&ticks));
    nl = date + strlen(date) - 1;
    if (*nl == '\n') {
	*nl = '\0';
    }
    Blt_PrintFormat(printable, "%%%%CreationDate: (%s)\n", date);
    Blt_PrintFormat(printable, "%%%%Title: (%s)\n", fileName);
    Blt_PrintAppend(printable, "%%%%DocumentData: Clean7Bit\n", (char *)NULL);
    Blt_PrintFormat(printable, "%%Orientation: %s\n",
	(psPtr->landscape) ? "Landscape" : "Portrait");
    Blt_PrintAppend(printable,
	"%%DocumentNeededResources: font Helvetica Courier\n", (char *)NULL);
    Blt_PrintAppend(printable, "%%EndComments\n", (char *)NULL);
    if (psPtr->addPreview) {
	PreviewImage(graphPtr, printable);
    }
    if (Blt_FileToPostScript(printable, "bltGraph.pro") != TCL_OK) {
	return TCL_ERROR;
    }
    if (psPtr->footer) {
	char *who;

	who = getenv("LOGNAME");
	if (who == NULL) {
	    who = "???";
	}
	Blt_PrintAppend(printable,
	    "8 /Helvetica SetFont\n",
	    "10 30 moveto\n",
	    "(Date: ", date, ") show\n",
	    "10 20 moveto\n",
	    "(File: ", fileName, ") show\n",
	    "10 10 moveto\n",
	    "(Created by: ", who, "@", Tcl_GetHostName(), ") show\n",
	    "0 0 moveto\n",
	    (char *)NULL);
    }
    /*
     * Set the conversion from PostScript to X11 coordinates.  Scale
     * pica to pixels and flip the y-axis (the origin is the upperleft
     * corner).
     */
    Blt_PrintAppend(printable,
	"% Transform coordinate system to use X11 coordinates\n\n",
	"% 1. Flip y-axis over by reversing the scale,\n",
	"% 2. Translate the origin to the other side of the page,\n",
	"%    making the origin the upper left corner\n", (char *)NULL);
    Blt_PrintFormat(printable, "%g -%g scale\n", xPixelsToPica, yPixelsToPica);
    /* Papersize is in pixels.  Translate the new origin *after*
     * changing the scale. */
    Blt_PrintFormat(printable, "0 %d translate\n\n", -paperHeightPixels);
    Blt_PrintAppend(printable, "% User defined page layout\n\n",
	"%% Set color level\n", (char *)NULL);
    Blt_PrintFormat(printable, "/CL %d def\n\n", psPtr->colorMode);
    Blt_PrintFormat(printable, "%% Set origin\n%d %d translate\n\n",
	psPtr->left, psPtr->bottom);
    if (psPtr->landscape) {
	Blt_PrintFormat(printable,
	    "%% Landscape orientation\n0 %g translate\n-90 rotate\n",
	    ((float)graphPtr->width * psPtr->pageScale));
    }
    if (psPtr->pageScale != 1.0) {
	Blt_PrintAppend(printable, "\n%% Setting graph scale factor\n",
	    (char *)NULL);
	Blt_PrintFormat(printable, " %g %g scale\n", psPtr->pageScale,
	    psPtr->pageScale);
    }
    Blt_PrintAppend(printable, "\n%%EndSetup\n\n", (char *)NULL);
    return TCL_OK;
}


static void
PrintMargins(graphPtr, printable)
    Graph *graphPtr;
    Printable printable;
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;
    XRectangle margin[4];

    margin[0].x = margin[0].y = margin[3].x = margin[1].x = 0;
    margin[0].width = margin[3].width = graphPtr->width;
    margin[0].height = graphPtr->y2;
    margin[3].y = graphPtr->y1;
    margin[3].height = graphPtr->height - graphPtr->y1;
    margin[2].y = margin[1].y = graphPtr->y2;
    margin[1].width = graphPtr->x1;
    margin[2].height = margin[1].height = graphPtr->y1 - graphPtr->y2;
    margin[2].x = graphPtr->x2;
    margin[2].width = graphPtr->width - graphPtr->x2;

    /* Clear the surrounding margins and clip the plotting surface */
    if (psPtr->decorations) {
	Blt_BackgroundToPostScript(printable,
	    Tk_3DBorderColor(graphPtr->border));
    } else {
	Blt_ClearBackgroundToPostScript(printable);
    }
    Blt_RectanglesToPostScript(printable, margin, 4);

    /* Interior 3D border */
    if ((psPtr->decorations) && (graphPtr->plotBW > 0)) {
	int x, y, width, height;

	x = graphPtr->x1 - graphPtr->plotBW;
	y = graphPtr->y2 - graphPtr->plotBW;
	width = (graphPtr->x2 - graphPtr->x1) + (2 * graphPtr->plotBW);
	height = (graphPtr->y1 - graphPtr->y2) + (2 * graphPtr->plotBW);
	Blt_Draw3DRectangleToPostScript(printable, graphPtr->border, x, y,
	    width, height, graphPtr->plotBW, graphPtr->plotRelief);
    }
    if (GetLegendSite(graphPtr) < LEGEND_PLOT) {
	/*
	 * Print the legend if we're using a site which lies in one
	 * of the margins (left, right, top, or bottom) of the graph.
	 */
	Blt_PrintLegend(graphPtr, printable);
    }
    if (graphPtr->titleText != NULL) {
	Blt_PrintText(printable, graphPtr->titleText,
	    &(graphPtr->titleStyle), graphPtr->titleX, graphPtr->titleY);
    }
    Blt_PrintAxes(graphPtr, printable);
}


static int
GraphToPostScript(graphPtr, ident, printable)
    Graph *graphPtr;
    char *ident;		/* Identifier string (usually the filename) */
    Printable printable;
{
    Legend *legendPtr = graphPtr->legendPtr;
    int x, y, width, height;
    int result = TCL_ERROR;
    LegendSite site;

    Tcl_DStringInit(printable->dStrPtr);
    result = PostScriptPreamble(graphPtr, ident, printable);
    if (result != TCL_OK) {
	goto error;
    }
    /*
     * Determine rectangle of the plotting area for the graph window
     */
    x = graphPtr->x1 - graphPtr->plotBW;
    y = graphPtr->y2 - graphPtr->plotBW;

    width = (graphPtr->x2 - graphPtr->x1 + 1) + (2 * graphPtr->plotBW);
    height = (graphPtr->y1 - graphPtr->y2 + 1) + (2 * graphPtr->plotBW);

    Blt_FontToPostScript(printable, graphPtr->titleStyle.font);
    if (graphPtr->postscript->decorations) {
	Blt_BackgroundToPostScript(printable, graphPtr->plotBg);
    } else {
	Blt_ClearBackgroundToPostScript(printable);
    }
    Blt_RectangleToPostScript(printable, x, y, width, height);
    Blt_PrintAppend(printable, "gsave clip\n\n", (char *)NULL);
    /* Draw the grid, elements, and markers in the plotting area. */
    site = GetLegendSite(graphPtr);

    if (!graphPtr->gridPtr->hidden) {
	Blt_PrintGrid(graphPtr, printable);
    }
    Blt_PrintMarkers(graphPtr, printable, TRUE);
    if ((site >= LEGEND_PLOT) && (!legendPtr->raised)) {
	/* Print legend underneath elements and markers */
	Blt_PrintLegend(graphPtr, printable);
    }
    Blt_PrintAxisLimits(graphPtr, printable);
    Blt_PrintElements(graphPtr, printable);
    if ((site >= LEGEND_PLOT) && (legendPtr->raised)) {
	/* Print legend above elements (but not markers) */
	Blt_PrintLegend(graphPtr, printable);
    }
    Blt_PrintMarkers(graphPtr, printable, FALSE);
    Blt_PrintActiveElements(graphPtr, printable);
    Blt_PrintAppend(printable, "\n",
	"% Unset clipping\n",
	"grestore\n\n", (char *)NULL);
    PrintMargins(graphPtr, printable);
    Blt_PrintAppend(printable,
	"showpage\n",
	"%Trailer\n",
	"grestore\n",
	"end\n",
	"%EOF\n", (char *)NULL);
  error:
    /* Reset height and width of graph window */
    graphPtr->width = Tk_Width(graphPtr->tkwin);
    graphPtr->height = Tk_Height(graphPtr->tkwin);
    graphPtr->flags = COORDS_WORLD;

    /*
     * Redraw the graph in order to re-calculate the layout as soon as
     * possible. This is in the case the crosshairs are active.
     */
    Blt_EventuallyRedrawGraph(graphPtr);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * OutputOp --
 *
 *      This procedure is invoked to print the graph in a file.
 *
 * Results:
 *      Standard TCL result.  TCL_OK if plot was successfully printed,
 *	TCL_ERROR otherwise.
 *
 * Side effects:
 *      A new PostScript file is created.
 *
 *----------------------------------------------------------------------
 */
static int
OutputOp(graphPtr, interp, argc, argv)
    Graph *graphPtr;		/* Graph widget record */
    Tcl_Interp *interp;
    int argc;			/* Number of options in argv vector */
    char **argv;		/* Option vector */
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;
    int result = TCL_ERROR;
    Tcl_DString dString;
    FILE *f = NULL;
    Printable printable;
    char *fileName;		/* Name of file to write PostScript output
                                 * If NULL, output is returned via
                                 * interp->result. */
    fileName = NULL;
    if (argc > 3) {
	if (argv[3][0] != '-') {
	    fileName = argv[3];	/* First argument is the file name. */
	    argv++, argc--;
	}
	if (Tk_ConfigureWidget(interp, graphPtr->tkwin, configSpecs, argc - 3,
		argv + 3, (char *)psPtr, TK_CONFIG_ARGV_ONLY) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (fileName != NULL) {
	    f = fopen(fileName, "w");
	    if (f == NULL) {
		Tcl_AppendResult(interp, "can't create \"", fileName, "\": ",
		    Tcl_PosixError(interp), (char *)NULL);
		return TCL_ERROR;
	    }
	}
    }
    Tcl_DStringInit(&dString);
    printable = Blt_PrintObject(graphPtr->interp, graphPtr->tkwin, &dString);
    printable->fontVarName = psPtr->fontVarName;
    printable->colorVarName = psPtr->colorVarName;
    printable->colorMode = psPtr->colorMode;

    if (GraphToPostScript(graphPtr, fileName, printable) != TCL_OK) {
	goto error;
    }
    /*
     * If a file name was given, write the results to that file
     */
    if (f != NULL) {
	fputs(Tcl_DStringValue(&dString), f);
	if (ferror(f)) {
	    Tcl_AppendResult(interp, "error writing file \"", fileName, "\": ",
		Tcl_PosixError(interp), (char *)NULL);
	    goto error;
	}
    } else {
	Tcl_DStringResult(interp, &dString);
    }
    result = TCL_OK;

  error:
    if (f != NULL) {
	fclose(f);
    }
    Tcl_DStringFree(&dString);
    free((char *)printable);

    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_CreatePostScript --
 *
 *      Creates a postscript structure.
 *
 * Results:
 *      Always TCL_OK.
 *
 * Side effects:
 *      A new PostScript structure is created.
 *
 *----------------------------------------------------------------------
 */
int
Blt_CreatePostScript(graphPtr)
    Graph *graphPtr;
{
    PostScript *psPtr;

    psPtr = (PostScript *)calloc(1, sizeof(PostScript));
    assert(psPtr);
    psPtr->colorMode = PS_MODE_COLOR;
    psPtr->center = TRUE;
    psPtr->decorations = TRUE;
    graphPtr->postscript = psPtr;

    if (Blt_ConfigureWidgetComponent(graphPtr->interp, graphPtr->tkwin,
	    "postscript", "Postscript", configSpecs, 0, (char **)NULL,
	    (char *)psPtr, 0) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Blt_PostScriptOp --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */
static Blt_OpSpec psOps[] =
{
    {"cget", 2, (Blt_Operation)CgetOp, 4, 4, "option",},
    {"configure", 2, (Blt_Operation)ConfigureOp, 3, 0, "?option value?...",},
    {"output", 1, (Blt_Operation)OutputOp, 3, 0,
	"?fileName? ?option value?...",},
};

static int nPsOps = sizeof(psOps) / sizeof(Blt_OpSpec);

int
Blt_PostScriptOp(graphPtr, interp, argc, argv)
    Graph *graphPtr;		/* Graph widget record */
    Tcl_Interp *interp;
    int argc;			/* # arguments */
    char **argv;		/* Argument list */
{
    Blt_Operation proc;
    int result;

    proc = Blt_GetOperation(interp, nPsOps, psOps, BLT_OPER_ARG2,
	argc, argv);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (graphPtr, interp, argc, argv);
    return result;
}
