#ifndef MX_H
#define MX_H

/* 
  mx -- Marc's eXtension modules for Python: basic macros

  This file is only meant to be included by the extension modules.
  DO NOT include it in the extension module's header file, since it
  will definitely cause troubles then.

  To enable debugging ceratin things, define one of these before
  including this file:

  MAL_REF_DEBUG -- debug reference counts (Py_MY_xxx) [this file]
  MAL_DEBUG     -- enable debug output (DPRINTF) [mxstdlib.h]
  MAL_MEM_DEBUG -- enable malloc output (new,cnew,free,...) [mxstdlib.h]

  (c) Marc-Andre Lemburg; All Rights Reserved; mailto: mal@lemburg.com
  See the documentation for further copyright information or contact
  the author.

*/

/* Include the generic mx header file */
#include "mxh.h"

/* Add some platform specific symbols to enable work-arounds */
#if defined(NeXT) || defined(sgi)
# define BAD_STATIC_FORWARD
#endif

/* Include nearly all Python symbols & definitions */
#include "Python.h"

/* Include other standard stuff */
#include "mxstdlib.h"

/* Include backward compatibility stuff */
#include "mxpyapi.h"

/* --- Declare macros ----------------------------------------------------- */

#define Py_NONE (Py_INCREF(Py_None),Py_None)

#ifdef MAL_REF_DEBUG
# define printref(x) printf("* refcount for "#x" = %i\n",(long) x->ob_refcnt);
#else
# define printref(x)
#endif

/* --- Error handling ----------------------------------------------------- */

#define Py_Do(x) {if (!(x)) goto onError;}
#define Py_ReturnOnError(errortype,errorstr) {PyErr_SetString(errortype,errorstr);return NULL;}

#define Py_Assert(x,errortype,errorstr) {if (!(x)) {PyErr_SetString(errortype,errorstr);goto onError;}}
#define Py_AssertWithArg(x,errortype,errorstr,a1) {if (!(x)) {PyErr_Format(errortype,errorstr,a1);goto onError;}}
#define Py_AssertWith2Args(x,errortype,errorstr,a1,a2) {if (!(x)) {PyErr_Format(errortype,errorstr,a1,a2);goto onError;}}
#define Py_AssertWith3Args(x,errortype,errorstr,a1,a2,a3) {if (!(x)) {PyErr_Format(errortype,errorstr,a1,a2,a3);goto onError;}}

#define Py_Error(errortype,errorstr) {PyErr_SetString(errortype,errorstr);goto onError;}
#define Py_ErrorWithArg(errortype,errorstr,a1) {PyErr_Format(errortype,errorstr,a1);goto onError;}
#define Py_ErrorWith2Args(errortype,errorstr,a1,a2) {PyErr_Format(errortype,errorstr,a1,a2);goto onError;}
#define Py_ErrorWith3Args(errortype,errorstr,a1,a2,a3) {PyErr_Format(errortype,errorstr,a1,a2,a3);goto onError;}

/* --- Reference counting ------------------------------------------------- */

#ifdef MAL_REF_DEBUG

static void _Py_INCREF(PyObject *v,
		       char *name,
		       char *filename,
		       int lineno)
{
    if (!Py_DebugFlag) {
	Py_XINCREF(v);
	return;
    }
    if (!v)
	mxDebugPrintf("[%s:%5i] Py_XINCREF( %-8s == NULL );\n",
		      filename,lineno,name);
    else {
	Py_INCREF(v);;
	mxDebugPrintf("[%s:%5i] Py_XINCREF( %-8s at 0x%x [%s]); "
		      "new refcount = %i\n",
		      filename,lineno,name,(int)v,v->ob_type->tp_name,
		      v->ob_refcnt);
    }
}

static void _Py_DECREF(PyObject *v,
		       char *name,
		       char *filename,
		       int lineno) 
{
    if (!Py_DebugFlag) {
	Py_XDECREF(v);
	return;
    }
    if (!v)
	mxDebugPrintf("[%s:%5i] Py_XDECREF( %-8s == NULL );\n",
		      filename,lineno,name);
    else {
	int refcnt = v->ob_refcnt;
	Py_DECREF(v);
	if (refcnt <= 1)
	    mxDebugPrintf("[%s:%5i] Py_XDECREF( %-8s at 0x%x [%s]); "
			  "object deleted\n",
			  filename,lineno,name,(int)v,v->ob_type->tp_name);
	else
	    mxDebugPrintf("[%s:%5i] Py_XDECREF( %-8s at 0x%x [%s]); "
			  "new refcount = %i\n",
			  filename,lineno,name,(int)v,v->ob_type->tp_name,
			  v->ob_refcnt);
    }
}

static void _Py_PRINT_REFCOUNT(PyObject *v,
			       char *name,
			       char *filename,
			       int lineno) 
{
    if (!v)
	mxDebugPrintf("[%s:%5i] Py_PRINT_REFCOUNT( %-8s == NULL );\n",
		      filename,lineno,name);
    else {
	mxDebugPrintf("[%s:%5i] Py_PRINT_REFCOUNT( %-8s at 0x%x [%s]) = %i;\n",
		      filename,lineno,name,(int)v,v->ob_type->tp_name,
		      v->ob_refcnt);
    }
}

# undef Py_INCREF
# define Py_INCREF(x) _Py_INCREF((PyObject *)x,#x,__FILE__,__LINE__)
# undef Py_DECREF
# define Py_DECREF(x) _Py_DECREF((PyObject *)x,#x,__FILE__,__LINE__)
# undef Py_XINCREF
# define Py_XINCREF(x) _Py_INCREF((PyObject *)x,#x,__FILE__,__LINE__)
# undef Py_XDECREF
# define Py_XDECREF(x) _Py_DECREF((PyObject *)x,#x,__FILE__,__LINE__)
# define Py_DELETE(x) {if (x->ob_refcnt > 1) mxDebugPrintf("[%s:%5i] Py_DELETE( "#x" ) WARNING: Refcount = %i > 0\n",__FILE__,__LINE__,(int)x->ob_refcnt-1);Py_DECREF(x);}
# define Py_PRINT_REFCOUNT(x) _Py_PRINT_REFCOUNT((PyObject *)x,#x,__FILE__,__LINE__)
#else
# define Py_DELETE(x) Py_DECREF(x)
# define Py_PRINT_REFCOUNT(x) 
#endif

#define Py_DEC_REF(x) {Py_XDECREF(x); x=0;} /* doing this once too often doesn't hurt */

/* --- Argument passing and checking -------------------------------------- */

/* No arguments expected; also use Py_MethodListEntryNoArgs() for this
   kind of fct */
#define Py_NoArgsCheck() {if (!PyArg_NoArgs(args)) goto onError;}

/* For functions with old style args (Py_MethodListEntrySingleArg) */
#define Py_GetArgObject(a) {a = args; if (!a) {PyErr_SetString(PyExc_TypeError,"function/method requires an argument"); goto onError;}}
#define Py_GetSingleArg(format,a1) {if (!PyArg_Parse(args,format,&a1)) goto onError;}

/* For functions with new style args: */
#define Py_GetArg(format,a1) {if (!PyArg_ParseTuple(args,format,&a1)) goto onError;}
#define Py_Get2Args(format,a1,a2) {if (!PyArg_ParseTuple(args,format,&a1,&a2)) goto onError;}
#define Py_Get3Args(format,a1,a2,a3) {if (!PyArg_ParseTuple(args,format,&a1,&a2,&a3)) goto onError;}
#define Py_Get4Args(format,a1,a2,a3,a4) {if (!PyArg_ParseTuple(args,format,&a1,&a2,&a3,&a4)) goto onError;}
#define Py_Get5Args(format,a1,a2,a3,a4,a5) {if (!PyArg_ParseTuple(args,format,&a1,&a2,&a3,&a4,&a5)) goto onError;}
#define Py_Get6Args(format,a1,a2,a3,a4,a5,a6) {if (!PyArg_ParseTuple(args,format,&a1,&a2,&a3,&a4,&a5,&a6)) goto onError;}
#define Py_Get7Args(format,a1,a2,a3,a4,a5,a6,a7) {if (!PyArg_ParseTuple(args,format,&a1,&a2,&a3,&a4,&a5,&a6,&a7)) goto onError;}
#define Py_Get8Args(format,a1,a2,a3,a4,a5,a6,a7,a8) {if (!PyArg_ParseTuple(args,format,&a1,&a2,&a3,&a4,&a5,&a6,&a7,&a8)) goto onError;}

/* For functions with keywords -- the first macro parameter must be
   the keywords array given as e.g.

   static char *keywords[] = {"first","second","third", 0};

   with an entry for every argument (in the correct order). The
   functions must be included in the method list using
   Py_MethodWithKeywordsListEntry() and be declared as
   Py_C_Function_WithKeywords().

*/
#define Py_KeywordGetArg(keywords,format,a1) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1)) goto onError;}
#define Py_KeywordGet2Args(keywords,format,a1,a2) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1,&a2)) goto onError;}
#define Py_KeywordGet3Args(keywords,format,a1,a2,a3) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1,&a2,&a3)) goto onError;}
#define Py_KeywordGet4Args(keywords,format,a1,a2,a3,a4) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1,&a2,&a3,&a4)) goto onError;}
#define Py_KeywordGet5Args(keywords,format,a1,a2,a3,a4,a5) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1,&a2,&a3,&a4,&a5)) goto onError;}
#define Py_KeywordGet6Args(keywords,format,a1,a2,a3,a4,a5,a6) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1,&a2,&a3,&a4,&a5,&a6)) goto onError;}
#define Py_KeywordGet7Args(keywords,format,a1,a2,a3,a4,a5,a6,a7) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1,&a2,&a3,&a4,&a5,&a6,&a7)) goto onError;}
#define Py_KeywordGet8Args(keywords,format,a1,a2,a3,a4,a5,a6,a7,a8) {if (!PyArg_ParseTupleAndKeywords(args,kw,format,keywords,&a1,&a2,&a3,&a4,&a5,&a6,&a7,&a8)) goto onError;}

/* New style macros fof functions supporting keywords -- the C
   variable names are used as template for the keyword list, i.e. they
   must match the Python keyword parameter names.

   Note that format strings with special parameters (e.g. "#s") are
   not allowed since they would cause the keyword list to be out of
   sync.

   The functions must be included in the method list using
   Py_MethodWithKeywordsListEntry() and be declared as
   Py_C_Function_WithKeywords().

   Example:

    Py_C_Function_WithKeywords(
        myfunction,
	"myfunction(filename,dupkeys=0,filemode=0,sectorsize=512)\n\n"
	"Returns a myobject"
	)
    {
	char *filename;
	int sectorsize = 512;
	int dupkeys = 0;
	int filemode = 0;

	Py_KeywordsGet4Args("s|iii",
			    filename,dupkeys,filemode,sectorsize);

	return (PyObject *)myobject_New(filename,
	                                filemode,
					sectorsize,
					dupkeys);
     onError:
	return NULL;
    }

*/
#define Py_KeywordsGetArg(format,a1) {static char *kwlist[] = {#a1,NULL}; if (!PyArg_ParseTupleAndKeywords(args,kw,format,kwlist,&a1)) goto onError;}
#define Py_KeywordsGet2Args(format,a1,a2) {static char *kwlist[] = {#a1,#a2,NULL}; if (!PyArg_ParseTupleAndKeywords(args,kw,format,kwlist,&a1,&a2)) goto onError;}
#define Py_KeywordsGet3Args(format,a1,a2,a3) {static char *kwlist[] = {#a1,#a2,#a3,NULL}; if (!PyArg_ParseTupleAndKeywords(args,kw,format,kwlist,&a1,&a2,&a3)) goto onError;}
#define Py_KeywordsGet4Args(format,a1,a2,a3,a4) {static char *kwlist[] = {#a1,#a2,#a3,#a4,NULL}; if (!PyArg_ParseTupleAndKeywords(args,kw,format,kwlist,&a1,&a2,&a3,&a4)) goto onError;}
#define Py_KeywordsGet5Args(format,a1,a2,a3,a4,a5) {static char *kwlist[] = {#a1,#a2,#a3,#a4,#a5,NULL}; if (!PyArg_ParseTupleAndKeywords(args,kw,format,kwlist,&a1,&a2,&a3,&a4,&a5)) goto onError;}
#define Py_KeywordsGet6Args(format,a1,a2,a3,a4,a5,a6) {static char *kwlist[] = {#a1,#a2,#a3,#a4,#a5,#a6,NULL}; if (!PyArg_ParseTupleAndKeywords(args,kw,format,kwlist,&a1,&a2,&a3,&a4,&a5,&a6)) goto onError;}
#define Py_KeywordsGet7Args(format,a1,a2,a3,a4,a5,a6,a7) {static char *kwlist[] = {#a1,#a2,#a3,#a4,#a5,#a6,#a7,NULL}; if (!PyArg_ParseTupleAndKeywords(args,kw,format,kwlist,&a1,&a2,&a3,&a4,&a5,&a6,&a7)) goto onError;}
#define Py_KeywordsGet8Args(format,a1,a2,a3,a4,a5,a6,a7,a8) {static char *kwlist[] = {#a1,#a2,#a3,#a4,#a5,#a6,#a7,#a8,NULL}; if (!PyArg_ParseTupleAndKeywords(args,kw,format,kwlist,&a1,&a2,&a3,&a4,&a5,&a6,&a7,&a8)) goto onError;}

/* --- Returning values to Python ----------------------------------------- */

/* XXX Don't always work: every time you have an 'O' in the BuildValue format
       string, you need to DECREF the variable *after* the tuple has been
       built !!!
*/

#define Py_ReturnNone() {Py_INCREF(Py_None);return Py_None;}
#define Py_ReturnTrue() {Py_INCREF(Py_True);return Py_True;}
#define Py_ReturnFalse() {Py_INCREF(Py_False);return Py_False;}
#define Py_ReturnArg(format,a1) return Py_BuildValue(format,a1);
#define Py_Return Py_ReturnArg
#define Py_Return2Args(format,a1,a2) return Py_BuildValue(format,a1,a2);
#define Py_Return2 Py_Return2Args
#define Py_Return3Args(format,a1,a2,a3) return Py_BuildValue(format,a1,a2,a3);
#define Py_Return3 Py_Return3Args
#define Py_Return4Args(format,a1,a2,a3) return Py_BuildValue(format,a1,a2,a3,a4);
#define Py_Return5Args(format,a1,a2,a3) return Py_BuildValue(format,a1,a2,a3,a4,a5);
#define Py_Return6Args(format,a1,a2,a3) return Py_BuildValue(format,a1,a2,a3,a4,a5,a6);
#define Py_Return7Args(format,a1,a2,a3) return Py_BuildValue(format,a1,a2,a3,a4,a5,a6,a7);

/* Build values */

#define Py_BuildNone() Py_NONE
#define Py_Build(format,x) Py_BuildValue(format,x)
#define Py_Build2(format,x,y) Py_BuildValue(format,x,y)
#define Py_Build3(format,x,y,z) Py_BuildValue(format,x,y,z)

/* --- Declaring Python builtin functions/methods ------------------------- */

/* Declare C function/method fct, having docstring docstr; may use vargargs */
#define Py_C_Function(fct,docstr) \
        static char fct##_docstring[] = docstr;\
        static PyObject *fct(PyObject *self, PyObject *args)

/* Declare C function/method fct, having keywords keywordsarray and a
   docstring docstr; may use vargargs & keywords */
#define Py_C_Function_WithKeywords(fct,docstr) \
        static char fct##_docstring[] = docstr;\
        static PyObject *fct(PyObject *self, PyObject *args, PyObject *kw)

/* These declare: self -- instance pointer for methods, NULL for functions
                  args -- argument tuple
		  kw   -- keywords dict (if applicable)
   plus as statics:
                  <function name>_docstring -- the docstring as given
		  <function name>_keywords  -- the keyword array as given

   Note: use the Py_GetArg macros for functions without keywords,
             and Py_KeywordGetArg macros for functions with keywords
*/

/* --- Method list entries for builtin functions/methods ------------------ */

/* Add a C function/method cname to the module dict as pyname; no
   doc-string */
#define Py_MethodListEntryAny(pyname,cname) {pyname,(PyCFunction)cname,METH_VARARGS}

/* Add a C function/method cname to the module dict as pyname; the
   function can use varargs */
#define Py_MethodListEntry(pyname,cname) {pyname,(PyCFunction)cname,METH_VARARGS,cname##_docstring}

/* Add a C function/method cname to the module dict as pyname; the
   function takes no args */
#define Py_MethodListEntryNoArgs(pyname,cname) {pyname,(PyCFunction)cname,0,cname##_docstring}

/* Add a C function/method cname to the module dict as pyname; the
   function takes one argument: the object is passed in directly
   (without wrapping it into a tuple first), i.e. don't use
   the Py_GetArg-macros or PyArg_ParseTuple(). */
#define Py_MethodListEntrySingleArg(pyname,cname) {pyname,(PyCFunction)cname,0,cname##_docstring}

/* Add a C function/method that uses keywords to the module dict */
#define Py_MethodWithKeywordsListEntry(pyname,cname) {pyname,(PyCFunction)cname,METH_VARARGS | METH_KEYWORDS,cname##_docstring}


/* --- Text macros -------------------------------------------------------- */

/* Check a given slice and apply the usual rules for negative indices */
#define Py_CheckBufferSlice(textlen,start,stop) {	\
	    if (stop > textlen)				\
		stop = textlen;				\
	    else {					\
		if (stop < 0)				\
		    stop += textlen;			\
		if (stop < 0)				\
		    stop = 0;				\
	    }						\
	    if (start < 0) {				\
		start += textlen;			\
		if (start < 0)				\
		    start = 0;				\
	    }						\
	    if (stop < start)				\
		start = stop;				\
	}

/* Dito for string objects */
#define Py_CheckSlice(textobj,start,stop) \
        Py_CheckBufferSlice(PyString_GET_SIZE(textobj),start,stop)

/* This assumes that fixed is a constant char array; the strcmp
   function is only called in case the attribute name length exceeds
   10 characters and the first 10 characters match; optimizing
   compilers should eliminate any unused parts of this comparison
   automatically */
#define Py_StringsCompareEqual(var,fixed)				\
     (var[0] == fixed[0] &&						\
      (var[0] == 0 ||							\
       (sizeof(fixed) >= 1 && var[1] == fixed[1] &&			\
	(var[1] == 0 ||							\
	 (sizeof(fixed) >= 2 && var[2] == fixed[2] &&			\
	  (var[2] == 0 ||						\
	   (sizeof(fixed) >= 3 && var[3] == fixed[3] &&			\
	    (var[3] == 0 ||						\
	     (sizeof(fixed) >= 4 && var[4] == fixed[4] &&		\
	      (var[4] == 0 ||						\
	       (sizeof(fixed) >= 5 && var[5] == fixed[5] &&		\
		(var[5] == 0 ||						\
		 (sizeof(fixed) >= 6 && var[6] == fixed[6] &&		\
		  (var[6] == 0 ||					\
		   (sizeof(fixed) >= 7 && var[7] == fixed[7] &&		\
		    (var[7] == 0 ||					\
		     (sizeof(fixed) >= 8 && var[8] == fixed[8] &&	\
		      (var[8] == 0 ||					\
		       (sizeof(fixed) >= 9 && var[9] == fixed[9] &&	\
			(var[9] == 0 || 				\
			 (sizeof(fixed) >= 10 && 			\
			  strcmp(&var[10],&fixed[10]) == 0		\
			 )))))))))))))))))))))

/* Fast character set member check; set must be a "static unsigned
   *char set" array of exactly 32 bytes length generated with
   TextTools.set() */
#define Py_CharInSet(chr,set)					\
        (((unsigned char)(set)[(unsigned char)(chr) >> 3] & 	\
	  (1 << ((unsigned char)(chr) & 7))) != 0)

/* --- Macros for getattr ------------------------------------------------- */

/* This assumes that name is a constant char array */
#define Py_WantAttr(var,name) Py_StringsCompareEqual(var,name)

/* --- Module init helpers ------------------------------------------------ */

/* Helper for startup type object initialization */

#define PyType_Init(x)						\
{								\
    x.ob_type = &PyType_Type; 					\
    Py_Assert(x.tp_basicsize >= sizeof(PyObject),		\
	      PyExc_SystemError,				\
	      "Internal error: tp_basicsize of "#x" too small");\
}

/* Error reporting for module init functions */

#define Py_ReportModuleInitError(modname) {			\
    PyObject *exc_type, *exc_value, *exc_tb;			\
    PyObject *str_type, *str_value;				\
								\
    /* Fetch error objects and convert them to strings */	\
    PyErr_Fetch(&exc_type, &exc_value, &exc_tb);		\
    if (exc_type && exc_value) {				\
	str_type = PyObject_Str(exc_type);			\
	str_value = PyObject_Str(exc_value);			\
    }								\
    else {							\
	str_type = NULL;					\
	str_value = NULL;					\
    }								\
    /* Try to format a more informative error message using the	\
       original error */					\
    if (str_type && str_value &&				\
	PyString_Check(str_type) && PyString_Check(str_value))	\
	PyErr_Format(						\
		PyExc_ImportError,				\
		"initialization of module "modname" failed "	\
		"(%s:%s)",					\
		PyString_AS_STRING(str_type),			\
		PyString_AS_STRING(str_value));			\
    else							\
	PyErr_SetString(					\
		PyExc_ImportError,				\
		"initialization of module "modname" failed");	\
    Py_XDECREF(str_type);					\
    Py_XDECREF(str_value);					\
    Py_XDECREF(exc_type);					\
    Py_XDECREF(exc_value);					\
    Py_XDECREF(exc_tb);						\
}

/* --- SWIG addons -------------------------------------------------------- */

/* Throw this error after having set the correct Python exception
   using e.g. PyErr_SetString(); */
#define mxSWIGError "mxSWIGError"

/* EOF */
#endif

