// ---------------------------------------------------------------------------
// - Integer.cpp                                                             -
// - standard object library - integer class implementation                  -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "cstring.hxx"
#include "Real.hpp"
#include "Method.hpp"
#include "Boolean.hpp"
#include "Character.hpp"
#include "Exception.hpp"

namespace aleph {

  // create a new default integer
  
  Integer::Integer (void) {
    d_value = 0;
  }

  // create a new integer from a long long integer

  Integer::Integer (const t_long value) {
    d_value = value;
  }

  // create a new integer from a string

  Integer::Integer (const String& value) {
    bool  status = false;
    const char* data = value.toChar ();
    d_value = c_atoll (data,0,status);
    delete [] data;
    if (status == false) 
      throw Exception ("literal-error",
		       "illegal string integer number",value);
  }
  
  // copy constructor for this integer

  Integer::Integer (const Integer& that) {
    d_value = that.d_value;
  }

  // return the class name

  String Integer::repr (void) const {
    return "Integer";
  }

  // return a literal representation of this integer

  String Integer::toLiteral (void) const {
    return toString ();
  }

  // get a string representation on this integer

  String Integer::toString (void) const {
    char*  buffer = c_lltoa (d_value);
    String result (buffer);
    delete [] buffer;
    return result;
  }

  // return a native integer value

  t_long Integer::toInteger (void) const {
    return d_value;
  }

  // set an integer with a value

  Integer& Integer::operator = (const long value) {
    d_value = value;
    return *this;
  }

  // set an integer with a value

  Integer& Integer::operator = (const Integer& value) {
    d_value = value.d_value;
    return *this;
  }

  // compare an integer with a native value

  bool Integer::operator == (const long value) const {
    return (d_value == value);
  }

  bool Integer::operator != (const long value) const {
    return (d_value != value);
  }

  // compare two integers

  bool Integer::operator == (const Integer& value) const {
    return (d_value == value.d_value);
  }

  bool Integer::operator != (const Integer& value) const {
    return (d_value != value.d_value);
  }

  bool Integer::operator < (const Integer& value) const {
    return (d_value < value.d_value);
  }

  bool Integer::operator <= (const Integer& value) const {
    return (d_value <= value.d_value);
  }
  
  bool Integer::operator > (const Integer& value) const {
    return (d_value > value.d_value);
  }

  bool Integer::operator >= (const Integer& value) const {
    return (d_value >= value.d_value);
  }

  // add two integers together from two integers

  Integer operator + (const Integer& x, const Integer& y) {
    return Integer (x.d_value + y.d_value);
  }

  // add an integer to this one

  Integer& Integer::operator += (const Integer& x) {
    d_value += x.d_value;
    return *this;
  }

  // substract an integer to this one
  
  Integer& Integer::operator -= (const Integer& x) {
    d_value -= x.d_value;
    return *this;
  }

  // substract two integers together from two integers
  
  Integer operator - (const Integer& x, const Integer& y) {
    return Integer (x.d_value - y.d_value);
  }
  
  // compute the opposite of the current integer
  
  Integer operator - (const Integer& x) {
    return Integer (-x.d_value);
  }
  
  // multiply two integers together from two integers
  
  Integer operator * (const Integer& x, const Integer& y) {
    return Integer (x.d_value * y.d_value);
  }
  
  // multiply an integer with this one
  
  Integer& Integer::operator *= (const Integer& x) {
    d_value *= x.d_value;
    return *this;
  }
  
  // divide two integers together from two integers
  
  Integer operator / (const Integer& x, const Integer& y) {
    if (y.d_value == 0) throw Exception ("division-by-zero");
    return Integer (x.d_value / y.d_value);
  }

  // return the absolute value of this number

  Integer Integer::abs (void) const {
  t_long result = (d_value < 0) ? -d_value : d_value;
  return Integer (result);
  }

  // return the remainder of this number with its argument

  Integer Integer::mod (const Integer& x) const {
    t_long result = d_value % x.d_value;
    return Integer (result);
  }

  // operate this integer with another object

  Object* Integer::oper (t_oper type, Object* object) {
    Integer* iobj = dynamic_cast <Integer*> (object);
    Real*    robj = dynamic_cast <Real*>    (object);
    switch (type) {
    case Object::ADD:
      if (iobj != nilp) return new Integer (d_value + iobj->d_value);
      if (robj != nilp) return new Real    (d_value + robj->d_value);
      break;
    case Object::SUB:
      if (iobj != nilp) return new Integer (d_value - iobj->d_value);
      if (robj != nilp) return new Real    (d_value - robj->d_value);
      break;
    case Object::MUL:
      if (iobj != nilp) return new Integer (d_value * iobj->d_value);
      if (robj != nilp) return new Real    (d_value * robj->d_value);
      break;
    case Object::DIV:
      if (iobj != nilp) return new Integer (*this / *iobj);
      if (robj != nilp) return new Real    (*this / *robj);
      break;
    case Object::EQL:
      if (iobj != nilp) return new Boolean (d_value == iobj->d_value);
      if (robj != nilp) return new Boolean (d_value == robj->d_value);
      break;
    case Object::NEQ:
      if (iobj != nilp) return new Boolean (d_value != iobj->d_value);
      if (robj != nilp) return new Boolean (d_value != robj->d_value);
      break;
    case Object::GEQ:
      if (iobj != nilp) return new Boolean (d_value >= iobj->d_value);
      if (robj != nilp) return new Boolean (d_value >= robj->d_value);
      break;
    case Object::GTH:
      if (iobj != nilp) return new Boolean (d_value > iobj->d_value);
      if (robj != nilp) return new Boolean (d_value > robj->d_value);
      break;
    case Object::LEQ:
      if (iobj != nilp) return new Boolean (d_value <= iobj->d_value);
      if (robj != nilp) return new Boolean (d_value <= robj->d_value);
      break;
    case Object::LTH:
      if (iobj != nilp) return new Boolean (d_value < iobj->d_value);
      if (robj != nilp) return new Boolean (d_value < robj->d_value);
      break;
    case Object::MINUS:
      return new Integer (-d_value);
      break;
    }
    throw Exception ("type-error", "invalid operand with integer",
		     Object::repr (object));
  }

  // create a new integer in a generic way

  Object* Integer::mknew (Vector* argv) {
    if ((argv == nilp) || (argv->length () == 0)) return new Integer;
    if (argv->length () != 1) 
      throw Exception ("argument-error", 
		       "too many argument with integer constructor");
    // try to map the integer argument
    Object* obj = argv->get (0);
    if (obj == nilp) return new Integer;

    // try an integer object
    Integer* ival = dynamic_cast <Integer*> (obj);
    if (ival != nilp) return new Integer (*ival);

    // try a real object
    Real* rval = dynamic_cast <Real*> (obj);
    if (rval != nilp) return new Integer (rval->toInteger ());

    // try a character object
    Character* cval = dynamic_cast <Character*> (obj);
    if (cval != nilp) return new Integer (cval->toCharacter ());

    // try a string object
    String* sval = dynamic_cast <String*> (obj);
    if (sval != nilp) return new Integer (*sval);

    // illegal object
    throw Exception ("type-error", "illegal object with integer constructor",
		     obj->repr ());
  }

  // set an object to this integer

  Object* Integer::vdef (Interp* interp, Nameset* nset, Object* object) {
    Integer* iobj = dynamic_cast <Integer*> (object);
    if (iobj != nilp) {
      d_value = iobj->d_value;
      return this;
    }
    Real* robj = dynamic_cast <Real*> (object);
    if (robj != nilp) {
      d_value = robj->toInteger ();
      return this;
    }
    throw Exception ("type-error", "invalid object with integer vdef",
		     Object::repr (object));
  }

  // evaluate this integer with a member name

  Object* Integer::eval (Interp* interp, Nameset* nset, const String& name) {
    return new Method (name, this);
  }

  // apply this integer with a method name

  Object* Integer::apply (Interp* interp, Nameset* nset, const String& name,
			  Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if ((name == "to-string") && (argc == 0)) {
      Object* result = new String (toLiteral ());
      delete argv;
      return result;
    }
    if ((name == "++") && (argc == 0)) {
      d_value++;
      delete argv;
      return this;
    }
    if ((name == "--") && (argc == 0)) {
      d_value--;
      delete argv;
      return this;
    }
    if ((name == "abs") && (argc == 0)) {
      t_long val = (d_value >= 0) ? d_value : -d_value;
      delete argv;
      return new Integer (val);
    }
    if ((name == "even-p") && (argc == 0)) {
      bool result = ((d_value & 1) == 0) ? true : false;
      delete argv;
      return new Boolean (result);
    }
    if ((name == "odd-p") && (argc == 0)) {
      bool result = ((d_value & 1) == 1) ? true : false;
      delete argv;
      return new Boolean (result);
    }
    if ((name == "zero-p") && (argc == 0)) {
      bool result = (d_value == 0);
      delete argv;
      return new Boolean (result);
    }
    if ((name == "not") && (argc == 1)) {
      Object* result = new Integer (~d_value);
      delete argv;
      return result;
    } 

    // dispatch one argument
    if ((name == "+") && (argc == 1)) {
      Object* result = oper (Object::ADD, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "-") && (argc == 1)) {
      Object* result = oper (Object::SUB, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "*") && (argc == 1)) {
      Object* result = oper (Object::MUL, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "/") && (argc == 1)) {
      Object* result = oper (Object::DIV, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "+=") && (argc == 1)) {
      t_long val = argv->getint (0);
      d_value += val;
      delete argv;
      return this;
    }
    if ((name == "-=") && (argc == 1)) {
      t_long val = argv->getint (0);
      d_value -= val;
      delete argv;
      return this;
    }
    if ((name == "*=") && (argc == 1)) {
      t_long val = argv->getint (0);
      d_value *= val;
      delete argv;
      return this;
    }
    if ((name == "/=") && (argc == 1)) {
      t_long val = argv->getint (0);
      if (val == 0) throw Exception ("divide-error", "division by zero");
      d_value = d_value / val;
      delete argv;
      return this;
    }
    if ((name == "==") && (argc == 1)) {
      Object* result = oper (Object::EQL, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "!=") && (argc == 1)) {
      Object* result = oper (Object::NEQ, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "<") && (argc == 1)) {
      Object* result = oper (Object::LTH, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "<=") && (argc == 1)) {
      Object* result = oper (Object::LEQ, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == ">") && (argc == 1)) {
      Object* result = oper (Object::GTH, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == ">=") && (argc == 1)) {
      Object* result = oper (Object::GEQ, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "mod") && (argc == 1)) {
      t_long val = argv->getint (0);
      Object* result = new Integer (d_value % val);
      delete argv;
      return result;
    }
    if ((name == "shl") && (argc == 1)) {
      t_long val = argv->getint (0);
      Object* result = new Integer (d_value << val);
      delete argv;
      return result;
    }
    if ((name == "shr") && (argc == 1)) {
      t_long val = argv->getint (0);
      Object* result = new Integer (d_value >> val);
      delete argv;
      return result;
    }
    if ((name == "xor") && (argc == 1)) {
      t_long val = argv->getint (0);
      Object* result = new Integer (d_value ^ val);
      delete argv;
      return result;
    }
    if ((name == "and") && (argc == 1)) {
      t_long val = argv->getint (0);
      Object* result = new Integer (d_value & val);
      delete argv;
      return result;
    }
    if ((name == "or") && (argc == 1)) {
      t_long val = argv->getint (0);
      Object* result = new Integer (d_value | val);
      delete argv;
      return result;
    }    

    // object method call
    Object* result = nilp;
    try {
      result =  Object::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    delete argv;
    return result;
  }
}
