// ---------------------------------------------------------------------------
// - Printer.cpp                                                             -
// - aleph engine - printer class implementation                             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Real.hpp"
#include "Interp.hpp"
#include "Printer.hpp"
#include "Boolean.hpp"
#include "Character.hpp"
#include "Exception.hpp"

namespace aleph {

  // this procedure print an object on the specified output stream.
  static inline void object_print (Output* os, Object* object) {
    if (os == nilp) return;
    // check for nilp object first
    if (object == nilp) {
      os->write ("nilp");
      return;
    }

    Character* cval = nilp;
    if ((cval = dynamic_cast <Character*> (object)) != nilp) {
      os->write (cval->toCharacter ());
      return;
    }
    
    String* sval = nilp;
    if ((sval = dynamic_cast <String*> (object)) != nilp) {
      os->write (*sval);
      return;
    }
    
    Boolean* bval = nilp;
    if ((bval = dynamic_cast <Boolean*> (object)) != nilp) {
      os->write (bval->toString());
      return;
    }
    
    Integer* ival = nilp;
    if ((ival = dynamic_cast <Integer*> (object)) != nilp) {
      os->write (ival->toString());
      return;
    }
    
    Real* rval = nilp;
    if ((rval = dynamic_cast <Real*> (object)) != nilp) {
      os->write (rval->toString());
      return;
    }
    
    throw Exception ("type-error","invalid object for printer function",
		     object->repr ());
  }

  // create a new printer class

  Printer::Printer (t_type type ) {
    d_type = type;
  }

  // return the class name

  String Printer::repr (void) const {
    return "Printer";
  }

  // apply this object on the interpreter streams

  Object* Printer::apply (Interp* interp, Nameset* nset, Cons* args) {
    // check for nilp as usual
    if (args == nilp) return nilp;

    // get the interpreter stream based on the printer type
    Output* os = nilp;
    if ((d_type == Printer::OUTPUT) || (d_type == Printer::OUTPUTLN)) 
      os  = interp->getos ();
    if ((d_type == Printer::ERROR) || (d_type == Printer::ERRORLN)) 
      os  = interp->getes ();
    
    // loop for each argument
    while (args != nilp) {
      Object* car = args->getcar ();
      Object* obj = (car == nilp) ? nilp : car->eval (interp,nset);
      object_print (os,obj);
      args = args->getcdr ();
    }

    // check for newline
    if ((d_type == Printer::OUTPUTLN) || (d_type == Printer::ERRORLN))
      os->newline ();
    return nilp;
  }
}
