// ---------------------------------------------------------------------------
// - Lexical.cpp                                                             -
// - aleph engine - lexical name class implementation                        -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Lexical.hpp"
#include "Nameset.hpp"
#include "Exception.hpp"

namespace aleph {

  // return true if the character is a valid lexical one

  bool Lexical::valid (const char c) {
    if ((c >= 'a') && (c <= 'z')) return true;
    if ((c >= 'A') && (c <= 'Z')) return true;
    if ((c >= '0') && (c <= '9')) return true;

    if (c == '.') return true;
    if (c == '+') return true;
    if (c == '-') return true;
    if (c == '*') return true;
    if (c == '/') return true;
    if (c == '!') return true;
    if (c == '=') return true;
    if (c == '.') return true;
    if (c == '>') return true;
    if (c == '<') return true;
    if (c == '?') return true;
    return false;
  }

  // return true if the name is valid lexical one

  bool Lexical::valid (const String& name) {
    long len = name.length ();
    if (len == 0) return false;
    for (long i = 0; i < len; i++)
      if (Lexical::valid (name[i]) == false) return false;
    return true;
  }

  // create a default lexical
 
  Lexical::Lexical (void) {
    d_lnum   = 0;
  }

  // create a lexical with a name

  Lexical::Lexical (const String& name) {
    if (Lexical::valid (name) == false) 
      throw Exception ("syntax-error","invalid lexical name", name);
    d_name = name;
    d_lnum = 0;
  }

  // create a lexical with a name and a line number

  Lexical::Lexical (const String& name, const long lnum) {
    if (Lexical::valid (name) == false) 
      throw Exception ("syntax-error","invalid lexical name", name);
    d_name = name;
    d_lnum = lnum;
  }
  
  // copy constructor for this lexical

  Lexical::Lexical (const Lexical& that) {
    d_name = that.d_name;
    d_lnum = that.d_lnum;
  }

  // return the class name

  String Lexical::repr (void) const {
    return "Lexical";
  }

  // set this lexical name as a const symbol

  Object* Lexical::cdef (Interp* interp, Nameset* nset, Object* object) {
    return nset->cdef (interp, nset, d_name, object);
  }

  // set this lexical name with an object

  Object* Lexical::vdef (Interp* interp, Nameset* nset, Object* object) {
    return nset->vdef (interp, nset, d_name, object);
  }

  // evaluate this lexical in the current environment

  Object* Lexical::eval (Interp* interp, Nameset* nset) {
    return nset->eval (interp,nset,d_name);
  }
}
