/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "HeightFieldWriter.h"
#include "HeightFieldIO.h"
#include "GlobalTrace.h"
#include "GlobalSanityCheck.h" 
#include "TFOptions.h"



/*
 *  constructor: initialize
 */
HeightFieldWriter::HeightFieldWriter (HeightField *HF, FileIO *Fio) 
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ HeightFieldWriter\n");

	SanityCheck::bailout ((!HF), "HF==NULL", "HeightFieldWriter::HeightFieldWriter");
	SanityCheck::bailout ((!HF->getData()), "HF->getData()==NULL", "HeightFieldWriter::HeightFieldWriter");
	SanityCheck::bailout ((!Fio), "Fio==NULL", "HeightFieldWriter::HeightFieldWriter");

	p_HF = HF;
	p_Fio = Fio;
}


/*
 *  destructor: nothing to clean up 
 */
HeightFieldWriter::~HeightFieldWriter ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- HeightFieldWriter\n");
}


/*
 *  writePGM: write a height field to a PGM ASCII file
 */
int HeightFieldWriter::writePGM ()
{
	int 	cnt; 			// count of #s printed
	int 	ix,iy;
	int	dotinterval=1;
	long 	idx;
	FILE 	*fp;
	int 	maxheight = 65535; 	// maximum value of array 
	char	*fname;
	char	buf[512];

	/* *** open PGM file *** */
	fp = p_Fio->open ("wb");
	fname = p_Fio->getName ();
	if (fp == NULL) 
		{
		fprintf (stderr, _("Can't open %s output file [%s]"), "PGM", fname);
		return(-1);
		}

	sprintf (buf, "Writing file %s (%d, %d) type PGM\nwriting: ", 
		fname, p_HF->getWidth(), p_HF->getHeight());
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);
	dotinterval=(int)(p_HF->getSize()/50);

	fprintf (fp, "P2\n");
	fprintf (fp, "%d %d\n", p_HF->getWidth(), p_HF->getHeight());
	fprintf (fp, "%d\n", maxheight);

	idx = 0;
	cnt = 0;
	for (iy = 0; iy<p_HF->getHeight(); iy++) 
		{
		for (ix = 0; ix<p_HF->getWidth(); ix++) 
			{
			fprintf(fp,"%d ",(int) (maxheight*p_HF->getEl(idx++)));
			if (++cnt > 10) 
				{ 
				fprintf (fp,"\n"); 
				cnt=0; 
				}
			if (!(idx%dotinterval) && 
			    GlobalTrace::trace(GlobalTrace::TRACE_VERBOSE, "."))
				;
    			}
    		if (cnt != 0) 
			{ 
			fprintf(fp,"\n"); 
			cnt = 0;
			}
		}

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "\n");

	p_Fio->close ();
	p_HF->setSaved (TRUE);
	return(0);
}


/*
 *  writePG8: write a height field to a PGM binary file
 */
int HeightFieldWriter::writePG8 ()
{
	int 	ix,iy;
	int	dotinterval=1;
	long 	idx;
	FILE 	*fp;
	int 	maxheight = 255; 
	char 	*fname;
	char	buf[512];
						/* open input file */
	/* *** open PG8 file *** */
	fp = p_Fio->open ("wb");
	fname = p_Fio->getName ();
	if (fp == NULL) 
		{
		fprintf (stderr, _("Can't open %s output file [%s]"), "PG8", fname);
		return(-1);
		}

	sprintf (buf, "Writing file %s (%d, %d) type PG8\nwriting: ", 
		fname, p_HF->getWidth(), p_HF->getHeight());
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);
	dotinterval=(int)(p_HF->getSize()/50);

	fprintf(fp, "P5\n");
	fprintf(fp, "%d %d\n", p_HF->getWidth(), p_HF->getHeight());
	fprintf(fp, "%d\n", maxheight);

	idx = 0;
	for (iy = 0; iy<p_HF->getHeight(); iy++) 
		for (ix = 0; ix<p_HF->getWidth(); ix++) 
			{
			fprintf(fp,"%c",(BYTE)(255*p_HF->getEl(idx++)));
			if (!(idx%dotinterval) && 
			    GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "."))
				;
			}

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "\n");

	p_Fio->close ();
	p_HF->setSaved (TRUE);
	return(0);
} 


/*
 *  writeTGA: write the Height Field to a TGA file 
 */
int HeightFieldWriter::writeTGA ()
{
	int 		ix, iy, dotinterval=1;
	FILE 		*fp;
	char		*fname;
	double 		temp;

	unsigned char 	red, green, blue;
	unsigned char 	IDLength;             /* length of Identifier String */
	unsigned char 	CoMapType;            /* 0 = no map */
	unsigned char 	ImgType;              /* image type (see below for values) */
	unsigned char 	Index_lo, Index_hi;   /* index of first color map entry */
	unsigned char 	Length_lo, Length_hi; /* number of entries in color map */
	unsigned char 	CoSize;               /* size colormap entry (15,16,24,32) */
	unsigned char 	X_org_lo, X_org_hi;   /* x origin of image */
	unsigned char 	Y_org_lo, Y_org_hi;   /* y origin of image */
	unsigned char 	Width_lo, Width_hi;   /* width of image */
	unsigned char 	Height_lo, Height_hi; /* height of image */
	unsigned char 	PixelSize;            /* pixel size (8,16,24,32) */
	long 		idx=0;

	/* *** open TGA file *** */
	fp = p_Fio->open ("wb");
	fname = p_Fio->getName ();
	if (fp == NULL) 
		{
		fprintf (stderr, _("Can't open %s output file [%s]"), "TGA", fname);
		return(-1);
		}

	if (GlobalTrace::isSet (GlobalTrace::TRACE_VERBOSE))
		{
		printf ("Writing file %s (%d, %d) type TGA\nwriting: ", 
			fname, p_HF->getWidth(), p_HF->getHeight());
		dotinterval=(int)(p_HF->getSize()/50);
		}

	/* *** Write out data to TGA format file *** */
	/* *** eighteen-byte header *** */
	/* *** b0-b7 *** */
	IDLength = 0; CoMapType = 0; ImgType = 2; Index_lo = 0; Index_hi=0;
	Length_lo = 0; Length_hi = 0; CoSize = 0; X_org_lo = 0; X_org_hi = 0;
	temp = p_HF->getWidth()/ 256;
	Width_hi = (unsigned char) temp;     /* low and high bytes of x width */
	Width_lo = p_HF->getWidth()% 256;
	temp = p_HF->getHeight()/ 256;
	Height_hi = (unsigned char) temp;
	Height_lo = p_HF->getHeight()% 256;
	PixelSize = 24;
	Y_org_lo = 0;
	Y_org_hi = 0;

	fprintf(fp ,"%c",IDLength);
	fprintf(fp ,"%c",CoMapType);
	fprintf(fp ,"%c",ImgType);
	fprintf(fp ,"%c",Index_lo);
	fprintf(fp ,"%c", Index_hi);
	fprintf(fp ,"%c",Length_lo);
	fprintf(fp ,"%c",Length_hi);
	fprintf(fp ,"%c",CoSize);
	fprintf(fp ,"%c",X_org_lo); 
	fprintf(fp ,"%c",X_org_hi);
	fprintf(fp ,"%c",Y_org_lo); 
	fprintf(fp ,"%c",Y_org_hi);
	fprintf(fp ,"%c",Width_lo);
	fprintf(fp ,"%c",Width_hi);              /* file in lo-byte, hi-byte order b12,b13*/
	fprintf(fp ,"%c",Height_lo);
	fprintf(fp ,"%c",Height_hi);    /* ysize b14, b15 */
	fprintf(fp ,"%c",PixelSize);
	fprintf(fp ,"%c",(unsigned char) 0x20);        /* descriptor byte b17 */

	blue = 0;
	for (iy = 0; iy < p_HF->getHeight(); iy++) 
		for (ix = 0; ix < p_HF->getWidth(); ix++) 
			{
			red = (UI)(65535.0*p_HF->getEl(ix,iy))/256; /* high byte */
			green = (UI)(65535.0*p_HF->getEl(ix,iy)) % 256; /* low byte */
			fprintf(fp ,"%c%c%c",blue,green,red);

			if (!(++idx%dotinterval) &&
			    GlobalTrace::trace(GlobalTrace::TRACE_VERBOSE, "."))
				;
			}

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "\n");

	p_Fio->close ();
	p_HF->setSaved (TRUE);
	return(0);
} 



/*
 * writeMAT: write out data matrix in Matlab Level 1.0 format
 * 	     creates data header fhdr from xsize, ysize and
 *	     writes data array 'hf'
 */
int HeightFieldWriter::writeMAT ()
{
	FILE 	*fp;
	Fmatrix fhdr;
	int 	rv,mn,i,j,c;
	char	*fname;
	char	*pname = "topo";
	int 	dotinterval=1;
	long	idx=0;
	PTYPE	v;

	/* *** open MAT file *** */
	fp = p_Fio->open ("wb");
	fname = p_Fio->getName ();
	if (fp == NULL) 
		{
		fprintf (stderr, _("Can't open %s output file [%s]"), "MAT", fname);
		return(-1);
		}

	if (GlobalTrace::isSet (GlobalTrace::TRACE_VERBOSE))
		{
		printf ("Writing file %s (%d, %d) type MAT\nwriting: ", 
			fname, p_HF->getWidth(), p_HF->getHeight());
		dotinterval=(int)(p_HF->getSize()/50);
		}

	fhdr.type = NUM_FMT*1000 + PREC*10;
	fhdr.ncols = p_HF->getWidth();
	fhdr.mrows = p_HF->getHeight();
	fhdr.imagf = FALSE;
	fhdr.namelen = strlen(pname) + 1;

	// write header 
	rv = fwrite (&fhdr, sizeof(Fmatrix),(size_t)1, fp); 
	if (rv != 1) 
		{
		perror("Error writing file.\n");
 		return(1);
		}

	// write mx name 
	rv = fwrite(pname, sizeof(char), (size_t)fhdr.namelen, fp);   
	if (rv != fhdr.namelen) 
		{
		perror("Error writing file.\n");
		return(1);
		}
	mn = fhdr.mrows * fhdr.ncols;

	for (i=0; i<p_HF->getWidth(); i++) 
		for (j=0; j<p_HF->getHeight(); j++) 
			{   
			v = p_HF->getEl(i,j);
       			c = fwrite (&v, sizeof(PTYPE), 1, fp);
       			if (c != 1) 
				{
				perror ("Trouble writing data to file.\n");
				return(1);
				}
			if (!(++idx%dotinterval) &&
			    GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "."))
				;
			} 

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "\n");

	if (fclose( fp ) == EOF) 
		{
		perror("Error closing file.\n");
		return(1);
		}

	p_Fio->close ();
	p_HF->setSaved (TRUE);
	return(0);
} 


/*
 *  writeOCT: write the Height Field to a OCT file 
 */
int HeightFieldWriter::writeOCT ()
{
	int 	ix,iy;
	long 	idx;
	FILE 	*fp;
	char 	*fname,
		*mname = "topo";
	int	dotinterval=1;

	/* *** open OCT file *** */
	fp = p_Fio->open ("wb");
	fname = p_Fio->getName ();
	if (fp == NULL) 
		{
		fprintf (stderr, _("Can't open %s output file [%s]"), "OCT", fname);
		return(-1);
		}

	if (GlobalTrace::isSet (GlobalTrace::TRACE_VERBOSE))
		{
		printf ("Writing file %s (%d, %d) type OCT\nwriting: ", 
			fname, p_HF->getWidth(), p_HF->getHeight());
		dotinterval=(int)(p_HF->getSize()/50);
		}

	fprintf (fp,"# name: %s\n", mname);
	fprintf (fp,"# type: matrix\n");
	fprintf (fp,"# rows: %d\n", p_HF->getHeight());
	fprintf (fp,"# columns: %d\n", p_HF->getWidth());
  
	idx = 0;
	for (iy = 0; iy<p_HF->getHeight(); iy++) 
		{
		for (ix = 0; ix<p_HF->getWidth(); ix++) 
			{
			fprintf (fp,"%e ", p_HF->getEl(idx++));
			if (!(idx%dotinterval) &&
			    GlobalTrace::trace(GlobalTrace::TRACE_VERBOSE, "."))
				;
			}
		fprintf(fp,"\n");
		}

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "\n");

	p_Fio->close ();
	p_HF->setSaved (TRUE);
	return 0;
}


/*
 * writeAC: write the Height Field to a AC file 
 * 	The strategy is to create a HeightFieldModel of the field and then to 
 * 	save the 3D Data to the file.... 
 *
 * function created by Roland Reckel (reckel@stud.montefiore.ulg.ac.be)
 */
int HeightFieldWriter::writeAC (HeightFieldDraw *HFD)
{
	FILE  			*fp;
	char			*fname;
	HeightFieldModel 	*HFM;
	GuiColormap		*cMap;
	TriMod3D 		*tModel3D;
	int 			i, limVer,
				resAdj = 2;		// resolution adjustment
	float			yScaleAdj = 0.5;

	
	fp = p_Fio->open ("wb");
	fname = p_Fio->getName ();
	if (fp == NULL) 
		{
		fprintf (stderr, _("Can't open %s output file [%s]"), "AC", fname);
		return -1;
		}

	// figure out scaling factors of Model
	if (TFOptions::s_fastHalfRes)
		resAdj = 2;
        else
		resAdj = 1;

        if (TFOptions::s_fastHalfYscale)
		yScaleAdj = 0.5;
        else
		yScaleAdj = 1;


	HFM = new HeightFieldModel(p_HF);
	cMap = HFD->getColormap ();
	//TODO: change resolution dialog etc......
	HFM->refresh(cMap, TFOptions::s_resolution*resAdj,
		TFOptions::s_yscale*yScaleAdj, TFOptions::s_fillSea); 
	tModel3D = HFM->get3DModel();
	limVer = HFM->getNumVer();

	fprintf (fp,"AC3Db\n");
	//TODO: set good materials
	fprintf (fp,"MATERIAL \"\" rgb 1 1 1  amb 0.2 0.2 0.2  emis 0 0 0 spec 0.5 0.5 0.5  shi 10  trans 0\n"); //TODO: set good materials
	fprintf (fp,"OBJECT world\n");
	fprintf (fp,"kids 1\n");
	fprintf (fp,"OBJECT poly\n");
	fprintf (fp,"name \"terrain\"\n");
	fprintf (fp,"numvert %d\n", limVer);

	for(i=0; i<limVer; i++)
		fprintf (fp,"%f %f %f\n", tModel3D->data[i].x,
			tModel3D->data[i].y, tModel3D->data[i].z);

	fprintf (fp,"numsurf %d\n", tModel3D->nTri);
	for(i=0; i<tModel3D->nTri; i++) 
		{
		fprintf (fp, "SURF 0x20\n");
		fprintf (fp, "mat 0\n");
		fprintf (fp, "refs 3\n");
		fprintf (fp, "%d 0 0\n", tModel3D->tri[i].v1);
		fprintf (fp, "%d 0 0\n", tModel3D->tri[i].v2);
		fprintf (fp, "%d 0 0\n", tModel3D->tri[i].v3);
		}
	fprintf (fp, "kids 0\n");

	delete HFM;
	p_Fio->close ();
	p_HF->setSaved (TRUE);
	return 0;
}
