/*
                 Recycled Objects Buffer Management

	Functions:

        int DBRecycleBufferInit(long entries)
        void DBRecycleBufferDeleteAll()

        int DBSaveRecycledObject(long object_num)
        long DBRecoverRecycledObject(char *name, long owner)

	---

 */

#include "swserv.h"
#include "../include/unvfile.h"


/*      
 *      Procedure to allocate recycled objects buffers.
 */
int DBRecycleBufferInit(long entries)
{
        long recbuf_count;
        int recbufs_allocated;


        /* Entries must be valid. */
        if(entries < 1)
        {
            fprintf(stderr,
  "DBAllocateRecycleBuffer(): Warning: Ignoring request to allocate none.\n"
            );
            return(-1);
        }
  
        /* Free all currently allocated recycle buffers. */
        DBRecycleBufferDeleteAll();

        recbufs_allocated = 0;
        
        /* ************************************************************** */
        
        /* Adjust global variable total_recycled_objects. */
        total_recycled_objects = entries;
 
        /* Allocate recycle buffer pointer array. */
        recycled_xsw_object = (xsw_object_struct **)
            realloc(recycled_xsw_object, entries *
                sizeof(xsw_object_struct *)
	);
        if(recycled_xsw_object == NULL)
        {
	    total_recycled_objects = 0;
            return(-1);
        }

        /* Allocate each recycle buffer. */
        for(recbuf_count = 0; recbuf_count < entries; recbuf_count++)
        {
            /* Allocate recycle buffer. */
            recycled_xsw_object[recbuf_count] = (xsw_object_struct *)calloc(
		1,
		sizeof(xsw_object_struct)
	    );
            if(recycled_xsw_object[recbuf_count] == NULL)
            {
		total_recycled_objects = recbuf_count;
                return(-1);   
            }

            recbufs_allocated++;

            /* Set type to garbage. */
            recycled_xsw_object[recbuf_count]->type = XSW_OBJ_TYPE_GARBAGE;
        }


        return(recbufs_allocated);
}


/*
 *      Procedure to deallocate all objects in recycled
 *      buffers list.
 */
void DBRecycleBufferDeleteAll()
{
        long i;


#ifdef DEBUG_MEM_FREE
printf("Recycled objects: Deleting %i...\n", total_recycled_objects);
#endif
        
        for(i = 0; i < total_recycled_objects; i++)
        {
#ifdef DEBUG_MEM_FREE
if(recycled_xsw_object[i] != NULL)
    printf("Recycled object %i: Free'ed.\n", i);
#endif
            DBDeleteObject(recycled_xsw_object[i]);
            recycled_xsw_object[i] = NULL;
        }
 
        /* Free recycle objects pointer. */
#ifdef DEBUG_MEM_FREE
if(recycled_xsw_object != NULL)
    printf("Recycled objects pointers: Free'ed.\n");
#endif
        free(recycled_xsw_object);
        recycled_xsw_object = NULL;
        
        total_recycled_objects = 0;


        return;
}



/*
 *      Copy object_num to the recycled objects list.
 */
int DBSaveRecycledObject(long object_num)
{
        int i, n;


        /* Error checks. */
        if(recycled_xsw_object == NULL)
            return(-1);
        if(DBIsObjectGarbage(object_num))
            return(-1);
        
    
        /* ********************************************************** */

	/* Look for available recycled object structure. */
	for(i = 0; i < total_recycled_objects; i++)
	{
 	    if(recycled_xsw_object[i] == NULL)
                continue;
	    if(recycled_xsw_object[i]->type <= XSW_OBJ_TYPE_GARBAGE)
		break;
	}
	if(i < total_recycled_objects)
	{
	    n = i;
	}
	else
	{
	    /*   Could not find available recycled object structure, so
	     *   make room for more.
	     */

            /* Deallocate sub resources of highest recycled object. */
            UNVResetObject(recycled_xsw_object[total_recycled_objects - 1]);

            /* `Shift' the recycle buffers. */
            for(i = total_recycled_objects - 1; i > 0; i--)
            {
                if(recycled_xsw_object[i] == NULL)
		    continue;
 
                memcpy(
		    recycled_xsw_object[i],         /* Destination. */
                    recycled_xsw_object[i - 1],     /* Source. */
                    sizeof(xsw_object_struct)
                );
            }

	    n = 0;
   	}

        /* ********************************************************** */
        /* Copy the object to available recycled object structure n. */

        /* Structure. */
        memcpy(
	    recycled_xsw_object[n],     /* Destination. */
            xsw_object[object_num],     /* Source. */
            sizeof(xsw_object_struct)
        );

        /* Weapons. */ 
        if(xsw_object[object_num]->weapons != NULL)
        {
            /* Allocate weapon pointers on recycle object. */
            recycled_xsw_object[n]->weapons = (xsw_weapons_struct **)calloc(
                1,
                recycled_xsw_object[n]->total_weapons *
                    sizeof(xsw_weapons_struct *)
            );
            if(recycled_xsw_object[n]->weapons == NULL)
	    {
		recycled_xsw_object[n]->total_weapons = 0;
		return(-1);
	    }

            /* Allocate each weapon */
            for(i = 0; i < recycled_xsw_object[n]->total_weapons; i++)
            {
                recycled_xsw_object[n]->weapons[i] = (xsw_weapons_struct *)calloc(
		    1,
		    sizeof(xsw_weapons_struct)
                );
                if(recycled_xsw_object[n]->weapons[i] == NULL)
		{
		    return(-1);
		}
                memcpy(
		    recycled_xsw_object[n]->weapons[i], /* Destination. */
                    xsw_object[object_num]->weapons[i],	/* Source. */
                    sizeof(xsw_weapons_struct)
                );
            } 
        }
            
        /* Scores. */
        if(xsw_object[object_num]->score != NULL)
        {     
            /* Allocate score structure. */
            recycled_xsw_object[n]->score = (xsw_score_struct *)calloc(
		1,
                sizeof(xsw_score_struct)
            );
            if(recycled_xsw_object[n]->score == NULL)
		return(-1);

            memcpy(
		recycled_xsw_object[n]->score,  /* Destination. */
                xsw_object[object_num]->score,	/* Source. */
                sizeof(xsw_score_struct)
            );
        }
 
        /* Economy. */
        if(xsw_object[object_num]->eco != NULL)
        {
            recycled_xsw_object[n]->eco = (xsw_ecodata_struct *)calloc(
		1,
                sizeof(xsw_ecodata_struct)
            );
            if(recycled_xsw_object[n]->eco == NULL)
		return(-1);

            memcpy(
		recycled_xsw_object[n]->eco,	/* Destination. */
                xsw_object[object_num]->eco,	/* Source. */
                sizeof(xsw_ecodata_struct)
            );

	    /* Economy products. */
	    if(recycled_xsw_object[n]->eco->total_products > 0)
	    {
                recycled_xsw_object[n]->eco->product =
		    (xsw_ecoproduct_struct **)calloc(
                    1,
                    recycled_xsw_object[n]->eco->total_products *
                        sizeof(xsw_ecoproduct_struct *)
                );
                if(recycled_xsw_object[n]->eco->product == NULL)
                {
                    recycled_xsw_object[n]->eco->total_products = 0;
                    return(-1);
                }

                for(i = 0; i < recycled_xsw_object[n]->eco->total_products; i++)
                {
                    recycled_xsw_object[n]->eco->product[i] =
			(xsw_ecoproduct_struct *)calloc(
			    1, sizeof(xsw_ecoproduct_struct)
                    );
                    if(recycled_xsw_object[n]->eco->product[i] == NULL)
                    {
                        return(-1);
                    }
                    memcpy(
                        recycled_xsw_object[n]->eco->product[i], /* Destination. */
                        xsw_object[object_num]->eco->product[i], /* Source. */
                        sizeof(xsw_ecoproduct_struct)
                    );
                }
	    }
	    else
	    {
		recycled_xsw_object[n]->eco->product = NULL;		
	    }
        }


        return(0);
}



/*
 *      Recover an object in the recycled objects list.
 */                   
long DBRecoverRecycledObject(char *name, long owner)
{
        long i, rec_object_num, object_num;   


        /* Error checks. */
        if(recycled_xsw_object == NULL)
            return(-1);

        /* Owner must be valid. */
        if(DBIsObjectGarbage(owner))
            return(-2);


        /*   Search for object in recycled objects buffer,
	 *   starting from oldest.
	 */
        for(i = total_recycled_objects - 1; i >= 0; i--)
        {
            /* Skip garbage objects. */
            if(recycled_xsw_object[i] == NULL)
                continue;
            if(recycled_xsw_object[i]->type <= XSW_OBJ_TYPE_GARBAGE)
                continue;

            /* Check if this recycled object is a player. */
            if(recycled_xsw_object[i]->type != XSW_OBJ_TYPE_PLAYER)
            {
		/* Recycled object is not a player. */

		/* Check if owner owns this recycled object. */
                if(recycled_xsw_object[i]->owner != owner)
                    continue;
            }
            else
            {
		/*   Recycled object is a player, so check if owner has
		 *   create player permission.
		 */
                if(xsw_object[owner]->permission.uid > ACCESS_UID_CREATEPLAYER)
                    continue;
            }

            /* Match name. */
            if(strstr(recycled_xsw_object[i]->name, name) == NULL)
                continue;

            /* Found recycled object that matches name and owner. */
            break;
        }
        rec_object_num = i;


        /* Did we find anything? */
        if((rec_object_num < 0) || (rec_object_num >= total_recycled_objects))
        {
            return(-1);
        }


        /* ******************************************************* */
        /* Recover object. */

	/* Create a new object. */
        object_num = DBCreateObject(
            recycled_xsw_object[rec_object_num]->imageset,
            recycled_xsw_object[rec_object_num]->type,
            recycled_xsw_object[rec_object_num]->owner,
            recycled_xsw_object[rec_object_num]->x,
            recycled_xsw_object[rec_object_num]->y,
            recycled_xsw_object[rec_object_num]->z,
            recycled_xsw_object[rec_object_num]->heading,
            recycled_xsw_object[rec_object_num]->pitch,
            recycled_xsw_object[rec_object_num]->bank
        );
        if(DBIsObjectGarbage(object_num))
        {
            /* Could not create object for recovery. */
            return(-3);
        }


        /* Copy the recycled object data to the recovered object. */
        memcpy(
	    xsw_object[object_num],			/* Destination. */
            recycled_xsw_object[rec_object_num],	/* Source. */
            sizeof(xsw_object_struct)
        );

        /*   NOTE: We copy the pointers to allocated substructures
         *   of the object in the recycled buffers list to the newly created
         *   recovered object.
         */


        /*   Reset the object in the recycle list so its substructure
         *   pointers are not free'ed.
         */
	memset(
	    recycled_xsw_object[rec_object_num],
	    0x00,
	    sizeof(xsw_object_struct)
	);
	recycled_xsw_object[rec_object_num]->type = XSW_OBJ_TYPE_GARBAGE;


	/* Player objects need to own themselves. */
	if(xsw_object[object_num]->type == XSW_OBJ_TYPE_PLAYER)
	    xsw_object[object_num]->owner = object_num;


        return(object_num);
}
