/*
	Avr.C
	
	Kit for AVR
	In-System Programmable
	Microcontrolers
	
	Uros Platise, (c) 1997
*/

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include "Avr.h"

TAvr::TAvr (TDev* device, TPartDB* _Part) : Part (_Part) {
  device->bitOrentation (true);
  enableAvr ();
  identify ();
}

void TAvr::enableAvr () {
  unsigned char prg  [4] = { 0xAC, 0x53, 0, 0 };
  int try_number = 32;
  do {
    prg[0]=0xAC; prg[1]=0x53; prg[2]=prg[3]=0;
    Send (prg, 4);
    if (prg[2] == 0x53)
      break;
    clk ();
  } while (try_number--);
  if (try_number<32){printf ("Succeeded after %d retries.\n", 32-try_number);}
}

TAvr::SPart TAvr::Parts [] = {
  { "at90s1200", 0x90 },
  { "at90s2313", 0x91 },
  { "at90s4414", 0x92 },
  { "at90s8515", 0x93 },
  { "",              TARGET_MISSING },
  { "device locked", DEVICE_LOCKED },
  { "", 0x0 }
};

void TAvr::identify () {
  VendorCode = getPart (0); 
  PartFamily = getPart (1);
  PartNumber = getPart (2);
  if (PartFamily==DEVICE_LOCKED &&
      PartNumber==0x02) { 
    deviceLocked=true;
    printf ("Cannot identify device because is it locked.\n");
    return;
  } else { deviceLocked=false; }
  if (PartFamily==TARGET_MISSING) {
    printf ("An error has occuried durring initilization.\n"
	    " * Target status:\n"
	    "   Vendor Code = %x, Part Family = %x, Part Number = %x\n\n",
	    VendorCode, PartFamily, PartNumber);
    throw Error_Device ("Probably the target is missing.");
  }
  int i;
  for(i=0; Parts [i].PartFamily != 0x0; i++) {
    if (PartFamily == Parts[i].PartFamily) {
      printf ("Device %s found.\n", Parts [i].name);
      Part->setPart (Parts [i].name);

      /* find flash end eeprom segments */
      for (int j=0; Part->segTableP [j].segName [0] != 0; j++) {
	if (strcmp (Part->segTableP [j].segName, "flash")==0) {
	  segFlash = &Part->segTableP [j]; 
	  printf ("FLASH: %ld bytes\n", segFlash->size);
	}
	if (strcmp (Part->segTableP [j].segName, "eeprom")==0) {
	  segEeprom = &Part->segTableP [j]; 
	  printf ("EEPROM: %ld bytes\n", segEeprom->size);
	}
      }
      return;
    } /* end if */
  }
  if (Parts [i].PartFamily == 0x0) {
    throw Error_Device ("Probably the AVR MCU is not in the RESET state.\n"
			"Check it out and run me again."); }
}

int TAvr::getPart (unsigned char addr) {
  unsigned char info [4] = { 0x30, 0, addr, 0 };
  Send(info, 4);
  return int(info [3]);
}

int TAvr::readEEPROM (unsigned int addr) {
  checkMemoryRange (addr, segEeprom->size);
  unsigned char eeprom [4] = { 0xA0, 
			       (unsigned char)((addr>>8)&0xff), 
			       (unsigned char)(addr&0xff), 
			       0 };
  Send(eeprom, 4);
  return int(eeprom [3]);
}

void TAvr::writeEEPROM (unsigned int addr, unsigned char byte) {
  checkMemoryRange (addr, segEeprom->size);
  unsigned char eeprom [4] = { 0xC0, 
			       (unsigned char)((addr>>8)&0xff), 
			       (unsigned char)(addr&0xff),
			       byte };
  Send(eeprom, 4);
  waitAfterWrite ();
}

int TAvr::readFLASH (unsigned int addr) {
  checkMemoryRange (addr, segFlash->size);
  unsigned char hl = (addr&1)?(lowByte):(highByte);
  addr>>=1;
  unsigned char flash [4] = { 0x20+hl, 
			      (unsigned char)((addr >> 8) & 0xff),
			      (unsigned char)(addr & 0xff),
			      0 };
  Send(flash, 4);
  return int(flash [3]);
}

void TAvr::writeFLASH (unsigned int addr, unsigned char byte) {
  checkMemoryRange (addr, segFlash->size);
  unsigned char hl = (addr&1)?(lowByte):(highByte);
  addr>>=1;
  unsigned char flash [4] = { 0x40+hl,
			      (unsigned char)((addr >> 8) & 0xff),
			      (unsigned char)(addr & 0xff),
			      byte };
  Send(flash, 4);
  waitAfterWrite ();
}

void TAvr::checkMemoryRange (unsigned int addr, unsigned int top_addr) {
  if (addr >= top_addr) { throw Error_MemoryRange (); }
}

void TAvr::chipErase () {
  unsigned char chip_erase [4] = { 0xAC, 0x80, 0x00, 0x00 };
  Send (chip_erase, 4);
  waitAfterWrite ();
  waitAfterWrite ();
  waitAfterWrite ();
  printf ("Togle RESET line to end chip erase cycle.\n");
}

void TAvr::waitAfterWrite () {
  timeval tv;
  tv.tv_sec = 0;
  tv.tv_usec = 4000;
  /*  select (0,NULL,NULL,NULL, &tv); */
  int i;
  for (i=0; i<200000; i++);
}

void TAvr::upload (TAout* aout, bool verifyOnly=false) {
  unsigned char read_buf [9*1024];
  char segName_buf [32];
  TDataQuery rdQ;
  rdQ.segName = segName_buf;
  rdQ.buf = read_buf;
  while (aout->readData (&rdQ)>0) {
    TSegmentName curSeg = parseSegment (&rdQ);
    printf ("Uploading: %xh bytes to %s at %xh",
	    rdQ.size, rdQ.segName, rdQ.offset);
    if (rdQ.keepOut==true) { printf (" - skipping\n"); continue; }
    putchar ('\n');
    if (curSeg==SEG_FLASH) {
      if (rdQ.size&1) {
	throw Error_Device ("Flash segment not correctly aligned."); }
    }
    for (unsigned int ib=0; ib<rdQ.size; ib++, rdQ.offset++) {
      switch (curSeg) {
      case SEG_FLASH:
	writeFLASH (rdQ.offset, rdQ.buf [ib]);
	  {
	    unsigned char byte = rdQ.buf [ib];
	    int rd = readFLASH (rdQ.offset);
	    if (byte != rd)
	      {
		fprintf (stderr,
			 "Error flash writing: Address 0x%04x (wr:%02x != rd:%02x)\n",
			 rdQ.offset,byte,rd);
		exit(1);
	      }
	  }
	break;
      case SEG_EEPROM: writeEEPROM (rdQ.offset, rdQ.buf [ib]); break;
      }
      if ((ib%8)==0) { printf ("\r%xh", ib); fflush (stdout); }
    }
    printf ("\r          \r"); fflush (stdout); /* delete status line */
  }
}

void TAvr::download (TAout* aout) {
  //TSegmentName curSeg = parseSegment (dataP);
}

TAvr::TSegmentName TAvr::parseSegment (TDataQuery* dataP) {
  if (strcmp (dataP->segName, "flash")==0) { return SEG_FLASH;
  } else if (strcmp (dataP->segName, "eeprom")==0) { return SEG_EEPROM;
  } else { throw Error_Device ("Invalid Segment", dataP->segName); }
}
