// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Radharamanan Radhakrishnan  ramanan@ece.uc.edu
//          Ardhendu Singh asingh@ececs.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: file_manager.cc,v 1.4 1999/10/05 03:01:51 dmartin Exp $
// 
//---------------------------------------------------------------------------


#include "file_manager.hh"
#include <sys/stat.h>
#include <unistd.h>
#include <iostream.h>
#include <fstream.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>

#define COMPARE_BUFFER_SIZE 2048

file_manager::file_status
file_manager::check_file_status( char *filename,  file_type MODE ){
  struct stat file_stat;
  if( stat( filename, &file_stat ) < 0 ){
    return NOT_FOUND;
  }
  
  if( MODE != DONTCARE ){
    switch( MODE ){
    case DIRECTORY:{
      if( !S_ISDIR( file_stat.st_mode ) ){
	return NOT_TYPE;
      }
      break;
    }
    
    case REGULAR_FILE:{
      if( !S_ISREG( file_stat.st_mode ) ){
	return NOT_TYPE;
      }
      break;
    }

    default:{
      cerr << "Don't know how to check for file type " << (int)MODE << endl;
      abort();
    }
    } 
  }
  
  return OK;
}


file_manager::file_status 
file_manager::make_directory( char *directory_name, mode_t mode ){
  if( mkdir( directory_name, mode ) == 0 ){
    return OK;
  }
  else{
    if( errno == EEXIST ){
      // Then the directory already exists - that's OK.
      return OK;
    }
    else{
      // Something else happened.
      return ERROR;
    }
  }
}


file_manager::file_status 
file_manager::change_directory( char *to_directory ){
  if( chdir( to_directory ) == 0 ){
    return OK;
  }
  else{
    return ERROR;
  }
}

file_manager::file_status
file_manager::unlink(const char* fileName) {
  if (::unlink(fileName) == 0) {
    return OK;
  }
  else {
    return ERROR;
  }
}

file_manager::file_status
file_manager::rename(const char* from, const char* to) {
  if (::rename(from, to) == 0) {
    return OK;
  }
  else {
    return ERROR;
  }
}

int
file_manager::file_compare(const char* fileName1, const char* fileName2) {
  ifstream file1;
  ifstream file2;
  int      difference = 0;
  int      fileSize1, fileSize2;
  int      bytesToRead;
  char*    compare_buffer1;
  char*    compare_buffer2;
  
  file1.open( fileName1, ios::in );
  file2.open( fileName2, ios::in );

  if (file2.good() == 0) {
    // Okay. The original file does not exists. Can't compare
    file1.close();
    return 256;
  }
  
  file1.seekg(0, ios::end);
  fileSize1 = file1.tellg();
  file1.seekg(0, ios::beg);
  
  file2.seekg(0, ios::end);
  fileSize2 = file2.tellg();
  file2.seekg(0, ios::beg);
  
  if (fileSize1 == fileSize2) {
    // Compare the files only if they are of the same size.
    
    compare_buffer1 = new char[COMPARE_BUFFER_SIZE];
    compare_buffer2 = new char[COMPARE_BUFFER_SIZE];
    
    while (fileSize1 > 0)  {
      bytesToRead = ((fileSize1 > COMPARE_BUFFER_SIZE) ? COMPARE_BUFFER_SIZE
		     : fileSize1);
      file1.read(compare_buffer1, bytesToRead);
      file2.read(compare_buffer2, bytesToRead);
      
      fileSize1 -= bytesToRead;
      
      if ((difference = memcmp(compare_buffer1, compare_buffer2, bytesToRead)) != 0) {
	break;
      }
    }
    
    delete [] compare_buffer1;
    delete [] compare_buffer2;
  }
  else {
    difference = 256;
  }
  
  file1.close();
  file2.close();
  
  return difference;
}

const char *
file_manager::get_directory_separator(){
  return "/";
}

const char **
file_manager::get_library_directories(){
  return lib_dirs;
}


const char *
file_manager::lib_dirs[] = {
  ".",
  "/usr/lib",
  "/usr/local/lib",
  0
};
