#ifndef DL_LIST_HH
#define DL_LIST_HH

// Copyright (c) 1994-1999 The University of Cincinnati.
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey            phil.wilsey@uc.edu
//          Tim McBrayer                tmcbraye@ece.uc.edu
//          Dale E. Martin              dmartin@ece.uc.edu

#include "list.hh"
#include "iterator.hh"

// This is to get the definition of "bool"
#include "savant.hh"

#include "wrapper_container.hh"

#ifndef NULL
#define NULL 0
#endif

#ifndef ASSERT
#ifdef DEVELOPER_ASSERTIONS
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#define ASSERT( x ) assert( x )
#else
#define ASSERT( x )
#endif
#endif

template <class type>
class dl_list {
  
public:

  dl_list() : my_iterator( &my_list ){}
  ~dl_list(){
    destroy_containers();
  }
    
  int num_elements(){
    return my_list.get_num_elements();
  }

  void append(type *to_append ){
    my_list.append( new wrapper_container(to_append) );
  }

  void prepend( type* to_prepend ) {
    my_list.prepend( new wrapper_container(to_prepend) );
  }

  void insert_after(type *after_me, type* new_object) {
    list_object *found = find( after_me );
    ASSERT( found != NULL );
    my_iterator.append_to_current( new wrapper_container( new_object ) );
  }
    
  bool remove( const type* to_remove ) {
    wrapper_container *found = find( to_remove );
    if( found == NULL ){
      return false;
    }
    else{
      ASSERT( my_iterator.get_current() == found );
      delete my_iterator.remove_current();
      return true;
    }
  }

  type* successor( const type* to_succeed ) {
    wrapper_container *current_container;
    current_container = find( to_succeed );
    current_container = (wrapper_container *)my_iterator.forward();

    if( current_container == NULL ){
      return NULL;
    }
    else{
      return (type *)current_container->get_data();
    }
  }
    
  type* predecessor( const type* to_precede ) {
    wrapper_container *current_container;
    current_container = find( to_precede );
    current_container = (wrapper_container *)my_iterator.backward();

    if( current_container == NULL ){
      return NULL;
    }
    else{
      return (type *)current_container->get_data();
    }
  }
    
  type* first() {
    wrapper_container *temp_head = (wrapper_container *)my_list.get_head();
    if( temp_head == NULL ){
      return NULL;
    }
    else{
      return (type *)temp_head->get_data();
    }
  }
    
  type* last() {
    wrapper_container *temp_tail = (wrapper_container *)my_list.get_tail();
    if( temp_tail == NULL ){
      return NULL;
    }
    else{
      return (type *)temp_tail->get_data();
    }
  }
    
  // This returns the ordinal position in the dl_list of the object passed
  // in.  It assumes you always append, and never prepend or insert_after.
  // Very brittle code.  Needed to give unique signal ids to each global
  // signal.  Returns -1 if signal not found in list.

  int get_position(type* data) {
    int i = 0;
    type *current = first();
    while( current != data && current != NULL ){
      i++;
      current = successor( current );
    }
    if( current == NULL ){
      return -1;
    }
    else{
      return i;
    }
  }

  type *get_nth_element( int pos_to_get ){
    int i = 0;
    type *current = first();
    for( i = 0; i < pos_to_get && current != NULL; i++ ){
      current = successor( current );
    }

    if( current == NULL ){
      return NULL;
    }
    else{
      return current;
    }
  }

  dl_list<type> &
  operator=( dl_list<type> &to_copy ){
    
    destroy_containers();

    type *current_thing = to_copy.first();
    while( current_thing != NULL ){
      append( current_thing );
      current_thing = to_copy.successor( current_thing );
    }
    
    return *this;
  }

  // Replace "to_replace" with replace with, without disturbing the rest
  // of the list.
  void 
  _replace( type *to_replace, type *replace_with ){
    wrapper_container *found = find( to_replace );
    ASSERT( found != NULL );
    found->set_data( replace_with );
  }

  void destroy_containers(){
    list_object *current = my_iterator.go_to_head();
    while( current != NULL ){
      delete my_iterator.remove_current();
      current = my_iterator.get_current();
    }
  } 
  
  bool in_list( const type *to_find ){
    if( find( to_find ) == NULL ){
      return false;
    }
    else{
      return true;
    }
  }

private:
  list my_list;
  iterator my_iterator;

  wrapper_container *find( const type *to_find ){
    if( my_list.get_num_elements() != 0 ){
      wrapper_container *found = 
	(wrapper_container *)my_iterator.get_current();
      
      if( found == NULL ){
	found = (wrapper_container *)my_iterator.go_to_head();
      }
      ASSERT( found != NULL );
      
      if( found->get_data() == to_find ){
	return found;
      }
      
      type *object = NULL;
      wrapper_container *start = found;
      do {
	found = (wrapper_container *)my_iterator.move_forward();     
	object = (type *)found->get_data();
      } while( found != start && object != to_find );
      
      if( found == start ){
	return NULL;
      }
      else{
	return (wrapper_container *)found;
      }
    }
    else{
      return NULL;
    }
  }
};

#endif
