// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajaske@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_UserAttribute.cc,v 1.5 1999/10/26 16:44:47 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include "IIRScram_UserAttribute.hh"
#include "IIR_Aggregate.hh"
#include "IIR_AssociationElementByExpression.hh"
#include "IIR_AttributeSpecification.hh"
#include "IIR_ConstantDeclaration.hh"
#include "IIR_QualifiedExpression.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "set.hh"
#include <strstream.h>

IIRScram_UserAttribute::~IIRScram_UserAttribute(){}

void 
IIRScram_UserAttribute::_publish_vhdl(ostream &_vhdl_out) {
  ASSERT(get_prefix() != NULL);
  ASSERT(get_prefix()->_is_resolved() == TRUE);
  ASSERT(get_suffix() != NULL);
  ASSERT(get_suffix()->_is_resolved() == TRUE);

  get_prefix()->_publish_vhdl(_vhdl_out);
  _vhdl_out << "'";
  get_suffix()->_publish_vhdl(_vhdl_out);
}


void 
IIRScram_UserAttribute::_publish_cc() {
  ASSERT(get_prefix() != NULL);
  ASSERT(get_prefix()->_is_resolved() == TRUE);
  ASSERT(get_suffix() != NULL);
  ASSERT(get_suffix()->_is_resolved() == TRUE);

  get_prefix()->_add_decl_into_cgen_symbol_table();
  get_suffix()->_add_decl_into_cgen_symbol_table();

  IIR_AttributeSpecification* attr_spec = get_prefix()->_get_attribute_specification(get_suffix());

  _publish_cc_prefix_string();
  attr_spec->_publish_cc();

  if(get_suffix()->_is_iir_declaration() == FALSE) {
    if ((get_suffix()->get_kind() != IIR_INDEXED_NAME) &&
	(get_suffix()->get_kind() != IIR_SLICE_NAME)) {
      _cc_out << ".";
    }
    
    get_suffix()->_publish_cc();
  }
}


void 
IIRScram_UserAttribute::_publish_cc_bounds() {
  ASSERT(get_prefix() != NULL);
  ASSERT(get_prefix()->_is_resolved() == TRUE);
  ASSERT(get_suffix() != NULL);
  ASSERT(get_suffix()->_is_resolved() == TRUE);
  _cc_out << "nullInfo";
}


// I'm not quite sure what to do with these yet...
IIR_TypeDefinition *
IIRScram_UserAttribute::_get_subtype(){
  IIR_TypeDefinition *retval = NULL;

  set<IIR_TypeDefinition> *suffix_rvals;
  ASSERT( get_suffix() != NULL );

  if( _is_qualified_expression() == FALSE ){

    suffix_rvals = get_suffix()->_get_rval_set();
    if( suffix_rvals == NULL ){
      report_undefined_symbol( get_suffix() );
      return NULL;
    }

    switch( suffix_rvals->num_elements() ){
    case 0:{
      ostrstream err;
      err << "|" << *get_suffix() << "| was not declared as an attribute in this scope." << ends;
      report_error( this, err );
      break;
    }
    case 1 :{
      retval = suffix_rvals->get_element();
      break;
    }

    default:{
      report_ambiguous_error( get_suffix(), suffix_rvals );
    }
    }
  }
  else{
    if( get_suffix()->get_kind() == IIR_AGGREGATE ){
      IIR_Aggregate *suffix_as_aggregate = (IIR_Aggregate *)get_suffix();
      if( suffix_as_aggregate->element_association_list.num_elements() == 1 ){
	IIR_AssociationElement *first_element;
	first_element = suffix_as_aggregate->element_association_list.first();
	if( first_element->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION ){
	  IIR_AssociationElementByExpression *by_expression;
	  by_expression = (IIR_AssociationElementByExpression *)first_element;
	  if( by_expression->get_formal() == NULL ){
	    set_suffix( by_expression->get_actual() );
	  }
	}
      }
    }
    
    suffix_rvals = get_suffix()->_get_rval_set();
    if( suffix_rvals == NULL ){
      report_undefined_symbol( get_suffix() );
      return NULL;
    }
    set<IIR_TypeDefinition> *prefix_rvals = get_prefix()->_get_rval_set();
    if( prefix_rvals != NULL ){
      reconcile_sets( prefix_rvals, suffix_rvals );
      switch( prefix_rvals->num_elements() ){
      case 0:{
	ostrstream err;
	
	err << "|" << *get_prefix() << "| and |" << *get_suffix() << "| aren't compatible, "
	    << "and therefore make an invalid qualified expression." << ends;
	
	report_error( this, err );
	break;
      }
      case 1:{
	retval = prefix_rvals->get_element();
	break;
      }
      default:{
	report_ambiguous_error( this, prefix_rvals );
	break;
      }
      }
    }
    else{
      ostrstream err;
      err << "Internal error in IIRScram_UserAttribute::_get_subtype() - no prefix rvals found." 
	  << ends;
      report_error( this, err );
      return NULL;
    }
  }

  return retval;
}


IIR_Declaration *
IIRScram_UserAttribute::_get_implicit_declaration(char *decl_name, IIR_TypeDefinition *decl_type){
  return _build_constant_declaration( decl_name, decl_type );
}


IIR_TextLiteral *
IIRScram_UserAttribute::_build_attribute_name(){
  ASSERT( get_suffix() != NULL );

  if( get_suffix()->_is_name() == TRUE ){
    return ((IIR_Name *)get_suffix())->_get_prefix_string();
  }
  else if ( get_suffix()->_is_iir_declaration() == TRUE ){
    return get_suffix()->_get_declarator();
  }
  else{
    ostrstream err;
    err << "Internal error in IIRScram_UserAttribute::_get_implicit_declaration ";
    err << "- got a " << get_suffix()->get_kind_text() << " suffix.\n";
    
    report_error( this, err );

    abort();

    return NULL;
  }
}


IIR *
IIRScram_UserAttribute::_semantic_transform( set<IIR_TypeDefinition> * ){
  IIR *retval = this;
  
  if( _is_qualified_expression() == TRUE ){
    // We have to type check the qualified expression right here to build it
    // correctly...
    set<IIR_TypeDefinition> *suffix_rvals = get_suffix()->_get_rval_set();;
    if( suffix_rvals == NULL ){
      report_undefined_symbol( get_suffix() );
    }

    set<IIR_TypeDefinition> *prefix_rvals = get_prefix()->_get_rval_set();
    if( prefix_rvals == NULL ){
      report_undefined_symbol( get_suffix() );
    }
    

    reconcile_sets( prefix_rvals, suffix_rvals );
    switch( prefix_rvals->num_elements() ){
    case 0:{
      ostrstream err;
      err << "|" << *get_prefix() << "| and |" << *get_suffix() << "| aren't compatible, "
	  << "and therefore make an invalid qualified expression." << ends;

      report_error( this, err );
      break;
    }
    case 1:{
      IIR_TypeDefinition *my_rval = prefix_rvals->get_element();
      IIR_QualifiedExpression *new_expr = new IIR_QualifiedExpression();
      copy_location( this, new_expr );
      new_expr->set_type_mark( my_rval );
      new_expr->set_expression( get_suffix() );
      retval = new_expr;
      delete this;
      break;
    }
    default:{
      report_ambiguous_error( get_prefix(), prefix_rvals );
      break;
    }
    }
    delete suffix_rvals;
  }
  else{
    _survived_transformation = TRUE;
  }
  
  return retval;
}


IIR_Boolean 
IIRScram_UserAttribute::_is_qualified_expression(){
  IIR_Boolean retval = FALSE;
  // An attribute name can have a suffix that is a simple name, or
  // a suffix that is an indexed name, with a simple name prefix.
  // In either case, the simple name MUST refer to an attribute.
  if( _survived_transformation == TRUE ){
    // Then retval remains false;
    return FALSE;
  }
  if(get_prefix()->get_kind() == IIR_SIMPLE_NAME || get_prefix()->get_kind() == IIR_SELECTED_NAME){
    set<IIR_Declaration> *prefix_decls = get_prefix()->_symbol_lookup( &IIR::_is_type );
    if( prefix_decls != NULL ){
      if( prefix_decls->num_elements() == 1 ){
	// Then it's unabiguously a type declaration.
	retval = TRUE;
      }
      delete prefix_decls;
    }
  }

  return retval;
}

void
IIRScram_UserAttribute::_resolve_suffix_special(){
    
  if( get_suffix() != NULL ){
    set<IIR_TypeDefinition> *suffix_rvals = 
      get_suffix()->_get_rval_set(&IIR::_is_iir_attribute_declaration);
    
    if( suffix_rvals == NULL ){
      report_undefined_symbol( get_suffix() );
    }
    
    //    reconcile_sets( suffix_rvals, context_set );
    switch( suffix_rvals->num_elements() ){
    case 0:{
      ostrstream err;
      err << "|" << *get_suffix() << "| was not declared as an attribute in this scope." 
	  << ends;
      report_error( this, err );
      break;
    }
    case 1:{
      IIR_TypeDefinition *my_rval = suffix_rvals->get_element();
      
      set_suffix( get_suffix()->_semantic_transform( my_rval ) );
      get_suffix()->_type_check( my_rval );
      set_suffix( get_suffix()->_rval_to_decl( my_rval ) );
      
      break;
    }
    default:{
      report_ambiguous_error( get_suffix(), suffix_rvals );
    }
    }
    delete suffix_rvals;
  }
}

IIR_Boolean 
IIRScram_UserAttribute::_is_locally_static_primary(){
  IIR_Declaration *implicit_decl = IIRScram_Attribute::_get_implicit_declaration();

  ASSERT( implicit_decl != NULL );
  ASSERT( implicit_decl->get_kind() == IIR_CONSTANT_DECLARATION );
  IIR_ConstantDeclaration *as_constant = (IIR_ConstantDeclaration *)implicit_decl;
  ASSERT( as_constant->get_value() != NULL );

  return as_constant->get_value()->_is_locally_static();
}

IIR_Boolean 
IIRScram_UserAttribute::_is_globally_static_primary(){
  IIR_Declaration *implicit_decl = IIRScram_Attribute::_get_implicit_declaration();

  ASSERT( implicit_decl != NULL );
  ASSERT( implicit_decl->get_kind() == IIR_CONSTANT_DECLARATION );
  IIR_ConstantDeclaration *as_constant = (IIR_ConstantDeclaration *)implicit_decl;
  ASSERT( as_constant->get_value() != NULL );

  return as_constant->get_value()->_is_globally_static();
}
